/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.dialog;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;

import jp.sourceforge.tomoyo.core.Utilities;
import jp.sourceforge.tomoyo.core.local.model.PolicyCacheManager;
import jp.sourceforge.tomoyo.core.local.model.domain.Domain;
import jp.sourceforge.tomoyo.core.local.resource.DomainPolicy;
import jp.sourceforge.tomoyo.core.local.resource.ProcResourceManager;
import jp.sourceforge.tomoyo.ui.editor.Activator;
import jp.sourceforge.tomoyo.ui.editor.ViewerExpandCollapseKeyListener;
import jp.sourceforge.tomoyo.ui.editor.CoreEditorOutlineMouseListener;
import jp.sourceforge.tomoyo.ui.editor.form.domain.DomainPolicyOutlineMouseListener;
import jp.sourceforge.tomoyo.ui.editor.text.outline.DomainPolicyOutlineContentProvider;
import jp.sourceforge.tomoyo.ui.editor.text.outline.DomainPolicyOutlineDecoratingLabelProvider;
import jp.sourceforge.tomoyo.ui.editor.text.outline.DomainPolicyOutlineLabelProvider;


import org.eclipse.core.resources.IProject;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.IBaseLabelProvider;
import org.eclipse.jface.viewers.ILabelDecorator;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableColorProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.PlatformUI;

public class SelectManagerDomainDialog extends TitleAreaDialog {

	private IProject project;
	
	public SelectManagerDomainDialog(Shell parentShell, IProject project) {
		super(parentShell);
		this.project = project;
	}
	
	private IProject getProject() {
		return project;
	}

	protected void configureShell(Shell shell) {
    	super.configureShell(shell);
    	shell.setText(Messages.SelectManagerDomainDialog_ShellTitle);
    }
    
    protected Point getInitialSize() {
    	int w_percent = 85;
    	int h_percent = 70;
    	Rectangle rect = Activator.getStandardDisplay().getClientArea();
    	
    	int width = rect.width * w_percent / 100;
    	int height = rect.height * h_percent / 100;
    	
    	return new Point(width, height);
    }
	
	protected int getShellStyle() {
//		return SWT.MAX | SWT.RESIZE;
		return SWT.SHELL_TRIM;
	}

	private static final String DEFAULT_MESSAGE = Messages.SelectManagerDomainDialog_DialogTitleDescription;
	
	protected Control createDialogArea(Composite parent) {
		setTitle(Messages.SelectManagerDomainDialog_DialogTitle);
		setTitleImage(Activator.getImage("newelm_wiz.gif")); //$NON-NLS-1$
		setMessage(DEFAULT_MESSAGE, IMessageProvider.NONE);

		Composite dialogArea = (Composite)super.createDialogArea(parent);
//		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true);
//		dialogArea.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));

		Composite container = new Composite(dialogArea, SWT.NULL);
		container.setBackground(parent.getBackground());
		GridLayout layout = new GridLayout(3, false);
		layout.marginWidth = 0;
//		layout.marginHeight = 0;
		container.setLayout(layout);
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		container.setLayoutData(data);

		createHeaderComposite(container);
		
		treeViewer = createTreeViewer(container);
		createDomainButtonContoler(container);
		tableViewer = createTableViewer(container);
		
		/*
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent,
			"jp.sourceforge.tomoyo.doc.select_manager_domain");		 //$NON-NLS-1$
		*/

		restoreDefault();
	    
		return dialogArea;
	}
	
	protected Control createContents(Composite parent) {
		try {
			return super.createContents(parent);
		} finally {
			recommendedManagerText = Utilities.getNeedToBeManagerDomain(getProject());
			super.getButton(IDialogConstants.OK_ID).setEnabled(false);
		}
	}

	private void restoreDefault() {
		DomainPolicy domainPolicy = ProcResourceManager.getInstance().getProcResourceSet(getProject()).getDomainPolicy();
		treeViewer.setInput(domainPolicy.getModel());
		treeViewer.expandAll();
		
		refreshTableViewer();
	}
	
	private String recommendedManagerText;

	private void refreshTableViewer() {
		tableViewer.setInput(Utilities.getManagers(getProject()));
	}

	private void createHeaderComposite(Composite container) {
		Label label1 = new Label(container, SWT.NULL);
		label1.setText(Messages.SelectManagerDomainDialog_DomainListTreeTitle);
		new Label(container, SWT.NULL);
		Label label3 = new Label(container, SWT.NULL);
		label3.setText(Messages.SelectManagerDomainDialog_ManagerDomainListTitle);
	}

	//------------------------------------------------------------------------------------------------
	// Domain Tree Composite
	//------------------------------------------------------------------------------------------------
	
	private TreeViewer treeViewer;
	
	private TreeViewer createTreeViewer(Composite parent) {

    	TreeViewer viewer = new TreeViewer(parent, SWT.SINGLE | SWT.BORDER);
		viewer.getControl().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
    	viewer.setLabelProvider(createDomainViewerLabelProvider());
    	
    	viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				handleDomainSelectionChanged();
			}
    	});
//    	viewer.setComparer(new ViewerComparer());
    	viewer.getControl().addKeyListener(new ViewerExpandCollapseKeyListener(viewer));
//    	viewer.getControl().addKeyListener(new DomainPolicyOutlineKeyListener(viewer, dialogObserver));
    	viewer.getControl().addMouseListener(new CoreEditorOutlineMouseListener(viewer));
    	viewer.getControl().addMouseListener(new DomainPolicyOutlineMouseListener(getProject(), viewer));
//    	viewer.getControl().addFocusListener(new TreeFocusAdapter(viewer));
    	
    	viewer.setContentProvider(new DomainPolicyOutlineContentProvider());
    	
		DomainPolicyOutlineDecoratingLabelProvider decoratingLabelProvider = (DomainPolicyOutlineDecoratingLabelProvider)viewer.getLabelProvider();
		DomainPolicyOutlineLabelProvider labelProvider = (DomainPolicyOutlineLabelProvider)decoratingLabelProvider.getLabelProvider();
    	labelProvider.setDomainVerbose(false);

    	return viewer;
	}
	
	private IBaseLabelProvider createDomainViewerLabelProvider() {
		HintOutlineLabelProvider labelProvider = new HintOutlineLabelProvider(getProject());
		ILabelDecorator decorator = PlatformUI.getWorkbench().getDecoratorManager().getLabelDecorator();
		return new DomainPolicyOutlineDecoratingLabelProvider(labelProvider, decorator);
	}
	
	private class HintOutlineLabelProvider extends DomainPolicyOutlineLabelProvider {
		public HintOutlineLabelProvider(IProject project) {
			super(project);
		}
		public Image getImage(Object object) {
			if (object instanceof Domain) {
				Domain domain = (Domain)object;
				if (recommendedManagerText != null) {
					if (domain.getText().equals(recommendedManagerText))
						return Activator.getImage("hint1.gif"); //$NON-NLS-1$
				}
			}
			return super.getImage(object);
		}
	}

	private void handleDomainSelectionChanged() {
		Domain domain = getSelectedDomain();
		if (domain == null) {
			addAsDomainButton.setEnabled(false);
			return;
		}
		if (isManagerInstalled(domain)) {
			setMessage(Messages.SelectManagerDomainDialog_AlreadyExistsErrorMessage, IMessageProvider.WARNING);
			addAsDomainButton.setEnabled(false);
		} else {
			setMessage(DEFAULT_MESSAGE);
			addAsDomainButton.setEnabled(true);
		}
	}
	
	private boolean isManagerInstalled(Domain domain) {
		TableViewerContentProvider contentProvider = (TableViewerContentProvider)tableViewer.getContentProvider();
		ArrayList<String> list = contentProvider.getManagerDomainList();
		Iterator<String> it = list.iterator();
		while (it.hasNext()) {
			String managerText = (String)it.next();
			if (managerText.equals(domain.getText()))
				return true;
		}
		return false;
	}

	private Domain getSelectedDomain() {
		if (treeViewer.getSelection() instanceof StructuredSelection) {
			StructuredSelection selection = (StructuredSelection)treeViewer.getSelection();
			if (selection.getFirstElement() instanceof Domain) {
				Domain domain = (Domain)selection.getFirstElement();
				return domain;
			}
		}
		return null;
	}

	private Button expandButton;
	private Button collapseButton;
	
	private Button hint1Button;
	private Button addAsDomainButton;
	
	private void createDomainButtonContoler(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		GridLayout layout = new GridLayout(1, false);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.marginLeft = 0;
		layout.marginRight = 0;
		container.setLayout(layout);
		GridData data = new GridData(SWT.FILL, SWT.FILL, false, true);
		container.setLayoutData(data);
		
		Composite topContainer = new Composite(container, SWT.NULL);
		GridLayout topLayout = new GridLayout(1, false);
		topLayout.marginHeight = 0;
		topLayout.marginWidth = 0;
		topLayout.marginLeft = 0;
		topLayout.marginRight = 0;
		topContainer.setLayout(topLayout);
		GridData topData = new GridData(GridData.VERTICAL_ALIGN_BEGINNING);
		topContainer.setLayoutData(topData);

		expandButton = this.createButton(topContainer, Activator.getImage("expandall.gif"), SWT.NULL); //$NON-NLS-1$
		collapseButton = this.createButton(topContainer, Activator.getImage("collapseall.gif"), SWT.NULL); //$NON-NLS-1$
		
		Composite centerContainer = new Composite(container, SWT.NULL);
		GridLayout centerLayout = new GridLayout(1, false);
		centerLayout.marginHeight = 0;
		centerLayout.marginWidth = 0;
		centerLayout.marginLeft = 0;
		centerLayout.marginRight = 0;
		centerContainer.setLayout(centerLayout);
		GridData centerData = new GridData(SWT.FILL, SWT.CENTER, false, true);
		centerContainer.setLayoutData(centerData);

		hint1Button = this.createButton(centerContainer, Activator.getImage("hint1.gif"), SWT.NULL); //$NON-NLS-1$
		hint1Button.setToolTipText(Messages.SelectManagerDomainDialog_AutoSelectionTooltip);
		
		addAsDomainButton = this.createButton(centerContainer, Activator.getImage("add.gif"), SWT.NULL); //$NON-NLS-1$
		addAsDomainButton.setToolTipText(Messages.SelectManagerDomainDialog_AddDomainTooltip);
	}

	private Button createButton(Composite parent, Image image, int style) {
        Button button = new Button(parent, style);
        button.setImage(image);
//        button.setBackground(parent.getBackground());
        button.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
        		handleRadioButtonSelected((Button)e.getSource());
        	}
        });
		GridData data = new GridData();
//		data.horizontalIndent = 0;
//		data.verticalIndent = 0;
		button.setLayoutData(data);
		return button;
    }
	
	private void handleRadioButtonSelected(Button button) {
		if (button == expandButton) {
			treeViewer.expandAll();
		}
		if (button == collapseButton) {
			treeViewer.collapseAll();
		}
		if (button == hint1Button) {
			selectRecommendedDomain();
		}
		if (button == addAsDomainButton) {
			Domain domain = getSelectedDomain();
			
			MessageDialog dialog = new MessageDialog(
					getShell(),
					Messages.SelectManagerDomainDialog_RegisterConfirmDialogTitle,
					null,
					Messages.SelectManagerDomainDialog_RegisterConfirmDialogDescription,
					MessageDialog.QUESTION,
					new String[] {Messages.SelectManagerDomainDialog_RegisterAsDomainButton, Messages.SelectManagerDomainDialog_RegisterAsProgramButton, Messages.SelectManagerDomainDialog_CancelButton},
					2);
			int pressedButton = dialog.open();
			if (pressedButton == 0)
				handleAddButtonSelected(domain.getText());
			if (pressedButton == 1)
				handleAddButtonSelected(domain.getProcessName());
		}
	}

	private void selectRecommendedDomain() {
		if (recommendedManagerText == null)
			return;
		String catDomainText = recommendedManagerText;
		Domain catDomain = PolicyCacheManager.getInstance().findDomain(getProject(), catDomainText);
		treeViewer.setSelection(new StructuredSelection(catDomain));
	}

	//	SimpleCommand command = CommandManager.getInstance().syncExec(new CheckManagerDomain(getProject(), proposedDomainText.getText()));
	//	if (command.isSuccess() && command.isOK()) {
	
	private void handleAddButtonSelected(String managerText) {
		if (managerText == null)
			return;
		if (Utilities.installManagerDomain(getProject(), managerText)) {
			refreshTableViewer();
			setManagerSelection(managerText);
			MessageDialog.openInformation(Activator.getStandardDisplay().getActiveShell(),
					Messages.SelectManagerDomainDialog_InfoDialogTitle, Messages.SelectManagerDomainDialog_AddOKMessage + System.getProperty("line.separator") + managerText); //$NON-NLS-3$ //$NON-NLS-1$
		} else {
			MessageDialog.openError(Activator.getStandardDisplay().getActiveShell(),
					Messages.SelectManagerDomainDialog_InfoDialogTitle, Messages.SelectManagerDomainDialog_AddNGMessage + System.getProperty("line.separator") + managerText); //$NON-NLS-3$ //$NON-NLS-1$
		}
	}

	private void setManagerSelection(String text) {
		tableViewer.setSelection(new StructuredSelection(text));
	}

	//------------------------------------------------------------------------------------------------
	// Manager Domain Table Composite
	//------------------------------------------------------------------------------------------------

	private TableViewer tableViewer;
	
	private static final String[] COLUMN_PROPERTIES = new String[] {
		"manager", //$NON-NLS-1$
    };

	private TableViewer createTableViewer(Composite parent) {
	    TableViewer viewer = new TableViewer(parent, SWT.BORDER);
	    Table table = viewer.getTable();
	    table.setLinesVisible(true);
	    table.setHeaderVisible(true);
	    table.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
	    viewer.setContentProvider(new TableViewerContentProvider());
	    viewer.setLabelProvider(new TableViewerLabelProvider());
	    viewer.setSorter(new TableViewerSorter());
	    viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				handleManagerDomainSelectionChanged();
			}
	    });
	    
		TableColumn column = new TableColumn(table,SWT.CENTER);
		column.setText("Registerd domains/programs");
		column.setWidth(400);
		
		viewer.setColumnProperties(COLUMN_PROPERTIES); 
		
		CellEditor[] editors = new CellEditor[] {
			    new TextCellEditor(table),
		};
		viewer.setCellEditors(editors);

	    return viewer;
	}

	private String managerTextCache = null;
	
	public String getManagerDomainText() {
		return managerTextCache;
	}
	
	private void handleManagerDomainSelectionChanged() {
		StructuredSelection selection = (StructuredSelection)tableViewer.getSelection();
		Object object = selection.getFirstElement();
		if (object == null) {
			managerTextCache = null;
			super.getButton(IDialogConstants.OK_ID).setEnabled(false);
			return;
		}
		String managerText = object.toString();
		managerTextCache = managerText;
		super.getButton(IDialogConstants.OK_ID).setEnabled(true);
	}

	private boolean isUndesirableManager(String managerText) {
		return managerText.startsWith("/root/ccstools"); //$NON-NLS-1$
	}

	private class TableViewerContentProvider implements IStructuredContentProvider {
		public Object[] getElements(Object inputElement) {
			if (inputElement instanceof ArrayList) {
				ArrayList<?> list = (ArrayList<?>)inputElement;
				return list.toArray();
			}
			return new Object[0];
        }
		public void dispose() {
		}
		private ArrayList<String> lines = new ArrayList<String>();
		@SuppressWarnings("unchecked")
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			lines = (ArrayList)newInput;
		}
		public ArrayList<String> getManagerDomainList() {
			return lines;
		}
	}
    
	private class TableViewerLabelProvider extends LabelProvider implements ITableLabelProvider, ITableColorProvider {
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}
		public String getColumnText(Object element, int columnIndex) {
			String result = ""; //$NON-NLS-1$
			switch (columnIndex) {
			case 0:
				result = element.toString();
				break;
			default:
				result = ""; //$NON-NLS-1$
				break;
			}
			return result;
		}
		public Color getBackground(Object element, int columnIndex) {
			if (element == null)
				return null;
			switch (columnIndex) {
			case 0:
				String managerText = element.toString();
				if (isUndesirableManager(managerText))
					return Activator.getStandardDisplay().getSystemColor(SWT.COLOR_GRAY);
			}
			return null;
		}
		public Color getForeground(Object element, int columnIndex) {
			switch (columnIndex) {
			case 0:
			}
			return null;
		}
	}
	
	private class TableViewerSorter extends ViewerSorter implements Comparator<Object> {
		public TableViewerSorter() {
		}
		public int compare(Viewer viewer, Object e1, Object e2) {
			return compare(e1, e2);
		}
		public int compare(Object e1, Object e2) {
			String s1 = (String)e1;
			String s2 = (String)e2;
			return s1.compareTo(s2);
		}
	}

	protected void okPressed() {
		super.okPressed();
	}

}
