/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor;

import jp.sourceforge.tomoyo.core.Utilities;

import jp.sourceforge.tomoyo.core.local.model.DomainPolicyModel;
import jp.sourceforge.tomoyo.core.local.model.ExceptPolicyModel;
import jp.sourceforge.tomoyo.core.local.model.SystemPolicyModel;
import jp.sourceforge.tomoyo.core.local.model.domain.Profile;
import jp.sourceforge.tomoyo.core.local.resource.DomainPolicy;
import jp.sourceforge.tomoyo.core.local.resource.ExceptPolicy;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;
import jp.sourceforge.tomoyo.core.local.resource.ProcResourceManager;
import jp.sourceforge.tomoyo.core.local.resource.ProcResourceSet;
import jp.sourceforge.tomoyo.core.local.resource.SystemPolicy;
import jp.sourceforge.tomoyo.ui.editor.form.PlainFormPage;
import jp.sourceforge.tomoyo.ui.editor.form.ProjectPropertySheetPage;
import jp.sourceforge.tomoyo.ui.editor.form.domain.DomainPolicyPage;
import jp.sourceforge.tomoyo.ui.editor.form.except.ExceptionPolicyPage;
import jp.sourceforge.tomoyo.ui.editor.form.outline.ProjectOutlineElement;
import jp.sourceforge.tomoyo.ui.editor.form.overview.ProjectOverviewPage;
import jp.sourceforge.tomoyo.ui.editor.form.profile.ProjectProfilePage;
import jp.sourceforge.tomoyo.ui.editor.form.system.SystemPolicyPage;
import jp.sourceforge.tomoyo.ui.editor.perspective.SwitchPerspectiveThread;
import jp.sourceforge.tomoyo.ui.editor.text.domain.DomainPolicyEditor;
import jp.sourceforge.tomoyo.ui.editor.text.except.ExceptPolicyEditor;
import jp.sourceforge.tomoyo.ui.editor.text.system.SystemPolicyEditor;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.IPageChangedListener;
import org.eclipse.jface.dialogs.PageChangedEvent;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;

import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.editors.text.TextEditor;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;
import org.eclipse.ui.views.properties.IPropertySheetPage;
import org.eclipse.ui.views.properties.PropertySheetPage;

/**
 * An example showing how to create a multi-page editor.
 * This example has 3 pages:
 * <ul>
 * <li>page 0 contains a nested text editor.
 * <li>page 1 allows you to change the font used in page 2
 * <li>page 2 shows the words in page 0 in sorted order
 * </ul>
 */
public class CoreEditor extends FormEditor implements IResourceChangeListener {

	/**
	 * Creates a multi-page editor example.
	 */
	public CoreEditor() {
		super();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
	}
	
    public void init(IEditorSite site, IEditorInput input) throws PartInitException {
		if (!(input instanceof IFileEditorInput))
			throw new PartInitException("Invalid Input: Must be IFileEditorInput");
		
		setSite(site);
		setInput(input);

		addPageChangedListener(new IPageChangedListener() {
			public void pageChanged(PageChangedEvent event) {
				handlePageChanged(event);
			}
		});
		
		Activator.getStandardDisplay().asyncExec(new SwitchPerspectiveThread());
	}

	private void handlePageChanged(PageChangedEvent event) {
		if (event.getSelectedPage() instanceof PlainFormPage) {
			PlainFormPage editorPage = (PlainFormPage)event.getSelectedPage();
			editorPage.notifyPageChanged(editorPage);
		}
		if (event.getSelectedPage() instanceof ICoreEditorOutlinePageProvider) {
			ICoreEditorOutlinePageProvider outlinePage = (ICoreEditorOutlinePageProvider)event.getSelectedPage();
			getContentOutline().setPageActive(outlinePage.getContentOutlinePage());
		}
		
		if (event.getSelectedPage() instanceof ICoreEditorPage) {
			ICoreEditorPage corePage = (ICoreEditorPage)event.getSelectedPage();
			corePage.notifyPageActivated();
		}
	}

	/**
	 * The <code>MultiPageEditorPart</code> implementation of this 
	 * <code>IWorkbenchPart</code> method disposes all nested editors.
	 * Subclasses may extend.
	 */
	public void dispose() {
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		super.dispose();
	}
	
	/**
	 * Saves the multi-page editor's document.
	 */
	public void doSave(IProgressMonitor monitor) {
		getEditor(0).doSave(monitor);
	}
	
	/**
	 * Saves the multi-page editor's document as another file.
	 * Also updates the text for page 0's tab, and updates this multi-page editor's input
	 * to correspond to the nested editor's.
	 */
	public void doSaveAs() {
		IEditorPart editor = getEditor(0);
		editor.doSaveAs();
		setPageText(0, editor.getTitle());
		setInput(editor.getEditorInput());
	}
	
	/* (non-Javadoc)
	 * Method declared on IEditorPart.	
	 */
	public boolean isSaveAsAllowed() {
		return true;
	}

    //-------------------------------------------------------------------------------------------
    // Listeners
    //-------------------------------------------------------------------------------------------

	/** The text editor used in page 0. */
	private TextEditor editor;

	/**
	 * Closes all project files on project close.
	 */
	public void resourceChanged(final IResourceChangeEvent event) {
		if (event.getType() == IResourceChangeEvent.PRE_CLOSE) {
			Display.getDefault().asyncExec(new Runnable(){
				public void run(){
					IWorkbenchPage[] pages = getSite().getWorkbenchWindow().getPages();
					for (int i = 0; i<pages.length; i++){
						if (editor == null)
							continue;
						if (((FileEditorInput)editor.getEditorInput()).getFile().getProject().equals(event.getResource())){
							IEditorPart editorPart = pages[i].findEditor(editor.getEditorInput());
							pages[i].closeEditor(editorPart,true);
						}
					}
				}            
			});
		}
	}

	//-------------------------------------------------------------------------------------------------
	// Pages
	//-------------------------------------------------------------------------------------------------	
	
	protected void addPages() {
		try {
			addPage(new ProjectOverviewPage(this));
			addPage(new DomainPolicyPage(this));
			addPage(new ExceptionPolicyPage(this));
			if (!Utilities.is20(getProject()))
				addPage(new SystemPolicyPage(this));
			addPage(new ProjectProfilePage(this));

			ProcResourceSet projectResource = ProcResourceManager.getInstance().getProcResourceSet(getProject());
			addEditorPage(new DomainPolicyEditor(this), projectResource.getDomainPolicy().getFile());
			addEditorPage(new ExceptPolicyEditor(this), projectResource.getExceptPolicy().getFile());
			if (!Utilities.is20(getProject()))
				addEditorPage(new SystemPolicyEditor(this), projectResource.getSystemPolicy().getFile());
		} catch (PartInitException e) {
			Activator.logException(e);
		}
	}
	
	private void addEditorPage(TextEditor editor, IFile file) throws PartInitException {
		FileEditorInput editorInput = new FileEditorInput(file);
		editor.init(getEditorSite(), editorInput);
		int index = addPage(editor, editor.getEditorInput());
		setPageText(index, editor.getTitle());
	}

	public void openEditorPage(LocalResource localResource) {
		if (localResource instanceof DomainPolicy)
			setActivePage(5);
		if (localResource instanceof ExceptPolicy)
			setActivePage(6);
		if (localResource instanceof SystemPolicy)
			setActivePage(7);
	}
	
	@SuppressWarnings("unchecked")
	public Object getAdapter(Class adapter) {
		if (adapter.equals(IContentOutlinePage.class)) {
			return getContentOutline();
		}
		if (adapter.equals(IPropertySheetPage.class)) {
			return getPropertySheet();
		}
		return super.getAdapter(adapter);
	}	

	private CoreEditorOutlineContainer outlinePage = null;

	/**
     * Returns the content outline.
     */
    public CoreEditorOutlineContainer getContentOutline() {
		if (outlinePage == null) {
			outlinePage = new CoreEditorOutlineContainer(this);	
			Display.getDefault().asyncExec(new Runnable(){
				public void run() {
//					outlinePage.notifyProjectSwitched(getProject());
				}
			});
			outlinePage.addSelectionChangedListener(new ProjectOutlineSelectionChangedListener());
		}
		return outlinePage;
	}


    private class ProjectOutlineSelectionChangedListener implements ISelectionChangedListener {
		public void selectionChanged(SelectionChangedEvent event) {
	    	StructuredSelection selections = (StructuredSelection)event.getSelection();
	    	if (selections.isEmpty())
	    		return;
	    	ProjectOutlineElement firstElement = (ProjectOutlineElement)selections.getFirstElement();
	    	if (firstElement.getObject().equals(DomainPolicyModel.class)) {
	    		setActivePage(DomainPolicyPage.PAGE_ID);
	    	}
	    	if (firstElement.getObject().equals(ExceptPolicyModel.class)) {
	    		setActivePage(ExceptionPolicyPage.PAGE_ID);
	    	}
	    	if (firstElement.getObject().equals(SystemPolicyModel.class)) {
	    		setActivePage(SystemPolicyPage.PAGE_ID);
	    	}
	    	if (firstElement.getObject().equals(Profile.class)) {
	    		setActivePage(ProjectProfilePage.PAGE_ID);
	    	}
		}
    }

    private PropertySheetPage propertySheet;
    
	/**
     * Returns the property sheet.
     */
    public PropertySheetPage getPropertySheet() {
		if (propertySheet == null) {
			propertySheet = new ProjectPropertySheetPage();
			Display.getDefault().asyncExec(new Runnable(){
				public void run() {
//					propertySheet.notifyProjectSwitched(getProject());
				}
			});
//			propertySheet.addSelectionChangedListener(new ProjectOutlineSelectionChangedListener());
		}
		return propertySheet;
	}

	public String getPartName() {
		return ((FileEditorInput)getEditorInput()).getFile().getProject().getName();
	}	

	public IProject getProject() {
		return ((FileEditorInput)getEditorInput()).getFile().getProject();
	}

	//-------------------------------------------------------------------------------
	// overwrites
	//-------------------------------------------------------------------------------
	
	protected FormToolkit createToolkit(Display display) {
		return new CoreEditorFormToolkit(display);
	}

}
