/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.dialog;

import java.util.Observable;
import java.util.Observer;

import jp.sourceforge.tomoyo.core.TomoyoCorePlugin;
import jp.sourceforge.tomoyo.core.local.model.PolicyCacheManager;
import jp.sourceforge.tomoyo.core.local.model.PolicyElement;
import jp.sourceforge.tomoyo.core.local.model.domain.Domain;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;
import jp.sourceforge.tomoyo.core.local.resource.ProcResourceManager;
import jp.sourceforge.tomoyo.core.server.CommandManager;
import jp.sourceforge.tomoyo.core.server.WriteCommand;
import jp.sourceforge.tomoyo.ui.editor.Activator;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

public class NewDomainDialog extends TitleAreaDialog {

	private IProject project;
	private String argDomainText = null;
	private boolean requireChildDomain = false;
	
	public NewDomainDialog(Shell parentShell, IProject project, String argDomainText, boolean requireChildDomain) {
		super(parentShell);
		this.project = project;
		this.argDomainText = argDomainText;
		this.requireChildDomain = requireChildDomain;
	}

    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText(Messages.NewDomainDialog_ShellTitle);
    }
    
	protected int getShellStyle() {
//		return SWT.MAX | SWT.RESIZE;
		return SWT.SHELL_TRIM;
	}

	protected Point getInitialSize() {
		return new Point(800, 200);
	}

	@Override
	protected Button createButton(Composite parent, int id, String label, boolean defaultButton) {
		if (id == IDialogConstants.OK_ID) {
			Button okButton = super.createButton(parent, id, label, defaultButton);
			okButton.setEnabled(false);
			return okButton;
		} else {
			return super.createButton(parent, id, label, defaultButton);
		}
	}

	protected Control createDialogArea(Composite parent) {
		setTitle(Messages.NewDomainDialog_DialogTitle);
		setTitleImage(Activator.getImage("coloring_wiz.gif")); //$NON-NLS-1$
		
		Composite composite = (Composite)super.createDialogArea(parent);
		createComposite(composite);

		initializeUI();

		/*
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent,
		"jp.sourceforge.tomoyo.doc.open_coloring_dialog");		 //$NON-NLS-1$
		*/

	    return composite;
	}
	
	private final String CHANGE_HERE = Messages.NewDomainDialog_PleaseChangeMessage;
	
	private void initializeUI() {
		if (argDomainText == null) {
			domainText.setText(CHANGE_HERE);
			domainText.selectAll();
		} else {
			if (requireChildDomain) {
				domainText.setText(argDomainText + " " + CHANGE_HERE); //$NON-NLS-1$
				domainText.setSelection(argDomainText.length() + 1, domainText.getText().length());
			} else {
				domainText.setText(argDomainText);
				domainText.setSelection(domainText.getText().length());
			}
		}
	}

	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		super.createButtonsForButtonBar(parent);
		handleModifyDomainText();
	}

	private Text domainText;
	
	private void createComposite(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		container.setBackground(parent.getBackground());
		GridLayout layout = new GridLayout(2, false);
		layout.horizontalSpacing = 5;
		container.setLayout(layout);
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, false);
		data.horizontalSpan = 2;
		data.horizontalIndent = 0;
		container.setLayoutData(data);
		
		Label domainIconLabel = new Label(container, SWT.NULL);
		domainIconLabel.setLayoutData(new GridData(GridData.BEGINNING));
		domainIconLabel.setImage(Activator.getImage("domain.gif")); //$NON-NLS-1$
		
		domainText = createText(container, SWT.BORDER);
		domainText.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
	}
	
	private Text createText(Composite parent, int style) {
		Text text = new Text(parent, style);
		text.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				handleModifyText((Text)e.getSource());
			}
        });
		GridData data = new GridData();
		text.setLayoutData(data);
		return text;
    }

	private void handleModifyText(Text eventText) {
		if (eventText == domainText) {
			handleModifyDomainText();
		}
	}

	private void handleModifyDomainText() {
		if (getButton(IDialogConstants.OK_ID) != null) {
			getButton(IDialogConstants.OK_ID).setEnabled(canSave(domainText.getText()));
		}
	}

	private boolean canSave(String argText) {
		String checkText = argText.trim();
		if (!checkText.startsWith("<kernel>")) { //$NON-NLS-1$
			setErrorMessage(Messages.NewDomainDialog_InvalidDomainFomatErrorMessage);
			return false;
		}
		String[] strs = checkText.split("[\\s]+"); //$NON-NLS-1$
		if (strs.length < 2) {
			setErrorMessage(Messages.NewDomainDialog_InvalidDomainFomatErrorMessage);
			return false;
		}
		for (int cnt = 0; cnt < strs.length; cnt++) {
			if (cnt == 0)
				continue;
			if (!strs[cnt].startsWith("/")) { //$NON-NLS-1$
				setErrorMessage(Messages.NewDomainDialog_MustStartWithSlashErrorMessage);
				return false;
			}
		}
		Domain cacheDomain = PolicyCacheManager.getInstance().findDomain(project, checkText);
		if (cacheDomain != null && !cacheDomain.isDeleted()) {
			setErrorMessage(null);
			setMessage(Messages.NewDomainDialog_AlreadyExistsMessage, IMessageProvider.WARNING);
			return false;
		}
		setErrorMessage(null);
		setMessage(Messages.NewDomainDialog_InputOKMessage, IMessageProvider.INFORMATION);
		return true;
	}

	protected void okPressed() {
		if (save()) {
			super.okPressed();
		} else {
			super.okPressed();
		}
	}

	private boolean save() {
		String saveText = domainText.getText();
		LocalResource localResource = ProcResourceManager.getInstance().getProcResourceSet(project).getDomainPolicy();

		WriteCommand command = CommandManager.getInstance().write(project, localResource, getInput(saveText));

		if (command.isSuccessful()) {
			if (ProcResourceManager.getInstance().synchronize(project, localResource)) {
				PolicyElement savedElement = PolicyCacheManager.getInstance().findElement(project, Domain.class, saveText);
				if (savedElement != null) {
					policyAdditionObservable.notifyObservers(savedElement);
					String message = Messages.NewDomainDialog_SaveOK + " : " + saveText; //$NON-NLS-2$
					TomoyoCorePlugin.getDefault().setStatusMessage(message);
					return true;
				} else {
					String message = Messages.NewDomainDialog_SaveNG + " : " + saveText; //$NON-NLS-2$
					TomoyoCorePlugin.getDefault().setStatusErrorMessage(message);
					return false;
				}
			}
		}	
		return false;
	}

	private String getInput(String saveText) {
		StringBuffer input = new StringBuffer();

		input.append(saveText);
		input.append("\n"); //$NON-NLS-1$
		
		return input.toString();
	}

	public void addObserver(Observer o) {
		policyAdditionObservable.addObserver(o);
	}
	
	private Observable policyAdditionObservable = new Observable() {
		@Override
		public void notifyObservers(Object arg) {
			setChanged();
			super.notifyObservers(arg);
		}
	};
	
}
