/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.form.domain;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Hashtable;
import java.util.Observable;
import java.util.Observer;

import jp.sourceforge.tomoyo.core.extensions.PolicyElementHistory;
import jp.sourceforge.tomoyo.core.local.model.PolicyCacheManager;
import jp.sourceforge.tomoyo.core.local.model.PolicyElement;
import jp.sourceforge.tomoyo.core.local.model.PolicyElementDefinition;
import jp.sourceforge.tomoyo.core.local.model.domain.AccessPermission;
import jp.sourceforge.tomoyo.core.local.model.domain.Domain;
import jp.sourceforge.tomoyo.core.local.model.domain.Profile;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileGroup;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileManager;
import jp.sourceforge.tomoyo.core.local.model.status.ProjectProfile;
import jp.sourceforge.tomoyo.core.local.resource.DomainPolicy;
import jp.sourceforge.tomoyo.core.local.resource.ExceptPolicy;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;
import jp.sourceforge.tomoyo.core.local.resource.ProcResourceManager;
import jp.sourceforge.tomoyo.core.othres.DomainTransitionTreeAsciiOrderSorter;
import jp.sourceforge.tomoyo.ui.editor.Activator;
import jp.sourceforge.tomoyo.ui.editor.ViewerExpandCollapseKeyListener;
import jp.sourceforge.tomoyo.ui.editor.CoreEditorOutlineLabelProvider;
import jp.sourceforge.tomoyo.ui.editor.CoreEditorOutlineMouseListener;
import jp.sourceforge.tomoyo.ui.editor.dialog.CcsTreeDialog;
import jp.sourceforge.tomoyo.ui.editor.dialog.ColoringDialog;
import jp.sourceforge.tomoyo.ui.editor.form.PlainFormPage;
import jp.sourceforge.tomoyo.ui.editor.form.ResourceFormPage;
import jp.sourceforge.tomoyo.ui.editor.text.outline.DomainPolicyOutlineContentProvider;
import jp.sourceforge.tomoyo.ui.editor.text.outline.DomainPolicyOutlineDecoratingLabelProvider;
import jp.sourceforge.tomoyo.ui.editor.text.outline.DomainPolicyOutlineLabelProvider;
import jp.sourceforge.tomoyo.ui.editor.text.outline.PolicyOutlineContentProvider;
import jp.sourceforge.tomoyo.ui.editor.text.outline.PolicyOutlineMenuListener;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.IBaseLabelProvider;
import org.eclipse.jface.viewers.ILabelDecorator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.ITreeViewerListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeExpansionEvent;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.SWTException;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.HyperlinkSettings;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.widgets.FormText;
import org.eclipse.ui.forms.widgets.Section;

public class DomainPolicyPage extends ResourceFormPage {

	public static final String PAGE_ID = "Domain"; //$NON-NLS-1$

	public DomainPolicyPage(FormEditor editor) {
		super(editor, PAGE_ID, Messages.DomainPolicyPage_PageTitle);
		setContentDescription(Messages.DomainPolicyPage_PageDescription);
	}

	// -----------------------------------------------------------------------------------------
	// Overwrides
	// -----------------------------------------------------------------------------------------

	protected LocalResource getLocalResource() {
		return ProcResourceManager.getInstance().getProcResourceSet(
				getProject()).getDomainPolicy();
	}

	// -----------------------------------------------------------------------------------------
	// Overwrides (form)
	// -----------------------------------------------------------------------------------------

	protected String getFormTitle() {
		return Messages.DomainPolicyPage_FormText;
	}

	// -----------------------------------------------------------------------------------------
	// Overwrides (TODO section)
	// -----------------------------------------------------------------------------------------

	protected Todo[] getTodoList() {
		Todo[] todoList = new Todo[] {
		/*
		 * new Todo( Activator.getImage("_show.gif"), //$NON-NLS-1$
		 * Messages.DomainPolicyPage_TodoCCSTreeTitle,
		 * Messages.DomainPolicyPage_TodoCCSTreeDescription, "href.ccstree"
		 * //$NON-NLS-1$ ), new Todo( Activator.getImage("_todo.gif"),
		 * //$NON-NLS-1$ Messages.DomainPolicyPage_TodoSearchDomainTitle,
		 * Messages.DomainPolicyPage_TodoSearchDomainDescription, "href.search"
		 * //$NON-NLS-1$ ), new Todo( Activator.getImage("_todo.gif"),
		 * //$NON-NLS-1$ Messages.DomainPolicyPage_TodoStartLearningTitle,
		 * Messages.DomainPolicyPage_TodoStartLearningDescription,
		 * "href.learning" //$NON-NLS-1$ ), new Todo(
		 * Activator.getImage("_todo2.gif"), //$NON-NLS-1$
		 * Messages.DomainPolicyPage_TodoStartPermissiveTitle,
		 * Messages.DomainPolicyPage_TodoStartPermissiveDescription,
		 * "href.permissive" //$NON-NLS-1$ ), new Todo(
		 * Activator.getImage("_todo.gif"), //$NON-NLS-1$
		 * Messages.DomainPolicyPage_TodoStartEnforcingTitle,
		 * Messages.DomainPolicyPage_TodoStartEnforcingDescription,
		 * "href.enforcing" //$NON-NLS-1$ ), new Todo(
		 * Activator.getImage("_todo2.gif"), //$NON-NLS-1$
		 * Messages.DomainPolicyPage_TodoSetInitializerTitle,
		 * Messages.DomainPolicyPage_TodoSetInitializerDescription,
		 * "href.initializer" //$NON-NLS-1$ ), new Todo(
		 * Activator.getImage("_todo2.gif"), //$NON-NLS-1$
		 * Messages.DomainPolicyPage_TodoSetKeepDomainTitle,
		 * Messages.DomainPolicyPage_TodoSetKeepDomainDescription,
		 * "href.keep_domain" //$NON-NLS-1$ ), new Todo(
		 * Activator.getImage("_todo.gif"), //$NON-NLS-1$
		 * Messages.DomainPolicyPage_TodoPatternizeTitle,
		 * Messages.DomainPolicyPage_TodoPatternizeDescription,
		 * "href.patternize" //$NON-NLS-1$ ),
		 */
		/*
		 * new Todo( Activator.getImage("_todo.gif"), //$NON-NLS-1$ "Action
		 * history", //$NON-NLS-1$ "Check and redo saved actions", //$NON-NLS-1$
		 * "href.saved_actions" //$NON-NLS-1$ ),
		 */
		};
		return todoList;
	}

	protected void handleTodoLinkEvent(String href) {
		if (href.equals("href.search")) { //$NON-NLS-1$
			handleSearchDomain();
		}
		if (href.equals("href.learning")) { //$NON-NLS-1$
		}
		if (href.equals("href.permissive")) { //$NON-NLS-1$
		}
		if (href.equals("href.enforcing")) { //$NON-NLS-1$
		}
		if (href.equals("href.initializer")) { //$NON-NLS-1$
		}
		if (href.equals("href.keep_domain")) { //$NON-NLS-1$
		}
		if (href.equals("href.ccstree")) { //$NON-NLS-1$
			handleCCSTree();
		}
	}

	// -----------------------------------------------------------------------------------------
	// Overwrides (othres)
	// -----------------------------------------------------------------------------------------

	protected void createClient(Composite body) {

		Composite composite = new Composite(body, SWT.NULL);
		composite.setBackground(body.getBackground());
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));

		SashForm sash = new SashForm(composite, SWT.HORIZONTAL);
		GridLayout layout2 = new GridLayout();
		sash.setBackground(body.getBackground());
		sash.setLayout(layout2);
		sash.setLayoutData(new GridData(GridData.FILL_BOTH));

		Composite left = getManagedForm().getToolkit().createComposite(sash,
				SWT.NULL);
		left.setLayout(setBorder0(new GridLayout()));
		left.setLayoutData(new GridData(GridData.FILL_BOTH));

		// createSection(left, SECTION_DOMAIN_VIEWER);
		createSection1(left);

		Composite right = getManagedForm().getToolkit().createComposite(sash,
				SWT.NULL);
		right.setLayout(setBorder0(new GridLayout()));
		right.setLayoutData(new GridData(GridData.FILL_BOTH));

		// createSection(right, SECTION_ACCESS_VIEWER);
		createSection2(right);

		sash.setWeights(new int[] { 3, 2 });
	}

	protected Section createSection1(Composite parent) {

		Section section = getManagedForm().getToolkit().createSection(parent,
				Section.TITLE_BAR);
		section.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		String title = Messages.DomainPolicyPage_DomainViewerSectionTitle;
		if (title != null)
			section.setText(title);

		Composite container = getManagedForm().getToolkit().createComposite(
				section);
		container.setLayout(setBorder0(new GridLayout()));
		container.setLayoutData(new GridData(GridData.FILL_BOTH));

		domainViewer = createDomainTreeViewer(container);

		// section.setTextClient(createToolbar(section));

		getManagedForm().getToolkit().paintBordersFor(container);
		section.setClient(container);

		return section;
	}

	protected Section createSection2(Composite parent) {

		Section section = getManagedForm().getToolkit().createSection(parent,
				Section.TITLE_BAR);

		section.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		String title = Messages.DomainPolicyPage_AccessViewerSectionTitle;
		if (title != null)
			section.setText(title);

		Composite container = getManagedForm().getToolkit().createComposite(
				section);
		container.setLayout(setBorder0(new GridLayout()));
		container.setLayoutData(new GridData(GridData.FILL_BOTH));

		accessViewer = createAccessTreeViewer(container);

		// section.setTextClient(createToolbar(section));

		getManagedForm().getToolkit().paintBordersFor(container);
		section.setClient(container);

		return section;
	}

	protected void restoreDefault() {
		super.restoreDefault();

		setAccessViewerTreeStyle(true);

		// accessElementOrderButton.setSelection(true);
		// setAccessViewerElementOrder(true);

		accessAscendButton.setSelection(true);
		setAccessViewerOrientation(true);

		coloringButton.setSelection(true);
		setDomainColoringEnabled(true);

		domainChildCountButton.setSelection(true);
		showChildDomainCount(true);

		accessChildCountButton.setSelection(true);
		showChildACLCount(true);

		domainViewer.setInput(createModel());
		expandTree(domainViewer);

		accessExpandButton.setSelection(true);
		accessViewer.expandAll();

		setDefaultSelection(domainViewer);

		domainViewer.getControl().setFocus();

		/*
		 * // !!! important !!! : expand and collapse tree invisibly let
		 * contentprovider run beforehand. Display.getDefault().asyncExec(new
		 * Runnable() { public void run() {
		 * domainViewer.getTree().setRedraw(false); domainViewer.expandAll();
		 * domainViewer.collapseAll(); domainViewer.getTree().setRedraw(true); }
		 * });
		 */
	}

	private void setDefaultSelection(TreeViewer viewer) {
		Object[] visibles = viewer.getVisibleExpandedElements();
		if (visibles == null || visibles.length == 0)
			return;
		viewer.setSelection(new StructuredSelection(visibles[0]));
	}

	/*
	 * protected void handleAuditLogSelectionChanged(ISelection selection) { if
	 * (selection instanceof StructuredSelection) { StructuredSelection
	 * structured = (StructuredSelection)selection; if (structured.size() == 1) {
	 * AuditLog auditlog = (AuditLog)structured.getFirstElement(); String domain =
	 * auditlog.getDomain(); String strPermission = auditlog.getPermission();
	 * String strPath = auditlog.getPath(); String strAccessPermission =
	 * strPermission + " " + strPath; Domain foundDomain =
	 * (Domain)ElementCacheManager.getInstance().findDomain(getProject(),
	 * domain); if (foundDomain != null) { setSelection(domainViewer,
	 * foundDomain); AccessPermission accesspermission =
	 * ElementCacheManager.getInstance().findAccessPermission(getProject(),
	 * foundDomain, strAccessPermission); if (accesspermission != null) {
	 * setSelection(accessViewer, accesspermission); } } } } }
	 */

	protected void handleHistoryElementSelectionChanged(ISelection selection) {
		if (selection instanceof StructuredSelection) {
			StructuredSelection structured = (StructuredSelection) selection;
			if (structured.size() == 1) {
				PolicyElementHistory history = (PolicyElementHistory) structured
						.getFirstElement();
				PolicyElement element = (PolicyElement) history
						.getPolicyElement();
				if (element instanceof Domain) {
					setSelection(domainViewer, element);
				}
				if (element instanceof AccessPermission) {
					Domain domain = (Domain) element.getParent();
					setSelection(domainViewer, domain);
					setSelection(accessViewer, element);
				}
			}
		}
	}

	private void handleCCSTree() {
		CcsTreeDialog dialog = new CcsTreeDialog(Activator.getStandardDisplay()
				.getActiveShell(), getProject());
		dialog.setParent(this);
		dialog.open();
	}

	// -------------------------------------------------------------------------------------------
	// Section : Domain tree viewer
	// -------------------------------------------------------------------------------------------

	private TreeViewer domainViewer;
	private Label iconLabel;

	private Composite createDomainTreeViewerHeaderArea(Composite parent) {
		Composite container = getManagedForm().getToolkit().createComposite(
				parent, SWT.NONE);
		/*
		 * TableWrapLayout layout = new TableWrapLayout(); layout.numColumns =
		 * 2; layout.topMargin = layout.bottomMargin = layout.leftMargin =
		 * layout.rightMargin = 0;
		 */
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		container.setLayout(layout);
		GridData gd = new GridData(SWT.FILL, SWT.FILL, true, false);
		gd.horizontalSpan = 2;
		gd.horizontalIndent = 0;
		gd.verticalIndent = 0;
		container.setLayoutData(gd);

		iconLabel = getManagedForm().getToolkit().createLabel(container, null);
		// iconLabel.setLayoutData(new TableWrapData());
		iconLabel.setLayoutData(new GridData(GridData.BEGINNING));
		iconLabel.setImage(Activator.getImage("blank.gif")); //$NON-NLS-1$

		formText = getManagedForm().getToolkit()
				.createFormText(container, true);
		try {
			formText.setText("", false, false); //$NON-NLS-1$
		} catch (SWTException e) {
			Activator.logException(e);
		}
		formText.setHyperlinkSettings(new HyperlinkSettings(Activator
				.getStandardDisplay()) {
			/**
			 * Note : Without returning null, you keep having busy cursor after
			 * work is done.
			 */
			public Cursor getBusyCursor() {
				return null;
			}
		});
		// formText.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB,
		// TableWrapData.FILL));
		formText.setLayoutData(new GridData(GridData.FILL_BOTH));
		formText.addHyperlinkListener(new HyperlinkAdapter() {
			public void linkActivated(HyperlinkEvent e) {
				/*
				 * obsolete int depth =
				 * Integer.parseInt(e.getHref().toString()); StructuredSelection
				 * selections =
				 * (StructuredSelection)domainViewer.getSelection(); if
				 * (selections.getFirstElement() instanceof Domain) { Domain
				 * selectDomain = (Domain)selections.getFirstElement(); Domain
				 * targetDomain =
				 * PolicyCacheManager.getInstance().findDomain(getProject(),
				 * selectDomain, depth); if (targetDomain != null) {
				 * setSelection(domainViewer, targetDomain); } }
				 */
				Domain targetDomain = PolicyCacheManager.getInstance()
						.findDomain(getProject(), e.getHref().toString());
				if (targetDomain != null) {
					setSelection(domainViewer, targetDomain);
				}
			}
		});

		return container;
	}

	private String createDomainLink(Domain domain) {
		StringBuffer formHTML = new StringBuffer();
		formHTML.append("<form>"); //$NON-NLS-1$
		formHTML.append("<p>"); //$NON-NLS-1$
		for (int cnt = 0; cnt < domain.getDepth(); cnt++) {
			int depth = cnt + 1;
			formHTML
					.append("<a href=\"" + escape(domain.getText2(depth)) + "\">"); //$NON-NLS-1$ //$NON-NLS-2$
			formHTML.append(escape(domain.getText(cnt)));
			formHTML.append("</a>"); //$NON-NLS-1$
			formHTML.append(" "); //$NON-NLS-1$
		}
		formHTML.append("</p>"); //$NON-NLS-1$
		formHTML.append("</form>"); //$NON-NLS-1$
		return formHTML.toString();
	}

	private String escape(String html) {
		String rep1 = html.replaceAll("<", "&lt;"); //$NON-NLS-1$ //$NON-NLS-2$
		String rep2 = rep1.replaceAll(">", "&gt;"); //$NON-NLS-1$ //$NON-NLS-2$
		return rep2;
	}

	private FormText formText;

	private void updateDomainText(PolicyElement element) {

		DomainPolicyOutlineDecoratingLabelProvider decoratingLabelProvider = (DomainPolicyOutlineDecoratingLabelProvider) domainViewer
				.getLabelProvider();
		CoreEditorOutlineLabelProvider labelProvider = (CoreEditorOutlineLabelProvider) decoratingLabelProvider
				.getLabelProvider();

		if (element == null) {
			formText.setText("", false, false); //$NON-NLS-1$
			iconLabel.setImage(null);
			return;
		}
		if (element instanceof Domain) {
			Domain domain = (Domain) element;
			formText.setText(createDomainLink(domain), true, false);
			iconLabel.setImage(labelProvider.getImage(element));
			return;
		}
		if (element instanceof AccessPermission) {
			AccessPermission permission = (AccessPermission) element;
			formText.setText(permission.getText(), false, false);
			iconLabel.setImage(labelProvider.getImage(element));
			return;
		}
		if (element instanceof Profile) {
			Profile profile = (Profile) element;
			formText.setText(profile.getLongName(), false, false);
			iconLabel.setImage(labelProvider.getImage(element));
			return;
		}
		/*
		 * if (element instanceof InitializedFrom) { InitializedFrom from =
		 * (InitializedFrom)element; formText.setText(from.getText(), false,
		 * false); iconLabel.setImage(labelProvider.getImage(element)); return; }
		 * if (element instanceof InitializingTo) { InitializingTo to =
		 * (InitializingTo)element; formText.setText(to.getText(), false,
		 * false); iconLabel.setImage(labelProvider.getImage(element)); return; }
		 */
		formText.setText("", false, false); //$NON-NLS-1$
	}

	private TreeViewer createDomainTreeViewer(Composite parent) {

		Composite container = new Composite(parent, SWT.NULL);
		container.setBackground(parent.getBackground());
		GridLayout layout = new GridLayout(2, false);
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		container.setLayout(layout);
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		container.setLayoutData(data);

		createDomainTreeViewerHeaderArea(container);

		TreeViewer viewer = new TreeViewer(container, SWT.MULTI | SWT.BORDER);

		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.heightHint = 50;
		gd.widthHint = 50;
		viewer.getControl().setLayoutData(gd);

		viewer.setLabelProvider(createDomainViewerLabelProvider());

		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				handleDomainSelectionChanged((TreeViewer) event.getSource());
			}
		});
		viewer.addTreeListener(new ITreeViewerListener() {
			public void treeCollapsed(TreeExpansionEvent event) {
				handleTreeExpansionStateChanged((TreeViewer) event.getSource());
			}

			public void treeExpanded(TreeExpansionEvent event) {
				handleTreeExpansionStateChanged((TreeViewer) event.getSource());
			}
		});
		// viewer.setComparer(new ViewerComparer());
		viewer.getControl().addKeyListener(
				new ViewerExpandCollapseKeyListener(viewer));
		viewer.getControl().addKeyListener(
				new DomainPolicyViewerKeyListener(getProject(), viewer, null,
						dialogObserver));
		viewer.getControl().addMouseListener(
				new CoreEditorOutlineMouseListener(viewer));
		viewer.getControl().addMouseListener(
				new DomainPolicyOutlineMouseListener(getProject(), viewer));
		viewer.getControl().addFocusListener(new TreeFocusAdapter(viewer));

		viewer.setContentProvider(new DomainPolicyOutlineContentProvider());

		DomainPolicyOutlineDecoratingLabelProvider decoratingLabelProvider = (DomainPolicyOutlineDecoratingLabelProvider) viewer
				.getLabelProvider();
		DomainPolicyOutlineLabelProvider labelProvider = (DomainPolicyOutlineLabelProvider) decoratingLabelProvider
				.getLabelProvider();
		labelProvider.setDomainVerbose(false);

		viewer.setSorter(new DomainTransitionTreeAsciiOrderSorter());

		setupDomainViewerMenuListener(parent, viewer);

		createDomainButtonContoler(container);

		container.setTabList(new Control[] { viewer.getControl() });

		return viewer;
	}
	
	private boolean checkViewers() {
		if (domainViewer == null || domainViewer.getControl().isDisposed())
			return true;
		if (accessViewer == null || accessViewer.getControl().isDisposed())
			return true;
		return false;
	}

	private void refreshViewers() {
		if (checkViewers())
			return;

		TreePath[] domainTreePaths = domainViewer.getExpandedTreePaths();
		StructuredSelection domainSelections = (StructuredSelection) domainViewer
				.getSelection();
		StructuredSelection domainParentSelection = getParentSelection(
				domainViewer, domainSelections);
		TreePath[] accessTreePaths = accessViewer.getExpandedTreePaths();
		StructuredSelection accessSelections = (StructuredSelection) accessViewer
				.getSelection();
		StructuredSelection accessParentSelection = getParentSelection(
				accessViewer, accessSelections);

		domainViewer.getTree().setRedraw(false);
		domainViewer.setInput(createModel());
		domainViewer.setExpandedTreePaths(domainTreePaths);
		updateSelection(domainViewer, domainSelections, domainParentSelection);
		domainViewer.getTree().setRedraw(true);

		Domain domain = (Domain)domainSelections.getFirstElement();
		
		if (domainSelections.getFirstElement() instanceof Domain) {
			accessViewer.getTree().setRedraw(false);
			AccessViewerLabelProvider labelProvider = (AccessViewerLabelProvider) accessViewer
					.getLabelProvider();
			labelProvider.inputChanged(domain);
			accessViewer.setInput(domain);
			accessViewer.setExpandedTreePaths(accessTreePaths);
			updateSelection(accessViewer, accessSelections,
					accessParentSelection);
			accessViewer.getTree().setRedraw(true);
		}
		
		updateDomainText(domain);
		updateProfileBar(domain);
	}

	private StructuredSelection getParentSelection(TreeViewer treeViewer, StructuredSelection selections) {
		if (selections.getFirstElement() instanceof PolicyElement) {
			PolicyElement firstElement = (PolicyElement) selections
					.getFirstElement();
			if (treeViewer.getContentProvider() instanceof PolicyOutlineContentProvider) {
				PolicyOutlineContentProvider cProvider = (PolicyOutlineContentProvider) treeViewer
						.getContentProvider();
				return cProvider.getParentSelection(firstElement);
			}
		}
		return null;
	}

	private void updateSelection(TreeViewer viewer, StructuredSelection selections, StructuredSelection parentSelection) {
		if (selections.getFirstElement() instanceof PolicyElement) {
			PolicyElement firstElement = (PolicyElement) selections
					.getFirstElement();
			if (PolicyCacheManager.getInstance().exists(getProject(),
					firstElement)) {
				viewer.setSelection(selections);
				viewer.reveal(selections.getFirstElement());
			} else {
				if (parentSelection != null) {
					viewer.setSelection(parentSelection);
					if (parentSelection.getFirstElement() != null) {
						viewer.reveal(parentSelection.getFirstElement());
					}
				}
			}
		}
	}

	private void setupDomainViewerMenuListener(Composite parent, TreeViewer viewer) {
		MenuManager menuManager = new MenuManager("#PopupMenu"); //$NON-NLS-1$
		menuManager.setRemoveAllWhenShown(true);
		menuManager.addMenuListener(new DomainPolicyOutlineMenuListener(
				getProject(), viewer));
		Menu menu = menuManager.createContextMenu(parent);
		viewer.getTree().setMenu(menu);
	}

	private class DomainPolicyOutlineMenuListener extends PolicyOutlineMenuListener {

		public DomainPolicyOutlineMenuListener(IProject project,
				StructuredViewer viewer) {
			super(project, viewer);
		}

		public void menuAboutToShow(IMenuManager manager) {
			Domain domain = null;
			String domainText = null;
			boolean withSuffix = true;
			if (getSelection(viewer).isEmpty()) {
				;
			} else {
				IStructuredSelection selection = getSelection(viewer);
				if (selection.getFirstElement() instanceof Domain) {
					domain = (Domain) selection.getFirstElement();
					domainText = domain.getText();
				}
			}

			Action workAction = null;

			workAction = createCreateDomainAction(domainText, withSuffix);
			workAction.setEnabled(getSelection(viewer).size() == 1);
			manager.add(workAction);

			manager.add(new Separator());

			workAction = createCreateAccessPermissionAction(domain, null);
			workAction.setEnabled(getSelection(viewer).size() == 1);
			manager.add(workAction);

			manager.add(new Separator());
			super.menuAboutToShow(manager);
		}

	}

	private void handleDomainSelectionChanged(TreeViewer viewer) {
		if (checkSameSelection(viewer))
			return;

		ISelection selection = viewer.getSelection();
		StructuredSelection selections = (StructuredSelection) selection;
		PolicyElement firstElement = (PolicyElement) selections
				.getFirstElement();

		updateDomainText(firstElement);

		if (selections.size() > 0)
			storeSelection(viewer, selections);

		if (firstElement instanceof Domain) {
			updateAccessSelection((Domain) firstElement);
		} else {
			updateAccessSelection(null);
		}
	}

	private boolean checkSameSelection(TreeViewer viewer) {
		StructuredSelection selections = getPersistentTreeSelection(viewer);
		if (selections == null)
			return false;
		return selections.getFirstElement() == ((StructuredSelection) viewer
				.getSelection()).getFirstElement();
	}

	private void handleTreeExpansionStateChanged(TreeViewer viewer) {
		Object[] expandedElements = viewer.getExpandedElements();
		storeTreeExpansionState(viewer, expandedElements);
	}

	private void updateAccessSelection(Domain domain) {
		AccessViewerLabelProvider labelProvider = (AccessViewerLabelProvider) accessViewer
				.getLabelProvider();
		labelProvider.inputChanged(domain);
		accessViewer.setInput(domain);
		accessViewer.refresh(true);
		accessViewer.expandAll();

		updateProfileBar(domain);
	}

	private void updateProfileBar(Domain domain) {
		if (domain == null || domain.getProfile() == null) {
			profileIconLabel.setImage(null); //$NON-NLS-1$
			prifileTextLabel.setText(""); //$NON-NLS-1$
			profileIconLabel.setBackground(Activator.getStandardDisplay()
					.getSystemColor(SWT.COLOR_WHITE));
			prifileTextLabel.setBackground(Activator.getStandardDisplay()
					.getSystemColor(SWT.COLOR_WHITE));
			profileIconLabel.getParent().setBackground(
					Activator.getStandardDisplay().getSystemColor(
							SWT.COLOR_WHITE));
		} else {
			Profile profile = domain.getProfile();
			profileIconLabel.setImage(Activator.getImage("profile.gif")); //$NON-NLS-1$
			prifileTextLabel.setText(profile.getLongName());
			int groupNo = Integer.parseInt(profile.getNo());
			ProjectProfile pp = ProfileManager.getInstance().getProjectProfile(
					getProject());
			ProfileGroup group = pp.getGroup(groupNo);
			if (group == null)
				return;
			RGB foreGround = group.getForecolor();
			RGB backGround = group.getBackcolor();
			if (foreGround != null)
				prifileTextLabel.setForeground(Activator.getDefault().getColor(
						foreGround));
			else
				prifileTextLabel.setForeground(Activator.getStandardDisplay()
						.getSystemColor(SWT.COLOR_BLACK));
			if (backGround != null) {
				profileIconLabel.setBackground(Activator.getDefault().getColor(
						backGround));
				prifileTextLabel.setBackground(Activator.getDefault().getColor(
						backGround));
				profileIconLabel.getParent().setBackground(
						Activator.getDefault().getColor(backGround));
			} else {
				profileIconLabel.setBackground(Activator.getStandardDisplay()
						.getSystemColor(SWT.COLOR_WHITE));
				prifileTextLabel.setBackground(Activator.getStandardDisplay()
						.getSystemColor(SWT.COLOR_WHITE));
				profileIconLabel.getParent().setBackground(
						Activator.getStandardDisplay().getSystemColor(
								SWT.COLOR_WHITE));
			}
		}
	}

	private Button domainExpandButton;
	private Button domainCollapseButton;
	private Button coloringButton;
	private Button showSearchButton;
	private Button showCCSTreeButton;
	private Button domainChildCountButton;

	private void createDomainButtonContoler(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		container.setBackground(parent.getBackground());
		GridLayout layout = new GridLayout(1, false);
		layout.horizontalSpacing = 0;
		layout.marginLeft = 0;
		layout.horizontalSpacing = 0;
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		container.setLayout(layout);
		GridData data = new GridData(SWT.RIGHT, SWT.FILL, false, true, 1, 1);
		data.horizontalIndent = 0;
		container.setLayoutData(data);

		domainExpandButton = createButton(container, Activator
				.getImage("expandall.gif"), SWT.NULL); //$NON-NLS-1$
		domainExpandButton
				.setToolTipText(Messages.DomainPolicyPage_ExpandTreeButtonTooltip);
		domainCollapseButton = createButton(container, Activator
				.getImage("collapseall.gif"), SWT.NULL); //$NON-NLS-1$
		domainCollapseButton
				.setToolTipText(Messages.DomainPolicyPage_CollapseTreeButtonTooltip);
		domainChildCountButton = createButton(container, Activator
				.getImage("count.gif"), SWT.TOGGLE); //$NON-NLS-1$
		domainChildCountButton
				.setToolTipText(Messages.DomainPolicyPage_ShowAccessPermissionCountButtonTooltip);
		coloringButton = createButton(container, Activator
				.getImage("coloring.gif"), SWT.TOGGLE); //$NON-NLS-1$
		coloringButton
				.setToolTipText(Messages.DomainPolicyPage_EnableColorDecorationButtonTooltip);
		showSearchButton = createButton(container, Activator
				.getImage("search.gif"), SWT.PUSH); //$NON-NLS-1$
		showSearchButton
				.setToolTipText(Messages.DomainPolicyPage_OpenSearchDialogButtonTooltip);
		showCCSTreeButton = createButton(container, Activator
				.getImage("ccstree.gif"), SWT.PUSH); //$NON-NLS-1$
		showCCSTreeButton
				.setToolTipText(Messages.DomainPolicyPage_OpenProdessTreeDialogButtonTooltip);
	}

	protected void handleButtonSelected(Button button) {
		if (button == domainChildCountButton) {
			showChildDomainCount(button.getSelection());
			domainViewer.refresh();
			return;
		}
		if (button == coloringButton) {
			setDomainColoringEnabled(button.getSelection());
			domainViewer.refresh();
			if (button.getSelection()) {
				showColoringDialog();
			}
			return;
		}
		if (button == showSearchButton) {
			handleSearchDomain();
			return;
		}
		if (button == showCCSTreeButton) {
			CcsTreeDialog dialog = new CcsTreeDialog(Activator
					.getStandardDisplay().getActiveShell(), getProject());
			dialog.setParent(this);
			dialog.open();
			return;
		}

		if (button == domainExpandButton) {
			expandTree(domainViewer);
		}
		if (button == domainCollapseButton) {
			collapseTree(domainViewer, 2);
		}

		if (button == accessChildCountButton) {
			showChildACLCount(button.getSelection());
			domainViewer.refresh();
			return;
		}
		if (button == accessElementOrderButton) {
			setAccessViewerElementOrder(button.getSelection());
			accessPathOrderButton.setSelection(!button.getSelection());
		}
		if (button == accessPathOrderButton) {
			setAccessViewerElementOrder(!button.getSelection());
			accessElementOrderButton.setSelection(!button.getSelection());
		}
		if (button == accessAscendButton) {
			setAccessViewerOrientation(button.getSelection());
		}
		if (button == accessExpandButton) {
			expandTree(accessViewer);
		}
		if (button == accessCollapseButton) {
			collapseTree(accessViewer, 0);
		}
	}

	private void showColoringDialog() {
		ColoringDialog dialog = new ColoringDialog(Activator
				.getStandardDisplay().getActiveShell(), getProject());
		dialog.addObserver(new ColoringDialogObserver());
		dialog.open();
	}

	private class ColoringDialogObserver implements Observer {
		public void update(Observable o, Object arg) {
			domainViewer.refresh();
		}
	}

	// ---------------------------------------------------------------------------------------
	// Section : Access viewer
	// ---------------------------------------------------------------------------------------

	private TreeViewer accessViewer;

	private TreeViewer createAccessTreeViewer(Composite parent) {

		Composite container = new Composite(parent, SWT.NULL);
		container.setBackground(parent.getBackground());
		GridLayout layout = new GridLayout(2, false);
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		container.setLayout(layout);
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true, 2, 1);
		container.setLayoutData(data);

		createAccessTreeViewerHeaderArea(container);

		TreeViewer viewer = new TreeViewer(container, SWT.MULTI | SWT.BORDER);

		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.heightHint = 50;
		gd.widthHint = 50;
		viewer.getControl().setLayoutData(gd);

		AccessControlViewerContentProvider contentProvider = new AccessControlViewerContentProvider();
		viewer.setContentProvider(contentProvider);
		viewer.setLabelProvider(new AccessViewerLabelProvider(getProject()));
		viewer.setSorter(createAccessControlSorter());

		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				TreeViewer viewer = (TreeViewer) event.getSource();
				ISelection selection = viewer.getSelection();
				StructuredSelection selections = (StructuredSelection) selection;
				if (selections.size() > 0)
					storeSelection(viewer, selections);
			}
		});
		viewer.getControl().addKeyListener(
				new ViewerExpandCollapseKeyListener(viewer));
		viewer.getControl().addKeyListener(
				new DomainPolicyViewerKeyListener(getProject(), domainViewer,
						viewer, null));
		viewer.getControl().addMouseListener(
				new CoreEditorOutlineMouseListener(viewer));
		viewer.getControl().addFocusListener(new TreeFocusAdapter(viewer));

		setupAccessViewerMenuListener(parent, viewer);

		createAccessButtonContoler(container);

		container.setTabList(new Control[] { viewer.getControl() });

		return viewer;
	}

	private void setupAccessViewerMenuListener(Composite parent,
			TreeViewer viewer) {
		MenuManager menuManager = new MenuManager("#PopupMenu"); //$NON-NLS-1$
		menuManager.setRemoveAllWhenShown(true);
		menuManager.addMenuListener(new AccessPermissionOutlineMenuListener(
				getProject(), viewer));
		Menu menu = menuManager.createContextMenu(parent);
		viewer.getTree().setMenu(menu);
	}

	private class AccessPermissionOutlineMenuListener extends
			PolicyOutlineMenuListener {

		public AccessPermissionOutlineMenuListener(IProject project,
				StructuredViewer viewer) {
			super(project, viewer);
		}

		public void menuAboutToShow(IMenuManager manager) {
			Domain domain = null;
			IStructuredSelection domainSelection = getSelection(domainViewer);
			IStructuredSelection accessSelection = getSelection(viewer);
			if (accessSelection.isEmpty()) {
				if (domainSelection.getFirstElement() instanceof Domain) {
					domain = (Domain)domainSelection.getFirstElement();
					manager.add(createCreateAccessPermissionAction(domain, null));
					manager.add(new Separator());
				}
			} else {
				if (accessSelection.getFirstElement() instanceof AccessPermission) {
					AccessPermission access = (AccessPermission)accessSelection.getFirstElement();
					if (access.getParent() instanceof Domain) {
						domain = (Domain) access.getParent();
					}
					manager.add(createCreateAccessPermissionAction(domain, access.getText()));
					manager.add(new Separator());
				} else if (accessSelection.getFirstElement() instanceof PolicyElementDefinition) {
					PolicyElementDefinition definition = (PolicyElementDefinition)accessSelection.getFirstElement();
					if (domainSelection.getFirstElement() instanceof Domain) {
						domain = (Domain)domainSelection.getFirstElement();
						manager.add(createCreateAccessPermissionAction(domain, definition.getDirectiveName()));
						manager.add(new Separator());
					}
				}
			}
			super.menuAboutToShow(manager);
		}

	}

	private Label profileIconLabel;
	private Label prifileTextLabel;

	private Composite createAccessTreeViewerHeaderArea(Composite parent) {
		Composite container = getManagedForm().getToolkit().createComposite(
				parent, SWT.NONE);
		/*
		 * TableWrapLayout layout = new TableWrapLayout(); layout.numColumns =
		 * 2; layout.topMargin = layout.bottomMargin = layout.leftMargin =
		 * layout.rightMargin = 0;
		 */
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		container.setLayout(layout);
		GridData gd = new GridData(SWT.FILL, SWT.FILL, true, false);
		gd.horizontalIndent = 0;
		gd.verticalIndent = 0;
		gd.horizontalSpan = 2;
		container.setLayoutData(gd);

		profileIconLabel = getManagedForm().getToolkit().createLabel(container,
				null);
		profileIconLabel.setLayoutData(new GridData(GridData.BEGINNING));

		prifileTextLabel = getManagedForm().getToolkit().createLabel(container,
				null);
		prifileTextLabel.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		return container;
	}

	private void expandTree(TreeViewer viewer) {
		viewer.getTree().setRedraw(false);
		viewer.expandAll();
		viewer.setSelection(getPersistentTreeSelection(viewer));
		viewer.getTree().setRedraw(true);
	}

	private void collapseTree(TreeViewer viewer, int level) {
		viewer.getTree().setRedraw(false);
		viewer.collapseAll();
		if (level > 0)
			viewer.expandToLevel(level);
		viewer.getTree().setRedraw(true);
	}

	private void storeSelection(TreeViewer viewer,
			StructuredSelection selections) {
		viewer.setData("PersistentSelection", selections); //$NON-NLS-1$
	}

	private void storeTreeExpansionState(TreeViewer viewer,
			Object[] expandedElements) {
		viewer.setData("PersistentExpansionState", expandedElements); //$NON-NLS-1$
	}

	private StructuredSelection getPersistentTreeSelection(TreeViewer viewer) {
		StructuredSelection selections = (StructuredSelection) viewer
				.getData("PersistentSelection"); //$NON-NLS-1$
		return selections;
	}

	public void setSelection(String domain) {
		Domain cacheDomain = PolicyCacheManager.getInstance().findDomain(
				getProject(), domain);
		setSelection(cacheDomain);
	}

	public void setSelection(Domain domain) {
		if (domain == null)
			return;
		setSelection(domainViewer, domain);
	}

	protected void setSelection(TreeViewer viewer, PolicyElement element) {
		super.setSelection(viewer, element);
		if (viewer == null)
			return;
		StructuredSelection selection = new StructuredSelection(element);
		storeSelection(viewer, selection);
	}

	/*
	 * private List getElementPath(PolicyElement element) { List pathList = new
	 * ArrayList(); _getElementPath(pathList, element); return pathList; }
	 * 
	 * private void _getElementPath(List pathList, PolicyElement element) { if
	 * (element == null) return; pathList.add(element); if (element.getParent() !=
	 * null && element.getParent() instanceof PolicyElement) { PolicyElement
	 * parent = element.getParent(); _getElementPath(pathList, parent); } }
	 * 
	 * private void expandPath(TreeViewer viewer, List elementPath) { for (int
	 * cnt = 0; cnt < elementPath.size(); cnt++) { PolicyElement element =
	 * (PolicyElement)elementPath.get(cnt); viewer.setExpandedState(element,
	 * true); } }
	 */
	private Button accessExpandButton;
	private Button accessCollapseButton;
	private Button accessChildCountButton;
	private Button accessElementOrderButton;
	private Button accessPathOrderButton;
	private Button accessAscendButton;

	private void createAccessButtonContoler(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		container.setBackground(parent.getBackground());
		GridLayout layout = new GridLayout(1, false);
		layout.horizontalSpacing = 0;
		layout.marginLeft = 0;
		layout.horizontalSpacing = 0;
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		container.setLayout(layout);
		GridData data = new GridData(SWT.RIGHT, SWT.FILL, false, true, 1, 1);
		data.horizontalIndent = 0;
		container.setLayoutData(data);

		accessExpandButton = createButton(container, Activator
				.getImage("expandall.gif"), SWT.NULL); //$NON-NLS-1$
		accessExpandButton
				.setToolTipText(Messages.DomainPolicyPage_ExpandTreeButtonTooltip);
		accessCollapseButton = createButton(container, Activator
				.getImage("collapseall.gif"), SWT.NULL); //$NON-NLS-1$
		accessCollapseButton
				.setToolTipText(Messages.DomainPolicyPage_CollapseTreeButtonTooltip);
		accessChildCountButton = createButton(container, Activator
				.getImage("count.gif"), SWT.TOGGLE); //$NON-NLS-1$
		accessChildCountButton
				.setToolTipText(Messages.DomainPolicyPage_ShowDirectiveAccessPermissionCountButtonTooltip);
		accessAscendButton = createButton(container, Activator
				.getImage("sort_asc.gif"), SWT.TOGGLE); //$NON-NLS-1$
		accessAscendButton
				.setToolTipText(Messages.DomainPolicyPage_SortButtonTooltip);
	}

	// ---------------------------------------------------------------------------------------
	// Event handler
	// ---------------------------------------------------------------------------------------

	private void setAccessViewerTreeStyle(boolean b) {
		AccessControlViewerContentProvider contentProvider = (AccessControlViewerContentProvider) accessViewer
				.getContentProvider();
		contentProvider.setTreeStyle(b);

		DomainPolicyOutlineLabelProvider labelProvider = (DomainPolicyOutlineLabelProvider) accessViewer
				.getLabelProvider();
		labelProvider.setAccessControlVerbose(!b);

		accessViewer.refresh(true);
		if (accessExpandButton.getSelection())
			accessViewer.expandAll();
	}

	private void setAccessViewerElementOrder(boolean b) {
		accessSorter.setElementOrder(b);
		accessViewer.refresh(true);
	}

	private void setAccessViewerOrientation(boolean b) {
		accessSorter.setOrientation(b);
		accessViewer.refresh();
	}

	private void setDomainColoringEnabled(boolean b) {
		DomainPolicyOutlineDecoratingLabelProvider decoratingLabelProvider = (DomainPolicyOutlineDecoratingLabelProvider) domainViewer
				.getLabelProvider();
		DomainPolicyOutlineLabelProvider labelProvider = (DomainPolicyOutlineLabelProvider) decoratingLabelProvider
				.getLabelProvider();
		labelProvider.setColoringEnabled(b);
	}

	private void showChildDomainCount(boolean selection) {
		DomainPolicyOutlineDecoratingLabelProvider decoratingLabelProvider = (DomainPolicyOutlineDecoratingLabelProvider) domainViewer
				.getLabelProvider();
		DomainPolicyOutlineLabelProvider labelProvider = (DomainPolicyOutlineLabelProvider) decoratingLabelProvider
				.getLabelProvider();
		labelProvider.showResourceCount(selection);
	}

	private void showChildACLCount(boolean selection) {
		AccessViewerLabelProvider lavelProvider = (AccessViewerLabelProvider) accessViewer
				.getLabelProvider();
		lavelProvider.showChildACLCount(selection);
		accessViewer.refresh();
	}

	// -----------------------------------------------------------------------------------------
	// Sorter
	// -----------------------------------------------------------------------------------------

	private AccessControlSorter accessSorter;

	private AccessControlSorter createAccessControlSorter() {
		if (accessSorter == null)
			accessSorter = new AccessControlSorter();
		return accessSorter;
	}

	private class AccessControlSorter extends ViewerSorter implements
			Comparator<Object> {
		private boolean elementOrder = false;
		private boolean ascend = false;

		public AccessControlSorter() {
		}

		public void setElementOrder(boolean b) {
			elementOrder = b;
		}

		public void setOrientation(boolean b) {
			ascend = b;
		}

		public int compare(Viewer viewer, Object e1, Object e2) {
			return compare(e1, e2);
		}

		public int compare(Object e1, Object e2) {
			String text1;
			String text2;
			if (e1 instanceof Profile)
				return 1;
			if (e2 instanceof Profile)
				return -1;
			if (e1 instanceof PolicyElementDefinition) {
				PolicyElementDefinition def1 = (PolicyElementDefinition)e1;
				PolicyElementDefinition def2 = (PolicyElementDefinition)e2;
				text1 = def1.getDirectiveName();
				text2 = def2.getDirectiveName();
				return text1.compareTo(text2);
			} else if (e1 instanceof PolicyElement) {
				PolicyElement element1 = (PolicyElement)e1;
				PolicyElement element2 = (PolicyElement)e2;
				if (elementOrder) {
					text1 = element1.getText();
					text2 = element2.getText();
				} else {
					text1 = element1.getTextFrom(1);
					text2 = element2.getTextFrom(1);
				}
			} else {
				return 0;
			}
			if (ascend)
				return text1.compareTo(text2);
			else
				return text2.compareTo(text1);
		}
	}

	// -----------------------------------------------------------------------------------------
	// Models
	// -----------------------------------------------------------------------------------------

	protected Object createModel() {
		DomainPolicy domainPolicy = ProcResourceManager.getInstance()
				.getProcResourceSet(getProject()).getDomainPolicy();
		return domainPolicy.getModel();
	}

	// -----------------------------------------------------------------------------------------
	// Content providers
	// -----------------------------------------------------------------------------------------

	public class AccessControlViewerContentProvider implements
			ITreeContentProvider {

		public AccessControlViewerContentProvider() {
		}

		private boolean deletedVisible = false;

		public void setDeletedVisible(boolean b) {
			deletedVisible = b;
		}

		private boolean treeStyle = false;

		public void setTreeStyle(boolean b) {
			treeStyle = b;
		}

		public Object[] getChildren(Object parent) {
			if (treeStyle)
				return getChildren_treeStyle(parent);
			else
				return getChildren_textStyle(parent);
		}

		protected Object[] getChildren_treeStyle(Object parent) {
			ArrayList<Object> children = new ArrayList<Object>();
			if (parent instanceof Domain) {
				Domain domain = (Domain) parent;
				Hashtable<String, PolicyElementDefinition> definitionTable = new Hashtable<String, PolicyElementDefinition>();
				for (int cnt = 0; cnt < domain.getChildrenCount(); cnt++) {
					PolicyElement child = domain.getChild(cnt);
					if (child.isDeleted())
						continue;
					if (child instanceof Profile)
						continue;
					String directive = child.getDirective();
					if (definitionTable.get(directive) == null) {
						definitionTable.put(directive, getLocalResource().getModel().getDefinitionByDirective(directive));
					}
				}
				children.addAll(definitionTable.values());
			}
			if (parent instanceof PolicyElementDefinition) {
				PolicyElementDefinition definition = (PolicyElementDefinition)parent;
				if (newInput instanceof Domain) {
					Domain domain = (Domain)newInput;
					for (int cnt = 0; cnt < domain.getChildrenCount(); cnt++) {
						PolicyElement child = domain.getChild(cnt);
						if (definition.isInstance(child)) {
							if (child.isDeleted()) {
								if (deletedVisible) {
									children.add(child);
								}
							} else {
								children.add(child);
							}
						}
					}
				}
			}
			return children.toArray();
		}

		protected Object[] getChildren_textStyle(Object parent) {
			ArrayList<PolicyElement> children = new ArrayList<PolicyElement>();
			if (parent instanceof PolicyElement) {
				PolicyElement element = (PolicyElement) parent;
				for (int cnt = 0; cnt < element.getChildrenCount(); cnt++) {
					PolicyElement child = element.getChild(cnt);
					// if (child.isChecked()) {
					if (!child.isDeleted()) {
						children.add(child);
					} else {
						if (deletedVisible) {
							children.add(child);
						}
					}
				}
				/*
				 * PolicyElement element = (PolicyElement)list.get(cnt); if
				 * (element.isChecked()) { retlist.add(element); } else { if
				 * (deletedVisible) { retlist.add(element); } }
				 */
			}
			return children.toArray();
		}

		public boolean hasChildren(Object obj) {
			return getChildren(obj).length > 0;
		}

		public Object[] getElements(Object inputElement) {
			return getChildren(inputElement);
		}

		public Object getParent(Object obj) {
			if (obj instanceof PolicyElement) {
				PolicyElement element = (PolicyElement) obj;
				return element.getParent();
			}
			return null;
		}

		public void dispose() {
		}

		// private Object oldInput;
		private Object newInput;

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			// this.oldInput = oldInput;
			this.newInput = newInput;
		}

	}

	// -----------------------------------------------------------------------------------------
	// Label providers
	// -----------------------------------------------------------------------------------------

	private IBaseLabelProvider createDomainViewerLabelProvider() {
		DomainPolicyOutlineLabelProvider labelProvider = new DomainPolicyOutlineLabelProvider(
				getProject());
		ILabelDecorator decorator = PlatformUI.getWorkbench()
				.getDecoratorManager().getLabelDecorator();
		return new DomainPolicyOutlineDecoratingLabelProvider(labelProvider,
				decorator);
	}

	private class AccessViewerLabelProvider extends
			DomainPolicyOutlineLabelProvider {

		public AccessViewerLabelProvider(IProject project) {
			super(project);
		}

		public String getText(Object object) {
			if (object instanceof PolicyElementDefinition) {
				PolicyElementDefinition definition = (PolicyElementDefinition)object;
				int count = PolicyCacheManager.getInstance().count(project,
						inputDomain, definition.getElementClass(), false);
				String directiveName = definition.getDirectiveName();
				if (showChildACLCount)
					return directiveName + " " + "(" + count + ")"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				else
					return directiveName;
			}
			return super.getText(object);
		}

		private Domain inputDomain;

		public void inputChanged(Domain inputDomain) {
			this.inputDomain = inputDomain;
		}

		private boolean showChildACLCount = false;

		public void showChildACLCount(boolean b) {
			showChildACLCount = b;
		}

	}

	// -----------------------------------------------------------------------------------------
	// Search dialog
	// -----------------------------------------------------------------------------------------

	private DomainSearchDialogObserver dialogObserver = new DomainSearchDialogObserver();

	private void handleSearchDomain() {
		DomainSearchDialog dialog = DomainSearchDialog.getInstance(Activator
				.getStandardDisplay().getActiveShell());
		dialogObserver.setDialog(dialog);
		dialogObserver.setViewer(domainViewer);
		dialog.open();
	}

	// -----------------------------------------------------------------------------------------
	// Notifications
	// -----------------------------------------------------------------------------------------

	public void notifySelectionChanged(ISelection selection) {
	}

	public void notifyPageChanged(PlainFormPage editorPage) {
		super.notifyPageChanged(editorPage);
		if (needToRefreshViewer) {
			refreshViewers();
			needToRefreshViewer = false;
		}
	}

	private boolean needToRefreshViewer = false;

	protected void notifyPolicyUpdated(IProject project, LocalResource localResource) {
		super.notifyPolicyUpdated(project, localResource);
		if (project == getProject()) {
			if (localResource instanceof DomainPolicy) {
				if (domainViewer.getTree().isVisible()) { // !!! important !!!
					refreshViewers();
				} else {
					needToRefreshViewer = true;
				}
			}
			if (localResource instanceof ExceptPolicy) {
				if (domainViewer.getTree().isVisible()) { // !!! important !!!
					refreshViewers();
				} else {
					needToRefreshViewer = true;
				}
			}
		}
	}

	protected void notifyProfileUpdated(IProject project) {
		if (!checkWidgets())
			return;
		if (project == getProject()) {
			domainViewer.refresh();
		}
	}

	protected String getHelpPath() {
		return "/jp.sourceforge.tomoyo.doc/" + "html/" + "page-domain.html"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	}

}
