/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.ui.editor.text.outline;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import jp.sourceforge.tomoyo.core.TomoyoCorePlugin;
import jp.sourceforge.tomoyo.core.local.model.AbstractExceptPolicyElement;
import jp.sourceforge.tomoyo.core.local.model.PolicyCacheManager;
import jp.sourceforge.tomoyo.core.local.model.PolicyElement;
import jp.sourceforge.tomoyo.core.local.model.AbstractSystemPolicyElement;
import jp.sourceforge.tomoyo.core.local.model.PolicyElementDefinition;
import jp.sourceforge.tomoyo.core.local.model.domain.AccessPermission;
import jp.sourceforge.tomoyo.core.local.model.domain.Domain;
import jp.sourceforge.tomoyo.core.local.resource.LocalResource;
import jp.sourceforge.tomoyo.core.local.resource.ProcResourceManager;
import jp.sourceforge.tomoyo.core.server.CommandManager;
import jp.sourceforge.tomoyo.core.server.ICommand;
import jp.sourceforge.tomoyo.core.server.ServerAction;
import jp.sourceforge.tomoyo.ui.editor.Activator;
import jp.sourceforge.tomoyo.ui.editor.dialog.ChangeProfileDialog;
import jp.sourceforge.tomoyo.ui.editor.dialog.NewAccessPermissionDialog;
import jp.sourceforge.tomoyo.ui.editor.dialog.NewDomainDialog;
import jp.sourceforge.tomoyo.ui.editor.dialog.PatternizeAccessPermissionsDialog;
import jp.sourceforge.tomoyo.ui.editor.form.NewDefinitionDialog;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;

public class PolicyActions {

	protected IProject project;

	public PolicyActions(IProject project) {
		this.project = project;
	}
	
	protected boolean isDomainOnly(IStructuredSelection selection) {
		if (selection.size() == 1) {
			if (selection.getFirstElement() instanceof Domain) {
				return true;
			}
		}
		return false;
	}
	
	protected boolean areDomainsOnly(IStructuredSelection selection) {
		if (selection.isEmpty())
			return false;
		Iterator<?> selections = selection.iterator();
		while (selections.hasNext()) {
			Object object = selections.next();
			if (!(object instanceof Domain))
				return false;
		}
		return true;
	}

	protected boolean isAccessPermissionOnly(IStructuredSelection selection) {
		if (selection.size() == 1) {
			if (selection.getFirstElement() instanceof AccessPermission) {
				return true;
			}
		}
		return false;
	}
	
	protected boolean areAccessPermissionsOnly(IStructuredSelection selection, boolean ignoreDefinition) {
		if (selection.isEmpty())
			return false;
		int vaildCount = 0;
		Iterator<?> selections = selection.iterator();
		while (selections.hasNext()) {
			Object object = selections.next();
			if (ignoreDefinition && object instanceof PolicyElementDefinition)
				continue;
			if (object instanceof AccessPermission)
				vaildCount++;
		}
		return vaildCount > 0;
	}

	protected boolean isExceptPolicyElementsOnly(IStructuredSelection selection) {
		if (selection.size() == 0)
			return false;
		Iterator<?> selections = selection.iterator();
		while (selections.hasNext()) {
			Object object = selections.next();
			if (!(object instanceof AbstractExceptPolicyElement))
				return false;
		}
		return true;
	}

	protected boolean isSystemPolicyElementsOnly(IStructuredSelection selection) {
		if (selection.size() == 0)
			return false;
		Iterator<?> selections = selection.iterator();
		while (selections.hasNext()) {
			Object object = selections.next();
			if (!(object instanceof AbstractSystemPolicyElement))
				return false;
		}
		return true;
	}

	/*
	protected MenuManager createChangeProfileAction(Domain domain) {
		MenuManager subMenu = new MenuManager(
				Messages.PolicyOutlineMenuListener_ChangeProfileMenu);
		ProfileGroup[] groups = ProfileManager.getInstance().getProjectProfile(
				project).toArray(true);
		for (int cnt = 0; cnt < groups.length; cnt++) {
			MenuManager profileMenu = new MenuManager(groups[cnt].getLongName());
			subMenu.add(profileMenu);
			profileMenu.add(new ChangeProfileAction(groups[cnt], domain, true));
			profileMenu
					.add(new ChangeProfileAction(groups[cnt], domain, false));
		}
		return subMenu;
	}
	*/

	protected Domain[] toDomainArray(IStructuredSelection selection) {
		ArrayList<Domain> domainList = new ArrayList<Domain>();
		Iterator<?> it = selection.iterator();
		while (it.hasNext()) {
			Object next = it.next();
			if (next instanceof Domain) {
				domainList.add((Domain) next);
			}
		}
		return domainList.toArray(new Domain[domainList.size()]);
	}
	
	protected AccessPermission[] toAccessPermissionArray(IStructuredSelection selection) {
		ArrayList<AccessPermission> acList = new ArrayList<AccessPermission>();
		Iterator<?> it = selection.iterator();
		while (it.hasNext()) {
			Object next = it.next();
			if (next instanceof AccessPermission) {
				acList.add((AccessPermission) next);
			}
		}
		return acList.toArray(new AccessPermission[acList.size()]);
	}
	
	protected PolicyElement[] toPolicyElementArray(IStructuredSelection selection) {
		ArrayList<PolicyElement> list = new ArrayList<PolicyElement>();
		Iterator<?> it = selection.iterator();
		while (it.hasNext()) {
			Object next = it.next();
			if (next instanceof PolicyElement) {
				list.add((PolicyElement) next);
			}
		}
		return list.toArray(new PolicyElement[list.size()]);
	}
	
	protected Action createDeleteChildAccessControlAction(IStructuredSelection selection) {
		Domain[] selectDomains = new Domain[selection.size()];
		int cnt = 0;
		Iterator<?> it = selection.iterator();
		while (it.hasNext()) {
			Object next = it.next();
			selectDomains[cnt++] = (Domain)next;
		}
		return new DeleteChildAccessControlAction(selectDomains);
	}

	protected Action createDeleteAccessPermissionAction(AccessPermission[] argArray) {
		return new DeleteAccessPermissionAction(argArray);
	}

	protected Action createPatternizeAccessPermissionAction(AccessPermission[] argArray) {
		Domain domain = null;
		if (argArray != null && argArray.length > 0) {
			AccessPermission argArray0 = (AccessPermission)argArray[0];
			domain = (Domain)argArray0.getParent();
		}
		return new PatternizeAccessPermissionAction(domain, argArray);
	}
	
	protected Action createDeleteElementAction(LocalResource localResource, IStructuredSelection selection) {
		PolicyElement[] argArray = toPolicyElementArray(selection);
		return new DeleteElementAction(localResource, argArray);
	}
	
	private abstract class DeleteElementServerAction extends ServerAction {
		public DeleteElementServerAction(IProject project) {
			super(project);
		}
		public void run() {
			ICommand command = exec();
			if (command.isSuccessful()) {
				synchronize(getLocalResource());
			} else {
				TomoyoCorePlugin.getDefault().setStatusErrorMessage(Messages.PolicyOutlineMenuListener_DeleteCommandErrorMessage);
			}
		}
		protected abstract ICommand exec();
		protected abstract LocalResource getLocalResource();
		private void synchronize(LocalResource localResource) {
			PolicyElement[] deletedElements = getDeletingElements();
			if (deletedElements == null || deletedElements.length == 0)
				return;
			if (ProcResourceManager.getInstance().synchronize(project, localResource)) {
				if (checkDeleted()) {
					String prefix = Messages.PolicyOutlineMenuListener_DeletingCompletedMessage;
					TomoyoCorePlugin.getDefault().setStatusMessage(decorateMessage(prefix));
				} else {
					TomoyoCorePlugin.getDefault().setStatusErrorMessage(
							Messages.PolicyOutlineMenuListener_DeletingErrorOccuredMessage);
				}
			} else {
				TomoyoCorePlugin.getDefault().setStatusErrorMessage(Messages.PolicyOutlineMenuListener_SynchronizedErrorMessage);
			}
		}
		private String decorateMessage(String prefix) {
			StringBuffer message = new StringBuffer(prefix);
			PolicyElement[] deletedElements = getDeletingElements();
			if (deletedElements != null || deletedElements.length > 0) {
				PolicyElement deletedElement = deletedElements[0];
				message.append(" : "); //$NON-NLS-1$
				message.append(deletedElement.getText());
				if (deletedElements.length > 1) {
					message.append(" ("); //$NON-NLS-1$
					message.append(Messages.PolicyOutlineMenuListener_DeletedMessage_prefix);
					message.append(" "); //$NON-NLS-1$
					message.append(deletedElements.length - 1);
					message.append(" "); //$NON-NLS-1$
					message.append(Messages.PolicyOutlineMenuListener_DeletedMessage_suffix);
					message.append(")"); //$NON-NLS-1$
				}
			}
			return message.toString();
		}
		private boolean checkDeleted() {
			PolicyElement[] deletingElements = getDeletingElements();
			ArrayList<PolicyElement> errorElemens = new ArrayList<PolicyElement>();
			for (int cnt = 0; cnt < deletingElements.length; cnt++) {
				PolicyElement checkElement = deletingElements[cnt];
				if (checkElement instanceof AccessPermission && checkElement.getParent() instanceof Domain) {
					AccessPermission access = (AccessPermission)checkElement;
					Domain domain = (Domain)checkElement.getParent();
					if (PolicyCacheManager.getInstance().exists(project, domain, access)) {
						errorElemens.add(checkElement);
					}
				}
			}
			return errorElemens.size() == 0;
		}
		protected abstract PolicyElement[] getDeletingElements();
    }

	protected class DeleteDomainAction extends DeleteElementServerAction {
		private Domain[] domains;
		private boolean recursive;
		private boolean includesRoot;

		public DeleteDomainAction(Domain[] domains, boolean recursive) {
			this(domains, recursive, false);
		}

		public DeleteDomainAction(Domain[] domains, boolean recursive, boolean includesRoot) {
			super(project);
			this.domains = domains;
			this.recursive = recursive;
			this.includesRoot = includesRoot;
			if (recursive) {
				if (includesRoot) {
					setText(Messages.PolicyOutlineMenuListener_DeleteDomainRecursiveMenu);
					setImageDescriptor(Activator
							.getImageDescriptor("recursive.gif")); //$NON-NLS-1$
				} else {
					setText(Messages.PolicyOutlineMenuListener_DeleteDomainRecursivelyWithoutSelectionMenu);
					setImageDescriptor(Activator
							.getImageDescriptor("recursive.gif")); //$NON-NLS-1$
				}
			} else {
				setText(Messages.PolicyOutlineMenuListener_DeleteDomainNonRecursiveMenu);
				setImageDescriptor(Activator
						.getImageDescriptor("non_recursive.gif")); //$NON-NLS-1$
			}
		}

		protected ICommand exec() {
			return CommandManager.getInstance().write(project, getLocalResource(), getInputText());
		}

		public String getInputText() {
			StringBuffer input = new StringBuffer();
			PolicyElement[] deletingDomains = getDeletingElements();
			input.append("delete"); //$NON-NLS-1$
			for (int cnt = 0; cnt < deletingDomains.length; cnt++) {
				input.append(" "); //$NON-NLS-1$
				input.append(deletingDomains[cnt].getText());
				input.append("\n"); //$NON-NLS-1$
			}
			return input.toString();
		}

		@Override
		protected LocalResource getLocalResource() {
			LocalResource domainPolicy = ProcResourceManager.getInstance()
			.getProcResourceSet(project).getDomainPolicy();
			return domainPolicy;
		}

		private PolicyElement[] deletingElementCache = null;
		              
		@Override
		protected PolicyElement[] getDeletingElements() {
			if (deletingElementCache == null) {
				Hashtable<String, Domain> uniqueDomains = new Hashtable<String, Domain>();

				for (int domcnt = 0; domcnt < domains.length; domcnt++) {
					Domain selectDomain = domains[domcnt];
					if (recursive) {
						if (uniqueDomains.get(selectDomain.getText()) == null) {
							if (includesRoot)
								uniqueDomains.put(selectDomain.getText(),selectDomain);
							List<Domain> children = PolicyCacheManager.getInstance().listDecendantDomains(project, selectDomain);
							for (int cnt = 0; cnt < children.size(); cnt++) {
								Domain childDomain = (Domain) children.get(cnt);
								uniqueDomains.put(childDomain.getText(), childDomain);
							}
						}
					} else {
						uniqueDomains.put(selectDomain.getText(), selectDomain);
					}
				}

				deletingElementCache = (Domain[])uniqueDomains.values()
						.toArray(new Domain[uniqueDomains.values().size()]);
			}
			return deletingElementCache;
		}
	}

	protected class DeleteChildAccessControlAction extends DeleteElementServerAction {
		private Domain domain;

		public DeleteChildAccessControlAction(Domain[] domains) {
			super(project);
			this.domain = domains[0];
			setEnabled(domains.length == 1);
			setText(Messages.PolicyOutlineMenuListener_DeleteAccessPermissionAction);
			setImageDescriptor(Activator.getImageDescriptor("trushbox.gif")); //$NON-NLS-1$
		}

		protected ICommand exec() {
			return CommandManager.getInstance().write(project, getLocalResource(), getInputText());
		}

		public String getInputText() {
			AccessPermission[] accessPermissions = (AccessPermission[])getDeletingElements();
			StringBuffer input = new StringBuffer();
			input.append(domain.getText());
			input.append("\n"); //$NON-NLS-1$
			for (int cnt = 0; cnt < accessPermissions.length; cnt++) {
				AccessPermission access = accessPermissions[cnt];
				input.append("delete"); //$NON-NLS-1$
				input.append(" "); //$NON-NLS-1$
				input.append(access.getText());
				input.append("\n"); //$NON-NLS-1$
			}
			return input.toString();
		}

		@Override
		protected LocalResource getLocalResource() {
			LocalResource domainPolicy = ProcResourceManager.getInstance()
			.getProcResourceSet(project).getDomainPolicy();
			return domainPolicy;
		}
		
		private PolicyElement[] deletingElementCache = null;

		protected PolicyElement[] getDeletingElements() {
			if (deletingElementCache == null) {
				ArrayList<AccessPermission> delegingList = new ArrayList<AccessPermission>();
				for (int cnt = 0; cnt < domain.getChildrenCount(); cnt++) {
					if (domain.getChild(cnt) instanceof AccessPermission) {
						delegingList.add((AccessPermission)domain.getChild(cnt));
					}
				}
				deletingElementCache = delegingList.toArray(new AccessPermission[delegingList.size()]);
			}
			return deletingElementCache;
		}
	}

	protected class DeleteAccessPermissionAction extends DeleteElementServerAction {
		
		private PolicyElement[] deletingElements = null;
        		
		public DeleteAccessPermissionAction(PolicyElement[] deletingElements) {
			super(project);
			setText(Messages.PolicyOutlineMenuListener_DeleteAccessPermissionAction);
			setImageDescriptor(Activator.getImageDescriptor("trushbox.gif")); //$NON-NLS-1$
			this.deletingElements = deletingElements;
			setEnabled(deletingElements != null && deletingElements.length > 0);
		}

		protected ICommand exec() {
			return CommandManager.getInstance().write(project, getLocalResource(), getInputText());
		}

		public String getInputText() {
			AccessPermission[] accessPermissions = (AccessPermission[])getDeletingElements();
			Domain domain = (Domain)accessPermissions[0].getParent();
			StringBuffer input = new StringBuffer();
			input.append("select"); //$NON-NLS-1$
			input.append(" "); //$NON-NLS-1$
			input.append(domain.getText());
			input.append("\n"); //$NON-NLS-1$
			for (int cnt = 0; cnt < accessPermissions.length; cnt++) {
				input.append("delete"); //$NON-NLS-1$
				input.append(" "); //$NON-NLS-1$
				input.append(accessPermissions[cnt].getText());
				input.append("\n"); //$NON-NLS-1$
			}
			return input.toString();
		}

		@Override
		protected LocalResource getLocalResource() {
			LocalResource domainPolicy = ProcResourceManager.getInstance()
				.getProcResourceSet(project).getDomainPolicy();
			return domainPolicy;
		}

		@Override
		protected PolicyElement[] getDeletingElements() {
			return deletingElements;
		}
	}

	private class PatternizeAccessPermissionAction extends ServerAction {
		
		private Domain domain;
		private AccessPermission[] pattenizeElements;
		
		public PatternizeAccessPermissionAction(Domain domain, AccessPermission[] pattenizeElements) {
			super(project);
			setText(Messages.PolicyActions_PatternizeAPActionTitle);
			setImageDescriptor(Activator.getImageDescriptor("patternize.gif")); //$NON-NLS-1$
			
			this.domain = domain;
			this.pattenizeElements = pattenizeElements;

			if (domain == null) {
				setEnabled(false);				
			} else {
				if (pattenizeElements != null && pattenizeElements.length > 0) {
					setEnabled(true);
				} else {
					setEnabled(false);
				}
			}
			
			setEnabled(pattenizeElements != null && pattenizeElements.length > 0);
		}

		public void run() {
			PatternizeAccessPermissionsDialog dialog = null;
			dialog = new PatternizeAccessPermissionsDialog(Activator.getStandardDisplay().getActiveShell(), project, domain, pattenizeElements);
	    	dialog.open();
		}
		
	}
	
	protected class DeleteElementAction extends DeleteElementServerAction {

		private PolicyElement[] deletingElements = null;

		private LocalResource localResource;

		public DeleteElementAction(LocalResource localResource, PolicyElement[] deletingElements) {
			super(project);
			setText(Messages.PolicyOutlineMenuListener_DeleteElementMenu);
			setImageDescriptor(Activator.getImageDescriptor("trushbox.gif")); //$NON-NLS-1$
			
			this.localResource = localResource;
			this.deletingElements = deletingElements;
		}

		protected ICommand exec() {
			return CommandManager.getInstance().write(project, getLocalResource(), getInputText());
		}

		public String getInputText() {
			StringBuffer input = new StringBuffer();
			for (int cnt = 0; cnt < deletingElements.length; cnt++) {
				input.append("delete"); //$NON-NLS-1$
				input.append(" "); //$NON-NLS-1$
				input.append(deletingElements[cnt].getText());
				input.append("\n"); //$NON-NLS-1$
			}
			return input.toString();
		}

		@Override
		protected LocalResource getLocalResource() {
			// TODO Auto-generated method stub
			return localResource;
		}
        
		@Override
		protected PolicyElement[] getDeletingElements() {
			return deletingElements;
		}

	}

	protected Action createCreateDomainAction(String domainText, boolean requireChildDomain) {
		return new CreateDomainAction(domainText, requireChildDomain);
	}

	private class CreateDomainAction extends ServerAction {
		
		private String domainText;
		private boolean requireChildDomain = false;
		
		public CreateDomainAction(String domainText, boolean requireChildDomain) {
			super(project);
			setText(Messages.PolicyActions_CreateNewDomainActionTitle);
			setImageDescriptor(Activator.getImageDescriptor("domain_new.gif")); //$NON-NLS-1$
			this.domainText = domainText;
			this.requireChildDomain = requireChildDomain;
		}

		public void run() {
			NewDomainDialog dialog = null;
			dialog = new NewDomainDialog(Activator.getStandardDisplay().getActiveShell(), project, domainText, requireChildDomain);
	    	dialog.open();
		}

	}
	
	protected Action createDeleteDomainAction(Domain[] domainArray) {
		return new _DeleteDomainAction(domainArray);
	}

	private class _DeleteDomainAction extends ServerAction {
		
		private Domain[] domainArray;
		
		public _DeleteDomainAction(Domain[] domainArray) {
			super(project);
			setText(Messages.PolicyActions_DeleteDomainActionTitle);
			setImageDescriptor(Activator.getImageDescriptor("trushbox.gif")); //$NON-NLS-1$
			this.domainArray = domainArray;
		}

		public void run() {
			MessageDialog dialog = new MessageDialog(
					Activator.getStandardDisplay().getActiveShell(),
					Messages.PolicyActions_DeleteDomainDialogTitle,
					null,
					Messages.PolicyActions_DeleteDomainDialogDescription,
					MessageDialog.QUESTION,
					new String[] {
						Messages.PolicyActions_OptionSelectionOnly,
						Messages.PolicyActions_Option2Bottom,
						Messages.PolicyActions_Option2BottomWithoutSelection
					},
					0);
			int pressedButton = dialog.open();
			if (pressedButton == -1)
				return;
			if (pressedButton == 0)
				new DeleteDomainAction(domainArray, false).run();
			if (pressedButton == 1)
				new DeleteDomainAction(domainArray, true, true).run();
			if (pressedButton == 2)
				new DeleteDomainAction(domainArray, true, false).run();

		}
	}
	
	protected Action createChangeProfileAction(Domain[] domainArray) {
		return new ChangeProfileAction(domainArray);
	}

	private class ChangeProfileAction extends ServerAction {
		
		private Domain[] domainArray;
		
		public ChangeProfileAction(Domain[] domainArray) {
			super(project);
			setText(Messages.PolicyActions_ChangeProfileActionTitle);
			setImageDescriptor(Activator.getImageDescriptor("profile.gif")); //$NON-NLS-1$
			this.domainArray = domainArray;

			if (domainArray != null && domainArray.length == 1) {
				setEnabled(true);
			} else {
				setEnabled(false);
			}
		}

		public void run() {
			ChangeProfileDialog dialog = new ChangeProfileDialog(Activator.getStandardDisplay().getActiveShell(), project, domainArray);
	    	dialog.open();
		}

	}
	
	protected Action createKeepDomainAction(Domain[] domainArray) {
		return new KeepDomainAction(domainArray);
	}

	private class KeepDomainAction extends ServerAction {
		
		private Domain[] domainArray;
		
		public KeepDomainAction(Domain[] domainArray) {
			super(project);
			setText(Messages.PolicyActions_KeepDomainActionTitle);
			setImageDescriptor(Activator.getImageDescriptor("keep_domain.gif")); //$NON-NLS-1$
			this.domainArray = domainArray;

			if (domainArray != null && domainArray.length == 1) {
				setEnabled(true);
			} else {
				setEnabled(false);
			}
		}

		public void run() {
			Domain domain = domainArray[0];
			ICommand command = CommandManager.getInstance().write(project, getLocalResource(), getInputText(domain));
			if (command.isSuccessful()) {
				if (ProcResourceManager.getInstance().synchronize(project, getLocalResource())) {
					if (checkKeepDomain(domain)) {
						TomoyoCorePlugin.getDefault().setStatusMessage(Messages.PolicyActions_KeepDomainOKMessage + " : " + domain.getText()); //$NON-NLS-2$ //$NON-NLS-1$
					} else {
						TomoyoCorePlugin.getDefault().setStatusErrorMessage(Messages.PolicyActions_KeepDomainNGMessage + " : " + domain.getText()); //$NON-NLS-2$ //$NON-NLS-1$
					}
				} else {
					TomoyoCorePlugin.getDefault().setStatusErrorMessage(Messages.PolicyOutlineMenuListener_SynchronizedErrorMessage);
				}
			} else {
				TomoyoCorePlugin.getDefault().setStatusErrorMessage(Messages.PolicyActions_KeepDomainNGMessage + " : " + domain.getText()); //$NON-NLS-2$ //$NON-NLS-1$
			}
		}

		private boolean checkKeepDomain(Domain domain) {
			return domain.isKeepDomain();
		}

		private LocalResource getLocalResource() {
			return ProcResourceManager.getInstance().getProcResourceSet(project).getExceptPolicy();
		}

		public String getInputText(Domain domain) {
			StringBuffer input = new StringBuffer();
			input.append("keep_domain "); //$NON-NLS-1$
			input.append(" "); //$NON-NLS-1$
			input.append(domain.getText());
			input.append("\n"); //$NON-NLS-1$
			return input.toString();
		}

	}
	
	protected Action createInializerAction(Domain[] domainArray) {
		return new InializerAction(domainArray);
	}

	private class InializerAction extends ServerAction {
		
		private Domain[] domainArray;
		
		public InializerAction(Domain[] domainArray) {
			super(project);
			setText(Messages.PolicyActions_InitializeDomainAction);
			setImageDescriptor(Activator.getImageDescriptor("initialize_domain.gif")); //$NON-NLS-1$
			this.domainArray = domainArray;

			if (domainArray != null && domainArray.length == 1) {
				setEnabled(true);
			} else {
				setEnabled(false);
			}
		}

		public void run() {
			Domain domain = domainArray[0];
			String processName = domain.getProcessName();
			ICommand command = CommandManager.getInstance().write(project, getLocalResource(), getInputText(domain));
			if (command.isSuccessful()) {
				if (ProcResourceManager.getInstance().synchronize(project, getLocalResource())) {
					if (checkInitializer(domain)) {
						TomoyoCorePlugin.getDefault().setStatusMessage(Messages.PolicyActions_InitializeDomainActionOKMessage + " : " + processName); //$NON-NLS-2$
					} else {
						TomoyoCorePlugin.getDefault().setStatusErrorMessage(Messages.PolicyActions_InitializeDomainActionNGMessage + " : " + processName); //$NON-NLS-2$
					}
				} else {
					TomoyoCorePlugin.getDefault().setStatusErrorMessage(Messages.PolicyOutlineMenuListener_SynchronizedErrorMessage);
				}
			} else {
				TomoyoCorePlugin.getDefault().setStatusErrorMessage(Messages.PolicyActions_InitializeDomainActionNGMessage + " : " + processName); //$NON-NLS-2$
			}
		}

		private boolean checkInitializer(Domain domain) {
			return domain.isInitializerSource();
		}

		private LocalResource getLocalResource() {
			return ProcResourceManager.getInstance().getProcResourceSet(project).getExceptPolicy();
		}

		public String getInputText(Domain domain) {
			StringBuffer input = new StringBuffer();
			input.append("initialize_domain"); //$NON-NLS-1$
			input.append(" "); //$NON-NLS-1$
			input.append(domain.getProcessName());
			input.append("\n"); //$NON-NLS-1$
			return input.toString();
		}

	}
	
	protected Action createCreateAccessPermissionAction(Domain domain, String accessPermissionText) {
		return new CreateAccessPermissionAction(domain, accessPermissionText);
	}

	private class CreateAccessPermissionAction extends ServerAction {
		
		private Domain domain;
		private String accessPermissionText;
		
		public CreateAccessPermissionAction(Domain domain, String accessPermissionText) {
			super(project);
			setText(Messages.PolicyActions_CreateNewAccessPermissionActionTitle);
			setImageDescriptor(Activator.getImageDescriptor("resource_new.gif")); //$NON-NLS-1$
			this.domain = domain;
			this.accessPermissionText = accessPermissionText;
			this.setEnabled(domain != null);
		}

		public void run() {
			NewAccessPermissionDialog dialog = new NewAccessPermissionDialog(
					Activator.getStandardDisplay().getActiveShell(), project, domain, accessPermissionText);
	    	dialog.open();
		}

	}

	protected Action createCreateDefinitionAction(LocalResource localResource, String directive, String context) {
		return new CreateDefinitionAction(localResource, directive, context);
	}

	private class CreateDefinitionAction extends ServerAction {
		
		private LocalResource localResource;
		private String directive;
		private String context;
		
		public CreateDefinitionAction(LocalResource localResource, String directive, String context) {
			super(project);
			setText(Messages.PolicyActions_CreateDefitionActionTitle);
			setImageDescriptor(Activator.getImageDescriptor("resource_new.gif")); //$NON-NLS-1$
			this.localResource = localResource;
			this.directive = directive;
			this.context = context;
		}

		public void run() {
			NewDefinitionDialog dialog = null;
			dialog = new NewDefinitionDialog(Activator.getStandardDisplay().getActiveShell(), project, localResource, directive, context);
			dialog.open();
		}

	}
	

}
