open OUnit
open Ast

let parse_file file_name = 
  let lexbuf = Lexing.from_channel (open_in ("../tests/parser/" ^ file_name)) in
    ParseRuby.rb_dump LexRuby.token lexbuf

let parse_fixnum _ = 
  assert_equal
    (NewlineExpr (("fixnum.rb",1),(IntExpr 1)))
    (parse_file "fixnum.nd")

let parse_symbol _ =
  assert_equal
    (NewlineExpr (("symbol.rb",1),(SymbolExpr 10121)))
    (parse_file "symbol.nd")

let parse_float _ = 
  assert_equal 
    (NewlineExpr (("float_lit.rb",1), (FloatExpr 1.0)))
    (parse_file "float_lit.nd")

let parse_array _ = 
  assert_equal
    (NewlineExpr (("array.rb",1),ArrayExpr [IntExpr 1; IntExpr 2; IntExpr 3]))
    (parse_file "array.nd")

let parse_call _ =
  assert_equal
    (NewlineExpr (("call.rb",1),(CallExpr ((IntExpr 1),(4345,"to_str"),[]))))
    (parse_file "call.nd");
  assert_equal 
    (NewlineExpr (("call2.rb",1),
		  (CallExpr (VarExpr((7659,"$stdout"),GlobalVar),
			     (7257,"puts"),
			     [(StringExpr "hello world")]))))
    (parse_file "call2.nd")

let parse_string _ = 
  assert_equal 
    (NewlineExpr (("string.rb",1),(StringExpr "hello ruby's string")))
    (parse_file "string.nd")
      
let parse_hash _ =
  assert_equal 
    (NewlineExpr (("hash.rb",1),(HashExpr [
				   ((StringExpr "one"), (IntExpr 1));
				   ((StringExpr "two"), (IntExpr 2));
				   ((StringExpr "three"), (IntExpr 3))
				 ])))
    (parse_file "hash.nd")

let parse_block _ =
  assert_equal
    (BlockExpr [
       (NewlineExpr (("block.rb",1),
		     (CallExpr ((IntExpr 1), (3161,"to_s"), []))));
       (NewlineExpr (("block.rb",2),
		     (CallExpr ((StringExpr "STRING"), (4257,"size"), []))));
       (NewlineExpr (("block.rb",3),
		     (CallExpr ((VarExpr ((7651,"$stdin"),GlobalVar)), (7257,"puts"), [(StringExpr "hoge")]))));
     ])
    (parse_file "block.nd")
	
let parse_var _ =
  assert_equal
    (BlockExpr [
       (NewlineExpr (("var.rb",1),
		     (AssignExpr ((2,"hoge"), LocalVar, (IntExpr 123)))));
       (NewlineExpr (("var.rb",2),
		     (AssignExpr ((10130,"@hoge"), InstanceVar, (IntExpr 123)))));
       (NewlineExpr (("var.rb",3),
		     (AssignExpr ((10139,"$hoge"), GlobalVar, (IntExpr 123)))));
       (NewlineExpr (("var.rb",5),
		     (VarExpr ((2,"hoge"), LocalVar))));
       (NewlineExpr (("var.rb",6),
		     (VarExpr ((10130,"@hoge"), InstanceVar))));
       (NewlineExpr (("var.rb",7),
		     (VarExpr ((10139,"$hoge"), GlobalVar))));
     ])
    (parse_file "var.nd")

let parse_if _ = 
  assert_equal 
    (NewlineExpr (("if.rb",1),
		  (IfExpr (
		     (CallExpr ((IntExpr 1), (321,"=="), [(IntExpr 2)])),
		     (NewlineExpr (("if.rb",2),
				   (IntExpr 3))),
		     (NewlineExpr (("if.rb",4),
				   (IntExpr 4)))
		   ))))
    (parse_file "if.nd");
  assert_equal
    (NewlineExpr (("if2.rb",1),
		  (IfExpr (
		     (IntExpr 1),
		     (NewlineExpr (("if2.rb",2),
				   (IntExpr 2))),
		     (NilExpr)))))
    (parse_file "if2.nd")

let parse_vcall _ =
  assert_equal
    (NewlineExpr (("vcall.rb",1),
		  (VCallExpr "self.a")))
    (parse_file "vcall.nd")

let parse_fcall _ =
  assert_equal
    (BlockExpr [
       (NewlineExpr (("fcall.rb",1),
		     (FCallExpr ((7257,"puts"), [(StringExpr "hello")]))));
       (NewlineExpr (("fcall.rb",2),
		     (FCallExpr ((7265,"gets"), []))));
     ])
    (parse_file "fcall.nd")

let parse_return _ =
  assert_equal 
    (NewlineExpr (("return.rb",1),
		  (ReturnExpr (IntExpr 123))))
    (parse_file "return.nd")

let parse_def _ = 
  assert_equal 
    (BlockExpr [
       (NewlineExpr (("def.rb",1),
		     (DefExpr (
			(10121,"plus"), 
			2, 
			(NewlineExpr (("def.rb",2),
				      (ReturnExpr 
					 (CallExpr (
					    (VarExpr ((2, "a"), LocalVar)),
					    (43,"+"),
					    [(VarExpr ((3,"b"), LocalVar))])))))))));
       (NewlineExpr (("def.rb",5),
		     (DefExpr (
			(10145, "one"),
			0,
			(NewlineExpr (("def.rb",6),
				      (ReturnExpr (IntExpr 1))))))));
     ])
    (parse_file "def.nd")

let parse_class _ =
  assert_equal 
    (NewlineExpr (("class.rb",1),
		  (ClassExpr ((10125, "C"), []))))
    (parse_file "class.nd");
  assert_equal
    (NewlineExpr (("class2.rb",1),
		  (ClassExpr ((10125,"Class1"), [
				(NewlineExpr (("class2.rb",2),
					      (DefExpr (
						 (10129, "hoge"),
						 0,
						 (NewlineExpr (("class2.rb",3),
							       (IntExpr 123)))))))
			      ]))))
    (parse_file "class2.nd");
  assert_equal
    (NewlineExpr (("class3.rb",1),
		  (ClassExpr (
		     (10125,"Class3"),
		     [
		       (NewlineExpr (("class3.rb",2),
				     (DefExpr (
					(10129, "hoge"),
					0,
					(NewlineExpr (("class3.rb",3),(IntExpr 123)))))));
		       (NewlineExpr (("class3.rb",5),
				     (DefExpr (
					(10137, "huga"),
					0,
					(NewlineExpr (("class3.rb",6),(StringExpr "huga")))))));
		     ]))))
    (parse_file "class3.nd");
  assert_equal
    (NewlineExpr (("class4.rb",1),
		  (ClassExpr ((10125,"Class4"), 
			      [
				(NewlineExpr (("class4.rb",2),(VarExpr ((10130, "@ivar"), InstanceVar))));
				(NewlineExpr (("class4.rb",3),
					      (DefExpr (
						 (10137,"hoge"),
						 0,
						 (NewlineExpr (("class4.rb",4),
							       (ReturnExpr (VarExpr ((10130, "@ivar"), InstanceVar)))))))));
			      ]))))
    (parse_file "class4.nd")

let parse_const _ = 
  assert_equal
    (BlockExpr [
       (NewlineExpr (("const.rb",1),
		     (CdeclExpr ((10125, "C"), (IntExpr 10)))));
       (NewlineExpr (("const.rb",2),
		     (ConstExpr (10125, "C"))));
     ])
    (parse_file "const.nd")
    

let suite = "parser suite" >:::
	      [
		"Fixnum:" >:: parse_fixnum;
		"Symbol:" >:: parse_symbol;
		"Float:" >:: parse_float;
		"Array:" >:: parse_array;
		"NODE_CALL:" >:: parse_call;
		"String:" >:: parse_string;
		"Hash:" >:: parse_hash;
		"Block:" >:: parse_block;
		"variable" >:: parse_var;
		"if:" >:: parse_if;
		"vcall" >:: parse_vcall;
		"FCALL" >:: parse_fcall;
		"return" >:: parse_return;
		"def" >:: parse_def;
		"class" >:: parse_class;
		"const" >:: parse_const;
	      ]
