open OUnit

let parse_file file_name = 
  let lexbuf = Lexing.from_channel (open_in ("../tests/ti/" ^ file_name)) in
    ParseRuby.rb_dump LexRuby.token lexbuf
      
let assert_equal_v =
  let c (v1,t1) (v2,t2) = (Ti.Env.equal v1 v2) && (t1 = t2) in
  let p (v,t) = "\n" ^ Ti.Env.to_string v ^ "\n" ^ Type.to_string t ^ "\n" in
    assert_equal ~cmp:c ~printer:p

let test_int _ =
  let _ = Type.init() in
  let expr = parse_file "int.nd" in
    begin
      assert_equal_v ((Ti.Env.empty), (Type.base "Integer")) (Ti.ti expr);
    end

let test_float _ =
  let _ = Type.init() in
  let expr = parse_file "float.nd" in
    begin
      assert_equal_v ((Ti.Env.empty), (Type.base "Float")) (Ti.ti expr);
    end

let test_string _ =
  let _ = Type.init() in
  let expr = parse_file "string.nd" in
    begin
      assert_equal_v ((Ti.Env.empty), (Type.base "String")) (Ti.ti expr);
    end

let test_nil _ =
  let _ = Type.init() in
  let expr = parse_file "nil.nd" in
    begin
      assert_equal_v (Ti.Env.empty, Type.base "NilClass") (Ti.ti expr);
    end

let test_assign _ =
  let _ = Type.init() in
  let expr = parse_file "var_assign.nd" in
    begin
      assert_equal_v (Ti.Env.empty, Type.base "Integer") (Ti.ti expr);
    end

let test_block _ =
  let _ = Type.init() in
  let expr = parse_file "block.nd" in
    begin
      assert_equal_v (Ti.Env.empty, Type.base "Integer") (Ti.ti expr);
    end

let test_block2 _ =
  let _ = Type.init() in
  let expr = parse_file "block2.nd" in
    begin
      assert_equal_v (Ti.Env.empty, Type.base "Integer") (Ti.ti expr);
    end

let test_block3 _ =
  let _ = Type.init() in
  let expr = parse_file "block3.nd" in
    begin
      assert_equal_v (Ti.Env.empty, Type.base "String") (Ti.ti expr);
    end

let test_call _ =
  let _ = Type.init() in
  let expr = parse_file "call.nd" in
    begin
      assert_equal_v
	(((SigDB.of_list [((Type.var 3), (Sig.create [("to_s", (Sig.Entry.create [] (Type.var 2)))]));
			  ((Type.var 9), (Sig.create [("foo", (Sig.Entry.create [(Type.var 5); (Type.var 6); (Type.var 7)] (Type.var 8)))]))]),
	  (TypeEnv.empty),
	  (Cstrs.of_list [(Cstrs.Subtype (Type.base "Integer", Type.var 3));
				(Cstrs.Subtype (Type.base "Integer", Type.var 5));
				(Cstrs.Subtype (Type.base "Integer", Type.var 6));
				(Cstrs.Subtype (Type.base "Integer", Type.var 7));
				(Cstrs.Subtype (Type.base "Integer", Type.var 9))]),
	  (PMap.empty)),
	 (Type.var 8))
	(Ti.ti expr);
    end

let test_call2 _ =
  let _ = Type.init() in
  let expr = parse_file "call2.nd" in
    begin
      assert_equal_v
	(((SigDB.of_list [(Type.var 4), (Sig.create [("foo", (Sig.Entry.create [Type.var 1; Type.var 2] (Type.var 3)))])]),
	  (TypeEnv.empty),
	  (Cstrs.of_list [(Cstrs.Subtype (Type.base "Integer", Type.var 4));
				(Cstrs.Subtype (Type.base "String", Type.var 1));
				(Cstrs.Subtype (Type.base "Integer", Type.var 2))]),
	  (PMap.empty)),
	 (Type.var 3))
	(Ti.ti expr);
    end

let test_def0 _ =
  let _ = Type.init() in
  let expr = parse_file "def0.nd" in
    begin
      assert_equal_v 
	(((SigDB.of_list [(Type.var 0), (Sig.create [("foo", (Sig.Entry.create [] (Type.base "Integer")))])]),
	  (TypeEnv.empty),
	  (Cstrs.empty),
	  (PMap.empty)),
	 Type.base "NilClass")
	(Ti.ti expr);
    end

let test_def1 _ =
  let _ = Type.init() in
  let expr = parse_file "def1.nd" in
    begin
      assert_equal_v
	(((SigDB.of_list [(Type.var 0), (Sig.create [("foo1", (Sig.Entry.create [Type.var 1] (Type.var 1)))])]),
	  (TypeEnv.empty),
	  (Cstrs.empty),
	  (PMap.empty)),
	 Type.base "NilClass")
	(Ti.ti expr);
    end

let test_def2 _ = 
  let _ = Type.init() in
  let expr = parse_file "def2.nd" in
    begin
      assert_equal_v
	(((SigDB.of_list [((Type.var 0), (Sig.create [("foo2", (Sig.Entry.create [Type.var 1; Type.var 2] (Type.var 4)))]));
			  ((Type.var 5), (Sig.create [("+", (Sig.Entry.create [Type.var 3] (Type.var 4)))]))]),
	  (TypeEnv.empty),
	  (Cstrs.of_list [(Cstrs.Subtype (Type.var 2, Type.var 3));
			  (Cstrs.Subtype (Type.var 1, Type.var 5))]),
	  (PMap.empty)),
	 Type.base "NilClass")
	(Ti.ti expr);
    end

let test_def3 _ =
  let _ = Type.init() in
  let expr = parse_file "def3.nd" in
    begin
      assert_equal_v (Ti.Env.empty, Type.base "NilClass") (Ti.ti expr);
    end

let test_fcall _ = 
  let _ = Type.init() in
  let expr = parse_file "fcall.nd" in
    begin
      assert_equal_v 
	(((SigDB.of_list [((Type.var 4), (Sig.create [("puts", (Sig.Entry.create [Type.var 2] (Type.var 3)))]))]),
	  (TypeEnv.empty),
	  (Cstrs.of_list [(Cstrs.Subtype (Type.var 0, Type.var 4));
				(Cstrs.Subtype (Type.base "String", Type.var 2))]),
	  (PMap.empty)),
	 (Type.var 3))
	(Ti.ti expr);
    end

let test_class _ =
  let _ = Type.init() in
  let expr = parse_file "class.nd" in
    begin
      assert_equal_v
	(((SigDB.of_list [(Type.base "_Hoge", (Sig.create [("new", (Sig.Entry.create [] (Type.base "Hoge")))]));
			  (Type.base "Hoge", (Sig.create [("foo", (Sig.Entry.create [] (Type.base "Integer")));
							    ("bar", (Sig.Entry.create [Type.var 1] (Type.var 1)))]))]),
	  (TypeEnv.of_list [("Hoge", Type.base "_Hoge")]),
	  (Cstrs.empty), PMap.empty),
	 (Type.base "NilClass"))
	(Ti.ti expr);
    end


let suite = "Test TI module" >:::
	      [
		"test_int" >:: test_int;
		"test_float" >:: test_float;
		"test_string" >:: test_string;
		"test_nil" >:: test_nil;
		"test_assign" >:: test_assign;
		"test_block" >:: test_block;
		"test_block2" >:: test_block2;
		(* "test_block3" >:: test_block3; *)   (* a = b = c is too complex to let TI algorithm support *)
		"test_call" >:: test_call;
		"test_call2" >:: test_call2;
		"test_def0" >:: test_def0;
		"test_def1" >:: test_def1;
		"test_def2" >:: test_def2;
		(* "test_def3" >:: test_def3; *) (* Too complex to write a test *)
		"test_fcall" >:: test_fcall;
		"test_class" >:: test_class;
	      ]
