%{
open Ast

let list_of_array = function
  | ArrayExpr(l) -> l
  | _ -> failwith "Invalid Array"

let unquote_string s = String.sub s 1 (String.length s - 2)

let block_of_list es = match (List.length es) with
    1 -> (List.hd es)
  | _ -> (BlockExpr es)

let list_of_block = function
  | BlockExpr(b) -> b
  | e -> [e]
%}

%token NODEDUMP

%token<int> INT
%token<string> STRING
%token<string> RSTRING
%token<float> FLOAT

%token NODE_NEWLINE
%token NODE_ARRAY
%token NODE_BLOCK
%token NODE_CALL
%token NODE_FCALL
%token NODE_HASH
%token NODE_IF
%token NODE_LIT
%token NODE_LASGN
%token NODE_LVAR
%token NODE_GASGN
%token NODE_GVAR
%token NODE_IASGN
%token NODE_IVAR
%token NODE_STR
%token NODE_VCALL
%token NODE_RETURN
%token NODE_SCOPE
%token NODE_DEFN
%token NODE_ARGS
%token NODE_CLASS
%token NODE_CONST
%token NODE_CDECL
%token NODE_NIL

%token NODE_Fixnum
%token NODE_Symbol
%token NODE_end
%token NODE_Parameters
%token NODE_Receiver
%token NODE_then
%token NODE_else
%token NODE_hash_arrow

%token LPAREN RPAREN
%token LPAREN1 RPAREN1
%token EOF

%start rb_dump
%type <Ast.expr> rb_dump

%%
rb_dump:
  NODEDUMP rb_expr EOF { $2 }
;
rb_expr_list:
    /* empty */             { [] }
  | rb_expr rb_expr_list    { $1 :: $2 }
;
rb_expr:
    rb_newline                                  { $1 }
  | NODE_SCOPE rb_expr                          { $2 }
  | rb_literal                                  { $1 }
  | rb_string                                   { $1 }
  | rb_array                                    { $1 }
  | rb_hash                                     { $1 }
  | rb_var                                      { $1 }
  | rb_vcall                                    { $1 }
  | rb_assign                                   { $1 }
  | rb_const                                    { $1 }
  | rb_cdecl                                    { $1 }
  | rb_call                                     { $1 }
  | rb_fcall                                    { $1 }
  | rb_block                                    { $1 }
  | rb_if                                       { $1 }
  | rb_return                                   { $1 }
  | rb_def                                      { $1 }
  | rb_class                                    { $1 }
  | rb_nil                                      { $1 }
;
rb_block:
  NODE_BLOCK rb_expr_list NODE_end            { BlockExpr($2) }
;
rb_string:
  NODE_STR RSTRING                   { StringExpr(unquote_string $2) }
;
rb_nil:
  NODE_NIL                                { NilExpr }
;
rb_literal:
    NODE_LIT NODE_Fixnum INT              { IntExpr($3) }
  | NODE_LIT NODE_Symbol INT paren_name   { SymbolExpr($3) }
  | NODE_LIT INT                          { IntExpr($2) }
  | NODE_LIT FLOAT                        { FloatExpr($2) }
;
rb_newline:
    NODE_NEWLINE file_line rb_expr        { NewlineExpr($2,$3) }
file_line:
  LPAREN1 STRING INT RPAREN1 { ($2,$3) }
;
paren_name:
  LPAREN STRING RPAREN { $2 }
;
rb_array:
  NODE_ARRAY STRING STRING INT rb_expr_list NODE_end  { ArrayExpr($5) }  
;
rb_call:
  NODE_CALL STRING STRING INT paren_name receiver parameters { CallExpr($6, ($4,$5),  $7) }
;
rb_fcall:
  NODE_FCALL STRING STRING INT paren_name parameters { FCallExpr(($4,$5), $6) }
;
receiver:
  NODE_Receiver rb_expr     { $2 }
;
parameters:
   NODE_Parameters NODE_end                 { [] } 
 | NODE_Parameters rb_array NODE_end   { list_of_array $2 }
;
rb_hash:
  NODE_HASH hash_elem_list NODE_end      { HashExpr($2) }
;
hash_elem_list:
    /* empty */                 { [] }
  | hash_elem hash_elem_list    { $1 :: $2 }
;
hash_elem:
  rb_expr NODE_hash_arrow rb_expr            { ($1,$3) }
;
rb_var:
    NODE_LVAR STRING INT paren_name          { VarExpr(($3, $4), LocalVar) }
  | NODE_IVAR STRING STRING INT paren_name   { VarExpr(($4, $5), InstanceVar) }
  | NODE_GVAR STRING INT paren_name          { VarExpr(($3, $4), GlobalVar) }
;
rb_vcall:
  NODE_VCALL STRING   { VCallExpr($2) }
;
rb_assign:
    NODE_LASGN rb_expr STRING STRING STRING INT paren_name { AssignExpr(($6, $7), LocalVar, $2) }
  | NODE_IASGN rb_expr STRING STRING STRING INT paren_name { AssignExpr(($6, $7), InstanceVar, $2) }
  | NODE_GASGN rb_expr STRING STRING STRING INT paren_name { AssignExpr(($6, $7), GlobalVar, $2) }
;
rb_if:
    NODE_IF rb_expr NODE_then rb_expr NODE_else NODE_end          { IfExpr($2,$4, NilExpr) }
  | NODE_IF rb_expr NODE_then rb_expr NODE_else rb_expr NODE_end  { IfExpr($2,$4,$6) }
;
rb_return:
  NODE_RETURN rb_expr      { ReturnExpr($2) }
;
rb_def:
    NODE_DEFN STRING INT paren_name NODE_SCOPE def_args    { DefExpr(($3,$4), $6, NilExpr) } /* no method body */
  | NODE_DEFN STRING INT paren_name NODE_SCOPE 
      NODE_BLOCK def_args rb_expr_list NODE_end            { DefExpr(($3,$4), $7, (block_of_list $8)) }
;
def_args:
  NODE_ARGS STRING STRING INT STRING STRING STRING   { $4 }
;
rb_class:
    NODE_CLASS STRING INT paren_name NODE_SCOPE NODE_end         { ClassExpr (($3,$4),[]) }
  | NODE_CLASS STRING INT paren_name NODE_SCOPE rb_expr NODE_end { ClassExpr (($3,$4), (list_of_block $6)) }
;
rb_const:
  NODE_CONST INT paren_name { ConstExpr ($2,$3) }
;
rb_cdecl:
  NODE_CDECL STRING INT paren_name rb_expr { CdeclExpr( ($3,$4), $5) }
;
