/*
 * ObjectInspectorTest class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.tester.function;

import ts.tester.function.coverage.FunctionTester;
import ts.tester.function.print.*;
import java.lang.reflect.*;
import java.util.*;

/**
 * <code>ObjectInspector</code>NX̋@\NXB
 *
 * @author  V. 
 * @version $Revision: 1.2 $, $Date: 2007/02/16 16:12:50 $
 */
public class ObjectInspectorTest extends FunctionTester
{
  public static void main(String[] args)
  {
    try {
      PrinterGroup group = new PrinterGroup();
      group.addPrinter(new ConsolePrinter());
      group.addPrinter(new HtmlPrinter("SATOH Takayuki"));
      setPrinter(group);

      run(ObjectInspectorTest.class, (args.length == 0) ? null : args[0]);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /* -- test case -- */

  public void constructor()
  {
    MSG("ʏ̑̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    NOTNULL(oi);
  }

  public void constructor_1()
  {
    MSG("k̏ꍇ");
    try {
      ObjectInspector oi = new ObjectInspector(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void expect()
  {
    MSG("ʏ̑̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    oi.expect("abc", Boolean.TRUE);
    oi.expect("def", new Integer(1234));
    oi.expect("ghi", "");

    try {
      EQUAL(oi.getExpected("abc"), Boolean.TRUE);
      EQUAL(oi.getExpected("def"), new Integer(1234));
      EQUAL(oi.getExpected("ghi"), "");
    } catch (Exception e) {
      NG(e);
    }
  }
  
  public void expect_1()
  {
    MSG("k̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    try {
      oi.expect(null, "vvv");
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void expect_2()
  {
    MSG("lk̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    try {
      oi.expect("abc", null);
      OK();
    } catch (Exception e) {
      NG(e);
    }
  }

  public void getExpected()
  {
    MSG("k̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    try {
      oi.getExpected(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void getExpected_1()
  {
    MSG("o^ĂȂw肵ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    try {
      oi.getExpected("abc");
      NG();
    } catch (NoSuchFieldException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void expect_String_boolean()
  {
    MSG("ʏ̑̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    oi.expect("abc", true);
    oi.expect("def", false);
    try {
      EQUAL(oi.getExpected("abc"), Boolean.TRUE);
      EQUAL(oi.getExpected("def"), Boolean.FALSE);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void expect_String_byte()
  {
    MSG("ʏ̑̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    oi.expect("abc", (byte) 0x00);
    oi.expect("def", (byte) 0xff);
    try {
      EQUAL(oi.getExpected("abc"), new Byte((byte)0x00));
      EQUAL(oi.getExpected("def"), new Byte((byte)0xff));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void expect_String_char()
  {
    MSG("ʏ̑̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    oi.expect("abc", 'a');
    oi.expect("def", '');
    try {
      EQUAL(oi.getExpected("abc"), new Character('a'));
      EQUAL(oi.getExpected("def"), new Character(''));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void expect_String_short()
  {
    MSG("ʏ̑̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    oi.expect("abc", (short)0);
    oi.expect("def", (short)127);
    oi.expect("ghi", (short)-128);
    try {
      EQUAL(oi.getExpected("abc"), new Short((short)0));
      EQUAL(oi.getExpected("def"), new Short((short)127));
      EQUAL(oi.getExpected("ghi"), new Short((short)-128));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void expect_String_int()
  {
    MSG("ʏ̑̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    oi.expect("abc", 0);
    oi.expect("def", Integer.MAX_VALUE);
    oi.expect("ghi", Integer.MIN_VALUE);
    try {
      EQUAL(oi.getExpected("abc"), new Integer(0));
      EQUAL(oi.getExpected("def"), new Integer(Integer.MAX_VALUE));
      EQUAL(oi.getExpected("ghi"), new Integer(Integer.MIN_VALUE));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void expect_String_long()
  {
    MSG("ʏ̑̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    oi.expect("abc", 0L);
    oi.expect("def", Long.MAX_VALUE);
    oi.expect("ghi", Long.MIN_VALUE);
    try {
      EQUAL(oi.getExpected("abc"), new Long(0L));
      EQUAL(oi.getExpected("def"), new Long(Long.MAX_VALUE));
      EQUAL(oi.getExpected("ghi"), new Long(Long.MIN_VALUE));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void expect_String_float()
  {
    MSG("ʏ̑̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    oi.expect("abc", 0.0f);
    oi.expect("def", Float.MAX_VALUE);
    oi.expect("ghi", Float.MIN_VALUE);
    try {
      EQUAL(oi.getExpected("abc"), new Float(0.0f));
      EQUAL(oi.getExpected("def"), new Float(Float.MAX_VALUE));
      EQUAL(oi.getExpected("ghi"), new Float(Float.MIN_VALUE));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void expect_String_float_1()
  {
    MSG("Ғlɖyє񐔒lݒ肵ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    oi.expect("abc", Float.NEGATIVE_INFINITY);
    oi.expect("def", Float.POSITIVE_INFINITY);
    oi.expect("ghi", Float.NaN);
    try {
      EQUAL(oi.getExpected("abc"), new Float(Float.NEGATIVE_INFINITY));
      EQUAL(oi.getExpected("def"), new Float(Float.POSITIVE_INFINITY));
      EQUAL(oi.getExpected("ghi"), new Float(Float.NaN));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void expect_String_double()
  {
    MSG("ʏ̑̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    oi.expect("abc", 0.0);
    oi.expect("def", Double.MAX_VALUE);
    oi.expect("ghi", Double.MIN_VALUE);
    try {
      EQUAL(oi.getExpected("abc"), new Double(0.0));
      EQUAL(oi.getExpected("def"), new Double(Double.MAX_VALUE));
      EQUAL(oi.getExpected("ghi"), new Double(Double.MIN_VALUE));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void expect_String_double_1()
  {
    MSG("Ғlɖyє񐔒lݒ肵ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    oi.expect("abc", Double.POSITIVE_INFINITY);
    oi.expect("def", Double.NEGATIVE_INFINITY);
    oi.expect("ghi", Double.NaN);
    try {
      EQUAL(oi.getExpected("abc"), new Double(Double.POSITIVE_INFINITY));
      EQUAL(oi.getExpected("def"), new Double(Double.NEGATIVE_INFINITY));
      EQUAL(oi.getExpected("ghi"), new Double(Double.NaN));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void ignore()
  {
    MSG("ʏ̑̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    oi.ignore("aaa");
    oi.ignore("bbb");
    oi.ignore("ccc");
    oi.ignore("ddd");

    Set<String> ignoreSet = new HashSet<String>();
    ignoreSet.add("aaa");
    ignoreSet.add("bbb");
    ignoreSet.add("ccc");
    ignoreSet.add("ddd");

    ObjectInspector oi2 = new ObjectInspector(this);
    oi2.expect("expectedMap_", new HashMap());
    oi2.expect("ignoredFieldSet_", ignoreSet);
    oi2.ignore("tester_");
    oi2.inspect(oi);
  }

  public void ignore_1()
  {
    MSG("k̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    try {
      oi.ignore(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void findField()
  {
    MSG("ʏ̑̏ꍇ");
    class AnObject {
      private boolean flag_ = true;
      private int code_ = 0;
      private String name_ = "";
      private Date date_ = new Date();
    }
    ObjectInspector oi = new ObjectInspector(this);
    try {
      Field field = oi.findField(AnObject.class, "flag_");
      EQUAL(field.getName(), "flag_");
      EQUAL(field.getType(), Boolean.TYPE);
    } catch (Exception e) {
      NG(e);
    }
    try {
      Field field = oi.findField(AnObject.class, "code_");
      EQUAL(field.getName(), "code_");
      EQUAL(field.getType(), Integer.TYPE);
    } catch (Exception e) {
      NG(e);
    }
    try {
      Field field = oi.findField(AnObject.class, "name_");
      EQUAL(field.getName(), "name_");
      EQUAL(field.getType(), String.class);
    } catch (Exception e) {
      NG(e);
    }
    try {
      Field field = oi.findField(AnObject.class, "date_");
      EQUAL(field.getName(), "date_");
      EQUAL(field.getType(), Date.class);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void findField_1()
  {
    MSG("k̏ꍇ");
    class AnObject {
      private boolean flag_ = true;
      private int code_ = 0;
      private String name_ = "";
      private Date date_ = new Date();
    }
    ObjectInspector oi = new ObjectInspector(this);
    try {
      oi.findField(null, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.findField(AnObject.class, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.findField(null, "code_");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void findField_2()
  {
    MSG("݂Ȃw肵ꍇ");
    class AnObject {
      private boolean flag_ = true;
      private int code_ = 0;
      private String name_ = "";
      private Date date_ = new Date();
    }
    ObjectInspector oi = new ObjectInspector(this);
    try {
      oi.findField(AnObject.class, "xxx");
      NG();
    } catch (NoSuchFieldException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.findField(AnObject.class, "flag");
      NG();
    } catch (NoSuchFieldException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.findField(AnObject.class, "lag_");
      NG();
    } catch (NoSuchFieldException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.findField(AnObject.class, "flag__");
      NG();
    } catch (NoSuchFieldException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void findField_3()
  {
    MSG("eNX̑w肵ꍇ");
    class AnObject {
      private boolean flag_ = true;
      private int code_ = 0;
      private String name_ = "";
      private Date date_ = new Date();
    }
    class AnObjectEx extends AnObject {
      private String code_ = "";
      private Float ratio_ = new Float(0.0f);
    }
    ObjectInspector oi = new ObjectInspector(this);
    try {
      Field field = oi.findField(AnObjectEx.class, "flag_");
      EQUAL(field.getName(), "flag_");
      EQUAL(field.getType(), Boolean.TYPE);
    } catch (Exception e) {
      NG(e);
    }
    try {
      Field field = oi.findField(AnObjectEx.class, "code_");
      EQUAL(field.getName(), "code_");
      EQUAL(field.getType(), String.class);
    } catch (Exception e) {
      NG(e);
    }
    try {
      Field field = oi.findField(AnObjectEx.class, "name_");
      EQUAL(field.getName(), "name_");
      EQUAL(field.getType(), String.class);
    } catch (Exception e) {
      NG(e);
    }
    try {
      Field field = oi.findField(AnObjectEx.class, "date_");
      EQUAL(field.getName(), "date_");
      EQUAL(field.getType(), Date.class);
    } catch (Exception e) {
      NG(e);
    }
    try {
      Field field = oi.findField(AnObjectEx.class, "ratio_");
      EQUAL(field.getName(), "ratio_");
      EQUAL(field.getType(), Float.class);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void isInheritance()
  {
    MSG("ʏ̑̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    TRUE(oi.isInheritance(String.class, Object.class));
    TRUE(oi.isInheritance(ArrayList.class, List.class));
    TRUE(oi.isInheritance(Object.class, Object.class));
    TRUE(oi.isInheritance(String.class, String.class));
    FALSE(oi.isInheritance(Integer.TYPE, Object.class));
    FALSE(oi.isInheritance(Map.class, Collection.class));
    FALSE(oi.isInheritance(Map.class, HashMap.class));
  }

  public void isInheritance_1()
  {
    MSG("k̏ꍇ");
    ObjectInspector oi = new ObjectInspector(this);
    try {
      oi.isInheritance(null, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
    try {
      oi.isInheritance(null, Object.class);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
    try {
      oi.isInheritance(String.class, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void inspect_object_string()
  {
    MSG("ʏ̑̏ꍇ");
    class AnObject {
      private int counter_ = 0;
      private void clearCounter() {
        counter_ = 0;
      }
      private void countUp() {
        counter_ ++;
      }
      private void countUp(int n) {
        counter_ += n;
      }
      private void setCounterValue(int v) {
        counter_ = v;
      }
      int counterValue() {
        return counter_ ;
      }
    }

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.inspect(obj, "counter_");
      NG();
    } catch (NoSuchFieldException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      oi.expect("counter_", 0);
      oi.inspect(obj, "counter_");
    } catch (Exception e) {
      NG(e);
    }
  }

  public void inspect_object_string_1()
  {
    MSG("k̏ꍇ");
    class AnObject {
      private int counter_ = 0;
      private void clearCounter() {
        counter_ = 0;
      }
      private void countUp() {
        counter_ ++;
      }
      private void countUp(int n) {
        counter_ += n;
      }
      private void setCounterValue(int v) {
        counter_ = v;
      }
      int counterValue() {
        return counter_ ;
      }
    }

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.inspect(null, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      oi.inspect(obj, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      oi.inspect(null, "counter_");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void inspect_object_string_2()
  {
    MSG("NXɑ݂Ȃw肵ꍇB");
    class AnObject {
      private int counter_ = 0;
      private void clearCounter() {
        counter_ = 0;
      }
      private void countUp() {
        counter_ ++;
      }
      private void countUp(int n) {
        counter_ += n;
      }
      private void setCounterValue(int v) {
        counter_ = v;
      }
      int counterValue() {
        return counter_ ;
      }
    }

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.expect("aaa", "bbb");
      oi.inspect(obj, "aaaa");
    } catch (NoSuchFieldException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void inspect_object_string_3()
  {
    MSG("̌^ƈႤҒlݒ肵ꍇ");
    class AnObject {
      private int counter_ = 0;
      private void clearCounter() {
        counter_ = 0;
      }
      private void countUp() {
        counter_ ++;
      }
      private void countUp(int n) {
        counter_ += n;
      }
      private void setCounterValue(int v) {
        counter_ = v;
      }
      int counterValue() {
        return counter_ ;
      }
    }

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.expect("counter_", 0.00123);
      oi.inspect(obj, "counter_");
      oi.expect("counter_", new ArrayList());
      oi.inspect(obj, "counter_");
      OK();
    } catch (Exception e) {
      NG(e);
    }
  }

  public void inspect_object_string_4()
  {
    MSG("̎擾Ɏsꍇ");

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester) {
      protected Field findField(Class cls, String fieldName) 
      throws NoSuchFieldException {
        throw new RuntimeException(fieldName);
      }
    };

    try {
      oi.expect("aaa", "");
      oi.inspect(new Object(), "aaa");
      OK();
    } catch (Exception e) {
      NG(e);
    }
  }

  public void inspect_object()
  {
    MSG("ʏ̑̏ꍇ");
    class AnObject {
      private boolean booleanValue = true;
      private byte byteValue = (byte) 0x1f;
      private char charValue = '$';
      private short shortValue = (short) 256;
      private int intValue = 123456; 
      private long longValue = 987654321L;
      private float floatValue = 1.23456f;
      private double doubleValue = 0.987654321;
      private String stringValue = "`abcdef";
      private List<String> list = new ArrayList<String>();
      AnObject() {
        list.add("Abc");
        list.add("Def");
        list.add("Ghi");
      }
    }

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.expect("booleanValue", true);
      oi.expect("byteValue", (byte) 0x1f);
      oi.expect("charValue", '$');
      oi.expect("shortValue", (short) 256);
      oi.expect("intValue", 123456);
      oi.expect("longValue", 987654321L);
      oi.expect("floatValue", 1.23456f);
      oi.expect("doubleValue", 0.987654321);
      oi.expect("stringValue", "`abcdef");
      List<String> l = new LinkedList<String>();
      l.add("Abc");
      l.add("Def");
      l.add("Ghi");
      oi.expect("list", l);

      oi.inspect(obj);
      OK();
    } catch (Exception e) {
      NG(e);
    }
  }

  public void inspect_object_1()
  {
    MSG("k̏ꍇ");
    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    try {
      oi.inspect(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void inspect_object_2()
  {
    MSG("hNX̏ꍇ");
    class AnObject {
      private boolean booleanValue = true;
      private byte byteValue = (byte) 0x1f;
      private char charValue = '$';
      private short shortValue = (short) 256;
      private int intValue = 123456; 
      private long longValue = 987654321L;
      private float floatValue = 1.23456f;
      private double doubleValue = 0.987654321;
      private String stringValue = "`abcdef";
      private List<String> list = new ArrayList<String>();
      AnObject() {
        list.add("Abc");
        list.add("Def");
        list.add("Ghi");
      }
    }
    class AnObjectEx extends AnObject {
      private long charValue = 12345L;
      private String strValue2 = "AAA";
    }

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObjectEx();
    try {
      oi.expect("booleanValue", true);
      oi.expect("byteValue", (byte) 0x1f);
      oi.expect("charValue", 12345L);
      oi.expect("shortValue", (short) 256);
      oi.expect("intValue", 123456);
      oi.expect("longValue", 987654321L);
      oi.expect("floatValue", 1.23456f);
      oi.expect("doubleValue", 0.987654321);
      oi.expect("stringValue", "`abcdef");
      oi.expect("strValue2", "AAA");
      List<String> l = new LinkedList<String>();
      l.add("Abc");
      l.add("Def");
      l.add("Ghi");
      oi.expect("list", l);

      oi.inspect(obj);
      OK();
    } catch (Exception e) {
      NG(e);
    }
  }

  public void inspect_object_3()
  {
    MSG("z^̃tB[hꍇ");
    class AnObject {
      boolean[] booleanValue = null;
      byte[] byteValue = null;
      char[] charValue = null;
      short[] shortValue = null;
      int[] intValue = null;
      long[] longValue = null;
      float[] floatValue = null;
      double[] doubleValue = null;
      String[] stringValue = null;
      Object[] objectValue = null;
      AnObject() {}
    }

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.expect("booleanValue", null);
      oi.expect("byteValue", null);
      oi.expect("charValue", null);
      oi.expect("shortValue", null);
      oi.expect("intValue", null);
      oi.expect("longValue", null);
      oi.expect("floatValue", null);
      oi.expect("doubleValue", null);
      oi.expect("stringValue", null);
      oi.expect("objectValue", null);
      oi.inspect(obj);
      OK();
    } catch (Exception e) {
      NG(e);
    }

    boolean[] bools0 = { true, false, true };
    boolean[] bools1 = { true, false, true };
    boolean[] bools2 = { false, true };
    byte[] bytes0 = { (byte)0x01, (byte)0x02, (byte)0x03 };
    byte[] bytes1 = { (byte)0x01, (byte)0x02, (byte)0x03 };
    byte[] bytes2 = { (byte)0x02, (byte)0x03 };
    char[] chars0 = { 'a', 'b' };
    char[] chars1 = { 'a', 'b' };
    char[] chars2 = { 'x', 'y' };
    short[] shorts0 = { (short) 1, (short) 2 };
    short[] shorts1 = { (short) 1, (short) 2 };
    short[] shorts2 = { (short) 1, (short) 2, (short) 3 };
    int[] ints0 = { 1, 2, 3 };
    int[] ints1 = { 1, 2, 3 };
    int[] ints2 = {};
    long[] longs0 = { 111L, 222L, 333L };
    long[] longs1 = { 111L, 222L, 333L };
    long[] longs2 = { 111L };
    float[] floats0 = { 0.1f, 0.2f };
    float[] floats1 = { 0.1f, 0.2f };
    float[] floats2 = { 0.2f, 0.1f, 0.0f };
    double[] doubles0 = { 0.0, 0.1, 0.2 };
    double[] doubles1 = { 0.0, 0.1, 0.2 };
    double[] doubles2 = { 0.1, 0.1, 0.1 };
    String[] strings0 = { "s", "ss", "sss" };
    String[] strings1 = { "s", "ss", "sss" };
    String[] strings2 = { "sss", "ss", "s" };
    Object[] objects0 = { Boolean.TRUE, "AAA" };
    Object[] objects1 = { Boolean.TRUE, "AAA" };
    Object[] objects2 = { Boolean.FALSE, "BBB" };

    try {
      oi.expect("booleanValue", bools0);
      oi.expect("byteValue", bytes0);
      oi.expect("charValue", chars0);
      oi.expect("shortValue", shorts0);
      oi.expect("intValue", ints0);
      oi.expect("longValue", longs0);
      oi.expect("floatValue", floats0);
      oi.expect("doubleValue", doubles0);
      oi.expect("stringValue", strings0);
      oi.expect("objectValue", objects0);
      oi.inspect(obj);
      OK();
    } catch (Exception e) {
      NG(e);
    }
    try {
      obj.booleanValue = bools1;
      obj.byteValue = bytes1;
      obj.charValue = chars1;
      obj.shortValue = shorts1;
      obj.intValue = ints1;
      obj.longValue = longs1;
      obj.floatValue = floats1;
      obj.doubleValue = doubles1;
      obj.stringValue = strings1;
      obj.objectValue = objects1;
      oi.expect("booleanValue", bools0);
      oi.expect("byteValue", bytes0);
      oi.expect("charValue", chars0);
      oi.expect("shortValue", shorts0);
      oi.expect("intValue", ints0);
      oi.expect("longValue", longs0);
      oi.expect("floatValue", floats0);
      oi.expect("doubleValue", doubles0);
      oi.expect("stringValue", strings0);
      oi.expect("objectValue", objects0);
      oi.inspect(obj);
      OK();
    } catch (Exception e) {
      NG(e);
    }
    try {
      obj.booleanValue = bools2;
      obj.byteValue = bytes2;
      obj.charValue = chars2;
      obj.shortValue = shorts2;
      obj.intValue = ints2;
      obj.longValue = longs2;
      obj.floatValue = floats2;
      obj.doubleValue = doubles2;
      obj.stringValue = strings2;
      obj.objectValue = objects2;
      oi.expect("booleanValue", bools0);
      oi.expect("byteValue", bytes0);
      oi.expect("charValue", chars0);
      oi.expect("shortValue", shorts0);
      oi.expect("intValue", ints0);
      oi.expect("longValue", longs0);
      oi.expect("floatValue", floats0);
      oi.expect("doubleValue", doubles0);
      oi.expect("stringValue", strings0);
      oi.expect("objectValue", objects0);
      oi.inspect(obj);
      OK();
    } catch (Exception e) {
      NG(e);
    }
  }

  public void inspect_object_4()
  {
    MSG("obt@^̌");
    class AnObject {
      StringBuffer buffer = null;
      AnObject() {}
    }
    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.expect("buffer", null);
      oi.inspect(obj);
      OK();
    } catch (Exception e) {
      NG(e);
    }

    StringBuffer buf0 = new StringBuffer("aaa");
    StringBuffer buf1 = new StringBuffer("aaa");
    StringBuffer buf2 = new StringBuffer("bbb");
    String str3 = "aaa";

    try {
      oi.expect("buffer", buf0);
      oi.inspect(obj);
    } catch (Exception e) {
      NG(e);
    }
    try {
      obj.buffer = buf1; 
      oi.expect("buffer", buf0);
      oi.inspect(obj);
    } catch (Exception e) {
      NG(e);
    }
    try {
      obj.buffer = buf2; 
      oi.expect("buffer", buf0);
      oi.inspect(obj);
      OK();
    } catch (Exception e) {
      NG(e);
    }
    try {
      obj.buffer = buf0; 
      oi.expect("buffer", str3);
      oi.inspect(obj);
      NG();
    } catch (ClassCastException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void invokeMethod_object_methodName()
  {
    MSG("ʏ̑̏ꍇ");
    class AnObject {
      private int counter_ = 0;
      private void clearCounter() {
        counter_ = 0;
      }
      private void countUp() {
        counter_ ++;
      }
      private void countUp(int n) {
        counter_ += n;
      }
      private void setCounterValue(int v) {
        counter_ = v;
      }
      private int counterValue() {
        return counter_ ;
      }
    }

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      EQUAL(oi.invokeMethod(obj, "counterValue"), 0);
      EQUAL(oi.invokeMethod(obj, "countUp"), Void.TYPE);
      EQUAL(oi.invokeMethod(obj, "counterValue"), 1);
      EQUAL(oi.invokeMethod(obj, "countUp"), Void.TYPE);
      EQUAL(oi.invokeMethod(obj, "counterValue"), 2);
      EQUAL(oi.invokeMethod(obj, "clearCounter"), Void.TYPE);
      EQUAL(obj.counterValue(), 0);
    } catch (Exception e) {
      NG(e);
    }
  }
  
  public void invokeMethod_object_methodName_1()
  {
    MSG("k̏ꍇ");
    class AnObject {
      private int counter_ = 0;
      private void clearCounter() {
        counter_ = 0;
      }
      private void countUp() {
        counter_ ++;
      }
      private void countUp(int n) {
        counter_ += n;
      }
      private void setCounterValue(int v) {
        counter_ = v;
      }
      int counterValue() {
        return counter_ ;
      }
    }

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.invokeMethod(null, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.invokeMethod(obj, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.invokeMethod(null, "countUp");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void invokeMethod_object_methodName_arg()
  {
    MSG("ʏ̑̏ꍇ");
    class AnObject {
      private int counter_ = 0;
      private void clearCounter() {
        counter_ = 0;
      }
      private void countUp() {
        counter_ ++;
      }
      private void countUp(int n) {
        counter_ += n;
      }
      private void setCounterValue(int v) {
        counter_ = v;
      }
      int counterValue() {
        return counter_ ;
      }
    }

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      Class TYPE = Integer.TYPE;
      Class VOID = Void.TYPE;
      EQUAL(oi.invokeMethod(obj, "counterValue"), new Integer(0));
      EQUAL(oi.invokeMethod(obj, "countUp", TYPE, new Integer(3)), VOID);
      EQUAL(oi.invokeMethod(obj, "counterValue"), new Integer(3));
      EQUAL(oi.invokeMethod(obj,"setCounterValue",TYPE, new Integer(5)),VOID);
      EQUAL(oi.invokeMethod(obj, "counterValue"), new Integer(5));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void invokeMethod_object_methodName_arg_1()
  {
    MSG("k̏ꍇ");
    class AnObject {
      private int counter_ = 0;
      private void clearCounter() {
        counter_ = 0;
      }
      private void countUp() {
        counter_ ++;
      }
      private void countUp(int n) {
        counter_ += n;
      }
      private void setCounterValue(int v) {
        counter_ = v;
      }
      int counterValue() {
        return counter_ ;
      }
    }
    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.invokeMethod(null, null, (Class)null, (Object)null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.invokeMethod(null, "countUp", Integer.TYPE, new Integer(1));
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.invokeMethod(obj, null, Integer.TYPE, new Integer(2));
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    } 
    try {
      oi.invokeMethod(obj, "countUp", null, new Integer(3));
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.invokeMethod(obj, "countUp", Integer.TYPE, null);
      NG();
    } catch (AssertionError e) {
      NG(e);
    } catch (Exception e) {
      OK(e);
    }
  }

  public void invokeMethod_object_methodName_argArray()
  {
    MSG("ʏ̑̏ꍇ");
    class AnObject {
      boolean booleanValue = true;
      byte byteValue = (byte) 0x1f;
      char charValue = '$';
      short shortValue = (short) 256;
      int intValue = 123456; 
      long longValue = 987654321L;
      float floatValue = 1.23456f;
      double doubleValue = 0.987654321;
      String stringValue = "`abcdef";
      void set(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = b;
        byteValue = t;
        charValue = c;
        shortValue = h;
        intValue = n;
        longValue = l;
        floatValue = f;
        doubleValue = d;
        stringValue = s;
      }
      int put(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = !b;
        byteValue = (byte)(t + t);
        charValue = (char)(c + 1);
        shortValue = (short)(h + h);
        intValue = n + n;
        longValue = l + l;
        floatValue = f + f;
        doubleValue = d + d;
        stringValue = s + s;
        return 0;
      }
    }

    Class[] types = { Boolean.TYPE, Byte.TYPE, Character.TYPE,
      Short.TYPE, Integer.TYPE, Long.TYPE, Float.TYPE,
      Double.TYPE, String.class };
    Object[] values = { Boolean.FALSE,new Byte((byte)0x01),new Character('x'),
      new Short((short)2), new Integer(4), new Long(6L), new Float(0.8f),
      new Double(0.01), "AA" };

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      EQUAL(obj.booleanValue, true);
      EQUAL(obj.byteValue, (byte) 0x1f);
      EQUAL(obj.charValue, '$');
      EQUAL(obj.shortValue, (short)256);
      EQUAL(obj.intValue, 123456);
      EQUAL(obj.longValue, 987654321L);
      EQUAL(obj.floatValue, 1.23456f);
      EQUAL(obj.doubleValue, 0.987654321);
      EQUAL(obj.stringValue, "`abcdef");

      EQUAL(oi.invokeMethod(obj, "set", types, values), Void.TYPE);
      EQUAL(obj.booleanValue, false);
      EQUAL(obj.byteValue, (byte) 0x01);
      EQUAL(obj.charValue, 'x');
      EQUAL(obj.shortValue, (short)2);
      EQUAL(obj.intValue, 4);
      EQUAL(obj.longValue, 6L);
      EQUAL(obj.floatValue, 0.8f);
      EQUAL(obj.doubleValue, 0.01);
      EQUAL(obj.stringValue, "AA");

      EQUAL(oi.invokeMethod(obj, "put", types, values), new Integer(0));
      EQUAL(obj.booleanValue, true);
      EQUAL(obj.byteValue, (byte) 0x02);
      EQUAL(obj.charValue, 'y');
      EQUAL(obj.shortValue, (short)4);
      EQUAL(obj.intValue, 8);
      EQUAL(obj.longValue, 12L);
      EQUAL(obj.floatValue, 1.6f);
      EQUAL(obj.doubleValue, 0.02);
      EQUAL(obj.stringValue, "AAAA");
    } catch (Exception e) {
      NG(e);
    }
  }

  public void invokeMethod_object_methodName_argArray_1()
  {
    MSG("k̏ꍇ");
    class AnObject {
      boolean booleanValue = true;
      byte byteValue = (byte) 0x1f;
      char charValue = '$';
      short shortValue = (short) 256;
      int intValue = 123456; 
      long longValue = 987654321L;
      float floatValue = 1.23456f;
      double doubleValue = 0.987654321;
      String stringValue = "`abcdef";
      void set(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = b;
        byteValue = t;
        charValue = c;
        shortValue = h;
        intValue = n;
        longValue = l;
        floatValue = f;
        doubleValue = d;
        stringValue = s;
      }
      int put(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = b;
        byteValue = t;
        charValue = c;
        shortValue = h;
        intValue = n;
        longValue = l;
        floatValue = f;
        doubleValue = d;
        stringValue = s;
        return 0;
      }
    }

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.invokeMethod(null, null, (Class[]) null, (Object[]) null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    Class[] types = { Boolean.TYPE, Byte.TYPE, Character.TYPE,
      Short.TYPE, Integer.TYPE, Long.TYPE, Float.TYPE,
      Double.TYPE, String.class };
    Object[] values = { Boolean.TRUE,new Byte((byte)0x01),new Character('x'),
      new Short((short)0), new Integer(0), new Long(0L), new Float(0.0f),
      new Double(0.0), "AA" };

    try {
      oi.invokeMethod(null, "set", types, values);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.invokeMethod(obj, null, types, values);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.invokeMethod(obj, "set", (Class[]) null, values);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.invokeMethod(obj, "set", types, (Object[])null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void invokeMethod_object_methodName_argArray_2()
  {
    MSG("݂Ȃ\bhw肵ꍇ");
    class AnObject {
      boolean booleanValue = true;
      byte byteValue = (byte) 0x1f;
      char charValue = '$';
      short shortValue = (short) 256;
      int intValue = 123456; 
      long longValue = 987654321L;
      float floatValue = 1.23456f;
      double doubleValue = 0.987654321;
      String stringValue = "`abcdef";
      void set(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = b;
        byteValue = t;
        charValue = c;
        shortValue = h;
        intValue = n;
        longValue = l;
        floatValue = f;
        doubleValue = d;
        stringValue = s;
      }
      int put(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = b;
        byteValue = t;
        charValue = c;
        shortValue = h;
        intValue = n;
        longValue = l;
        floatValue = f;
        doubleValue = d;
        stringValue = s;
        return 0;
      }
    }

    Class[] types = { Boolean.TYPE, Byte.TYPE, Character.TYPE,
      Short.TYPE, Integer.TYPE, Long.TYPE, Float.TYPE,
      Double.TYPE, String.class };
    Object[] values = { Boolean.TRUE,new Byte((byte)0x01),new Character('x'),
      new Short((short)0), new Integer(0), new Long(0L), new Float(0.0f),
      new Double(0.0), "AA" };

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.invokeMethod(obj, "get", types, values);
      NG();
    } catch (NoSuchMethodException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void invokeMethod_object_methodName_argArray_3()
  {
    MSG("̌^ƒl̐Ȃꍇ");
    class AnObject {
      boolean booleanValue = true;
      byte byteValue = (byte) 0x1f;
      char charValue = '$';
      short shortValue = (short) 256;
      int intValue = 123456; 
      long longValue = 987654321L;
      float floatValue = 1.23456f;
      double doubleValue = 0.987654321;
      String stringValue = "`abcdef";
      void set(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = b;
        byteValue = t;
        charValue = c;
        shortValue = h;
        intValue = n;
        longValue = l;
        floatValue = f;
        doubleValue = d;
        stringValue = s;
      }
      int put(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = b;
        byteValue = t;
        charValue = c;
        shortValue = h;
        intValue = n;
        longValue = l;
        floatValue = f;
        doubleValue = d;
        stringValue = s;
        return 0;
      }
    }

    Class[] types = { Boolean.TYPE, Byte.TYPE, Character.TYPE,
      Short.TYPE, Integer.TYPE, Long.TYPE, Float.TYPE,
      Double.TYPE, String.class };
    Class[] types1 = { Boolean.TYPE, Byte.TYPE, Character.TYPE,
      Short.TYPE, Integer.TYPE, Long.TYPE, Float.TYPE,
      Double.TYPE };
    Class[] types2 = { Boolean.TYPE, Byte.TYPE, Character.TYPE,
      Short.TYPE, Integer.TYPE, Long.TYPE, Float.TYPE,
      Double.TYPE, String.class, String.class };
    Object[] values = { Boolean.TRUE,new Byte((byte)0x01),new Character('x'),
      new Short((short)0), new Integer(0), new Long(0L), new Float(0.0f),
      new Double(0.0), "AA" };
    Object[] values1 = { Boolean.TRUE,new Byte((byte)0x01),new Character('x'),
      new Short((short)0), new Integer(0), new Long(0L), new Float(0.0f),
      new Double(0.0) };
    Object[] values2 = { Boolean.TRUE,new Byte((byte)0x01),new Character('x'),
      new Short((short)0), new Integer(0), new Long(0L), new Float(0.0f),
      new Double(0.0), "AA", "BB" };

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.invokeMethod(obj, "set", types1, values);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.invokeMethod(obj, "set", types2, values);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.invokeMethod(obj, "set", types, values1);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      oi.invokeMethod(obj, "set", types, values2);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void invokeMethod_object_methodName_argArray_4()
  {
    MSG("\bhŗOX[ꂽꍇ");
    class AnObject {
      boolean booleanValue = true;
      byte byteValue = (byte) 0x1f;
      char charValue = '$';
      short shortValue = (short) 256;
      int intValue = 123456; 
      long longValue = 987654321L;
      float floatValue = 1.23456f;
      double doubleValue = 0.987654321;
      String stringValue = "`abcdef";
      void set(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        throw new UnsupportedOperationException("xyz");
      }
    }

    Class[] types = { Boolean.TYPE, Byte.TYPE, Character.TYPE,
      Short.TYPE, Integer.TYPE, Long.TYPE, Float.TYPE,
      Double.TYPE, String.class };
    Object[] values = { Boolean.TRUE,new Byte((byte)0x01),new Character('x'),
      new Short((short)0), new Integer(0), new Long(0L), new Float(0.0f),
      new Double(0.0), "AA" };

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.invokeMethod(obj, "set", types, values);
      OK();
    } catch (UnsupportedOperationException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void invokeMethod_object_methodName_argArray_5()
  {
    MSG("όŎw肵ꍇ");
    class AnObject {
      boolean booleanValue = true;
      byte byteValue = (byte) 0x1f;
      char charValue = '$';
      short shortValue = (short) 256;
      int intValue = 123456; 
      long longValue = 987654321L;
      float floatValue = 1.23456f;
      double doubleValue = 0.987654321;
      String stringValue = "`abcdef";
      void set(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = b;
        byteValue = t;
        charValue = c;
        shortValue = h;
        intValue = n;
        longValue = l;
        floatValue = f;
        doubleValue = d;
        stringValue = s;
      }
      int put(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = !b;
        byteValue = (byte) (t + t);
        charValue = (char) (c + 1);
        shortValue = (short)(h + h);
        intValue = n + n;
        longValue = l + l;
        floatValue = f + f;
        doubleValue = d + d;
        stringValue = s + s;
        return 0;
      }
    }

    Class[] types = { Boolean.TYPE, Byte.TYPE, Character.TYPE,
      Short.TYPE, Integer.TYPE, Long.TYPE, Float.TYPE,
      Double.TYPE, String.class };
    Object[] values = { Boolean.FALSE,new Byte((byte)0x01),new Character('x'),
      new Short((short)2), new Integer(4), new Long(6L), new Float(0.8f),
      new Double(0.01), "AA" };

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      EQUAL(obj.booleanValue, true);
      EQUAL(obj.byteValue, (byte) 0x1f);
      EQUAL(obj.charValue, '$');
      EQUAL(obj.shortValue, (short)256);
      EQUAL(obj.intValue, 123456);
      EQUAL(obj.longValue, 987654321L);
      EQUAL(obj.floatValue, 1.23456f);
      EQUAL(obj.doubleValue, 0.987654321);
      EQUAL(obj.stringValue, "`abcdef");

      EQUAL(oi.invokeMethod(obj, "set", types,
        Boolean.FALSE, new Byte((byte)0x01), new Character('x'),
        new Short((short)2), new Integer(4), new Long(6L),
        new Float(0.8f), new Double(0.01), "AA"),
        Void.TYPE);
      EQUAL(obj.booleanValue, false);
      EQUAL(obj.byteValue, (byte) 0x01);
      EQUAL(obj.charValue, 'x');
      EQUAL(obj.shortValue, (short)2);
      EQUAL(obj.intValue, 4);
      EQUAL(obj.longValue, 6L);
      EQUAL(obj.floatValue, 0.8f);
      EQUAL(obj.doubleValue, 0.01);
      EQUAL(obj.stringValue, "AA");

      EQUAL(oi.invokeMethod(obj, "put", types, 
        Boolean.FALSE, new Byte((byte)0x01), new Character('x'),
        new Short((short)2), new Integer(4), new Long(6L),
        new Float(0.8f), new Double(0.01), "AA"),
        new Integer(0));
      EQUAL(obj.booleanValue, true);
      EQUAL(obj.byteValue, (byte) 0x02);
      EQUAL(obj.charValue, 'y');
      EQUAL(obj.shortValue, (short)4);
      EQUAL(obj.intValue, 8);
      EQUAL(obj.longValue, 12L);
      EQUAL(obj.floatValue, 1.6f);
      EQUAL(obj.doubleValue, 0.02);
      EQUAL(obj.stringValue, "AAAA");
    } catch (UnsupportedOperationException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void invokeMethod_object_methodName_argList()
  {
    MSG("ʏ̑̏ꍇ");
    class AnObject {
      boolean booleanValue = true;
      byte byteValue = (byte) 0x1f;
      char charValue = '$';
      short shortValue = (short) 256;
      int intValue = 123456; 
      long longValue = 987654321L;
      float floatValue = 1.23456f;
      double doubleValue = 0.987654321;
      String stringValue = "`abcdef";
      void set(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = b;
        byteValue = t;
        charValue = c;
        shortValue = h;
        intValue = n;
        longValue = l;
        floatValue = f;
        doubleValue = d;
        stringValue = s;
      }
      int put(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = !b;
        byteValue = (byte) (t + t);
        charValue = (char) (c + 1);
        shortValue = (short)(h + h);
        intValue = n + n;
        longValue = l + l;
        floatValue = f + f;
        doubleValue = d + d;
        stringValue = s + s;
        return 0;
      }
    }

    Class[] types = { Boolean.TYPE, Byte.TYPE, Character.TYPE,
      Short.TYPE, Integer.TYPE, Long.TYPE, Float.TYPE,
      Double.TYPE, String.class };

    List<Class> typL = Arrays.asList(types);

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      EQUAL(obj.booleanValue, true);
      EQUAL(obj.byteValue, (byte) 0x1f);
      EQUAL(obj.charValue, '$');
      EQUAL(obj.shortValue, (short)256);
      EQUAL(obj.intValue, 123456);
      EQUAL(obj.longValue, 987654321L);
      EQUAL(obj.floatValue, 1.23456f);
      EQUAL(obj.doubleValue, 0.987654321);
      EQUAL(obj.stringValue, "`abcdef");

      EQUAL(oi.invokeMethod(obj, "set", typL, 
        Boolean.FALSE,new Byte((byte)0x01),new Character('x'),
        new Short((short)2), new Integer(4), new Long(6L), new Float(0.8f),
        new Double(0.01), "AA"),
        Void.TYPE);
      EQUAL(obj.booleanValue, false);
      EQUAL(obj.byteValue, (byte) 0x01);
      EQUAL(obj.charValue, 'x');
      EQUAL(obj.shortValue, (short)2);
      EQUAL(obj.intValue, 4);
      EQUAL(obj.longValue, 6L);
      EQUAL(obj.floatValue, 0.8f);
      EQUAL(obj.doubleValue, 0.01);
      EQUAL(obj.stringValue, "AA");

      EQUAL(oi.invokeMethod(obj, "put", types, 
        Boolean.FALSE,new Byte((byte)0x01),new Character('x'),
        new Short((short)2), new Integer(4), new Long(6L), new Float(0.8f),
        new Double(0.01), "AA"),
        new Integer(0));
      EQUAL(obj.booleanValue, true);
      EQUAL(obj.byteValue, (byte) 0x02);
      EQUAL(obj.charValue, 'y');
      EQUAL(obj.shortValue, (short)4);
      EQUAL(obj.intValue, 8);
      EQUAL(obj.longValue, 12L);
      EQUAL(obj.floatValue, 1.6f);
      EQUAL(obj.doubleValue, 0.02);
      EQUAL(obj.stringValue, "AAAA");
    } catch (Exception e) {
      NG(e);
    }
  }

  public void invokeMethod_object_methodName_argList_1()
  {
    MSG("k̏ꍇ");
    class AnObject {
      boolean booleanValue = true;
      byte byteValue = (byte) 0x1f;
      char charValue = '$';
      short shortValue = (short) 256;
      int intValue = 123456; 
      long longValue = 987654321L;
      float floatValue = 1.23456f;
      double doubleValue = 0.987654321;
      String stringValue = "`abcdef";
      void set(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = b;
        byteValue = t;
        charValue = c;
        shortValue = h;
        intValue = n;
        longValue = l;
        floatValue = f;
        doubleValue = d;
        stringValue = s;
      }
      int put(boolean b, byte t, char c, short h, int n, long l,
      float f, double d, String s) {
        booleanValue = !b;
        byteValue = (byte) (t + t);
        charValue = (char) (c + 1);
        shortValue = (short)(h + h);
        intValue = n + n;
        longValue = l + l;
        floatValue = f + f;
        doubleValue = d + d;
        stringValue = s + s;
        return 0;
      }
    }

    Class[] types = { Boolean.TYPE, Byte.TYPE, Character.TYPE,
      Short.TYPE, Integer.TYPE, Long.TYPE, Float.TYPE,
      Double.TYPE, String.class };

    List<Class> typL = Arrays.asList(types);

    FunctionTester tester = new FunctionTester();
    tester.initTesting();
    ObjectInspector oi = new ObjectInspector(tester);
    AnObject obj = new AnObject();
    try {
      oi.invokeMethod(null, null, (List<Class>)null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      oi.invokeMethod(null, "set", typL, 
        Boolean.FALSE,new Byte((byte)0x01),new Character('x'),
        new Short((short)2), new Integer(4), new Long(6L), new Float(0.8f),
        new Double(0.01), "AA");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      oi.invokeMethod(obj, null, typL, 
        Boolean.FALSE,new Byte((byte)0x01),new Character('x'),
        new Short((short)2), new Integer(4), new Long(6L), new Float(0.8f),
        new Double(0.01), "AA");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      oi.invokeMethod(obj, "set", (List<Class>) null, 
        Boolean.FALSE,new Byte((byte)0x01),new Character('x'),
        new Short((short)2), new Integer(4), new Long(6L), new Float(0.8f),
        new Double(0.01), "AA");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      oi.invokeMethod(obj, "set", typL,
        null  ,new Byte((byte)0x01),new Character('x'),
        new Short((short)2), new Integer(4), new Long(6L), new Float(0.8f),
        new Double(0.01), "AA");
      NG();
    } catch (Exception e) {
      OK(e);
    }
  }

  public void havePriorExpectedOverIgnore()
  {
    class C {
      private int n = 1024;
      private boolean b = true;
      private String s = "abc";
    }
    C c = new C();

    ObjectInspector oi = new ObjectInspector(this);
    oi.ignore("n");
    oi.ignore("b");
    oi.ignore("s");
    oi.inspect(c);

    oi.expect("n", 1024);
    oi.inspect(c);

    oi.expect("b", true);
    oi.inspect(c);

    oi.expect("s", "abc");
    oi.inspect(c);
  }
}
