/*
 * ObjectWriterTest class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.tester.coverage;

import ts.tester.function.coverage.FunctionTester;
import ts.tester.function.print.*;
import ts.tester.util.ObjectSerializer;
import java.io.*;
import java.util.*;

/**
 * <code>ObjectWriter</code>NX̋@\NXB
 *
 * @author  VB
 * @version $Revision: 1.3 $, $Date: 2007/05/07 15:19:59 $
 */
public class ObjectWriterTest extends FunctionTester
{
  public static void main(String[] args)
  {
    try { 
      PrinterGroup group = new PrinterGroup();
      group.addPrinter(new ConsolePrinter());
      group.addPrinter(new HtmlPrinter("SATOH Takayuki"));
      setPrinter(group);

      run(ObjectWriterTest.class, (args.length == 0) ? null : args[0]);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  protected void preTesting()
  {
    MSG("ObjectWriterNX̋@\sB");
  }

  public void outputObject()
  {
    MSG("IuWFNg̏o͂̊mFB");

    PrintStream oldOut = System.out;

    ByteArrayOutputStream baos = null;
    Object obj = null;
    try {
      baos = new ByteArrayOutputStream();
      PrintStream ps = new PrintStream(baos);
      System.setOut(ps);

      ObjectWriter.outputObject(new Integer(123));
      byte[] bs = baos.toByteArray();
      obj = ObjectSerializer.deserialize(new String(bs, 1, bs.length-2));
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      System.setOut(oldOut);
      if (baos != null) try { baos.close(); } catch (Exception e) {}
    }

    TRUE(obj instanceof Integer);
    EQUAL(obj.toString(), "123");
  }

  public void outputObject_1()
  {
    MSG("k̏ꍇB");

    PrintStream oldOut = System.out;

    Object obj = null;
    ByteArrayOutputStream baos = null;
    try {
      baos = new ByteArrayOutputStream();
      PrintStream ps = new PrintStream(baos);
      System.setOut(ps);

      ObjectWriter.outputObject(null);
      byte[] bs = baos.toByteArray();
      obj = ObjectSerializer.deserialize(new String(bs, 1, bs.length-2));
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      System.setOut(oldOut);
      if (baos != null) try { baos.close(); } catch (Exception e) {}
    }

    EQUAL(obj.getClass().getName(), "ts.tester.coverage.ObjectWriter$Null");
  }

  public void writeObjectOrStream()
  {
    MSG("Xg[̏o͂̊mFB");

    PrintStream oldOut = System.out;

    String s = null;
    try {
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      PrintStream ps = new PrintStream(baos);
      System.setOut(ps);

      ObjectWriter.prepareOutput();

      ObjectWriter.outputObject(new Integer(123));
      byte[] bs = baos.toByteArray();

      final ByteArrayOutputStream baos2 = new ByteArrayOutputStream();
      final OutputStreamWriter osw = new OutputStreamWriter(baos2);

      class ObjectWriterEx extends ObjectWriter {
        protected void writeObject(Object obj) {
          try {
            osw.write("[" + String.valueOf(obj) + "]");
            osw.flush();
          } catch (Exception e) {
            e.printStackTrace();
          }
        }
      }
      ObjectWriterEx owex = new ObjectWriterEx();
      owex.setStreamWriter(osw);

      s = new String(bs);
      char[] cs = new char[s.length()];
      for (int i=0; i<cs.length; i++) {
        cs[i] = s.charAt(i);
      }
      owex.writeObjectOrStream(cs, cs.length);
      s = new String(baos2.toByteArray());
    } catch (Exception e) {
      NG(e);
    } finally {
      System.setOut(oldOut);
    }
    EQUAL(s, "[123]");
  }

  public void writeObjectOrStream_1()
  {
    MSG("obt@ɑ̕񂪍ĂꍇB");

    PrintStream oldOut = System.out;

    String s = null;
    try {
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      PrintStream ps = new PrintStream(baos);
      System.setOut(ps);

      ObjectWriter.prepareOutput();

      System.out.print("");
      ObjectWriter.outputObject(new Double(3.14159));
      System.out.print("");
      byte[] bs = baos.toByteArray();

      final ByteArrayOutputStream baos2 = new ByteArrayOutputStream();
      final OutputStreamWriter osw = new OutputStreamWriter(baos2);

      class ObjectWriterEx extends ObjectWriter {
        protected void writeObject(Object obj) {
          try {
            osw.write("[" + String.valueOf(obj) + "]");
            osw.flush();
          } catch (Exception e) {
            e.printStackTrace();
          }
        }
      }
      ObjectWriterEx owex = new ObjectWriterEx();
      owex.setStreamWriter(osw);

      String ss = new String(bs);
      char[] cs = new char[ss.length()];
      for (int i=0; i<cs.length; i++) {
        cs[i] = ss.charAt(i);
      }
      owex.writeObjectOrStream(cs, cs.length);
      s = new String(baos2.toByteArray());
    } catch (Exception e) {
      NG(e);
    } finally {
      System.setOut(oldOut);
    }

    EQUAL(s, "[3.14159]");
  }

  public void writeObjectOrStream_2()
  {
    MSG("NX̃VACY񂪕ɕĎ擾ꍇ");

    PrintStream oldOut = System.out;

    String str = null;
    try {
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      PrintStream ps = new PrintStream(baos);
      System.setOut(ps);

      ObjectWriter.prepareOutput();

      System.out.print("");
      ObjectWriter.outputObject(new Float(1.618));
      System.out.print("͂܂");
      ObjectWriter.outputObject(new Long(987654321L));
      System.out.print("");
      System.out.flush();
      byte[] bs = baos.toByteArray();

      String s = new String(bs);
      int slice = s.length() / 5;
      char[] cs1 = new char[slice];
      char[] cs2 = new char[slice];
      char[] cs3 = new char[slice];
      char[] cs4 = new char[slice];
      char[] cs5 = new char[s.length() - slice * 4];

      int i=0;
      for (; i<cs1.length; i++) {
        cs1[i] = s.charAt(i);
      }
      for (int j=0; j<cs2.length; i++, j++) {
        cs2[j] = s.charAt(i);
      }
      for (int j=0; j<cs3.length; i++, j++) {
        cs3[j] = s.charAt(i);
      }
      for (int j=0; j<cs4.length; i++, j++) {
        cs4[j] = s.charAt(i);
      }
      for (int j=0; j<cs5.length; i++, j++) {
        cs5[j] = s.charAt(i);
      }

      final ByteArrayOutputStream baos2 = new ByteArrayOutputStream();
      final OutputStreamWriter osw = new OutputStreamWriter(baos2);

      class ObjectWriterEx extends ObjectWriter {
        protected void writeObject(Object obj) {
          try {
            osw.write("[" + String.valueOf(obj) + "]");
            osw.flush();
          } catch (Exception e) {
            e.printStackTrace();
          }
        }
      }
      ObjectWriterEx owex = new ObjectWriterEx();
      owex.setStreamWriter(osw);

      owex.writeObjectOrStream(cs1, cs1.length);
      owex.writeObjectOrStream(cs2, cs2.length);
      owex.writeObjectOrStream(cs3, cs3.length);
      owex.writeObjectOrStream(cs4, cs4.length);
      owex.writeObjectOrStream(cs5, cs5.length);

      str = new String(baos2.toByteArray());
    } catch (Exception e) {
      NG(e);
    } finally {
      System.setOut(oldOut);
    }

    EQUAL(str, "[1.618]͂܂[987654321]");
  }

  public void writeObjectOrStream_3()
  {
    MSG("̃Xg[񂪃k̏ꍇB");

    PrintStream oldOut = System.out;

    Throwable t = null;
    try {
      final ByteArrayOutputStream baos2 = new ByteArrayOutputStream();
      final OutputStreamWriter osw = new OutputStreamWriter(baos2);

      class ObjectWriterEx extends ObjectWriter {
        protected void writeObject(Object obj) {
          try {
            osw.write("[" + String.valueOf(obj) + "]");
            osw.flush();
          } catch (Exception e) {
            e.printStackTrace();
          }
        }
      }
      ObjectWriterEx owex = new ObjectWriterEx();
      owex.setStreamWriter(osw);

      try {
        owex.writeObjectOrStream(null, 0);
      } catch (AssertionError e) {
        t = e;
      } catch (Exception e) {
      }
    } catch (Exception e) {
      NG(e);
    } finally {
      System.setOut(oldOut);
    }

    NOTNULL(t);
  }

  public void writeObjectOrStream_4()
  {
    MSG("̃Xg[̒sȏꍇB");

    PrintStream oldOut = System.out;

    Throwable t1 = null, t2 = null;
    try {
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      PrintStream ps = new PrintStream(baos);
      System.setOut(ps);

      ObjectWriter.prepareOutput();

      ObjectWriter.outputObject(new Integer(123));
      byte[] bs = baos.toByteArray();

      final ByteArrayOutputStream baos2 = new ByteArrayOutputStream();
      final OutputStreamWriter osw = new OutputStreamWriter(baos2);

      class ObjectWriterEx extends ObjectWriter {
        protected void writeObject(Object obj) {
          try {
            osw.write("[" + String.valueOf(obj) + "]");
            osw.flush();
          } catch (Exception e) {
            e.printStackTrace();
          }
        }
      }
      ObjectWriterEx owex = new ObjectWriterEx();
      owex.setStreamWriter(osw);

      String s = new String(bs);
      char[] cs = new char[s.length()];
      for (int i=0; i<cs.length; i++) {
        cs[i] = s.charAt(i);
      }

      try {
        owex.writeObjectOrStream(cs, -1);
      } catch (AssertionError e) {
        t1 = e;
      } catch (Exception e) {
      }

      try {
        owex.writeObjectOrStream(cs, cs.length + 1);
      } catch (AssertionError e) {
        t2 = e;
      } catch (Exception e) {
      }
    } catch (Exception e) {
      NG(e);
    } finally {
      System.setOut(oldOut);
    }

    NOTNULL(t1);
    NOTNULL(t2);
  }

  public void writeObjectOrStream_5()
  {
    MSG("ꕶĂꍇB");

    PrintStream oldOut = System.out;

    String s = null;
    try {
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      PrintStream ps = new PrintStream(baos);
      System.setOut(ps);

      ObjectWriter.prepareOutput();

      System.out.print("/\u0002\u0003\u001b/");
      ObjectWriter.outputObject("\u001b\u0003\u0002");
      byte[] bs = baos.toByteArray();

      final ByteArrayOutputStream baos2 = new ByteArrayOutputStream();
      final OutputStreamWriter osw = new OutputStreamWriter(baos2);

      class ObjectWriterEx extends ObjectWriter {
        protected void writeObject(Object obj) {
          try {
            osw.write("[" + String.valueOf(obj) + "]");
            osw.flush();
          } catch (Exception e) {
            e.printStackTrace();
          }
        }
      }
      ObjectWriterEx owex = new ObjectWriterEx();
      owex.setStreamWriter(osw);

      s = new String(bs);
      char[] cs = new char[s.length()];
      for (int i=0; i<cs.length; i++) {
        cs[i] = s.charAt(i);
      }
      owex.writeObjectOrStream(cs, cs.length);
      s = new String(baos2.toByteArray());
    } catch (Exception e) {
      NG(e);
    } finally {
      System.setOut(oldOut);
    }

    EQUAL(s.getBytes(), "/\u0002\u0003\u001b/[\u001b\u0003\u0002]".getBytes());
  }

  public void objectPrintStream_str()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      ObjectWriter.prepareOutput();

      System.err.print("abc\u0002\u0003\u001b");
      System.err.flush();
      bsq = baos.toByteArray();
    }
    finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "abc\u001b\u0002\u001b\u0003\u001b\u001b".getBytes());
  }

  public void objectPrintStream_boolean()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      ObjectWriter.prepareOutput();
      System.err.print(true);
      System.err.print(false);
      System.err.flush();
      bsq = baos.toByteArray();
    } finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "truefalse".getBytes());
  }

  public void objectPrintStream_char()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      ObjectWriter.prepareOutput();
      System.err.print('a');
      System.err.print('\u0002');
      System.err.print('\u0003');
      System.err.print('\u001b');
      System.err.flush();
      bsq = baos.toByteArray();
    } finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "a\u001b\u0002\u001b\u0003\u001b\u001b".getBytes());
  }

  public void objectPrintStream_chars()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      final char[] cs = { 'x', '\u0002', 'y', '\u0003', 'z', '\u001b' };
      ObjectWriter.prepareOutput();
      System.err.print(cs);
      System.err.flush();
      bsq = baos.toByteArray();
    } finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "x\u001b\u0002y\u001b\u0003z\u001b\u001b".getBytes());
  }

  public void objectPrintStream_object()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    Object o = new Object() {
      public String toString() {
        return "!\u0002\u0003\u001b!";
      }
    };

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      ObjectWriter.prepareOutput();
      System.err.print(o);
      System.err.flush();
      bsq = baos.toByteArray();
    } finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "!\u001b\u0002\u001b\u0003\u001b\u001b!".getBytes());
  }

  public void write_bytes_int_int()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    byte[] b = { (byte) 0x02, (byte) 0x03, (byte) 0x1b, (byte)'!' };

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      ObjectWriter.prepareOutput();
      System.err.write(b, 0, b.length);
      System.err.flush();
      bsq = baos.toByteArray();
    } finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "\u001b\u0002\u001b\u0003\u001b\u001b!".getBytes());
  }

  public void write_int()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      ObjectWriter.prepareOutput();
      System.err.write(0x02);
      System.err.write(0x03);
      System.err.write(0x1b);
      System.err.write((int) '!');
      System.err.flush();
      bsq = baos.toByteArray();
    } finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "\u001b\u0002\u001b\u0003\u001b\u001b!".getBytes());
  }

  public void append_char()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      ObjectWriter.prepareOutput();
      System.err.append('\u0002');
      System.err.append('\u0003');
      System.err.append('\u001b');
      System.err.flush();
      bsq = baos.toByteArray();
    } finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "\u001b\u0002\u001b\u0003\u001b\u001b".getBytes());
  }

  public void append_charsequence()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      ObjectWriter.prepareOutput();
      System.err.append("\u0002\u0003\u001b");
      System.err.flush();
      bsq = baos.toByteArray();
    } finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "\u001b\u0002\u001b\u0003\u001b\u001b".getBytes());
  }

  public void append_charsequence_int_int()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      ObjectWriter.prepareOutput();
      System.err.append("\u0002\u0003\u001b", 0, 3);
      System.err.flush();
      bsq = baos.toByteArray();
    } finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "\u001b\u0002\u001b\u0003\u001b\u001b".getBytes());
  }

  public void format_Locale_String_Object()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      ObjectWriter.prepareOutput();
      System.err.format(Locale.JAPAN, "%1$s", "\u0002\u0003\u001b");
      System.err.flush();
      bsq = baos.toByteArray();
    } finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "\u001b\u0002\u001b\u0003\u001b\u001b".getBytes());
  }

  public void format_String_Object()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      ObjectWriter.prepareOutput();
      System.err.format("%1$s", "\u0002\u0003\u001b");
      System.err.flush();
      bsq = baos.toByteArray();
    } finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "\u001b\u0002\u001b\u0003\u001b\u001b".getBytes());
  }

  public void printf_Locale_String_Object()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      ObjectWriter.prepareOutput();
      System.err.printf(Locale.JAPAN, "%1$s", "\u0002\u0003\u001b");
      System.err.flush();
      bsq = baos.toByteArray();
    } finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "\u001b\u0002\u001b\u0003\u001b\u001b".getBytes());
  }

  public void printf_String_Object()
  {
    byte[] bsq = null;

    PrintStream psOut = System.out;
    PrintStream psErr = System.err;

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    PrintStream ps = new PrintStream(baos);
    System.setErr(ps);
    try {
      ObjectWriter.prepareOutput();
      System.err.printf("%1$s", "\u0002\u0003\u001b");
      System.err.flush();
      bsq = baos.toByteArray();
    } finally {
      System.setOut(psOut);
      System.setErr(psErr);
    }

    EQUAL(bsq, "\u001b\u0002\u001b\u0003\u001b\u001b".getBytes());
  }
}
