/**********************************************************************
 * pk.c                                                     August 2005
 *
 * ASYM: An implementation of Asymetric Cryptography in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#ifdef __KERNEL__
#include <linux/kernel.h>
#else
#include "compat.h"
#endif

#include "pk.h"

/*
 * pubKey = number of 32bit words || modulus loaded MSB first
 * priKey = number of 32bit words || modulus loaded MSB first || decExp loaded MSB first
 */

const char* pk_errstr(int err) {
  switch(err) {
	  case PK_OK:
		  return("OK");
	  case PK_BAD_ALGO:
		  return("Invalid Encryption Algorithm");
	  case PK_KEYGEN_FAILED:
		  return("Key Generation Failed");
	  case PK_INVALID_KEY:
		  return("Invalid Key Argument");
	  case PK_INVALID_INPUT:
		  return("Invalid Input Argument");
	  case PK_UNWRAP_FAILED:
		  return("Padding check failed on unwrap");
	  case -ENOMEM:
		  return("No memory");
	  case -EINVAL:
		  return("Invalid argument");
  }
  return NULL;
}


int pk_init(size_t bitLen, unsigned options) {
  return(unsx_init(bitLen));
}


void pk_deinit(void) {
  unsx_deinit();
}


/* OpenPGP Message Format Key ID: The lower 64 buits of the public key
 * See: RFC 2440 */
int pk_getKeyID(u8 *id, const u8 *pubKey, size_t pubKeyLen,
                       unsigned options) 
{
  if (options & PK_ALGO_RSA || options & PK_ALGO_ELGAMAL) {
    memcpy(id, pubKey+pubKeyLen-8, 8);
  }
  else {
    return PK_BAD_ALGO;
  }

  return PK_OK;
}

static inline int __pk_keygen_rsa(u8 **pubKey, size_t *pubKeyLen,
		u8 **priKey, size_t *priKeyLen, size_t bitLen,
		unsigned options)
{
  int ret = -ENOMEM;
  rsa_key_t pub=NULL_RSA_KEY;
  rsa_key_t pri=NULL_RSA_KEY;

  if (rsa_keygen(&pub, &pri, bitLen, options)) {
    return(PK_KEYGEN_FAILED);
  }

  *pubKey = NULL;
  *priKey = NULL;

  *pubKeyLen = 2 + (2 * pub.len * sizeof(unsx));
  *pubKey = (u8 *) kmalloc(*pubKeyLen, GFP_KERNEL);
  if(!*pubKey) {
    goto leave;
  }
  **pubKey = (pub.len & 0xff); 
  *(*pubKey+1) = ((pub.len>>8) & 0xff); 
  unload_unsx(*pubKey+2, pub.n, pub.len); 
  unload_unsx(*pubKey+2+(pub.len * sizeof(unsx)), pub.e, pub.len); 
	
  *priKeyLen = 2 + (2 * pub.len * sizeof(unsx));
  *priKey = (u8 *) kmalloc(*priKeyLen, GFP_KERNEL);
  if(!*pubKey) {
    goto leave;
  }
  **priKey = (pri.len & 0xff);
  *(*priKey+1) = ((pri.len>>8) & 0xff);
  unload_unsx(*priKey+2, pri.n, pri.len); 
  unload_unsx(*priKey+2+(pub.len * sizeof(unsx)), pri.d, pri.len); 

#if 0
  {
    time_t a,b;
    int i;
    UNSX_NEW(OUT,pub.len);
    UNSX_NEW(IN,pub.len);

    if(!OUT || !IN) {
      if(OUT) {
        UNSX_FREE(OUT);
      }
      if(IN) {
        UNSX_FREE(OUT);
      }
      goto leave;
    }
    printf("BenchMark!\n");
    for (i=0; i<pub.len-1; i++) {
      IN[i] = (rand() << 16) | rand();
    }
    time(&a);
    #define LOOPS 1000
    for (i=0; i<LOOPS; i++) {
      ret = unsx_modPow(OUT, IN, pub.e, pub.n, pub.len);
      if(ret < 0)
        goto leave;
    }
    time(&b);
    printk(KERN_DEBUG "%u / (%lu - %lu) = %f\n", 
		    LOOPS, b,a, (double)LOOPS/(b-(double)a));
  }
#endif

  ret = 0;
leave:
  rsa_key_destroy_data(&pub);
  rsa_key_destroy_data(&pri);
  if(ret) {
    if(*pubKey) {
      kfree(*pubKey);
      *pubKey = NULL;
    }
    if(*priKey) {
       kfree(*priKey);
       *priKey = NULL;
    }
  }
  return(ret);
}

static inline int __pk_keygen_elgamal(u8 **pubKey, size_t *pubKeyLen,
		u8 **priKey, size_t *priKeyLen, size_t bitLen,
		unsigned options)
{
    int ret = -ENOMEM;
    elgamal_key_t pub=NULL_ELGAMAL_KEY;
    elgamal_key_t pri=NULL_ELGAMAL_KEY;

    if (elgamal_keygen(&pub, &pri, bitLen, options)) {
      return PK_KEYGEN_FAILED;
    }

    *pubKey = NULL;
    *priKey = NULL;

    *pubKeyLen = 2 + (3 * pub.len * sizeof(unsx));
    *pubKey = (u8*) kmalloc(*pubKeyLen, GFP_KERNEL);
    if(!pubKey)
      goto leave;
    **pubKey = (pub.len & 0xff); 
    *(*pubKey+1) = ((pub.len>>8) & 0xff); 
    unload_unsx(*pubKey + 2, pub.p,  pub.len); 
    unload_unsx(*pubKey + 2 + (pub.len * sizeof(unsx)), pub.g,  pub.len); 
    unload_unsx(*pubKey + 2 + (2 * pub.len * sizeof(unsx)), pub.ga, pub.len); 
 
    *priKeyLen = 2 + (4 * pri.len * sizeof(unsx));
    *priKey = (u8*) kmalloc(*priKeyLen, GFP_KERNEL);
    if(!priKey)
      goto leave;
    **priKey = (pri.len & 0xff); 
    *(*priKey+1) = ((pri.len>>8) & 0xff); 
    unload_unsx(*priKey+2, pri.p,  pri.len); 
    unload_unsx(*priKey+2+(pri.len * sizeof(unsx)), pri.g,  pri.len); 
    unload_unsx(*priKey+2+(2 * pri.len * sizeof(unsx)), pri.ga, pri.len); 
    unload_unsx(*priKey+2+(3 * pri.len * sizeof(unsx)), pri.a,  pri.len); 

    ret = 0;
leave:
    elgamal_key_destroy_data(&pri);
    elgamal_key_destroy_data(&pub);
    if(ret) {
      if(*pubKey) {
	kfree(*pubKey);
	*pubKey = NULL;
      }
      if(*pubKey) {
	kfree(*pubKey);
	*pubKey = NULL;
      }
    }
    return(ret);
}

int pk_keygen(u8 **pubKey, size_t *pubKeyLen,
                     u8 **priKey, size_t *priKeyLen,
                     size_t bitLen, unsigned options) {
  int ret = 0;
  
  ret = unsx_init(bitLen);
  if(ret < 0) {
    return(ret);
  }

  if (options & PK_ALGO_RSA) {
    return(__pk_keygen_rsa(pubKey, pubKeyLen, priKey, priKeyLen, bitLen,
			    options));
  }
  else if (options & PK_ALGO_ELGAMAL) {
    return(__pk_keygen_elgamal(pubKey, pubKeyLen, priKey, priKeyLen, bitLen,
			    options));
  }
  else {
    return PK_BAD_ALGO;
  }

  return PK_OK;
}

static inline int __pk_encrypt_rsa(u8 *output, size_t *outputLen,
                      const u8 *input, size_t inputLen,
                      const u8 *pubKey, size_t pubKeyLen, int options) 
{
  int ret;
  rsa_key_t pub = NULL_RSA_KEY;

  pub.len = (pubKey[1]&0xff)<<8 | (pubKey[0]&0xff);

  printk(KERN_DEBUG "__pk_encrypt_rsa enter\n");

  if (pub.len > 2+(2*pubKeyLen*sizeof(unsx)))
    return PK_INVALID_KEY;
  if (inputLen > (pub.len*sizeof(unsx)-4))
    return PK_INVALID_INPUT;

  pub.n = alloc_load_unsx(pubKey+2, pub.len);
  if(!pub.n) {
    ret = -ENOMEM;
    goto leave;
  }

  pub.e = alloc_load_unsx(pubKey+2+(pub.len*sizeof(unsx)), pub.len);
  if(!pub.e) {
    ret = -ENOMEM;
    goto leave;
  }

  ret = rsa_wrap(output, outputLen, &pub, pub.e, input, inputLen,
                 options & (PK_PKCS1_ENCRYPT|PK_PKCS1_SIGN));

leave:
  rsa_key_destroy_data(&pub);
  return ret;
}

static inline int __pk_encrypt_elgamal(u8 *output, size_t *outputLen,
                      const u8 *input, size_t inputLen,
                      const u8 *pubKey, size_t pubKeyLen, int options) 
{
  int ret = -ENOMEM;

  elgamal_key_t pub = NULL_ELGAMAL_KEY;
  int pubIdx;

  pub.len = (pubKey[1]&0xff)<<8 | (pubKey[0]&0xff);

  if (pub.len > (2+3*pubKeyLen*sizeof(unsx)))
    return(PK_INVALID_KEY);
  if (inputLen > (pub.len*sizeof(unsx)-4))
    return(PK_INVALID_INPUT);

  pubIdx = 2;
  pub.p = alloc_load_unsx(pubKey+pubIdx, pub.len); 
  if(!pub.p)
    goto leave;
  pubIdx += pub.len*sizeof(unsx);

  pub.g = alloc_load_unsx(pubKey+pubIdx, pub.len); 
  if(!pub.g)
    goto leave;
  pubIdx += pub.len*sizeof(unsx);

  pub.ga = alloc_load_unsx(pubKey+pubIdx, pub.len); 
  if(!pub.ga)
    goto leave;
  pubIdx += pub.len*sizeof(unsx);

  ret = elgamal_encrypt(output, outputLen, &pub, input, inputLen,
                        options & (PK_PKCS1_ENCRYPT|PK_PKCS1_SIGN));

leave:
  elgamal_key_destroy_data(&pub);
  return ret;
}

int pk_encrypt(u8 *output, size_t *outputLen,
                      const u8 *input, size_t inputLen,
                      const u8 *pubKey, size_t pubKeyLen, 
		      unsigned options) {
  printk(KERN_DEBUG "pk_encrypt enter\n");
  if (options & PK_ALGO_RSA) {
    return(__pk_encrypt_rsa(output, outputLen, input, inputLen,
		    pubKey, pubKeyLen, options));
  }
  else if (options & PK_ALGO_ELGAMAL) {
    return(__pk_encrypt_elgamal(output, outputLen, input, inputLen,
		    pubKey, pubKeyLen, options));
  }
  else {
    return PK_BAD_ALGO;
  }

  return PK_OK;
}

static int  __pk_decrypt_rsa(u8 *output, size_t *outputLen,
                      const u8 *input, size_t inputLen,
                      const u8 *priKey, size_t priKeyLen, int options) 
{
  int ret = -ENOMEM;
  rsa_key_t pri = NULL_RSA_KEY;

  pri.len = (priKey[1]&0xff)<<8 | (priKey[0]&0xff);

  if (2 + (2 * pri.len * sizeof(unsx)) != priKeyLen) {
    printk(KERN_DEBUG "__pk_decrypt_rsa: length mismatch %d != %d\n",
		    2 + (2 * pri.len * sizeof(unsx)), priKeyLen);
    return PK_INVALID_KEY;
  }

  pri.n = alloc_load_unsx(priKey+2, pri.len);
  if(!pri.n)
    goto leave;
  pri.d = alloc_load_unsx(priKey+2+(pri.len*sizeof(unsx)), pri.len);
  if(!pri.d)
    goto leave;

  ret = rsa_unwrap(output, outputLen, &pri, pri.d, input, inputLen,
                   options & (PK_PKCS1_ENCRYPT|PK_PKCS1_SIGN));

leave:
  rsa_key_destroy_data(&pri);
  return ret;
}

static inline int __pk_decrypt_elgamal(u8 *output, size_t *outputLen,
                      const u8 *input, size_t inputLen,
                      const u8 *priKey, size_t priKeyLen,
                      unsigned options) {
  int ret = -ENOMEM;
  elgamal_key_t pri = NULL_ELGAMAL_KEY;
  int priIdx;

  pri.len = (priKey[1]&0xff)<<8 | (priKey[0]&0xff);

  if (2 + (4 * pri.len * sizeof(unsx)) != priKeyLen)
    return PK_INVALID_KEY;

  priIdx = 2;
  pri.p  = alloc_load_unsx(priKey+priIdx, pri.len); 
  if(!pri.p)
    goto leave;
  priIdx += pri.len*sizeof(unsx);

  pri.g  = alloc_load_unsx(priKey+priIdx, pri.len); 
  if(!pri.g)
    goto leave;
  priIdx += pri.len*sizeof(unsx);

  pri.ga = alloc_load_unsx(priKey+priIdx, pri.len); 
  if(!pri.ga)
    goto leave;
  priIdx += pri.len*sizeof(unsx);

  pri.a  = alloc_load_unsx(priKey+priIdx, pri.len); 
  if(!pri.a)
    goto leave;
  priIdx += pri.len*sizeof(unsx);

  ret = elgamal_decrypt(output, outputLen, &pri, input, inputLen,
                        options & (PK_PKCS1_ENCRYPT|PK_PKCS1_SIGN));

leave:
  elgamal_key_destroy_data(&pri);
  return ret;
}

int pk_decrypt(u8 *output, size_t *outputLen,
                      const u8 *input, size_t inputLen,
                      const u8 *priKey, size_t priKeyLen,
                      unsigned options) {
  if (options & PK_ALGO_RSA) {
    return(__pk_decrypt_rsa(output, outputLen, input, inputLen,
			    priKey, priKeyLen, options));
  }
  else if (options & PK_ALGO_ELGAMAL) {
    return(__pk_decrypt_elgamal(output, outputLen, input, inputLen,
			    priKey, priKeyLen, options));
  }
  else {
    return PK_BAD_ALGO;
  }

  return PK_OK;
}

static inline int __pk_verify_rsa(const u8 *payload, size_t payloadLen,
		const u8 *input, size_t inputLen,
		const u8 *pubKey, size_t pubKeyLen, int options) 
{
    int ret = -ENOMEM;
    rsa_key_t pub = NULL_RSA_KEY;

    u8 *output;
    size_t outputLen = inputLen;

    pub.len = (pubKey[1]&0xff)<<8 | (pubKey[0]&0xff);

    if (pub.len > (2+pubKeyLen*sizeof(unsx)))
      return PK_INVALID_KEY;

    output = (u8*) kmalloc(inputLen, GFP_KERNEL);
    if(!output)
      goto leave;
    pub.n = alloc_load_unsx(pubKey+2, pub.len);
    if(!pub.n)
      goto leave;
    pub.e = (unsx*) kmalloc(pub.len*sizeof(unsx), GFP_KERNEL);
    if(!pub.n)
      goto leave;
    unsx_setLow(pub.e, (1<<16) | 1, pub.len);

    pub.e = (unsx*) kmalloc(pub.len*sizeof(unsx), GFP_KERNEL);
    if(!pub.n) {
      ret = -ENOMEM;
      goto leave;
    }
    unsx_setLow(pub.e, (1<<16) | 1, pub.len);

    memset(output, 0, outputLen);
    ret = rsa_unwrap(output, &outputLen, &pub, pub.e, input, inputLen,
                     options & (PK_PKCS1_ENCRYPT|PK_PKCS1_SIGN));

    if(ret) {
      if (outputLen != payloadLen) {
        ret = PK_UNWRAP_FAILED;
      } else {
        ret = memcmp(output, payload, payloadLen) ?  PK_UNWRAP_FAILED : PK_OK;
      }
    }

leave:
    if (output) 
      kfree(output);
    rsa_key_destroy_data(&pub);
    return ret;
}


static inline int __pk_verify_elgamal(const u8 *payload, size_t payloadLen,
                     const u8 *input, size_t inputLen,
                     const u8 *pubKey, size_t pubKeyLen, int options) 
{
  int ret = -ENOMEM;
  elgamal_key_t pub = NULL_ELGAMAL_KEY;
  u8 *output;
  int outputLen = inputLen;
  int pubIdx;

  pub.len = (pubKey[1]&0xff)<<8 | (pubKey[0]&0xff);

  if (pub.len > (2+pubKeyLen*sizeof(unsx))) {
    return PK_INVALID_KEY;
  }

  output = (u8*)kmalloc(inputLen, GFP_KERNEL);
  if(!output)
    goto leave;

  pubIdx = 2;
  pub.p  = alloc_load_unsx(pubKey+pubIdx, pub.len); 
  if(!pub.p)
    goto leave;
  pubIdx += pub.len*sizeof(unsx);
  pub.g  = alloc_load_unsx(pubKey+pubIdx, pub.len); 
  if(!pub.g)
    goto leave;
  pubIdx += pub.len*sizeof(unsx);
  pub.ga = alloc_load_unsx(pubKey+pubIdx, pub.len); 
  if(!pub.ga)
    goto leave;
  pubIdx += pub.len*sizeof(unsx);

  memset(output, 0, outputLen);
  ret = elgamal_verify(payload, payloadLen, &pub, input, inputLen,
                       options & (PK_PKCS1_ENCRYPT|PK_PKCS1_SIGN));

leave:
  if (output) 
    kfree(output);
  elgamal_key_destroy_data(&pub);
  return ret;
}


int pk_verify(const u8 *payload, size_t payloadLen,
                     const u8 *input, size_t inputLen,
                     const u8 *pubKey, size_t pubKeyLen, 
		     unsigned options) 
{
  if (options & PK_ALGO_RSA) {
    return(__pk_verify_rsa(payload, payloadLen, input, inputLen, 
			    pubKey, pubKeyLen, options));
  }
  else if (options & PK_ALGO_ELGAMAL) {
    return(__pk_verify_elgamal(payload, payloadLen, input, inputLen, 
			    pubKey, pubKeyLen, options));
  }
  else {
    return PK_BAD_ALGO;
  }

  return PK_OK;
}


static inline int __pk_sign_rsa(u8 *output, size_t *outputLen,
                   const u8 *payload, size_t payloadLen,
                   const u8 *priKey, size_t priKeyLen, int options) {
  int ret = -ENOMEM;

  rsa_key_t pri = NULL_RSA_KEY;

  pri.len = (priKey[1]&0xff)<<8 | (priKey[0]&0xff);

  if (pri.len > 2+(2*priKeyLen*sizeof(unsx)))
    return PK_INVALID_KEY;
  if (payloadLen > (pri.len*sizeof(unsx)-4))
    return PK_INVALID_INPUT;

  pri.n = alloc_load_unsx(priKey+2, pri.len);
  if(!pri.n)
    goto leave;
  pri.d = alloc_load_unsx(priKey+2+(pri.len*sizeof(unsx)), pri.len);
  if(!pri.d)
    goto leave;

  ret = rsa_wrap(output, outputLen, &pri, pri.d, payload, payloadLen,
                 options & (PK_PKCS1_ENCRYPT|PK_PKCS1_SIGN));

leave:
  rsa_key_destroy_data(&pri);
  return ret;
}


static inline int __pk_sign_elgamal(u8 *output, size_t *outputLen,
                   const u8 *payload, size_t payloadLen,
                   const u8 *priKey, size_t priKeyLen,
                   int options) {
  int ret = -ENOMEM;
  elgamal_key_t pri = NULL_ELGAMAL_KEY;
  int priIdx;

  pri.len = (priKey[1]&0xff)<<8 | (priKey[0]&0xff);

  if (pri.len > 2+(4*priKeyLen*sizeof(unsx)))
    return PK_INVALID_KEY;
  if (payloadLen > (pri.len*sizeof(unsx)-4))
    return PK_INVALID_INPUT;

  priIdx = 2;
  pri.p  = alloc_load_unsx(priKey+priIdx, pri.len); 
  if(!pri.p)
    goto leave;
  priIdx += pri.len*sizeof(unsx);
  pri.g  = alloc_load_unsx(priKey+priIdx, pri.len); 
  if(!pri.g)
    goto leave;
  priIdx += pri.len*sizeof(unsx);
  pri.ga = alloc_load_unsx(priKey+priIdx, pri.len); 
  if(!pri.ga)
    goto leave;
  priIdx += pri.len*sizeof(unsx);
  pri.a  = alloc_load_unsx(priKey+priIdx, pri.len); 
  if(!pri.a)
    goto leave;
  priIdx += pri.len*sizeof(unsx);

  ret = elgamal_sign(output, outputLen, &pri, payload, payloadLen,
                     options & (PK_PKCS1_ENCRYPT|PK_PKCS1_SIGN));

leave:
  elgamal_key_destroy_data(&pri);
  return ret;
}


int pk_sign(u8 *output, size_t *outputLen,
                   const u8 *payload, size_t payloadLen,
                   const u8 *priKey, size_t priKeyLen,
                   unsigned options) {
  if (options & PK_ALGO_RSA) {
    return(__pk_sign_rsa(output, outputLen, payload, payloadLen, 
			    priKey, priKeyLen, options));
  }
  else if (options & PK_ALGO_ELGAMAL) {
    return(__pk_sign_elgamal(output, outputLen, payload, payloadLen, 
			    priKey, priKeyLen, options));
  }
  else {
    return PK_BAD_ALGO;
  }

  return PK_OK;
}

