/**********************************************************************
 * unsx.c                                                   August 2005
 *
 * ASYM: An implementation of Asymetric Cryptography in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#ifndef UNSX_C
#define UNSX_C

#ifdef __KERNEL__
#include <linux/types.h>
#include <linux/slab.h>
#else
#include "compat.h"
#endif


#include "unsx.h"

/* used to find maximum stack usage */
#define STACK_MACRO() {\
  char *X=NULL;\
  unsx_stackMaxDepth = (unsx_stackMaxDepth < (unsx_stack-((uns32)&X)-sizeof(X))\
                     ? (unsx_stack-((uns32)&X)-sizeof(X))\
                     : unsx_stackMaxDepth);\
/*{\
int i;\
for (i=0; i<(unsx_stack-((uns32)&X)-sizeof(X)); i+=20) putc(' ', stdout);\
printf("%1.3u %s\n", (unsx_stack-((uns32)&X)-sizeof(X)), __FUNCTION__);\
}*/\
}

uns32 unsx_stack=0, unsx_stackMaxDepth=0;

static unsx **unsx_Trial_Prime=NULL;
int UNSX_LENGTH = 0;
unsx *unsx_ZERO=NULL;
unsx *unsx_ONE=NULL;
unsx *unsx_TWO=NULL;
unsx *unsx_THREE=NULL;
unsx *unsx_FIVE=NULL;
unsx *unsx_SEVEN=NULL;
unsx *unsx_ELEVEN=NULL;
unsx *unsx_THIRTEEN=NULL;
unsx *unsx_TWENTYTHREE=NULL;
unsx *unsx_1662803=NULL;

static const uns8 unsx_Prime_unsx_add[unsx_Trial_Prime_size] =
#if (unsx_Trial_Prime_size == 2*3*5*7)
{1, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4,
 3,  2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2,
 1,  2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2,
 1,  6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4,
 3,  2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2,
 1,  2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2,
 1,  6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4,
 3,  2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4,
 3,  2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2,
 1,  6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 10,
 9,  8, 7, 6, 5, 4, 3, 2, 1, 2},
#else /* (unsx_Trial_Prime_size == 2*3*5*7) */
  #if (unsx_Trial_Prime_size == 2*3*5*7*11)
{1,12,11,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1,
 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2,
 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1,14,13,12,11,10, 9, 8, 7, 6, 5,
 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4,
 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1,12,11,10, 9, 8, 7,
 6, 5, 4, 3, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2,
 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3,
 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4,
 3, 2, 1, 6, 5, 4, 3, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5,
 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1,10,
 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7,
 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1,12,11,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2,
 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1,12,11,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1,
 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4,
 3, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3,
 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4,
 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1,
 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2,
 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1,
 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2,
 1, 2, 1, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3,
 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6,
 5, 4, 3, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1,
 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6,
 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3,
 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4,
 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3,
 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4,
 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1,
 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2,
 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1,
 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2,
 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5,
 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 8, 7, 6,
 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5,
 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6,
 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1,
 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2,
 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3,
 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4,
 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1,
10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2,
 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1,
 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1,12,11,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4,
 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1,12,11,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3,
 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4,
 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1,
 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6,
 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8, 7, 6, 5, 4, 3,
 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4,
 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1,10, 9,
 8, 7, 6, 5, 4, 3, 2, 1,12,11,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1, 6,
 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2, 1,
 6, 5, 4, 3, 2, 1, 4, 3, 2, 1,14,13,12,11,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 8,
 7, 6, 5, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3,
 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2, 1, 6, 5, 4, 3, 2, 1, 2, 1, 6, 5, 4, 3, 2, 1, 4, 3, 2, 1, 2, 1, 4, 3, 2,
 1,12,11,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 2};
  #else /* (unsx_Trial_Prime_size == 2*3*5*7*11) */
    #error unsx_Trial_Prime_size of unknown size
  #endif /* (unsx_Trial_Prime_size == 2*3*5*7*11) */
#endif /* (unsx_Trial_Prime_size == 2*3*5*7) */

/*
** YOU MUST RUN THIS BEFORE YOU USE ANY unsx
** ROUTINES!!!
*/

static int __unsx_init(int bitLength);

int unsx_init(int bitLength) {
  if (unsx_ONE && UNSX_LENGTH==bitLength)
    return(0);

  unsx_deinit();
  return(__unsx_init(bitLength));
}

static int __unsx_init(int bitLength) {
  int i, arri, trial_prime_count;
  int status;
  uns2x t;
  unsx n;
  unsx arr[10];
  STACK_MACRO();

  if (unsx_ONE != NULL)
    return(0);

  UNSX_LENGTH = bitLength / unsx_Bit_Length;
  if (UNSX_LENGTH <= 0)
	  goto error;
       
  unsx_ZERO         = (unsx*)kmalloc(UNSX_LENGTH * sizeof(unsx), GFP_KERNEL);
  unsx_ONE          = (unsx*)kmalloc(UNSX_LENGTH * sizeof(unsx), GFP_KERNEL);
  unsx_TWO          = (unsx*)kmalloc(UNSX_LENGTH * sizeof(unsx), GFP_KERNEL);
  unsx_THREE        = (unsx*)kmalloc(UNSX_LENGTH * sizeof(unsx), GFP_KERNEL);
  unsx_FIVE         = (unsx*)kmalloc(UNSX_LENGTH * sizeof(unsx), GFP_KERNEL);
  unsx_SEVEN        = (unsx*)kmalloc(UNSX_LENGTH * sizeof(unsx), GFP_KERNEL);
  unsx_ELEVEN       = (unsx*)kmalloc(UNSX_LENGTH * sizeof(unsx), GFP_KERNEL);
  unsx_THIRTEEN     = (unsx*)kmalloc(UNSX_LENGTH * sizeof(unsx), GFP_KERNEL);
  unsx_TWENTYTHREE  = (unsx*)kmalloc(UNSX_LENGTH * sizeof(unsx), GFP_KERNEL);
  unsx_1662803      = (unsx*)kmalloc(UNSX_LENGTH * sizeof(unsx), GFP_KERNEL);
  unsx_Trial_Prime  = (unsx**)kmalloc(unsx_Trial_Prime_Sets * sizeof(unsx*), 
		  GFP_KERNEL);

  if(!unsx_ZERO || !unsx_ONE || !unsx_TWO || !unsx_THREE || !unsx_FIVE ||
		  !unsx_SEVEN || !unsx_ELEVEN || ! unsx_THIRTEEN ||
		  !unsx_TWENTYTHREE || !unsx_1662803 || !unsx_Trial_Prime) {
    goto error;
  }

  memset(unsx_ZERO,        0, UNSX_LENGTH * sizeof(unsx));
  memset(unsx_ONE,         0, UNSX_LENGTH * sizeof(unsx));
  memset(unsx_TWO,         0, UNSX_LENGTH * sizeof(unsx));
  memset(unsx_THREE,       0, UNSX_LENGTH * sizeof(unsx));
  memset(unsx_FIVE,        0, UNSX_LENGTH * sizeof(unsx));
  memset(unsx_SEVEN,       0, UNSX_LENGTH * sizeof(unsx));
  memset(unsx_ELEVEN,      0, UNSX_LENGTH * sizeof(unsx));
  memset(unsx_THIRTEEN,    0, UNSX_LENGTH * sizeof(unsx));
  memset(unsx_TWENTYTHREE, 0, UNSX_LENGTH * sizeof(unsx));
  memset(unsx_1662803,     0, UNSX_LENGTH * sizeof(unsx));
  memset(unsx_Trial_Prime, 0, unsx_Trial_Prime_Sets * sizeof(unsx*));

  /* unsx_ZERO[0] = 0; */
  unsx_ONE[0] = 1;
  unsx_TWO[0] = 2;
  unsx_THREE[0] = 3;
  unsx_FIVE[0] = 5;
  unsx_SEVEN[0] = 7;
  unsx_ELEVEN[0] = 11;
  unsx_THIRTEEN[0] = 13;
  unsx_TWENTYTHREE[0] = 23;
  unsx_1662803[0] = 1662803;

  /*
   ** unsx_Trial_Prime[i] = {length(unsx_Trial_Prime),
   **                        prod(unsx_Trial_Prime[i]
   **                                     [j = 2:length(unsx_Trial_Prime)]),
   **                        prime_k,
   **                        prime_k+1,
   **                        prime_k+2,
   **                        ... };
  */
  trial_prime_count = arri = 0;
  n = unsx_Trial_Prime_Sets_Start+1;                 /* start somewhere */
  n |= 1;                                            /* if even, make odd */

  /* while n is composite, step up by 2 
   * (could be faster, but primes are dense when n is small) */
  while (1) {
    yield();
    status = unsx_isPrimeSPRPn(&n,unsx_TWO,1);
    if(status < 0)
      goto error;
    if(status)
      break;
    n += 2;
  }
  arr[arri++] = (unsx)(t = n);

  for (i=0; i<unsx_Trial_Prime_Sets; ) {
    n += 2;
    while (1) {
      yield();
      status = unsx_isPrimeSPRPn(&n,unsx_TWO,1);
      if(status < 0)
        goto error;
      if(status)
        break;
      n += 2;
    }
    trial_prime_count++;

    if (t*n > unsx_Max) {
      t *= (arr[arri++] = n);
      continue;
    }
    unsx_Trial_Prime[i] = (unsx*)kmalloc((arri+2) * sizeof(unsx),
		      GFP_KERNEL); /* Make array */
    if(!unsx_Trial_Prime[i])
      goto error;
    *unsx_Trial_Prime[i] = arri+2;   /* Store Length */
    *(unsx_Trial_Prime[i]+1) = (unsx) t; /* Store master modulo */

    memcpy(unsx_Trial_Prime[i]+2, arr, arri);

    arr[0] = (unsx)(t = n);
    arri = 1;
    i++;
  }

  return(0);

error:
    unsx_deinit();
    return(-ENOMEM);
}

void unsx_deinit(void) {
  int i;
  STACK_MACRO();

  if (unsx_ONE == NULL)
    return;
  
  UNSX_SAFE_FREE(unsx_ZERO);
  UNSX_SAFE_FREE(unsx_ONE);
  UNSX_SAFE_FREE(unsx_TWO);
  UNSX_SAFE_FREE(unsx_THREE);
  UNSX_SAFE_FREE(unsx_FIVE);
  UNSX_SAFE_FREE(unsx_SEVEN);
  UNSX_SAFE_FREE(unsx_ELEVEN);
  UNSX_SAFE_FREE(unsx_THIRTEEN);
  UNSX_SAFE_FREE(unsx_TWENTYTHREE);
  UNSX_SAFE_FREE(unsx_1662803);

  if (unsx_Trial_Prime != NULL) {
    for (i=0; i<unsx_Trial_Prime_Sets; i++) {
      UNSX_SAFE_FREE(unsx_Trial_Prime[i]);
    }
    UNSX_SAFE_FREE(unsx_Trial_Prime);
  }

  UNSX_LENGTH = 0;
}

/*!
** a = b + 1
** Returns carry
** a[len] b[len]
*/
unsx unsx_inc(unsx *a, const unsx *b, int len) {
  int i=0;
  STACK_MACRO();

  while (i < len) {
    if (b[i] != unsx_Max) {
      a[i] = b[i] + 1;
      break;
    }
    a[i++] = 0;
  }
  if (i==len)
    return 1;

  for(/**/; i<len; i++) {
    a[i] = b[i];
  }

  #ifdef UNSX_SECURE
  #endif

  return 0;
}

/*!
** a = b + c
** Returns carry
** a[len] b[len] c[len]
*/
unsx unsx_add(unsx *a, const unsx *b, const unsx *c, int len) {
  unsx carry;
  int i;
  STACK_MACRO();

  carry = 0;
  for (i=0; i<len; i++) {
    if ((a[i] = b[i] + carry) < carry) {
      a[i] = c[i];
    } else {
      carry = ((a[i] += c[i]) < c[i]);
    }
  }

  #ifdef UNSX_SECURE
  #endif

  return carry;
}

/*
** a = b + c*d
** Returns carry
** a[len] b[len] d[len]
*/
unsx unsx_addMultSingle(unsx *a, const unsx *b, unsx c, const unsx *d, int len) {
  unsx carry;
  unsx th, tl;
  int i;

  STACK_MACRO();

  carry = 0;
  for (i=0; i<len; i++) {
    UNSX_DOUBLEMULT(th,tl, c, d[i]);
    carry = ((a[i] = b[i] + carry) < carry);
    carry += ((a[i] += tl) < tl);
    carry += th;
  }

  #ifdef UNSX_SECURE
    th = tl = 0;
  #endif

  return carry;
}

/*
** a = b - 1
** Returns carry
** a[len] b[len]
*/
unsx unsx_dec(unsx *a, const unsx *b, int len) {
  int i=0;
  STACK_MACRO();

  while (i < len) {
    if (b[i]) { /* b[i] != 0 */
      a[i] = b[i] - 1;
      break;
    }
    a[i++] = unsx_Max;
  }
  if (i == len)
    return 1;

  while (++i < len) {
    a[i] = b[i];
  }
  
  #ifdef UNSX_SECURE
  #endif

  return 0;
}

/*
** a = b - c
** Returns borrow
** a[len] b[len] c[len]
*/
unsx unsx_sub(unsx *a, const unsx *b, const unsx *c, int len) {
  unsx borrow;
  int i;
  STACK_MACRO();

  borrow = 0;
  for (i=0; i<len; i++) {
    if ((a[i] = b[i] - borrow) > (unsx_Max - borrow)) {
      a[i] = unsx_Max - c[i];
    } else {
      borrow = ((a[i] -= c[i]) > (unsx_Max - c[i]));
    }
  }

  #ifdef UNSX_SECURE
  #endif

  return borrow;
}

/*
** a = b - c*d
** Returns carry
** a[len] b[len] d[len]
*/
unsx unsx_subMultSingle(unsx *a, const unsx *b, unsx c, const unsx *d, int len) {
  unsx borrow;
  unsx th, tl;
  int i;
  STACK_MACRO();

  borrow = 0;
  for (i=0; i<len; i++) {
    UNSX_DOUBLEMULT(th,tl, c, d[i]);;
    borrow = ((a[i] = b[i] - borrow) > (unsx_Max - borrow));
    borrow += ((a[i] -= tl) > (unsx_Max - tl));
    borrow += th;
  }

  #ifdef UNSX_SECURE
    th = tl = 0;
  #endif

  return borrow;
}

/*
** a = b * c
** a[2*len] b[len] c[len]
** len < UNSX_LENGTH
*/
int unsx_mult(unsx *a, const unsx *b, const unsx *c, int len) {
  int bLen, i, cLen;
  UNSX_NEW(t, 2*len);
  STACK_MACRO();

  if(!t) {
    return(-ENOMEM);
  }

  bLen = unsx_countArr(b, len);
  cLen = unsx_countArr(c, len);

  unsx_setZero(t, 2*len);
  for (i=0; i<bLen; i++) {
    t[i+cLen] += unsx_addMultSingle(&t[i], &t[i], b[i], c, cLen);
  }
  unsx_set(a, t, 2*len);

  UNSX_FREE(t, bLen+cLen);

  #ifdef UNSX_SECURE
    bLen = cLen = i = 0;
  #endif

  return(0);
}

/*
** a = c / d
** b = c % d.
** a[cLen] b[dLen] c[cLen] d[dLen]
** 0 < d
** cLen < (2*UNSX_LENGTH)
** dLen < UNSX_LENGTH
** if a == NULL, then it will not be written to.
*/
#define RETURN(RET) do { ret = (RET); goto unsx_divi_ret; }while(0)
int unsx_divi(unsx *a, unsx *b,
               const unsx *c, int cLen,
               const unsx *d, int dLen) {
  unsx t, ai;
  int d2Len, shift, i;
  int ret;
  UNSX_NEW(c2, 2* UNSX_LENGTH +1);
  UNSX_NEW(d2, UNSX_LENGTH);

  STACK_MACRO();

  if(!c2 || !d2) {
    RETURN(-ENOMEM);
  }

  d2Len = unsx_countArr(d,dLen);

  if (unsx_countArr(c,cLen) == 1) {
    if (d2Len == 1) {
      ai = c[0]/d[0];
      t = c[0]%d[0];

      if (a != NULL) unsx_setLow(a, ai, cLen);
      unsx_setLow(b, t, dLen);
    } else {
      if (a != NULL) unsx_setZero(a, cLen);
      unsx_setLow(b, c[0], dLen);
    }
    RETURN(0);
  }

  if (d2Len == 0) {
    RETURN(0);
  }

  shift = unsx_Bit_Length - unsx_countBits(d[d2Len-1]);
  unsx_setZero(c2, d2Len);
  c2[cLen] = unsx_shl(c2, c, shift, cLen);
  unsx_shl(d2, d, shift, d2Len);
  t = d2[d2Len-1];
  d2Len = unsx_countArr(d,d2Len);

  if (a != NULL) unsx_setZero(a, cLen);

  for (i=cLen-d2Len; 0<=i; i--) {
    if (t == unsx_Max) {
      ai = c2[i+d2Len];
    } else {
      UNSX_DOUBLEDIVI(ai, c2[i+d2Len],c2[i+d2Len-1], t+1);
    }
    c2[i+d2Len] -= unsx_subMultSingle(&c2[i], &c2[i], ai, d2, d2Len);

    while (c2[i+d2Len] || unsx_cmp(&c2[i], d2, d2Len) >= 0) {
      ai++;
      c2[i+d2Len] -= unsx_sub(&c2[i], &c2[i], d2, d2Len);
    }

    if (a != NULL) a[i] = ai;
  }

  unsx_setZero(b, dLen);
  unsx_shr(b, c2, shift, d2Len);

  ret = 0;
unsx_divi_ret:
  #ifdef UNSX_SECURE
    t = ai = i = d2Len = shift = 0;
  #endif
	
  if(c2)
    UNSX_FREE(c2, 2*UNSX_LENGTH+1);
  if(d2)
    UNSX_FREE(d2, UNSX_LENGTH);

  return(ret);
}
#undef RETURN

/*
** if(a< b)   <==>   if(unsx_cmp(a,b,len) <  0)
** if(a<=b)   <==>   if(unsx_cmp(a,b,len) <= 0)
** if(a==b)   <==>   if(unsx_cmp(a,b,len) == 0)
** if(a>=b)   <==>   if(unsx_cmp(a,b,len) >= 0)
** if(a> b)   <==>   if(unsx_cmp(a,b,len) >  0)
** a[len] b[len]
*/
int __inline unsx_cmp(const unsx *a, const unsx *b, int len) {
  int i;
  STACK_MACRO();

  for (i=len-1; 0<=i; i--) {
    if (a[i] > b[i]) { return 1; }
    if (a[i] < b[i]) { return -1; }
  }
  return 0;
}

/*
** Returns unsx-length of a
** a[len]
*/
int __inline unsx_countArr(const unsx *a, int len) {
  int i;
  STACK_MACRO();

  for (i=len-1; 0<=i; i--) {
    if (a[i]) { break; }
  }
  return i+1;
}

/*
** Returns the bit-length of a
*/
int __inline unsx_countBits(unsx a) {
  int i;
  STACK_MACRO();

  for (i=0; i<unsx_Bit_Length; i++,a>>=1) {
    if (a == 0) { break; }
  }
  return i;
}

/*
** Returns bit-length of a
** a[len]
*/
int __inline unsx_countBitsArr(const unsx *a, int len) {
  int i = len;
  STACK_MACRO();

  if ((i = unsx_countArr(a, i)) == 0) {
    return 0;
  }
  return ((i-1) * unsx_Bit_Length + unsx_countBits(a[i-1]));
}

/*
** a = unsx_gcd(b,c)
** a[len] b[len] c[len]
** c < b
** len < UNSX_LENGTH
*/
int unsx_gcd(unsx *a, const unsx *b, const unsx *c, int len) {
  int status = -ENOMEM;

  UNSX_NEW(t,UNSX_LENGTH);
  UNSX_NEW(u,UNSX_LENGTH);
  UNSX_NEW(v,UNSX_LENGTH);
  STACK_MACRO();

  if(!t || !u || !v) {
    goto leave;
  }

  unsx_set(u, b, len);
  unsx_set(v, c, len);

  while ( !unsx_isZero(v, len) ) {
    status = unsx_mod(t, u, len, v, len);
    if(status) {
      goto leave;
    }
    unsx_set(u, v, len);
    unsx_set(v, t, len);
  }
  unsx_set(a, u, len);

  status = 0;
leave:
  if(t)
    UNSX_FREE(t,UNSX_LENGTH);
  if(u)
    UNSX_FREE(u,UNSX_LENGTH);
  if(v)
    UNSX_FREE(v,UNSX_LENGTH);
  return(status);
}

int unsx_isPrimeFast(const unsx *a, int len) {
  int ret=0;
  unsx t, r;
  STACK_MACRO();

  if (UNSX_ISEVEN(a, len))
    return 0;

  len = unsx_countArr(a,len);

  t = unsx_Trial_Prime_size/2;
  ret = unsx_mod(&r, a, len, &t,1);
  if(ret < 0) {
    return(ret);
  }

  if (r%3 && r%5 && r%7
#if (unsx_Trial_Prime_size == 2*3*5*7*11)
        && r%11
#endif
  ) {
    /*
    ret = unsx_isPrimeTrialLarge(a,len);
    if(ret == 1)
    */
      ret = unsx_isPrimeSPRPn(a,unsx_TWO,len);
  }

  #ifdef UNSX_SECURE
    t = r = 0;
  #endif

  return ret;
}

/* Is this used? I don't think so - NTT COMWARE */
/* NB: Need to check return values for errors before this can be used */
#if 0
/*
** There is no need for this kind of test.
** MAJOR OVERKILL!
*/
int unsx_isPrimeFull(const unsx *a, int len) {
  STACK_MACRO();
  if (unsx_isPrimeTrial_2to29(a,len)) {
  if (unsx_isPrimeTrialLarge(a,len)) {
  if (unsx_isPrimeProb(a,10,len)) {
  if (unsx_isPrimePRPn(a,unsx_TWO,len)) {
  if (unsx_isPrimePRPn(a,unsx_THREE,len)) {
  if (unsx_isPrimePRPn(a,unsx_FIVE,len)) {
  if (unsx_isPrimePRPn(a,unsx_SEVEN,len)) {
  if (unsx_isPrimePRPn(a,unsx_ELEVEN,len)) {
  if (unsx_isPrimePRPn(a,unsx_THIRTEEN,len)) {
  if (unsx_isPrimePRPn(a,unsx_TWENTYTHREE,len)) {
  if (unsx_isPrimePRPn(a,unsx_1662803,len)) {
  if (unsx_isPrimeSPRPn(a,unsx_TWO,len)) {
  if (unsx_isPrimeSPRPn(a,unsx_THREE,len)) {
  if (unsx_isPrimeSPRPn(a,unsx_FIVE,len)) {
  if (unsx_isPrimeSPRPn(a,unsx_SEVEN,len)) {
  if (unsx_isPrimeSPRPn(a,unsx_ELEVEN,len)) {
  if (unsx_isPrimeSPRPn(a,unsx_THIRTEEN,len)) {
  if (unsx_isPrimeSPRPn(a,unsx_TWENTYTHREE,len)) {
  if (unsx_isPrimeSPRPn(a,unsx_1662803,len)) {
    return 1;
  }} } }}}}}}}} }}}}}}}}

  return 0;
}
#endif

/*
** Fermats Little Theorum. (Pseudo Primality)
** True if a is 2-PRP, 3-PRP, 5-PRP and 7-PRP
** a[len]
*/
int unsx_isPrimePRP(const unsx *a, int len) {
  int ret = 0;
  UNSX_NEW(tmp,UNSX_LENGTH);
  UNSX_NEW(aMinus1,UNSX_LENGTH);
  STACK_MACRO();

  if(!tmp || !aMinus1) {
    ret = -ENOMEM;  
    goto leave;
  }

  unsx_dec(aMinus1, a, len);

  unsx_modPow(tmp, unsx_TWO,aMinus1,a, len);
  if (unsx_isOne(tmp,len)) {       /* if (2^(a-1) MOD a == 1) { */
    unsx_modPow(tmp, unsx_THREE,aMinus1,a, len);
    if (unsx_isOne(tmp,len)) {     /* if (3^(a-1) MOD a == 1) { */
      unsx_modPow(tmp, unsx_FIVE,aMinus1,a, len);
      if (unsx_isOne(tmp,len)) {   /* if (5^(a-1) MOD a == 1) { */
        unsx_modPow(tmp, unsx_SEVEN,aMinus1,a, len);
        if (unsx_isOne(tmp,len)) { /* if (7^(a-1) MOD a == 1) { */
          ret = 1;
        } /* 7-PRP */
      } /* 5-PRP */
    } /* 3-PRP */
  } /* 2-PRP */

leave:
  if(tmp)
    UNSX_FREE(tmp,UNSX_LENGTH);
  if(aMinus1)
    UNSX_FREE(aMinus1,UNSX_LENGTH);
  return ret;
}

/*
** Fermats Little Theorum. (Pseudo Primality)
**   True if a is b-PRP
**   a[len] b[len]
*/
int unsx_isPrimePRPn(const unsx *a, const unsx *b, int len) {
  int ret = 0;
  UNSX_NEW(tmp, UNSX_LENGTH);
  UNSX_NEW(aMinus1, UNSX_LENGTH);
  STACK_MACRO();

  if(!tmp || !aMinus1) {
    ret = -ENOMEM;  
    goto leave;
  }

  unsx_dec(aMinus1, a,len);
  ret = unsx_modPow(tmp, b,aMinus1,a, len);
  if(ret < 0)
    goto leave;
  if (unsx_isOne(tmp,len)) {  /* if  b^(a-1) MOD a == 1 */
    ret = 1;
  } /* b-PRP */

leave:
  if(tmp)
    UNSX_FREE(tmp,UNSX_LENGTH);
  if(aMinus1)
    UNSX_FREE(aMinus1,UNSX_LENGTH);
  return ret;
}

/* Is this used? I don't think so - NTT COMWARE */
/* NB: Need to check return values for errors before this can be used */
#if 0
/*
** Iterative Primality Test
**   1  = prime   (1 - 0.5^iter) certain.
**   0 = if not prime (100% certain)
*/
#define RETURN(RET)\
{\
  ret = RET;\
  goto unsx_isPrimeProb_ret;\
}
int unsx_isPrimeProb(const unsx *a, int certainty, int len) {
  int n = (certainty+1)/2,
      i, ai, bitLen, ret = 1;
  UNSX_NEW(aMinus1,UNSX_LENGTH);
  UNSX_NEW(rnd,UNSX_LENGTH);
  UNSX_NEW(m,UNSX_LENGTH);
  UNSX_NEW(z,UNSX_LENGTH);
  UNSX_NEW(tmp,UNSX_LENGTH);
  STACK_MACRO();

  if (n <= 0) {
    RETURN(1);
  }
  if (!UNSX_TESTBIT(a,0)) {
    RETURN(0);
  }
/*  Should we catch a==1, a==2, a==3 and waste CPU time? */
/*  if (unsx_cmp(a,unsx_THREE,len) <= 0) */
/*    return (unsx_cmp(a,unsx_ONE,len) == 0) ? 0 : 1; */

  unsx_dec(aMinus1, a,len);       /* aMinus1 = a-1 */
  unsx_set(m, aMinus1, len);          /* m = a-1 */
  ai = unsx_lowestSetBit(m,len);      /* ai = lowest unsx_set bit */
  unsx_shr(m, m,ai,len);              /* m = m>>ai */
  bitLen = unsx_countBitsArr(a,len);  /* bitLen = bitLength = log(a)/log(2) */
  len = unsx_countArr(a,len);

  for (i=0; i<n; i++) {
    unsx_setZero(rnd, len);
    rnd[0] = rand() & 0x00010001;
    unsx_modPow(z, rnd,m,a, len);     /* z = rnd^m mod a */

    if (unsx_isOne(z,len) || unsx_isEqual(z,aMinus1,len)) {
      RETURN(1);
    }
    for (; 0<ai; ai--) {
      unsx_mult(tmp, z,z, len);  /* tmp = z^2 */
      unsx_mod(z,                /* z = z^2 mod a */
               tmp, len,
               a,   len);
      if (unsx_isOne(z,len)) {
        RETURN(1);
      }
      if (unsx_isEqual(z,aMinus1,len)) {
        RETURN(0);
      }
    }
  }

unsx_isPrimeProb_ret:
  #ifdef UNSX_SECURE
    bitLen = ai = 0;
  #endif
  UNSX_FREE(aMinus1, UNSX_LENGTH);
  UNSX_FREE(rnd, UNSX_LENGTH);
  UNSX_FREE(m, UNSX_LENGTH);
  UNSX_FREE(z, UNSX_LENGTH);
  UNSX_FREE(tmp, UNSX_LENGTH);

  return ret;
}
#undef RETURN
#endif

/*
** Enhancement on Fermats Little Theorum.  (Strong Pseudo Primality)
** Returns 1 if a is 2-SPRP, 13-SPRP, 23-SPRP and 1662803-SPRP
**   This is a PROOF for primality if a < 1,000,000,000
**   This can be sped up in a messy way...
** a[len]
*/
int unsx_isPrimeSPRP(const unsx *a, int len) {
  int s;

  STACK_MACRO();
  if ((s = unsx_isPrimeSPRPn(a,unsx_TWO,len)) > 0)         /* if 2-SPRP */
  if ((s = unsx_isPrimeSPRPn(a,unsx_THIRTEEN,len)) > 0)    /* if 13-SPRP */
  if ((s = unsx_isPrimeSPRPn(a,unsx_TWENTYTHREE,len)) > 0) /* if 23-SPRP */
  if ((s = unsx_isPrimeSPRPn(a,unsx_1662803,len)) > 0)     /* if 1662803-SPRP */
    return 1;
  return s;
}

/*
** Enhancement on Fermats Little Theorum.  (Strong Pseudo Primality)
** Returns 1 if a is b-SPRP
** a[len] b[len]
*/
#define RETURN(RET)\
  do {\
    ret = (RET);\
    goto unsx_isPrimeSPRPn_ret;\
  } while(0);
int unsx_isPrimeSPRPn(const unsx *a, const unsx *b, int len) {
  int c,i,ret=0;
  UNSX_NEW(tmp2,2*UNSX_LENGTH);
  UNSX_NEW(test,UNSX_LENGTH);
  UNSX_NEW(aMinus1,UNSX_LENGTH);
  UNSX_NEW(t,UNSX_LENGTH);
  STACK_MACRO();

  if(!tmp2 || !test || !aMinus1 || !t) {
    RETURN(-ENOMEM);
  }

  len = unsx_countArr(a,len);

  if (unsx_isEqual(a,b,len)) {      /* if (a==b) */
    RETURN(1);
  }

  ret = unsx_mod(tmp2,                    /* tmp2 = a % b */
           a,    len,
           b,    unsx_countArr(b,len));
  if(ret < 0) {
    RETURN(ret);
  }
  if (unsx_isZero(tmp2,len)) {      /* if ((a%b) == 0) */
    RETURN(0);
  }

  unsx_dec(aMinus1, a,len);         /* aMinus1 = a-1 */
  unsx_set(t, aMinus1, len);        /* t = a-1; */

  c = 0;
  while (0 == UNSX_ISEVEN(t,len)) { /* while (t is Even) */
    unsx_shr(t, t, 1, len);         /* t = t/2; */
    c++;                            /* c = c+1; */
  }

  ret = unsx_modPow(test, b,t,a, len);    /* test = b^t MOD a */
  if(ret < 0) {
    RETURN(ret);
  }
  /* if (test==1 || test==(a-1))  */
  if (unsx_isOne(test,len) || unsx_isEqual(test,aMinus1,len)) {
    RETURN(1);
  }

  for (i=0; i<c; i++) {
    yield();
    ret = unsx_mult(tmp2, test,test,len); /* test = test^2 MOD a */
    if(ret < 0) {
      RETURN(ret);
    }
    yield();
    ret = unsx_mod(test, tmp2,2*len, a,len);
    if(ret < 0) {
      RETURN(ret);
    }
    if (unsx_isEqual(test,aMinus1,len)) { /* if (test==(a-1)) */
      RETURN(1);
    }
  }

  ret = 0;
unsx_isPrimeSPRPn_ret:
  #ifdef UNSX_SECURE
    c = i = 0;
  #endif
  if(tmp2)
    UNSX_FREE(tmp2, 2*UNSX_LENGTH);
  if(test)
    UNSX_FREE(test, UNSX_LENGTH);
  if(aMinus1)
    UNSX_FREE(aMinus1, UNSX_LENGTH);
  if(t)
    UNSX_FREE(t, UNSX_LENGTH);

  return ret;
}
#undef RETURN

/* I don't think this works - NTT COMWARE */
#if 0
/*
** Trial division
*/
int unsx_isPrimeTrialLarge(const unsx *a, int len) {
  unsx X;
  int i, ret=1;
  unsigned int j;
  STACK_MACRO();

  for (i=0; i<unsx_Trial_Prime_Sets; i++) {
    ret = unsx_mod(&X, a, len, &unsx_Trial_Prime[i][1], 1);
    if(ret < 0) {
      return(ret);
    }
    for (j=2; j<unsx_Trial_Prime[i][0]; j++) {
      if (X % unsx_Trial_Prime[i][j] == 0) {
        ret = 0;
        break;
      }
    }
  }

  #ifdef UNSX_SECURE
    X = i = j = 0;
  #endif

  return ret;
}
#endif

#if 0
/* Is this used? I don't think so - NTT COMWARE */
/* NB: Need to check return values for errors before this can be used */
#if 0
/*
** Trial division 11,13,17,19,23,29,31
*/
int unsx_isPrimeTrial_11to31(const unsx *a, int len) {
  int ret=0;
  unsx N, X;
  STACK_MACRO();

  if (UNSX_ISODD(a,len)) {     /* Test for an even number because its cheep */
    N = 955049953;
    unsx_mod(&X, a,len, &N,1); /* 11*13*17*19*23*29*31 = 955049953 */
                               /*  So we do one division by the largest */
    if (X%11)                  /*  number that is a product of primes from 3 and up */
    if (X%13)                  /*  that is less than 2^32 and all the modulus classes */
    if (X%17)                  /*  are preserved and we can test them at the register */
    if (X%19)                  /*  level.  Sweet eh! */
    if (X%23)
    if (X%29)
    if (X%31)                  /* Same as "if (X%31 != 0)" sunsx_ince unsx_ZERO is 0 */
    {
      ret = 1;
    }
  }

#ifdef UNSX_SECURE
  X = 0;
#endif

  return ret;
}
#endif

/* Is this used? I don't think so - NTT COMWARE */
/* NB: Need to check return values for errors before this can be used */
#if 0
/*
** Trial division 2,3,5,7,11,13,17,19,23,29
*/
int unsx_isPrimeTrial_2to29(const unsx *a, int len) {
  int ret=0;
  unsx N, X;
  STACK_MACRO();

  if (UNSX_TESTBIT(a,0)) {      /* Test for an even number because its cheep */
    N = 0xc0cfd797;
    unsx_mod(&X, a,len, &N,1);  /* 3*5*7*11*13*17*19*23*29 = 3234846615 */
                                /*  So we do one division by the largest */
    if (X%3)                    /*  number that is a product of primes from 3 and up */
    if (X%5)                    /*  that is less than 2^32 and all the modulus classes */
    if (X%7)                    /*  are preserved and we can test them at the register */
    if (X%11)                   /*  level.  Sweet eh! */
    if (X%13)
    if (X%17)
    if (X%19)
    if (X%23)
    if (X%29)                   /* Same as "if (X%29 != 0)" since false is 0 */
    {
      ret = 1;
    }
  }

#ifdef UNSX_SECURE
  X = 0;
#endif

  return ret;
}
#endif

/*
** Returns index of lowest set bit
** a[len]
*/
int unsx_lowestSetBit(const unsx *a, int len) {
  int i, k;
  unsx t;
  STACK_MACRO();

  for (i=0,k=0; i<len && a[i]==0; i++) {
    k+=unsx_Bit_Length;
  }
  t = a[i];
  for (; !(t&1); t>>=1) {
    k++;
  }

  #ifdef UNSX_SECURE
    t = 0;
  #endif

  return k;
}
#endif

/*
** Finds a for: (a * b) % c = 1
** a[len] b[len] c[len]
** unsx_gcd(b,c) = 1
** len < UNSX_LENGTH
*/
int unsx_modInv(unsx *a,
                 const unsx *b, const unsx *c, int len) {
  int sign;
  int status = -ENOMEM;
  UNSX_NEW(q, UNSX_LENGTH);
  UNSX_NEW(w, 2*UNSX_LENGTH);
  UNSX_NEW(t1, UNSX_LENGTH);
  UNSX_NEW(t3, UNSX_LENGTH);
  UNSX_NEW(u1, UNSX_LENGTH);
  UNSX_NEW(u3, UNSX_LENGTH);
  UNSX_NEW(v1, UNSX_LENGTH);
  UNSX_NEW(v3, UNSX_LENGTH);
  STACK_MACRO();

  if(!q || !w || !t1 || !t3 || !u1 || !u3 || !v1 || !v3) 
    goto leave;

  unsx_setLow(u1, 1, len);
  unsx_setZero(v1, len);
  unsx_set(u3, b, len);
  unsx_set(v3, c, len);
  sign = 1;

  while ( !unsx_isZero(v3,len) ) {
    status = unsx_divi(q, t3, u3, len, v3, len);  /* t3 = u3 mod v3,  
						     q = u3 / v3 */
    if(status < 0)
      goto leave;
    status = unsx_mult(w, q, v1, len);            /* w = q * v1 */
    if(status < 0)
      goto leave;
    unsx_add(t1, u1, w, len);                     /* t1 = u1 + w */
    unsx_set(u1, v1, len);
    unsx_set(v1, t1, len);
    unsx_set(u3, v3, len);
    unsx_set(v3, t3, len);
    sign = -sign;
  }

  if (sign < 0) {
    unsx_sub(a, c, u1, len);
  } else {
    unsx_set(a, u1, len);
  }

  status = 0;
leave:
  if(q)
    UNSX_FREE(q, UNSX_LENGTH);
  if(w)
    UNSX_FREE(w, 2*UNSX_LENGTH);
  if(t1)
    UNSX_FREE(t1, UNSX_LENGTH);
  if(t3)
    UNSX_FREE(t3, UNSX_LENGTH);
  if(u1)
    UNSX_FREE(u1, UNSX_LENGTH);
  if(u3)
    UNSX_FREE(u3, UNSX_LENGTH);
  if(v1)
    UNSX_FREE(v1, UNSX_LENGTH);
  if(v3)
    UNSX_FREE(v3, UNSX_LENGTH);
  sign = 0;

  return(status);
}

/*
** a = b * c mod d
** a[len] b[len] c[len] d[len]
** 0 < d
** len < UNSX_LENGTH
*/
int unsx_modMult(unsx *a, const unsx *b, const unsx *c, const unsx *d, 
		int len) {
  int status;
  UNSX_NEW(tmp1, 2*UNSX_LENGTH);
  STACK_MACRO();

  if(!tmp1)
    return(-ENOMEM);

  status = unsx_mult(tmp1, b, c, len);         /* tmp1 = b*c */
  if(status < 0)
    goto leave;
  status = unsx_mod(a, tmp1, 2*len, d, len);   /* a = (b*c) mod d */
  if(status < 0)
    goto leave;

leave:
  UNSX_FREE(tmp1, 2*UNSX_LENGTH);
  return(status);
}

/*
** a = b^c % d
** a[dLen], b[dLen], c[cLen], d[dLen].
** 0 < d
** 0 < cLen
** dLen < UNSX_LENGTH
** simple BinExp algo, doesn't use much stack space...
** ...perfect fro kernel crypto
*/
int unsx_modPowJL(unsx *a,
                   const unsx *b,
                   const unsx *c,
                   const unsx *d, int len) {
  int status = -ENOMEM;
  UNSX_NEW(tmp1, 2*UNSX_LENGTH);
  UNSX_NEW(b1, UNSX_LENGTH);
  UNSX_NEW(c1, UNSX_LENGTH);
  STACK_MACRO();

  if(!tmp1 || !b1 || !c1) {
    status = -ENOMEM;
    goto leave;
  }

  unsx_set(b1, b, len);                     /* b1 = b                */
  unsx_set(c1, c, len);                     /* c1 = c                */
  unsx_setOne(a,len);                       /* a  = 1                */
  while ( !unsx_isZero(c1,len) ) {          /* while (c1 != 0) {     */
    if (UNSX_ISODD(c1,len)) {               /*   if (c1 is odd) {    */
      status = unsx_mult(tmp1, a,b1, len);  /*     tmp1 = a * b1;    */
      if(status < 0)                        /*                       */
	goto leave;                         /*                       */
      status = unsx_mod(a,                  /*     a = (a*b1) mod d; */
               tmp1, 2*len,                 /*                       */
               d,    len);                  /*                       */ 
      if(status < 0)                        /*                       */
	goto leave;                         /*                       */
    }                                       /*   }                   */
    status = unsx_mult(tmp1, b1,b1, len);   /*   tmp1 = b1^2;        */
    if(status < 0)                          /*                       */
      goto leave;                           /*                       */
    status = unsx_mod(b1,                   /*   b1 = b1^2 mod d;    */
             tmp1, 2*len,                   /*                       */
             d,    len);                    /*                       */
    if(status < 0)                          /*                       */
      goto leave;                           /*                       */
    unsx_shr(c1, c1, 1, len);               /*   c1 = c1 / 2;        */
  }                                         /* }                     */

leave:
  if(tmp1)
    UNSX_FREE(tmp1, 2*UNSX_LENGTH);
  if(b1)
    UNSX_FREE(b1, UNSX_LENGTH);
  if(c1)
    UNSX_FREE(c1, UNSX_LENGTH);

  return(status);
}

/* Is this used? I don't think so - NTT COMWARE */
/* NB: Need to check return values for errors before this can be used */
#if 0
#define ORDER_LOG\
  2
#define ORDER\
  ( (1 << ORDER_LOG) -1 )
#define DIGIT_MSB(x)\
  (int)(((x) >> (unsx_Bit_Length - ORDER_LOG)) & ORDER)
/*
** a = b^c % d
** a[dLen], b[dLen], c[cLen], d[dLen].
** 0 < d
** 0 < cLen
** dLen < UNSX_LENGTH
** uses alot of stack space...
** ...not good for kernel crypto
*/
void unsx_modPowRSAREF(unsx *a,
                       const unsx *b,
                       const unsx *c, int cDigits,
                       const unsx *d, int dDigits) {
  unsx ci;
  unsx *bPower[ORDER];
  int i, ciBits, j, s;
  UNSX_NEW(t, UNSX_LENGTH);
  STACK_MACRO();

  for (i=0; i<ORDER; i++)
    bPower[i] = (unsx*)kmalloc(UNSX_LENGTH*sizeof(unsx), GFP_KERNEL);

  /* Store b, b^2 mod d, and b^3 mod d. */
  unsx_set(bPower[0], b, dDigits);
  for (i=1; i<ORDER; i++)
    unsx_modMult(bPower[i], bPower[i-1], b, d, dDigits);
  unsx_setOne(t, dDigits);

  cDigits = unsx_countArr(c, cDigits);
  for (i=cDigits-1; 0<=i; i--) {
    ci = c[i];
    ciBits = unsx_Bit_Length;

    /* Scan past leading zero bits of most significant digit. */
    if (i == (int)(cDigits - 1)) {
      while (!DIGIT_MSB(ci)) {
        ci <<= ORDER_LOG;
        ciBits -= ORDER_LOG;
      }
    }

    for (j=0; j<ciBits; j+=ORDER_LOG,ci<<=ORDER_LOG) {
      /* Compute t = t^4 * b^s mod d, where s = ORDER_LOG MSBs of ci. */
      unsx_modMult(t, t, t, d, dDigits);
      unsx_modMult(t, t, t, d, dDigits);
      if ((s=DIGIT_MSB(ci)) != 0) {
        unsx_modMult(t, t, bPower[s-1], d, dDigits);
      }
    }
  }

  unsx_set(a, t, dDigits);

  #ifdef UNSX_SECURE
    ci = i = j = s = ciBits = 0;
  #endif
  for (i=0; i<ORDER; i++)
    UNSX_FREE(bPower[i], UNSX_LENGTH);
  UNSX_FREE(t, UNSX_LENGTH);
}
#undef ORDER
#undef DIGIT_MSB
#endif

/* Is this used? I don't think so - NTT COMWARE */
#if 0
/*!
** unsx_set a[len] to a prime number of blen bits length.
*/
int unsx_newPrime(unsx *a, int len, int blen) {
  STACK_MACRO();
  if (blen > (UNSX_LENGTH*unsx_Bit_Length)) {
    unsx_setZero(a,len);
    return;
  }

  UNSX_SETBIT(a,blen-1);  /* asure it's the right size */
  a[0] |= 1;              /* asure it's odd */
  return(unsx_nextPrime(a,len));
}
#endif

/*
** Incerments a and will only end once a becomes prime
** a[len]
*/
int unsx_nextPrime(unsx *a, int len) {
  int status;
  UNSX_NEW(tmp1, UNSX_LENGTH);
  unsx tmpN;
  unsx trials = unsx_Trial_Prime_size;
  STACK_MACRO();

  if(!tmp1) {
    return(-ENOMEM);
  }

  /* Where is a on the MOD unsx_Trial_Prime_size ring? */
  /* tmpN = a mod unsx_Trial_Prime_size */
  unsx_mod(&tmpN, a, len, &trials, 1);

  /* since unsx_Trial_Prime_size < 2^32, tmpN = a mod unsx_Trial_Prime_size */
  /* Now distance to next number not unsx_divisable 
   * by factors of unsx_Trial_Prime_size */
  unsx_setLow(tmp1, unsx_Prime_unsx_add[tmpN], len);
  /* ... to a */
  unsx_add(a, a, tmp1, len);
  /* go to the next index... */
  tmpN = (tmpN+unsx_Prime_unsx_add[tmpN]) % unsx_Trial_Prime_size;

  do {
    tmp1[0] = unsx_Prime_unsx_add[tmpN];
    tmpN = (tmpN+tmp1[0]) % unsx_Trial_Prime_size;
    unsx_add(a, a, tmp1, len);
    status = unsx_isPrimeFast(a, len);
    /* status = unsx_isPrimePRPn(a, unsx_TWO, len); */
    if(status < 0) {
      goto leave;
    }
    if(status) {
      break;
    }
  } while (1);

  status = 0;
leave:
  #ifdef UNSX_SECURE
    tmpN = 0;
  #endif

  UNSX_FREE(tmp1, UNSX_LENGTH);
  return(status);
}

/*
** a = 2^b
** a[len]
** b < (len*unsx_Bit_Length)
*/
void unsx_set2Exp(unsx *a, int b, int len) {
  STACK_MACRO();
  unsx_setZero(a, len);
  if (b >= len * unsx_Bit_Length) {
    return;
  }
  a[b >> unsx_log] = 1 << (b & (unsx_Bit_Length-1));
}

/*
** a = b * 2^c
** Returns carry
** a[len] b[len]
** c < unsx_Bit_Length
*/
unsx unsx_shl(unsx *a, const unsx *b, int c, int len) {
  unsx ai,
       carry;
  int i, t;
  STACK_MACRO();

  if (c == 0) {
    unsx_set(a,b,len);
    return 0;
  }

  t = unsx_Bit_Length - c;
  carry = 0;
  for (i=0; i<len; i++) {
    ai = b[i];
    a[i] = (ai << c) | carry;
    carry = ai >> t;
  }

  #ifdef UNSX_SECURE
    ai = t = 0;
  #endif

  return carry;
}

/*
** a = b / 2^c
** Returns carry
** a[len] b[len]
** c < unsx_Bit_Length
*/
unsx unsx_shr(unsx *a, const unsx *b, int c, int len) {
  unsx ai,
       carry;
  int i, t;
  STACK_MACRO();

  if (c == 0) {
    unsx_set(a,b,len);
    return 0;
  }

  t = unsx_Bit_Length - c;
  carry = 0;
  for (i=len-1; 0<=i; i--) {
    ai = b[i];
    a[i] = (ai >> c) | carry;
    carry = c ? (ai << t) : 0;
  }

  #ifdef UNSX_SECURE
    ai = t = 0;
  #endif

  return carry;
}

#endif /* UNSX_C */
