/**********************************************************************
 * handshake_t.h                                            August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#ifndef _HANDSHAKE_T_H
#define _HANDSHAKE_T_H

#include "types/base_t.h"
#include "types/cipher_suite_t.h"
#include "types/change_cipher_spec_t.h"
#include "types/security_parameters_t.h"
#include "types/record_t.h"


/**********************************************************************
 * Specialised Types
 * Used in handshake messages
 **********************************************************************/

#define RANDOM_NLEN  RANDOM_NBYTES


typedef struct {
	opaque_t random_bytes[RANDOM_NBYTES];
} random_t;

typedef struct {
	u8 len;
	opaque_t *id;  /* <0..32> */
} session_id_t;

typedef struct {
	pos24_t len;    /* Only 24 bits are used */
	opaque_t *cert; /* <2^24-1> */
} asn1_cert_t;

typedef struct {
	protocol_version_t client_version;
	opaque_t random[46];
} pre_master_secret_t;

typedef struct {
	u16 len;
	opaque_t *encrypted_data;
	u16 offset;                     /* Not part of spec */
} encrypted_pre_master_secret_t;

typedef struct {
	opaque_t *encrypted_data;
} encrypted_pre_master_secret_ssl3_t;

typedef struct {
	union {
		/* struct {} implicit; */
		struct {
			u16 len;
			opaque_t *dh_yc;
		} explicit;
	} dh_public;
} diffie_hellman_client_public_value_t;

typedef enum {
	cct_rsa_sign=1,
	cct_dss_sign=2,
	cct_fixed_dh=3,
	cct_dss_fixed_dh=4,
	cct_last=255
} client_certificate_type_t;

typedef struct {
	u16 len;
	opaque_t dn;
} distinguished_name_t;

typedef union {
	/* struct {} anonymous; */
	struct {
		opaque_t md5_hash[16];
		opaque_t sha_hash[20];
	} rsa;
	struct {
		opaque_t sha_hash[20];
	} dsa;
} signature_t;

typedef struct {
	struct {
		u16 len;
		opaque_t modulus;
	} rsa_modulus;
	struct {
		u16 len;
		opaque_t exponent;
	} rsa_exponent;
} server_rsa_params_t;

typedef struct {
	struct {
		u16 len;
		opaque_t p;
	} dh_p;
	struct {
		u16 len;
		opaque_t g;
	} dh_g;
	struct {
		u16 len;
		opaque_t ys;
	} dh_ys;
} server_dh_params_t;


/**********************************************************************
 * Handshake Messages
 **********************************************************************/

/* typedef struct { } server_hello_done_t */


typedef struct {
	protocol_version_t client_version;
	random_t random;
	session_id_t session_id;
	struct {
		u16 len;
		cipher_suite_t *cs; /* < 2..2^16-1> */
	} cipher_suites;
	struct {
		u8 len;
		u8 *cm; /* <1..2^8-1> */
	} compression_methods;
} client_hello_t;


typedef struct {
	protocol_version_t server_version;
	random_t random;
	session_id_t session_id;
	cipher_suite_t cipher_suite;
	compression_method_t compression_method;
} server_hello_t;

#define SERVER_HELLO_BODY_NLEN(session_id_len)                               \
	( PROTOCOL_VERSION_NLEN + RANDOM_NLEN + (1 + (session_id_len)) +     \
	  CIPHER_SUITE_NLEN + COMPRESSION_METHOD_NLEN )

#define SERVER_HELLO_NLEN(session_id_len)                                    \
	( HANDSHAKE_HEAD_NLEN + SERVER_HELLO_BODY_NLEN(session_id_len) )


typedef struct {
	struct {
		pos24_t len;
		asn1_cert_t cert;
	} certificate_list;
} certificate_t;


typedef struct {
	struct {
		server_dh_params_t params;
		signature_t signed_params;
	} diffie_hellman;
	struct {
		server_rsa_params_t params;
		signature_t signed_params;
	} rsa;
} server_key_exchange_t;


typedef struct {
	struct {
		u8 len;
		client_certificate_type_t *ct;
	} certificate_types;
	struct {
		u16 len;
		distinguished_name_t *ca;
	} certificate_authorities;
} certificate_request_t;


typedef struct {
	signature_t signature;
} certificate_verify_t;


typedef struct {
	union {
		encrypted_pre_master_secret_t rsa;
		diffie_hellman_client_public_value_t diffie_hellman;
	} exchange_keys;
} client_key_exchange_t;


typedef union {
	struct {
		opaque_t md5_hash[16];
		opaque_t sha1_hash[20];
	} ssl3;
	struct {
		opaque_t verify_data[12];
	} tls;
} finished_t;


/**********************************************************************
 * Handshake
 **********************************************************************/

typedef enum {
	ht_hello_request=0,
	ht_client_hello=1,
	ht_server_hello=2,
	ht_certificate=11,
	ht_server_key_exchange=12,
	ht_certificate_request=13,
	ht_server_hello_done=14,
	ht_certificate_verify=15,
	ht_client_key_exchange=16,
	ht_finished=20,
	ht_last=255
} handshake_type_t;

typedef struct {
	handshake_type_t msg_type;
	pos24_t length;
	union {
		/* hello_request_t hello_request; Empty Body */
		client_hello_t client_hello;
		server_hello_t server_hello;
		certificate_t certificate;
		server_key_exchange_t server_key_exchange;
		certificate_request_t certificate_request;
		/* server_hello_done_t server_hello_done; Empty Body */
		certificate_verify_t certificate_verify;
		client_key_exchange_t client_key_exchange;
		finished_t finished;
	} body;
} handshake_t;

/* buf must be at least 4 bytes long */
static inline void handshake_head_to_buf(const handshake_t *handshake, 
		u8 *buf)
{
	u32 len;

	len = htonl(handshake->length);
	memcpy(buf, &len, sizeof(u32));
	*buf = handshake->msg_type;
}

/* buf must be at least 4 bytes long */
static inline void handshake_head_parts_to_buf(u32 len, 
		handshake_type_t msg_type, u8 *buf)
{
	len = htonl(len);
	memcpy(buf, &len, sizeof(u32));
	*buf = msg_type;
}

/* buf must be at least 4 bytes long */
static inline void handshake_head_from_buf(handshake_t *handshake, 
		const u8 *buf)
{
	u32 len;
	u8 *tmp;

	tmp = (u8 *)&len;

	memcpy(tmp, buf, sizeof(u32));
	*tmp = 0;
	handshake->length = htonl(len);
	handshake->msg_type = *buf;
}

#define HANDSHAKE_HEAD_NLEN 4

#endif /* _HANDSHAKE_T_H */
