/**********************************************************************
 * aep.c                                                    August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include <linux/kernel.h>
#include <linux/init.h>

#include "aep.h"
#include "socket.h"
#include "log.h"

#define AEP_F_NOCTXCMD 0x00000020
#define AEP_F_ASIC     0x00000100
#define AEP_SOCK_NAME "/usr/aep/aep_socket"

static struct socket *aep_sock = NULL;

typedef struct {
	u64 app_id;
	u64 trans_hndl;
	u32 command;
	u32 status;
	u32 flags;
	u32 datalen;
} aep_msg_hdr_t;


static int
kssl_aep_open(void)
{
	int status;
	
	status = kssl_socket_unix_open_connect(AEP_SOCK_NAME, &aep_sock);
	if (status < 0) {
		KSSL_DEBUG(10, "kssl_aep_open: "
				"kssl_socket_unix_open_connect\n");
		return status;
	}

	return 0;
}

ssize_t
kssl_aep_write(void *buf, size_t len)
{
	size_t i;
	int status;
	struct msghdr msg;
	struct iovec iov;

	if (!aep_sock) {
		int status;
		status = kssl_aep_open();
		if (status < 0)
			return status;
	}

	memset(&msg, 0, sizeof(msg));
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
	msg.msg_flags = MSG_DONTWAIT;
	iov.iov_base = (void *)buf;
	iov.iov_len = len;
	

	for (i = 0; i < 2; i++) {
		status = sock_sendmsg(aep_sock, &msg, len);
		if (status >= 0 || status == -EAGAIN)
			break;

		if (aep_sock) {
			kssl_socket_close(aep_sock);
			aep_sock = NULL;
		}

		if (status != -EPIPE)
			break;

		if(kssl_aep_open() < 0)
			break;
	}

	return status;
}


ssize_t
kssl_aep_read(void *buf, size_t len)
{
	size_t i;
	int status;	
	struct msghdr msg;
	struct iovec iov;

	if (!aep_sock) {
		int status;
		status = kssl_aep_open();
		if (status < 0)
			return status;
	}

	memset(&msg, 0, sizeof(msg));
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
	msg.msg_flags = MSG_DONTWAIT;
	iov.iov_base = (void *)buf;
	iov.iov_len = len;

	for (i = 0; i < 2; i++) {
		status = sock_recvmsg(aep_sock, &msg, len, 0);
		if (status >= 0 || status == -EAGAIN)
			break;

		if (aep_sock) {
			kssl_socket_close(aep_sock);
			aep_sock = NULL;
		}

		if (status != -EPIPE)
			break;

		if(kssl_aep_open() < 0)
			break;
	}

	return status;
}


void
kssl_aep_reverse_8(u8 *dst, const u8 *src, size_t len) 
{
	size_t i;

	for (i = 0; i < len; i++) {
		*(dst+len-i-1) = *(src+i);
	}
}


static inline size_t 
pad_32bit(size_t len)
{
	size_t pad;

	pad = 4 - (len % 4);
	if (pad != 4)
		return len + pad;
	
	return len;
}


#define REQ_MEM_CPY(dest, src, len)                                          \
do {                                                                         \
	if (flag & KSSL_AEP_REVERSE)                                         \
		kssl_aep_reverse_8(dest, src, len);                          \
	else                                                                 \
		memcpy(dest, src, len);                                      \
} while(0)

int
kssl_aep_msg_make_req_crt(u64 id, u64 handle, const u8 *a, size_t a_len,
		const u8 *p, size_t p_len, const u8 *q, size_t q_len,
		const u8 *dmpi, size_t dmpi_len, 
		const u8 *dmqi, size_t dmqi_len,
		const u8 *iqmp, size_t iqmp_len, 
		u8 *buf, size_t *buf_len, u32 flag)
{
	size_t offset;
	size_t len;
	aep_msg_hdr_t hdr;
	u32 tmp;

	len = pad_32bit(36 + pad_32bit(a_len) + pad_32bit(p_len) + 
			pad_32bit(q_len) + pad_32bit(dmpi_len) + 
			pad_32bit(dmqi_len) + pad_32bit(iqmp_len));

	if (len  + 32 > *buf_len) {
		KSSL_DEBUG(3, "message_make_req_crt: buffer too short\n");
		return -1;
	}

	*buf_len = len + 32;

	hdr.app_id = id;
	hdr.trans_hndl = handle;
	hdr.command = 0x2;
	hdr.status = 0x10000;
	hdr.flags = AEP_F_NOCTXCMD | AEP_F_ASIC;
	hdr.datalen = len;

	memset(buf, 0, len);

	memcpy(buf, &hdr, 32);
	offset = 32;

	tmp = 2;
	memcpy(buf+offset, &tmp, 4);
	offset += 12;

	tmp = pad_32bit(a_len);
	memcpy(buf+offset, &tmp, 4);
	offset += 4;
	REQ_MEM_CPY(buf+offset, a, a_len);
	offset += tmp;

	tmp = pad_32bit(p_len);
	memcpy(buf+offset, &tmp, 4);
	offset += 4;
	REQ_MEM_CPY(buf+offset, p, p_len);
	offset += tmp;

	tmp = pad_32bit(q_len);
	memcpy(buf+offset, &tmp, 4);
	offset += 4;
	REQ_MEM_CPY(buf+offset, q, q_len);
	offset += tmp;

	tmp = pad_32bit(dmpi_len);
	memcpy(buf+offset, &tmp, 4);
	offset += 4;
	REQ_MEM_CPY(buf+offset, dmpi, dmpi_len);
	offset += tmp;

	tmp = pad_32bit(dmqi_len);
	memcpy(buf+offset, &tmp, 4);
	offset += 4;
	REQ_MEM_CPY(buf+offset, dmqi, dmqi_len);
	offset += tmp;

	tmp = pad_32bit(iqmp_len);
	memcpy(buf+offset, &tmp, 4);
	offset += 4;
	REQ_MEM_CPY(buf+offset, iqmp, iqmp_len);
	offset += tmp;

	return 0;
}


int
kssl_aep_msg_make_req(u64 id, u64 handle, const u8 *b, size_t b_len,
		const u8 *e, size_t e_len, const u8 *n, size_t n_len,
		u8 *buf, size_t *buf_len, u32 flag)
{
	size_t offset;
	size_t len;
	aep_msg_hdr_t hdr;
	u32 tmp;

	len = pad_32bit(12 + pad_32bit(b_len) + pad_32bit(e_len) + 
			pad_32bit(n_len));

	if (len  + 32 > *buf_len) {
		KSSL_DEBUG(3, "message_make_req_crt: buffer too short\n");
		return -1;
	}

	*buf_len = len + 32;

	hdr.app_id = id;
	hdr.trans_hndl = handle;
	hdr.command = 0x1;
	hdr.status = 0x10000;
	hdr.flags = AEP_F_NOCTXCMD | AEP_F_ASIC;
	hdr.datalen = len;

	memset(buf, 0, len);

	memcpy(buf, &hdr, 32);
	offset = 32;

	tmp = pad_32bit(b_len);
	memcpy(buf+offset, &tmp, 4);
	offset += 4;
	REQ_MEM_CPY(buf+offset, b, b_len);
	offset += tmp;

	tmp = pad_32bit(e_len);
	memcpy(buf+offset, &tmp, 4);
	offset += 4;
	REQ_MEM_CPY(buf+offset, e, e_len);
	offset += tmp;

	tmp = pad_32bit(n_len);
	memcpy(buf+offset, &tmp, 4);
	offset += 4;
	REQ_MEM_CPY(buf+offset, n, n_len);
	offset += tmp;

	return(0);
}

void kssl_aep_cleanup(void)
{
	if (aep_sock)
		kssl_socket_close(aep_sock);

	return;
}                       

