/**********************************************************************
 * l7vsadm.c                                                August 2005
 *
 * L7VSADM: Virtual Server Administration Program for L7vsd
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#define _GNU_SOURCE     /* for getopt_long() */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/un.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <getopt.h>
#include <vanessa_logger.h>
#include <vanessa_adt.h>
#include "glib.h"
#include "l7vs.h"

#define L7VSADM_NAME            "l7vsadm"
#define L7VSADM_LIST_VS_BUFSIZE 1024
#define L7VSADM_LIST_RS_RSNUM   16
#define L7VSADM_DEFAULT_BACKLOG 127
#define L7VSADM_DEFAULT_SCHEDULER "rr"

#define L7VSADM_RECV_FAILED             (-1)
#define L7VSADM_RECV_TOO_LARGE          (-2)

struct l7vsadm_options {
        /* XXX not yet */
#define L7VSADM_OP_LIST         0       /* run as default */
#define L7VSADM_OP_ADD_VS       1
#define L7VSADM_OP_ADD_RS       2
#define L7VSADM_OP_DEL_VS       3
#define L7VSADM_OP_DEL_RS       4
#define L7VSADM_OP_EDIT_VS      5
#define L7VSADM_OP_EDIT_RS      6
#define L7VSADM_OP_FLUSH_VS     7
#define L7VSADM_OP_VERBOSE	8
#define L7VSADM_OP_KEY		9
        int operation;
        struct sockaddr_in vaddr;
        struct sockaddr_in raddr;
        char *protomodname;
        char *schedmod;
        struct l7vs_protomod *protomod;
        int persist;
        int backlog;
        int weight;
        int isnumeric;
        vanessa_dynamic_array_t *pm_args;
};

struct l7vsadm_operations {
        int code;
        int (*func)(int, struct l7vsadm_options *);
};

extern void *l7vs_module_load(char *modname, char *type);

static void usage(FILE *fp);
static int parse_args(struct l7vsadm_options *opt, int argc, char *argv[]);
static int parse_service(struct sockaddr_in *addr, const char *srvstr);
static char *get_servicename(struct sockaddr_in *addr, int isnumeric);
static int create_socket(void);
static void destroy_socket(void);
static int send_request(int s, void *req, size_t len);
static int sendv_request(int s, struct iovec *iov, int iovlen);
static int recv_response(int s, struct iovec *iov, size_t iovlen);
static const char *config_strerror(int err);
static int list_vs(int s, struct l7vsadm_options *opt);
static int list_rs(int s, struct l7vs_service_arg *sarg, int isnumeric);
static int operate_vs(int s, struct l7vsadm_options *opt);
static int operate_rs(int s, struct l7vsadm_options *opt);
static void sig_exit_handler(int sig);
static int set_signal(void);

static const char l7vs_config_sockname[] = L7VS_CONFIG_SOCK_PATH "/l7vs";
static const char *l7vs_version_string = L7VS_VERSION_STRING;
static char *local_sockname;

static void
usage(FILE *fp)
{
        fprintf(fp,
                "Usage: \n"
                "  l7vsadm -A|E -t service-address -m proto-module [module-args] [-s scheduler]\n"
                "  l7vsadm -D -t service-address -m proto-module [module-args]\n"
                "  l7vsadm -C\n"
#if 0
                "  l7vsadm -R\n"
                "  l7vsadm -S [-n]\n"
#endif
                "  l7vsadm -a|e -t service-address -m proto-module [module-args]\n"
                "          -r server-address\n"
                "  l7vsadm -d -t service-address -m proto-module [module-args]\n"
                "          -r server-address\n"
                "  l7vsadm -L|l [-n]\n"
                "  l7vsadm -V\n"
                "  l7vsadm -K\n"
                "  l7vsadm -h\n\n");
        fprintf(fp,
                "Commands:\n"
                "  --add-service     -A        add virtual service with options\n"
                "  --edit-service    -E        edit virtual service with options\n"
                "  --delete-service  -D        delete virtual service\n"
#if 0
                "  --restore         -R        restore rules from stdin\n"
                "  --save            -S        save rules to stdout\n"
#endif
                "  --add-server      -a        add real server with options\n"
                "  --edit-server     -e        edit real server with options\n"
                "  --list            -L|l      list the table\n\n"
                "  --verbose         -V        list the table in verbose format\n"
                "  --key             -K        list the table in key setting format\n\n"
                );
        fprintf(fp,
                "Options:\n"
                "  --tcp-service  -t service-address   service-address is host:port\n"
                "  --scheduler    -s scheduler         one of rr,lc\n");
}

/*
 * XXX Comment here.
 */
static int
parse_args(struct l7vsadm_options *opt, int argc, char *argv[])
{
#define OPTION_STRING   "-lVKADECadenr:m:t:s:w:"

        static struct option options[] = {
                /* operation options (toplevel) */
                {"list",                no_argument,            NULL, 'l'},
                {"verbose",		no_argument,            NULL, 'V'},
		{"key",			no_argument,		NULL, 'K'},
                {"add-service",         no_argument,            NULL, 'A'},
                {"delete-service",      no_argument,            NULL, 'D'},
                {"edit-service",        no_argument,            NULL, 'E'},
                {"flush",               no_argument,            NULL, 'C'},
                {"add-server",          no_argument,            NULL, 'a'},
                {"delete-server",       no_argument,            NULL, 'd'},
                {"edit-server",         no_argument,            NULL, 'e'},

                {"real-server",         required_argument,      NULL, 'r'},
                {"proto-module",        required_argument,      NULL, 'm'},

                {"tcp-service",         required_argument,      NULL, 't'},
                {"scheduler",           required_argument,      NULL, 's'},

                {"weight",              required_argument,      NULL, 'w'},

                {"numeric",             no_argument,            NULL, 'n'},

                {"help",                no_argument,            NULL, 'h'},

                {NULL,                  0,                      NULL, 0}
        };

        int c;
        int ret;

#define SET_OP(OPTP, OP) \
do {                                                                    \
        if ((OPTP)->operation) {                                        \
                VANESSA_LOGGER_ERR("Operation option conflicts");       \
                exit(1);                                                \
        }                                                               \
        (OPTP)->operation = (OP);                                       \
                                                                        \
} while (0)

        opterr = 0;
        while ((c = getopt_long(argc, argv, OPTION_STRING, options, NULL)) != -1) {
                switch (c) {
                case 'l':
                        SET_OP(opt, L7VSADM_OP_LIST);
                        break;
                case 'V':
                        SET_OP(opt, L7VSADM_OP_VERBOSE);
                        break;
		case 'K':
			SET_OP(opt, L7VSADM_OP_KEY);
			break;
                case 'A':
                        SET_OP(opt, L7VSADM_OP_ADD_VS);
                        break;
                case 'D':
                        SET_OP(opt, L7VSADM_OP_DEL_VS);
                        break;
                case 'E':
                        SET_OP(opt, L7VSADM_OP_EDIT_VS);
                        break;
                case 'C':
                        SET_OP(opt, L7VSADM_OP_FLUSH_VS);
                        break;
                case 'a':
                        SET_OP(opt, L7VSADM_OP_ADD_RS);
                        break;
                case 'd':
                        SET_OP(opt, L7VSADM_OP_DEL_RS);
                        break;
                case 'e':
                        SET_OP(opt, L7VSADM_OP_EDIT_RS);
                        break;
                case 't':
                        ret = parse_service(&opt->vaddr, optarg);
                        if (ret < 0) {
                                VANESSA_LOGGER_ERR_UNSAFE("Could not parse"
                                                " service string: %s",
                                                optarg);
                                exit(1);
                        }
                        break;
                case 'r':
                        ret = parse_service(&opt->raddr, optarg);
                        if (ret < 0) {
                                VANESSA_LOGGER_ERR_UNSAFE("Could not parse"
                                                " service string: %s",
                                                optarg);
                                exit(1);
                        }
                        break;

                case 's':
                        if (opt->schedmod != NULL) {
                                VANESSA_LOGGER_ERR("scheduler module option"
                                                   "conflicts");
                                exit(1);
                        }
                        if (strlen(optarg) >= L7VS_MODNAME_LEN - 1) {
                                VANESSA_LOGGER_ERR_UNSAFE("%s: name too long",
                                                          optarg);
                                exit(1);
                        }
                        opt->schedmod = strdup(optarg);
                        if (opt->schedmod == NULL) {
                                VANESSA_LOGGER_ERR("Could not allocate memory");
                                exit(1);
                        }
                        break;

                case 'm':
                        if (opt->protomodname != NULL) {
                                VANESSA_LOGGER_ERR("protocol module option"
                                                   "conflicts");
                                exit(1);
                        }
                        if (strlen(optarg) >= L7VS_MODNAME_LEN - 1) {
                                VANESSA_LOGGER_ERR_UNSAFE("%s: name too long",
                                                          optarg);
                                exit(1);
                        }
                        opt->protomodname = strdup(optarg);
                        if (opt->protomodname == NULL) {
                                VANESSA_LOGGER_ERR("Could not allocate memory");
                                exit(1);
                        }
                        opt->protomod = (struct l7vs_protomod *)
                                l7vs_module_load(opt->protomodname,
                                                 "protomod");
                        if (opt->protomod == NULL) {
                                exit(1);
                        }

                        break;

                case 'w':
                        opt->weight = atoi(optarg);
                        break;
                
                case 'n':
                        opt->isnumeric = 1;
                        break;

                case 'h':
                        usage(stdout);
                        exit(0);
                        break;

                case '1':
                default:
//                      VANESSA_LOGGER_DEBUG_UNSAFE("added %s", argv[optind-1]);
                        if (vanessa_dynamic_array_add_element(
                                opt->pm_args, argv[optind - 1]) == NULL) {
                                VANESSA_LOGGER_ERR("Could not allocate memory");
                                exit(1);
                                
                        }
                        break;
                }

        }
#undef SET_OP

        return 0;
}

static int
parse_service(struct sockaddr_in *addr, const char *srvstr)
{
        struct addrinfo *res, hints;
        struct sockaddr_in *sin;
        char *s, *t;
        int ret;

        s = strdup(srvstr);
        t = strrchr(s, ':');
        if (t == NULL) {
                free(s);
                return -1;
        }
        *t++ = '\0';

        memset(&hints, 0, sizeof(hints));
        hints.ai_family = PF_INET; /* inet only. no support for inet6 (yet) */
        hints.ai_socktype = SOCK_STREAM;
        ret = getaddrinfo(s, t, &hints, &res);
        if (ret != 0) {
                if (ret == EAI_NONAME) {
                        hints.ai_flags = AI_NUMERICHOST;
                        ret = getaddrinfo(s, t, &hints, &res);
                        if (ret == 0) {
                                goto found;
                        }
                }
                VANESSA_LOGGER_ERR_UNSAFE("%s:%s: %s",
                                          s, t, gai_strerror(ret));
                free(s);
                return -1;
        }
found:
        free(s);

        /*
         * We always use the first entry, because we can't distinguish
         * which entry you want to specify if we have multiple entries.
         */
        sin = (struct sockaddr_in *)res->ai_addr;
        if (sin->sin_addr.s_addr == htonl(INADDR_ANY)) {
                VANESSA_LOGGER_ERR("You can't specify INADDR_ANY"
                                   " for virtual service");
                freeaddrinfo(res);
                return -1;
        }
        
        if (sin->sin_port == htons(0)) {
                VANESSA_LOGGER_ERR("You can't specify port number 0");
                freeaddrinfo(res);
                return -1;
        }

        *addr = *sin;
        freeaddrinfo(res);

        return 0;
}

static char *
get_servicename(struct sockaddr_in *addr, int isnumeric)
{
        char *sname;
        char hostname[40], portname[20];
        int flags;
        int ret;

        flags = 0;
        if (isnumeric) {
                flags = NI_NUMERICHOST | NI_NUMERICSERV;
        }

        ret = getnameinfo((struct sockaddr *)addr, sizeof(*addr),
                          hostname, sizeof(hostname),
                          portname, sizeof(portname), flags);
        if (ret != 0) {
                VANESSA_LOGGER_ERR_UNSAFE("getnameinfo: %s",
                                          gai_strerror(ret));
                return NULL;
        }

        ret = asprintf(&sname, "%s:%s", hostname, portname);
        if (ret < 0) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return NULL;
        }

        return sname;
}

static int
create_socket(void)
{
        struct sockaddr_un addr;
        char *sockname;
        int opt;
        int ret;
        int s;

        ret = asprintf(&sockname, "%s/l7vsadm-%d",
                       L7VS_CONFIG_SOCK_PATH, getpid());
        if (ret < 0) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return ret;
        }

        unlink(sockname);
        if (ret >= sizeof(addr.sun_path)) {
                VANESSA_LOGGER_ERR("Internal error. socket name too long.");
                free(sockname);
                return -1;
        }

        memset(&addr, 0, sizeof(addr));
        memcpy(addr.sun_path, sockname, ret);
        addr.sun_family = AF_LOCAL;

        s = socket(PF_LOCAL, SOCK_DGRAM, 0);
        if (s < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("socket: %s", strerror(errno));
                free(sockname);
                return s;
        }

        opt = 1;
        ret = setsockopt(s, SOL_SOCKET, SO_PASSCRED, &opt, sizeof(opt));
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("setsockopt SO_PASSCRED: %s",
                                          strerror(errno));
                free(sockname);
                close(s);
                return ret;
        }

        ret = bind(s, (struct sockaddr *)&addr, SUN_LEN(&addr));
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("bind on %s: %s",
                                          sockname, strerror(errno));
                close(s);
                return ret;
        }

        local_sockname = sockname;

        memset(&addr, 0, sizeof(addr));
        addr.sun_family = AF_LOCAL;
        strcpy(addr.sun_path, l7vs_config_sockname);

        ret = connect(s, (struct sockaddr *)&addr, SUN_LEN(&addr));
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("connect to daemon: %s",
                                          strerror(errno));
                close(s);
                return ret;
        }

        return s;
}

static void
destroy_socket(void)
{
        if (local_sockname == NULL) {
                return;
        }

        unlink(local_sockname);
}

static int
send_request(int s, void *req, size_t len)
{
        int ret;

        ret = send(s, req, len, 0);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("send to daemon: %s",
                                          strerror(errno));
        }

        return ret;
}

static int
sendv_request(int s, struct iovec *iov, int iovlen)
{
        int ret;

        ret = writev(s, iov, iovlen);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("send to daemon: %s",
                                          strerror(errno));
        }

        return ret;
}

static int
recv_response(int s, struct iovec *iov, size_t iovlen)
{
        struct msghdr msg;
        struct ucred *cred;
        struct cmsghdr *cmsg;
        unsigned char cbuf[CMSG_LEN(sizeof(struct ucred))];
        int ret;

        memset(&msg, 0, sizeof(msg));
        msg.msg_control = cbuf;
        msg.msg_controllen = sizeof(cbuf);
        msg.msg_iov = iov;
        msg.msg_iovlen = iovlen;
        cmsg = (struct cmsghdr *)cbuf;

        ret = recvmsg(s, &msg, 0);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("recvmsg: %s", strerror(errno));
                return L7VSADM_RECV_FAILED;
        }

        if (msg.msg_flags & MSG_CTRUNC) {
                VANESSA_LOGGER_ERR("Invalid response from l7vsd");
                return L7VSADM_RECV_FAILED;
        }

        if (msg.msg_flags & MSG_TRUNC) {
                return L7VSADM_RECV_TOO_LARGE;
        }

        if (! (cmsg->cmsg_level == SOL_SOCKET &&
               cmsg->cmsg_type == SCM_CREDENTIALS)) {
                VANESSA_LOGGER_ERR("Could not receive a remote credential");
                return L7VSADM_RECV_FAILED;
        }

        cred = (struct ucred *)CMSG_DATA(cmsg);
        if (cred->uid != 0) {
                VANESSA_LOGGER_ERR("Response from unprivileged user");
                return L7VSADM_RECV_FAILED;
        }

        return ret;
}

static const char *
config_strerror(int err)
{
        switch (err) {
        case 0:
                return "No error";
        case L7VS_CONFIG_ERR_INVALID_COMMAND:
                return "Invalid command";
        case L7VS_CONFIG_ERR_NOMEM:
                return "Could not allocate memory";
        case L7VS_CONFIG_ERR_VS_EXISTS:
                return "Virtual service already exists";
        case L7VS_CONFIG_ERR_RS_EXISTS:
                return "Real server already exists";
        case L7VS_CONFIG_ERR_NOVS:
                return "No such virtual service";
        case L7VS_CONFIG_ERR_NORS:
                return "No such real server";
        case L7VS_CONFIG_ERR_NOSCHED:
                return "No such scheduler";
        case L7VS_CONFIG_ERR_NOSOCK:
                return "Could not create a service socket";
        default:
                return "Unknown error";
        }
}

static int
list_vs(int s, struct l7vsadm_options *opt)
{
        struct l7vs_config_req_list_vs req;
        struct l7vs_config_rsp_list_vs rsp;
        struct l7vs_service_arg *sarg;
        struct iovec iov[2];
        void *buf;
        char *srvname;
        int bufsize;
        int i, off;
        int ret;

        bufsize = L7VSADM_LIST_VS_BUFSIZE;

retry:
        buf = malloc(bufsize);
        if (buf == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return -1;
        }

	if (bufsize == L7VSADM_LIST_VS_BUFSIZE) {
        	printf("Layer-7 Virtual Server version %s\n", l7vs_version_string);
        	printf("Prot LocalAddress:Port Scheduler ProtoMod\n"
        	       "  -> RemoteAddress:Port           Forward Weight ActiveConn InactConn\n");
	}

        req.cmd = L7VS_CONFIG_LIST_VS;
        ret = send_request(s, &req, sizeof(req));
        if (ret < 0) {
                free(buf);
                return -1;
        }

        iov[0].iov_base = &rsp;
        iov[0].iov_len = sizeof(rsp);
        iov[1].iov_base = buf;
        iov[1].iov_len = bufsize;

        ret = recv_response(s, iov, sizeof(iov) / sizeof(iov[0]));
        if (ret < 0) {
                free(buf);
                if (ret == L7VSADM_RECV_TOO_LARGE) {
                        bufsize += L7VSADM_LIST_VS_BUFSIZE;
                        goto retry;
                }
                return -1;
        }

        if (rsp.code != 0) {
                VANESSA_LOGGER_ERR(config_strerror(rsp.code));
                free(buf);
                return -1;
        }

        /* now display it */
        sarg = (struct l7vs_service_arg *)buf;
        off = 0;
        for (i = 0; i < rsp.num; i++) {
                /* XXX should be replaced with pretty-print routine. */
                srvname = get_servicename(&sarg->addr, opt->isnumeric);
                if (srvname == NULL) {
                        break;
                }

		/* If it ie verbose, display more details else simple details */
		if (opt->operation == L7VSADM_OP_VERBOSE) {
                	printf("TCP %s %s %s %d %s\n",
                       srvname, sarg->protomod, sarg->schedmod, sarg->reschedule, sarg->protomod_opt_string);
		} else if (opt->operation == L7VSADM_OP_KEY) {
                	printf("TCP %s %s %s %d %s\n",
                       srvname, sarg->protomod, sarg->schedmod, sarg->reschedule, sarg->protomod_key_string);
		} else {
                	printf("TCP %s %s %s\n",
                       srvname, sarg->protomod, sarg->schedmod);
		}

                free(srvname);
                list_rs(s, sarg, opt->isnumeric);

                off += sarg->len;
                if (off >= bufsize) {
                        VANESSA_LOGGER_INFO("Message size did not match");
                        VANESSA_LOGGER_INFO("version mismatch between"
                                            " l7vsd and l7vsadm?");
                        break;
                }
                sarg = (struct l7vs_service_arg *)((char *)buf + off);
        }

        free(buf);

        return 0;
}

static int
list_rs(int s, struct l7vs_service_arg *sarg, int isnumeric)
{
        struct l7vs_config_req_list_rs req;
        struct l7vs_config_rsp_list_vs rsp;
        struct l7vs_dest_arg *darg;
        struct iovec iov[2];
        char *srvname;
        int ret;
        int i, num;

        num = L7VSADM_LIST_RS_RSNUM;
retry:
        darg = (struct l7vs_dest_arg *)malloc(num * sizeof(*darg));
        if (darg == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return -1;
        }

        req.cmd = L7VS_CONFIG_LIST_RS;
        iov[0].iov_base = &req;
        iov[0].iov_len = sizeof(req);
        iov[1].iov_base = sarg;
        iov[1].iov_len = sarg->len;
        ret = sendv_request(s, iov, sizeof(iov) / sizeof(iov[0]));
        if (ret < 0) {
                free(darg);
                return -1;
        }

        iov[0].iov_base = &rsp;
        iov[0].iov_len = sizeof(rsp);
        iov[1].iov_base = darg;
        iov[1].iov_len = num * sizeof(*darg);

        ret = recv_response(s, iov, sizeof(iov) / sizeof(iov[0]));
        if (ret < 0) {
                free(darg);
                if (ret == L7VSADM_RECV_TOO_LARGE) {
                        num = rsp.num;
                        goto retry;
                }
                return -1;
        }

        if (rsp.code != 0) {
                VANESSA_LOGGER_ERR(config_strerror(rsp.code));
                return -1;
        }
        /* now display it */
        num = rsp.num;
        for (i = 0; i < num; i++) {
                srvname = get_servicename(&darg[i].addr, isnumeric);
                if (srvname == NULL) {
                        break;
                }
                printf("  -> %-28s %-7s %-6d %-10d 0\n",
                       srvname,
                       "Masq",
                       darg[i].weight,
                       darg[i].nactive);
                free(srvname);
        }

        free(darg);
        return 0;
}

static int
operate_vs(int s, struct l7vsadm_options *opt)
{
        struct l7vs_protomod *pm = opt->protomod;
        struct l7vs_service_arg *sarg;
        struct l7vs_config_req_operate_vs req;
        struct l7vs_config_rsp_operate_vs rsp;
        struct iovec iov[2];
        char *opstr;
        int ret;

        assert(opt->operation == L7VSADM_OP_ADD_VS ||
               opt->operation == L7VSADM_OP_DEL_VS ||
               opt->operation == L7VSADM_OP_EDIT_VS);

        switch (opt->operation) {
        case L7VSADM_OP_ADD_VS:
                req.cmd = L7VS_CONFIG_ADD_VS;
                opstr = "add";
                break;
        case L7VSADM_OP_DEL_VS:
                req.cmd = L7VS_CONFIG_DEL_VS;
                opstr = "delete";
                break;
        case L7VSADM_OP_EDIT_VS:
                req.cmd = L7VS_CONFIG_EDIT_VS;
                opstr = "edit";
                break;
        }

        if (pm == NULL) {
                VANESSA_LOGGER_ERR_UNSAFE("Need -m option"
                                   " to %s a virtual service", opstr);
                return -1;
        }

        sarg = pm->create_sa();
        sarg->addr = opt->vaddr;
        sarg->proto = IPPROTO_TCP;
        sarg->persist = opt->persist;
        sarg->backlog = opt->backlog;
        if (opt->schedmod == NULL) {
                opt->schedmod = strdup(L7VSADM_DEFAULT_SCHEDULER);
                if (opt->schedmod == NULL) {
                        VANESSA_LOGGER_ERR("Could not allocate memory");
                        exit(1);
                }
        }
        strcpy(sarg->schedmod, opt->schedmod);
        
        ret = pm->parse(sarg, vanessa_dynamic_array_get_count(opt->pm_args),
                        (char **)
                            vanessa_dynamic_array_get_vector(opt->pm_args));
        if (ret < 0) {
                VANESSA_LOGGER_ERR("protocol-module parameter error!");
                l7vs_module_unload(pm->handle);
                free(sarg);
                return -1;
        }

        iov[0].iov_base = &req;
        iov[0].iov_len = sizeof(req);
        iov[1].iov_base = sarg;
        iov[1].iov_len = sarg->len;

        ret = sendv_request(s, iov, sizeof(iov) / sizeof(iov[0]));
        if (ret < 0) {
                l7vs_module_unload(pm->handle);
                free(sarg);
                return -1;
        }

        iov[0].iov_base = &rsp;
        iov[0].iov_len = sizeof(rsp);
        ret = recv_response(s, iov, 1);
        if (ret < 0) {
                l7vs_module_unload(pm->handle);
                free(sarg);
                return -1;
        }

        if (rsp.code != 0) {
                VANESSA_LOGGER_ERR(config_strerror(rsp.code));
                l7vs_module_unload(pm->handle);
                free(sarg);
                return -1;
        }

        l7vs_module_unload(pm->handle);
        free(sarg);

        return 0;
}

static int
operate_rs(int s, struct l7vsadm_options *opt)
{
        struct l7vs_protomod *pm;
        struct l7vs_service_arg *sarg;
        struct l7vs_config_req_operate_rs req;
        struct l7vs_config_rsp_operate_rs rsp;
        struct iovec iov[2];
        char *opstr;
        int ret;

        assert(opt->operation == L7VSADM_OP_ADD_RS ||
               opt->operation == L7VSADM_OP_DEL_RS ||
               opt->operation == L7VSADM_OP_EDIT_RS);

        switch (opt->operation) {
        case L7VSADM_OP_ADD_RS:
                req.cmd = L7VS_CONFIG_ADD_RS;
                opstr = "add";
                break;
        case L7VSADM_OP_DEL_RS:
                req.cmd = L7VS_CONFIG_DEL_RS;
                opstr = "delete";
                break;
        case L7VSADM_OP_EDIT_RS:
                req.cmd = L7VS_CONFIG_EDIT_RS;
                opstr = "edit";
                break;
        }

        req.darg.addr = opt->raddr;
        if (opt->weight == 0) {
                req.darg.weight = 1;
        } else {
                req.darg.weight = opt->weight;
        }

        if (opt->protomodname == NULL) {
                VANESSA_LOGGER_ERR_UNSAFE("Need -m option"
                                   " to %s a virtual service", opstr);
                return -1;
        }

        pm = (struct l7vs_protomod *)
                l7vs_module_load(opt->protomodname, "protomod");
        if (pm == NULL) {
                return -1;
        }

        sarg = pm->create_sa();
        sarg->addr = opt->vaddr;
        sarg->proto = IPPROTO_TCP;
        
        optind = 0;
        ret = pm->parse(sarg, vanessa_dynamic_array_get_count(opt->pm_args),
                        (char **)
                            vanessa_dynamic_array_get_vector(opt->pm_args));
        if (ret < 0) {
                VANESSA_LOGGER_ERR("protocol-module parameter error!");
                l7vs_module_unload(pm->handle);
                free(sarg);
                return -1;
        }

        iov[0].iov_base = &req;
        iov[0].iov_len = sizeof(req);
        iov[1].iov_base = sarg;
        iov[1].iov_len = sarg->len;

        ret = sendv_request(s, iov, sizeof(iov) / sizeof(iov[0]));
        if (ret < 0) {
                l7vs_module_unload(pm->handle);
                free(sarg);
                return -1;
        }

        iov[0].iov_base = &rsp;
        iov[0].iov_len = sizeof(rsp);
        ret = recv_response(s, iov, 1);
        if (ret < 0) {
                l7vs_module_unload(pm->handle);
                free(sarg);
                return -1;
        }

        if (rsp.code != 0) {
                VANESSA_LOGGER_ERR(config_strerror(rsp.code));
                l7vs_module_unload(pm->handle);
                free(sarg);
                return -1;
        }

        l7vs_module_unload(pm->handle);
        free(sarg);

        return 0;
}

static int
flush_vs(int s, struct l7vsadm_options *opt)
{
        struct l7vs_config_req_flush_vs req;
        struct l7vs_config_rsp_flush_vs rsp;
        struct iovec iov;
        int ret;

        req.cmd = L7VS_CONFIG_FLUSH_VS;
        ret = send_request(s, &req, sizeof(req));
        if (ret < 0) {
                return ret;
        }

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        ret = recv_response(s, &iov, 1);
        if (ret < 0) {
                return ret;
        }

        if (rsp.code != 0) {
                VANESSA_LOGGER_ERR(config_strerror(rsp.code));
                return -1;
        }

        return 0;
}

static void
sig_exit_handler(int sig)
{
        VANESSA_LOGGER_DEBUG_UNSAFE("signal %d received", sig);
        destroy_socket();
        _exit(1);
}

struct l7vsadm_sig {
        int sig;
        void (*func)(int sig);
};

static int
set_signal(void)
{
        static struct l7vsadm_sig sigs[] = {
                {SIGHUP,        sig_exit_handler},
                {SIGINT,        sig_exit_handler},
                {SIGQUIT,       sig_exit_handler},
                {SIGPIPE,       sig_exit_handler},
                {SIGTERM,       sig_exit_handler},
                {SIGUSR1,       sig_exit_handler},
                {SIGUSR2,       sig_exit_handler},
                {0,             NULL}
        };
        struct l7vsadm_sig *s;
        struct sigaction act, oact;
        int ret;

        for (s = sigs; s->sig != 0; s++) {
                ret = sigaction(s->sig, NULL, &oact);
                if (ret < 0) {
                        VANESSA_LOGGER_ERR_UNSAFE("sigaction: %s",
                                                  strerror(errno));
                        return ret;
                }

                act = oact;
                act.sa_handler = s->func;
                ret = sigaction(s->sig, &act, NULL);
                if (ret < 0) {
                        VANESSA_LOGGER_ERR_UNSAFE("sigaction: %s",
                                                  strerror(errno));
                        return ret;
                }
        }

        return 0;
}

int
main(int argc, char *argv[])
{
        /* XXX test version */
        struct l7vsadm_options options;
        vanessa_logger_t *vl;
        int s;
        int ret;
        struct l7vsadm_operations *op;

        static struct l7vsadm_operations oplist[] = {
                {L7VSADM_OP_LIST,       list_vs},
                {L7VSADM_OP_ADD_VS,     operate_vs},
                {L7VSADM_OP_DEL_VS,     operate_vs},
                {L7VSADM_OP_EDIT_VS,    operate_vs},
                {L7VSADM_OP_ADD_RS,     operate_rs},
                {L7VSADM_OP_DEL_RS,     operate_rs},
                {L7VSADM_OP_EDIT_RS,    operate_rs},
                {L7VSADM_OP_FLUSH_VS,   flush_vs},
                {L7VSADM_OP_VERBOSE,	list_vs},
                {L7VSADM_OP_KEY,	list_vs},
                {-1, NULL}
        };

        vl = vanessa_logger_openlog_filehandle(stderr, "l7vsadm",
                                               LOG_DEBUG, 0);
        vanessa_logger_set(vl);
        if (set_signal() < 0) {
                exit(1);
        }

        memset(&options, 0, sizeof(options));
        options.backlog = L7VSADM_DEFAULT_BACKLOG;
        options.pm_args = vanessa_dynamic_array_create(0,
                                                       VANESSA_DESS,
                                                       VANESSA_DUPS,
                                                       VANESSA_DISS,
                                                       VANESSA_LENS);
        if (options.pm_args == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                exit(1);
        }

        /* Add argv[0] first because getopt_long() skip argv[0]... */
        if (vanessa_dynamic_array_add_element(options.pm_args, argv[0]) == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                exit(1);
        }

        l7vs_module_init(NULL);
        ret = parse_args(&options, argc, argv);
	if (ret < 0) {
		usage(stderr);
	}

        s = create_socket();
        if (s < 0) {
                exit(1);
        }

        for (op = oplist; op->code != -1; op++) {
                if (op->code == options.operation) {
                        break;
                }
        }

        if (op->code == -1) {
                destroy_socket();
                usage(stderr);
                exit(1);
        }

        ret = op->func(s, &options);

        destroy_socket();

	/* free for strdup() */
	free(options.schedmod);
	free(options.protomodname);

        l7vs_module_fini();

	/* free for vannesa_dynamic_array_create() */
	vanessa_dynamic_array_destroy(options.pm_args);

        if (ret < 0) {
                exit(1);
        }

        return 0;
}
