/**********************************************************************
 * iomux.c                                                  August 2005
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/time.h>
#include <sys/select.h>
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs.h"

static void l7vs_iomux_set_fds(struct l7vs_iomux *iom, int flags);

static GList *iomux_list;
static my_fd_set iomux_readfds, iomux_writefds, iomux_exceptfds;
static int iomux_maxfd;

void
l7vs_iomux_init(void)
{
        FD_ZERO(&iomux_readfds);
        FD_ZERO(&iomux_writefds);
        FD_ZERO(&iomux_exceptfds);
}

void
l7vs_iomux_fini(void)
{
}

static void
l7vs_iomux_set_fds(struct l7vs_iomux *iom, int flags)
{
        if (iomux_maxfd < iom->fd) {
                iomux_maxfd = iom->fd;
        }

        if (flags & L7VS_IOMUX_READ) {
                FD_SET(iom->fd, (fd_set *)&iomux_readfds);
        } else {
                FD_CLR(iom->fd, (fd_set *)&iomux_readfds);
        }

        if (flags & L7VS_IOMUX_WRITE) {
                FD_SET(iom->fd, (fd_set *)&iomux_writefds);
        } else {
                FD_CLR(iom->fd, (fd_set *)&iomux_writefds);
        }

        if (flags & L7VS_IOMUX_EXCEPT) {
                FD_SET(iom->fd, (fd_set *)&iomux_exceptfds);
        } else {
                FD_CLR(iom->fd, (fd_set *)&iomux_exceptfds);
        }

}

void
l7vs_iomux_add(struct l7vs_iomux *iom, int flags)
{
        l7vs_iomux_set_fds(iom, flags);
        iomux_list = g_list_append(iomux_list, iom);
}

void
l7vs_iomux_remove(struct l7vs_iomux *iom)
{
        l7vs_iomux_set_fds(iom, 0);
        iomux_list = g_list_remove(iomux_list, iom);
}

void
l7vs_iomux_change_flags(struct l7vs_iomux *iom, int flags)
{
        l7vs_iomux_set_fds(iom, flags);
}

int
l7vs_iomux_select(struct timeval *timo)
{
        int ret;
        int flags;
        my_fd_set readfds, writefds, exceptfds;
        GList *l, *next, *restart;
        struct l7vs_iomux *iom;

        readfds = iomux_readfds;
        writefds = iomux_writefds;
        exceptfds = iomux_exceptfds;
        ret = select(iomux_maxfd + 1, (fd_set *)&readfds, (fd_set *)&writefds, (fd_set *)&exceptfds, timo);
        if (ret <= 0) {
                /* Return immediately to keep errno. */
                return ret;
        }

        for (restart = l = g_list_first(iomux_list); l != NULL; ) {
                /*
                 * The list entry may be removed in the callback.
                 * So we keep list_next first.
                 */
                next = g_list_next(l);
                iom = (struct l7vs_iomux *)l->data;
                flags = 0;
                if (FD_ISSET(iom->fd, (fd_set *)&readfds)) {
                        flags |= L7VS_IOMUX_READ;
                        FD_CLR(iom->fd, (fd_set *)&readfds);
                }
                    
                if (FD_ISSET(iom->fd, (fd_set *)&writefds)) {
                        flags |= L7VS_IOMUX_WRITE;
                        FD_CLR(iom->fd, (fd_set *)&writefds);
                }
                    
                if (FD_ISSET(iom->fd, (fd_set *)&exceptfds)) {
                        flags |= L7VS_IOMUX_EXCEPT;
                        FD_CLR(iom->fd, (fd_set *)&exceptfds);
                }
                    
                ret = L7VS_IOMUX_LIST_UNCHANGED;
                if (flags) {
                        ret = iom->callback(iom, flags);
                }

                switch (ret) {
                case L7VS_IOMUX_LIST_UNCHANGED:
                        restart = l;
                        l = next;
                        break;
                case L7VS_IOMUX_LIST_REMOVED_MYSELF:
                        l = restart;
                        break;
                case L7VS_IOMUX_LIST_REMOVED_OTHER:
                        restart = l = g_list_first(iomux_list);
                        break;
                }
        }

        return ret;
}
