/**********************************************************************
 * lsock.c                                                  August 2005
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs.h"

/* static functions */
static void l7vs_lsock_accept(struct l7vs_lsock *lsock);
static void l7vs_lsock_table_add(struct l7vs_lsock *lsock);
static void l7vs_lsock_table_remove(struct l7vs_lsock *lsock);
static gint l7vs_lsock_addr_cmp(gconstpointer a, gconstpointer b);
static int l7vs_lsock_callback(struct l7vs_iomux *iom, int flags);

static GList *l7vs_lsock_list;

/*
 * init/fini functions
 */
int
l7vs_lsock_init(void)
{
        l7vs_lsock_list = NULL;
        return 0;
}

void
l7vs_lsock_fini(void)
{
        /* XXX  MUST clean-up lsocks, services, and so on... */
}

/*
 * get/put
 */
struct l7vs_lsock *
l7vs_lsock_get(struct sockaddr_in *sin, u_int8_t proto, int backlog)
{
        struct l7vs_lsock *lsock;
        int stype;
        int ret;

        lsock = l7vs_lsock_table_lookup(sin, proto);
        if (lsock != NULL) {
                lsock->refcnt++;
                return lsock;
        }
                                                         
        switch (proto) {
        case IPPROTO_TCP:
                stype = SOCK_STREAM;
                break;
        case IPPROTO_UDP:
                stype = SOCK_DGRAM;
                break;
        default:
                VANESSA_LOGGER_ERR_UNSAFE("Protocol number should be"
                                          " TCP or UDP (%d)",
                                          proto);
                return NULL;
        }

        lsock = (struct l7vs_lsock *)calloc(1, sizeof(*lsock));
        if (lsock == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate lsock");
                return lsock;
        }

        VANESSA_LOGGER_DEBUG_UNSAFE("creating lsock %p", lsock);
        lsock->proto = proto;
        lsock->iom.fd = socket(PF_INET, stype, proto);  //create a socket to get connection request from the client
        if (lsock->iom.fd < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("socket: %s", strerror(errno));
                free(lsock);
                return NULL;
        }

        ret = bind(lsock->iom.fd, (struct sockaddr *)sin, sizeof(*sin)); //binding the socket for incoming client request
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("Could not bind socket: %s",
                                          strerror(errno));
                close(lsock->iom.fd);
                free(lsock);
                return NULL;
        }
        lsock->addr = *sin;

        ret = listen(lsock->iom.fd, backlog); //listening for client requests
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("Could not listen: %s\n",
                                          strerror(errno));
                close(lsock->iom.fd);
                free(lsock);
                return NULL;
        }

        lsock->refcnt = 1;

        lsock->iom.callback = l7vs_lsock_callback;
        lsock->iom.data = lsock;
        l7vs_lsock_table_add(lsock);  //Add socket in the list. It may be used for maintaining session (Am not sure)
        l7vs_iomux_add(&lsock->iom, L7VS_IOMUX_READ);

        return lsock;
}

void
l7vs_lsock_put(struct l7vs_lsock *lsock)
{
        if (--lsock->refcnt > 0)
                return;

        VANESSA_LOGGER_DEBUG_UNSAFE("removing lsock %p", lsock);
        l7vs_iomux_remove(&lsock->iom);
        l7vs_lsock_table_remove(lsock);

        close(lsock->iom.fd);
        free(lsock);
}

/*
 * table operations
 */
static void
l7vs_lsock_table_add(struct l7vs_lsock *lsock)
{
        l7vs_lsock_list = g_list_append(l7vs_lsock_list, (gpointer)lsock);
}

static void
l7vs_lsock_table_remove(struct l7vs_lsock *lsock)
{
        l7vs_lsock_list = g_list_remove(l7vs_lsock_list, (gpointer)lsock);
}

struct l7vs_lsock *
l7vs_lsock_table_lookup(struct sockaddr_in *sin, u_int8_t proto)
{
        struct l7vs_lsock tmpl;
        GList *l;

        tmpl.addr = *sin;
        tmpl.proto = proto;

        l = g_list_find_custom(l7vs_lsock_list, (gpointer)&tmpl,
                               l7vs_lsock_addr_cmp);
        if (l == NULL)
                return NULL;
        return (struct l7vs_lsock *)l->data;  //for checking up and finding the socket in the list for a particular client socket(not sure)
}

static gint
l7vs_lsock_addr_cmp(gconstpointer a, gconstpointer b)
{
        struct l7vs_lsock *la = (struct l7vs_lsock *)a;
        struct l7vs_lsock *lb = (struct l7vs_lsock *)b;

        if (la->addr.sin_addr.s_addr != lb->addr.sin_addr.s_addr)
                return la->addr.sin_addr.s_addr - lb->addr.sin_addr.s_addr;
        if (la->addr.sin_port != lb->addr.sin_port)
                return la->addr.sin_port - lb->addr.sin_port;
        if (la->proto != lb->proto)
                return la->proto - lb->proto;

        return 0;
}

int
l7vs_lsock_select_service(struct l7vs_lsock *lsock, 
                          struct l7vs_conn *conn,
                          char *buf,
                          size_t len,
                          struct l7vs_service **srv_ret,
                          struct l7vs_dest **dest_ret,
                          int *tcps_ret)
{
        GList *l;
        struct l7vs_service *srv;
        struct l7vs_dest *dest;
        int ret, val;
        int tcps;

        val = -1;
        l = lsock->srv_list;
        while (l != NULL) {
                srv = (struct l7vs_service *)l->data;
                dest = NULL;
                tcps = 1;
                ret = srv->pm->match_cldata(srv, conn, buf, &len, &dest, &tcps);
                if (ret == 0) {
                        val = 1;
			if (len > conn->cldata_len + L7VS_PROTOMOD_MAX_ADD_BUFSIZE){
				VANESSA_LOGGER_ERR("bufsize too long modified by protomod ");
				return -1;
			} else {
				conn->cldata_len = len;
			}
                        *srv_ret = srv;
                        *dest_ret = dest;
                        *tcps_ret = tcps;
                        break;
                } else if (ret < 0) {
                        val = ret;
                        break;
                }

                /* continue checking if ret > 0 */
                l = g_list_next(l);
        }

        return val;
}

/*
 * service registration
 */
void
l7vs_lsock_add_service(struct l7vs_lsock *lsock,
                       struct l7vs_service *srv)
{
        lsock->srv_list = g_list_append(lsock->srv_list, srv);
}

void
l7vs_lsock_remove_service(struct l7vs_lsock *lsock,
                          struct l7vs_service *srv)
{
        lsock->srv_list = g_list_remove(lsock->srv_list, srv);
}

/*
 * socket operations
 */
static void
l7vs_lsock_accept(struct l7vs_lsock *lsock)
{
        l7vs_conn_create(lsock->iom.fd, lsock);
}

static int
l7vs_lsock_callback(struct l7vs_iomux *iom, int flags)
{
        if ((flags & L7VS_IOMUX_READ) == 0) {
                return L7VS_IOMUX_LIST_UNCHANGED;
        }

        l7vs_lsock_accept((struct l7vs_lsock *)iom->data); //for accepting data from clients

        return L7VS_IOMUX_LIST_UNCHANGED;
}
