/**********************************************************************
 * module.c                                                 August 2005
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#define _GNU_SOURCE
#include <sys/param.h>
#include <sys/types.h>
#include <dlfcn.h>
#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs.h"

#define L7VS_MODULE_INITFN              "init"

char l7vs_module_path[MAXPATHLEN];

int
l7vs_module_init(char *modpath)
{
        if (modpath == NULL) {
                strcpy(l7vs_module_path, L7VS_MODULE_PATH);
        } else {
                if (strlen(modpath) >= sizeof(l7vs_module_path) - 1) {
                        return -1;
                }
                strcpy(l7vs_module_path, modpath);
        }

        return 0;
}

void
l7vs_module_fini(void)
{
        return;
}

void *
l7vs_module_load(char *modname, char *type)
{
        int ret;
        void *h;
        void *modhandle;
        char *modpath;
        void *(*initf)(void *);

        ret = asprintf(&modpath, "%s/%s_%s.so", 
                       l7vs_module_path, type, modname);
        if (ret < 0) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return NULL;
        }

        h = dlopen(modpath, RTLD_LAZY);
        if (h == NULL) {
                VANESSA_LOGGER_ERR_UNSAFE("Could not open %s module: %s",
                                          modpath, dlerror());
                free(modpath);
                return NULL;
        }
        free(modpath);

        initf = dlsym(h, L7VS_MODULE_INITFN);
        if (initf == NULL) {
                VANESSA_LOGGER_ERR_UNSAFE("Could not find symbol %s: %s",
                                          L7VS_MODULE_INITFN, dlerror());
                dlclose(h);
                return NULL;
        }

        modhandle = (*initf)(h);
        if (modhandle == NULL) {
                VANESSA_LOGGER_ERR("Module initialization failed");
                dlclose(h);
                return NULL;
        }

        return modhandle;
}

void
l7vs_module_unload(void *handle)
{
        dlclose(handle);
}

void
l7vs_module_register(GList **modlist, void *mod)
{
        *modlist = g_list_append(*modlist, (gpointer)mod);
}

void
l7vs_module_remove(GList **modlist, void *mod)
{
        *modlist = g_list_remove(*modlist, mod);
}

void *
l7vs_module_lookup(GList *modlist, void *key, GCompareFunc cmp)
{
        GList *l;

        l = g_list_find_custom(modlist, (gpointer)key, cmp);
        if (l == NULL) {
                return NULL;
        }

        return (void *)l->data;
}
