/* 
 * @file    l7vs_service.h
 * @bref    service that brings function of l7vsd together.
 * @bref    It has sets of conn of ther service.
 * @bref    and sets of read servers.
 * @bref    and protocol module
 * @bref    and schedulers.
 *
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/
#ifndef L7VS_SERVICE_H
#define L7VS_SERVICE_H

#include <stdint.h>
#include <glib.h>
#include "l7vs_conn.h"
#include "l7vs.h"

struct l7vs_lsock;
struct l7vs_scheduler;
struct l7vs_protomod;
struct l7vs_dest;
struct l7vs_dest_arg;

typedef uint32_t handle_t;

#define TEMP_SERVICEHANDLE      UINT32_MAX
#define BPS_DEFAULT_INTERVAL    1000000ULL

// virtual service definition datatype
struct l7vs_service{
    handle_t               handle;     //! l7vs_service_handle
    struct l7vs_lsock*     lsock;      //! listen socket struct pointer
    GList*                 dest_list;  //! real server list
//    GList*                 conn_list;
    GHashTable*            conn_hash;  //! connection table
    struct l7vs_scheduler* scheduler;  //! scheduler pointer
    void*                  sched_data; //! scheduler data pointer
    struct l7vs_protomod*  pm;         //! protocol module pointer

    // ...and per-module match-data follows
    char              protomod_opt_string[L7VS_PROTOMOD_OPT_LEN]; //! protocol module option string
    char              protomod_key_string[L7VS_PROTOMOD_KEY_LEN]; //! protocol module key string
    int               reschedule;               //! reschedule flag
    int               sorry_cc;                 //! sorry connection count limit
    struct l7vs_dest* sorry_dest;               //! sorry-server destination
    int               sorry_flag;               //! sorry flag

    unsigned long long qos_threshold_up;            //! QoS threshold value from client to real server (UP Direction)
    unsigned long long qos_threshold_down;          //! QoS threshold value from real server to client (DOWN Direction)
    unsigned long long throughput_to_server;        //! Throughput to real server
    unsigned long long throughput_to_client;        //! Throughput to client
    size_t             pre_recvsize_from_client;    //! data recive size from client (before 1 sec)
    size_t             cur_recvsize_from_client;    //! data recive size from client (current)
    unsigned long long pre_recvtime_from_client;    //! Receiving time from client
    size_t             pre_recvsize_from_server;    //! data recive size from real server (before 1 sec)
    size_t             cur_recvsize_from_server;    //! data recive size from real server (current)
    unsigned long long pre_recvtime_from_server;    //! Receiving time from real server
    int                qos_up_flag;                 //! QoS to real server intervention flag (UP Direction)
    int                qos_down_flag;               //! QoS to client intervention flag (DOWN Direction)
};

// virtual service definition, external representation.
struct l7vs_service_arg{
    size_t             len;     //! structure size
    struct sockaddr_in addr;    //! server address
    uint8_t            proto;   //! server transport protocol
    int                persist; //! persistency
    int                backlog; //! listen(2) backlog length

    char               protomod[L7VS_MODNAME_LEN]; //! protocol module name
    char               schedmod[L7VS_MODNAME_LEN]; //! scheduler

    // ... and protocol-module defined data follows
    char                    protomod_opt_string[L7VS_PROTOMOD_OPT_LEN]; //! protocol module option string
    char                    protomod_key_string[L7VS_PROTOMOD_KEY_LEN]; //! protocol module key string
    int                     reschedule;               //! reshcedule
    int                     sorry_cc;                 //! sorry connection count limit
    struct sockaddr_storage sorry_addr;               //! sorry-server address
    int                     sorry_flag;               //! sorry flag

    unsigned long long qos_threshold_up;          //! QoS Threshold value from client to real server (UP Direction)
    unsigned long long qos_threshold_down;        //! QoS Threshold value from real server to client (DOWN Direction)
    unsigned long long throughput_to_server; //! Throughput to real server
    unsigned long long throughput_to_client; //! Throughput to client
};

// message of virtual service and protocol module definition, external representation.
struct l7vs_service_arg_multi{
    struct l7vs_service_arg srv_arg;
    char                    protomod_arg[L7VS_PROTOMOD_MAX_SERVICE_ARG];
};

// replication data
struct l7vs_service_repdata{
    struct sockaddr_in addr;  //! server address
    uint8_t            proto; //! server transport protocol

    char protomod[L7VS_MODNAME_LEN]; //! protocol module name
    char protomod_arg[L7VS_PROTOMOD_MAX_SERVICE_ARG];

    int                     sorry_cc;   //! sorry connection count limit
    struct sockaddr_storage sorry_addr; //! sorry-server address
    int                     sorry_flag; //! sorry flag

    unsigned long long qos_threshold_up;   //! QoS Threshold value from client to real server
    unsigned long long qos_threshold_down; //! QoS Threshold value from real server to client
};

//
// functions
//
extern struct l7vs_service* l7vs_service_create( struct l7vs_service_arg_multi*, int* );
extern int                  l7vs_service_destroy( struct l7vs_service* );
extern struct l7vs_service* l7vs_service_lookup( struct l7vs_service_arg_multi* );
extern int                  l7vs_service_add_dest( struct l7vs_service*, struct l7vs_dest_arg* );
extern int                  l7vs_service_remove_dest( struct l7vs_service*, struct l7vs_dest_arg* );
extern int                  l7vs_service_schedule( struct l7vs_service*, struct l7vs_conn* );
extern int                  l7vs_service_establish( struct l7vs_service*, struct l7vs_conn* );
extern struct l7vs_dest*    l7vs_service_lookup_dest( struct l7vs_service*, struct sockaddr_in* );
extern int                  l7vs_service_list_service_arg( struct l7vs_service_arg_multi**, int* );
extern int                  l7vs_service_list_dest_arg( struct l7vs_service*, struct l7vs_dest_arg** );
extern int                  l7vs_service_register_conn( struct l7vs_service*, struct l7vs_conn* );
extern int                  l7vs_service_remove_conn( struct l7vs_service*, struct l7vs_conn* );
extern void                 l7vs_service_flush_all();
extern int                  l7vs_service_set_QoS_Threshold( struct l7vs_service*, struct l7vs_service_arg_multi* );
extern int                  l7vs_service_set_SorryServer_Values( struct l7vs_service*, struct l7vs_service_arg_multi* );
extern unsigned int         l7vs_service_get_VSnum();
extern struct l7vs_service* l7vs_service_get_VSInfo_byNum( unsigned int );
extern struct l7vs_service* l7vs_service_get_VSInfo_byHandler( const handle_t );
extern void                 l7vs_service_update_throughput( struct l7vs_service*, unsigned long long );
extern struct l7vs_service_arg_multi * l7vs_service_get_service_arg(struct l7vs_service *);

/*!
 * make l7vs_service struct dump string
 * @param[out] retstr char pointer
 * @param[in]  srv l7vs_service struct pointer
 */
inline void
l7vs_service_c_str( char* retstr, struct l7vs_service* srv )
{
    if (srv == NULL) {
        snprintf(retstr, DEBUG_STR_LEN, "NULL");
    }else{
        //handle
        sprintf( retstr, "%s handle = %d", retstr, srv->handle );
        //lsock pointer
        //scheduler pointer
        //scheduler data pointer
        //protocol module pointer
        //protocol module option string
        sprintf( retstr, "%s protomod option string = %s", retstr, srv->protomod_opt_string );
        //protocol module key string
        sprintf( retstr, "%s protomod key string = %s", retstr, srv->protomod_key_string );
        //reschedule
        sprintf( retstr, "%s reschedule = %d", retstr, srv->reschedule );
        //sorry
        sprintf( retstr, "%s sorry conn count limit = %d", retstr, srv->sorry_cc );
        sprintf( retstr, "%s sorry flag = %d", retstr, srv->sorry_flag );
        //QoS
        sprintf( retstr, "%s QoS to server = %llu", retstr, srv->qos_threshold_up );
        sprintf( retstr, "%s QoS to client = %llu", retstr, srv->qos_threshold_down );
        //throughput
        sprintf( retstr, "%s Throughput to server = %llu", retstr, srv->throughput_to_server );
        sprintf( retstr, "%s Throughput to client = %llu", retstr, srv->throughput_to_client );
        if( NULL != srv->sorry_dest ){
            switch( srv->sorry_dest->addr.sin_family ){
            case AF_UNIX:
                sprintf( retstr, "%s socket family = AF_UNIX(PF_UNIX)", retstr );
                break;
            case AF_INET:
                sprintf( retstr, "%s socket family = AF_INET(PF_INET)", retstr );
                break;
            default:
                sprintf( retstr, "%s socket family = %d", retstr, srv->sorry_dest->addr.sin_family );
            }
            sprintf( retstr, "%s port no = %d", retstr, ntohs( srv->sorry_dest->addr.sin_port ) );
            sprintf( retstr, "%s address = %s", retstr, inet_ntoa( srv->sorry_dest->addr.sin_addr ) );
        }
        //The last receiving time
        sprintf( retstr, "%s recv time from client = %llu", retstr, srv->pre_recvtime_from_client );
        sprintf( retstr, "%s recv time from server = %llu", retstr, srv->pre_recvtime_from_server );
        //The last receiving size
        sprintf( retstr, "%s pre recv size from client = %zu", retstr, srv->pre_recvsize_from_client );
        sprintf( retstr, "%s pre recv size from server = %zu", retstr, srv->pre_recvsize_from_server );
        sprintf( retstr, "%s recv size from client = %zu", retstr, srv->cur_recvsize_from_client );
        sprintf( retstr, "%s recv size from server = %zu", retstr, srv->cur_recvsize_from_server );
    }
}

/*!
 * make l7vs_service_arg struct dump string
 * @param[out] retstr  char pointer
 * @param[in]  srv_arg l7vs_service_arg struct pointer
 */
inline void
l7vs_service_arg_c_str( char* retstr, struct l7vs_service_arg* srv_arg )
{
    if (srv_arg == NULL) {
        snprintf(retstr, DEBUG_STR_LEN, "NULL");
    }else{
        //data length
        sprintf( retstr, "%s struct data length = %zu", retstr, srv_arg->len );
        //service address dump
        switch( srv_arg->addr.sin_family ){
        case AF_UNIX:
            sprintf( retstr, "%s socket family = AF_UNIX(PF_UNIX)", retstr );
            break;
        case AF_INET:
            sprintf( retstr, "%s socket family = AF_INET(PF_INET)", retstr );
            break;
        default:
            sprintf( retstr, "%s socket family = %d", retstr, srv_arg->addr.sin_family );
        }
        sprintf( retstr, "%s port no = %d", retstr, ntohs( srv_arg->addr.sin_port ) );
        sprintf( retstr, "%s address = %s", retstr, inet_ntoa( srv_arg->addr.sin_addr ) );
        //proto
        switch( srv_arg->proto ){
        case IPPROTO_TCP:
            sprintf( retstr, "%s proto = IPPROTO_TCP", retstr );
            break;
        case IPPROTO_UDP:
            sprintf( retstr, "%s proto = IPPROTO_UDP", retstr );
            break;
        default:
            sprintf( retstr, "%s proto = %d", retstr, ntohs( srv_arg->proto ) );
        }
        //persist
        sprintf( retstr, "%s proto = %d", retstr, srv_arg->proto );
        //backlog
        sprintf( retstr, "%s backlog = %d", retstr, srv_arg->backlog );
        //protomod name
        sprintf( retstr, "%s protomod name = %s", retstr, srv_arg->protomod );
        //scheduler name
        sprintf( retstr, "%s scheduler name = %s", retstr, srv_arg->schedmod );
        //protomod option string
        sprintf( retstr, "%s protomod option string = %s", retstr, srv_arg->protomod_opt_string );
        //protomod key string
        sprintf( retstr, "%s protomod key string = %s", retstr, srv_arg->protomod_key_string );
        //reschedule
        sprintf( retstr, "%s backlog = %d", retstr, srv_arg->reschedule );
        //sorry_cc
        sprintf( retstr, "%s sorry conn count limit = %d", retstr, srv_arg->sorry_cc );
        //sorry address dump
        struct sockaddr_in* sorryaddr = (struct sockaddr_in*)&(srv_arg->sorry_addr);
        if( NULL != sorryaddr ){
            switch( sorryaddr->sin_family ){
            case AF_UNIX:
                sprintf( retstr, "%s sorry server socket family = AF_UNIX(PF_UNIX)", retstr );
                break;
            case AF_INET:
                sprintf( retstr, "%s sorry server socket family = AF_INET(PF_INET)", retstr );
                break;
            default:
                sprintf( retstr, "%s sorry server socket family = %d", retstr, sorryaddr->sin_family );
            }
            sprintf( retstr, "%s sorry server port no = %d", retstr, ntohs( sorryaddr->sin_port ) );
            sprintf( retstr, "%s sorry server address = %s", retstr, inet_ntoa( sorryaddr->sin_addr ) );
        }
        //sorry_flag
        sprintf( retstr, "%s sorry flag = %d", retstr, srv_arg->sorry_flag );
        //QoS setting value
        sprintf( retstr, "%s QoS control to server = %llu", retstr, srv_arg->qos_threshold_up );
        sprintf( retstr, "%s QoS control to client = %llu", retstr, srv_arg->qos_threshold_down );
        //throughput
        sprintf( retstr, "%s Throughput to server = %llu", retstr, srv_arg->throughput_to_server );
        sprintf( retstr, "%s Throughput to client = %llu", retstr, srv_arg->throughput_to_client );
    }
}

/*!
 * make l7vs_service_arg_multi struct dump string
 * @param[out] retstr  char pointer
 * @param[in]  srv_arg_multi l7vs_service_repdata struct pointer
 */
inline void
l7vs_service_arg_multi_c_str( char* retstr, struct l7vs_service_arg_multi* srv_arg_multi )
{
    if (srv_arg_multi == NULL) {
        snprintf(retstr, DEBUG_STR_LEN, "NULL");
    }else{
        //service_arg dump
        l7vs_service_arg_c_str( retstr, &srv_arg_multi->srv_arg );
        //protomodarg
        sprintf( retstr, "%s protomod args = %s", retstr, srv_arg_multi->protomod_arg );
    }
}

/*!
 * make l7vs_service_repdata struct dump string
 * @param[out] retstr  char pointer
 * @param[in]  srv_repdata l7vs_service_repdata struct pointer
 */
inline void
l7vs_service_repdata_c_str( char* retstr, struct l7vs_service_repdata* srv_repdata )
{
    if (srv_repdata == NULL) {
        snprintf(retstr, DEBUG_STR_LEN, "NULL");
    }else{
        //addr dump
        switch( srv_repdata->addr.sin_family ){
        case AF_UNIX:
            sprintf( retstr, "%s socket family = AF_UNIX(PF_UNIX)", retstr );
            break;
        case AF_INET:
            sprintf( retstr, "%s socket family = AF_INET(PF_INET)", retstr );
            break;
        default:
            sprintf( retstr, "%s socket family = %d", retstr, srv_repdata->addr.sin_family );
        }
        sprintf( retstr, "%s port no = %d", retstr, ntohs( srv_repdata->addr.sin_port ) );
        sprintf( retstr, "%s address = %s", retstr, inet_ntoa( srv_repdata->addr.sin_addr ) );
        //proto
        switch( srv_repdata->proto ){
        case IPPROTO_TCP:
            sprintf( retstr, "%s proto = IPPROTO_TCP", retstr );
            break;
        case IPPROTO_UDP:
            sprintf( retstr, "%s proto = IPPROTO_UDP", retstr );
            break;
        default:
            sprintf( retstr, "%s proto = %d", retstr, ntohs( srv_repdata->proto ) );
        }
        //protocol module name
        sprintf( retstr, "%s protomod name = %s", retstr, srv_repdata->protomod );
        //protomod argument
        sprintf( retstr, "%s protomod args = %s", retstr, srv_repdata->protomod_arg );
        //number of connection limit
        sprintf( retstr, "%s sorry conn count limit = %d", retstr, srv_repdata->sorry_cc );
        //sorry addr dump
        struct sockaddr_in* sorryaddr = (struct sockaddr_in*)&(srv_repdata->sorry_addr);
        switch( sorryaddr->sin_family ){
        case AF_UNIX:
            sprintf( retstr, "%s socket family = AF_UNIX(PF_UNIX)", retstr );
            break;
        case AF_INET:
            sprintf( retstr, "%s socket family = AF_INET(PF_INET)", retstr );
            break;
        default:
            sprintf( retstr, "%s socket family = %d", retstr, sorryaddr->sin_family );
        }
        sprintf( retstr, "%s port no = %d", retstr, ntohs( sorryaddr->sin_port ) );
        sprintf( retstr, "%s address = %s", retstr, inet_ntoa( sorryaddr->sin_addr ) );
        //sorry flag
        sprintf( retstr, "%s sorry flag = %d", retstr, srv_repdata->sorry_flag );
        //QoS setting value
        sprintf( retstr, "%s QoS control to server = %llu", retstr, srv_repdata->qos_threshold_up );
        sprintf( retstr, "%s QoS control to client = %llu", retstr, srv_repdata->qos_threshold_down );
    }
}

#endif //L7VS_SERVICE_H
