use strict;
use warnings;
no warnings qw(redefine once);
use lib qw(t/lib lib);
use subs qw(print);
use Cwd;
use L7lib;
use Test::More tests => 33;

L7lib::chdir();
L7lib::comment_out();
require './l7directord';
override();

close *STDOUT;
open  *STDOUT, '>', '/dev/null';
close *STDERR;
open  *STDERR, '>', '/dev/null';

#...............................................
# test start
#   - command_wrapper
{
    my @got = command_wrapper();
    is_deeply \@got, [$?, ''], 'command_wrapper - args is undef';
}
{
    my @got = command_wrapper('ls');
    is $got[0], 0, 'command_wrapper - ls is ok';
    like $got[1], qr/Makefile.*l7directord.*t/s, 'command_wrapper - ls result';
}
{
    my @got = command_wrapper('env LANG=C ls /proc/100000 2>&1');
    is_deeply \@got, [512, "ls: /proc/100000: No such file or directory\n"], 'command_wrapper - ls /proc/100000(string) is ok';
}
{
    my @got = command_wrapper('/etc');
    is_deeply \@got, [-1, undef], 'command_wrapper - no such command';
}
#   - system_wrapper
{
    my $got = system_wrapper();
    is $got, 65280, 'system_wrapper - args is undef';
}
{
    my $got = system_wrapper('ls');
    is $got, 0, 'system_wrapper - ls is ok';
}
{
    my $got = system_wrapper('ls /proc/100000');
    is $got, 512, 'system_wrapper - ls /proc/100000(string) is ok';
}
{
    my $got = system_wrapper('ls', '/proc/100000');
    is $got, 512, 'system_wrapper - ls /proc/100000(array) is ok';
}
{
    my $got = system_wrapper('/etc');
    is $got, -1, 'system_wrapper - no such command';
}
#   - exec_wrapper
{
    my $got = exec_wrapper();
    is $got, 0, 'exec_wrapper - args is undef';
}
{
    my $got = exec_wrapper('/etc');
    is $got, 0, 'exec_wrapper - no such command';
}
#   - ld_gethostbyname
{
    my $got = ld_gethostbyname();
    is $got, undef, 'ld_gethostbyname - name is undef';
}
{
    my $got = ld_gethostbyname('localhost');
    is $got, '127.0.0.1', 'ld_gethostbyname - resolve localhost';
}
{
    my $got = ld_gethostbyname('0.0.0.0');
    is $got, '0.0.0.0', 'ld_gethostbyname - already resolved';
}
{
    my $got = ld_gethostbyname('any host name');
    is $got, undef, 'ld_gethostbyname - cannot resolve';
}
#   - ld_getservbyname
{
    my $name = undef;
    my $protocol = 'tcp';
    my $got = ld_getservbyname($name, $protocol);
    is $got, undef, 'ld_getservbyname - name is undef';
}
{
    my $name = 'http';
    my $protocol = undef;
    my $got = ld_getservbyname($name, $protocol);
    is $got, 80, 'ld_getservbyname - protocol is undef';
}
{
    my $name = 0;
    my $protocol = 'tcp';
    my $got = ld_getservbyname($name, $protocol);
    is $got, 0, 'ld_getservbyname - name is 0(min)';
}
{
    my $name = 65535;
    my $protocol = 'tcp';
    my $got = ld_getservbyname($name, $protocol);
    is $got, 65535, 'ld_getservbyname - name is 65535(max)';
}
{
    my $name = 65536;
    my $protocol = 'tcp';
    my $got = ld_getservbyname($name, $protocol);
    is $got, undef, 'ld_getservbyname - name is 65536(error)';
}
{
    my $name = 'foobar';
    my $protocol = 'tcp';
    my $got = ld_getservbyname($name, $protocol);
    is $got, undef, 'ld_getservbyname - no such service name';
}
{
    my $name = 'http';
    my $protocol = 'foo';
    my $got = ld_getservbyname($name, $protocol);
    is $got, undef, 'ld_getservbyname - no such protocol name';
}
{
    my $name = 'http';
    my $protocol = 'tcp';
    my $got = ld_getservbyname($name, $protocol);
    is $got, 80, 'ld_getservbyname - resolve http tcp';
}
{
    my $name = 'http';
    my $protocol = 'udp';
    my $got = ld_getservbyname($name, $protocol);
    is $got, 80, 'ld_getservbyname - resolve http udp';
}
#   - ld_getservhostbyname
{
    my $hostserv = 'localhost:http';
    my $protocol = 'tcp';
    my $got = ld_gethostservbyname($hostserv, $protocol);
    is_deeply $got, { ip => '127.0.0.1', port => 80 }, 'ld_getservhostbyname - resolve localhost:http tcp';
}
{
    my $hostserv = 'localhost:http';
    my $protocol = 'udp';
    my $got = ld_gethostservbyname($hostserv, $protocol);
    is_deeply $got, { ip => '127.0.0.1', port => 80 }, 'ld_getservhostbyname - resolve localhost:http udp';
}
{
    my $hostserv = 'localhost:http';
    my $protocol = undef;
    my $got = ld_gethostservbyname($hostserv, $protocol);
    is_deeply $got, { ip => '127.0.0.1', port => 80 }, 'ld_getservhostbyname - protocol is undef';
}
{
    my $hostserv = 'localhost:http';
    my $protocol = 'foo';
    my $got = ld_gethostservbyname($hostserv, $protocol);
    is $got, undef, 'ld_getservhostbyname - no such protocol';
}
{
    my $hostserv = 'foo-bar-host:http';
    my $protocol = 'tcp';
    my $got = ld_gethostservbyname($hostserv, $protocol);
    is $got, undef, 'ld_getservhostbyname - invalid host address';
}
{
    my $hostserv = 'localhost:foobar';
    my $protocol = 'tcp';
    my $got = ld_gethostservbyname($hostserv, $protocol);
    is $got, undef, 'ld_getservhostbyname - invalid service name';
}
{
    my $hostserv = undef;
    my $protocol = 'tcp';
    my $got = ld_gethostservbyname($hostserv, $protocol);
    is $got, undef, 'ld_getservhostbyname - hostserv is undef';
}
{
    my $hostserv = 'localhost';
    my $protocol = 'tcp';
    my $got = ld_gethostservbyname($hostserv, $protocol);
    is $got, undef, 'ld_getservhostbyname - hostserv is invalid format';
}
# test end
#...............................................

L7lib::comment_in();

sub set_default {
}
sub override {
    *ld_log = \&__ld_log;
}
sub __ld_log {
}
