/*
 * @file  module_http.c
 * @brief Library of HTTP protocol for protocol module.
 * @brief this library provide HTTP header control.
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2008  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include "module_http.h"

/*!
 * create a expire string of HTTP cookie header field.
 * @param  expire_t expire time
 * @param  expire_c return expire string
 * @return void
 */
extern void
http_cookie_expire(time_t expire_t, char* expire_c)
{
	struct tm *date;

	date = gmtime(&expire_t);
	strftime(expire_c, EXPIRE_MAXSIZE, "%a, %d-%b-%Y %H:%M:%S GMT", date);
}

/*!
 * check the HTTP status that is included in a HTTP Response header.
 * (RFC2616 http://www.ietf.org/rfc/rfc2616.txt)
 * @param  *res HTTP Response header
 * @retval 0    HTTP status is 2xx or 3xx, or 1xx.
 * @retval -1   HTTP status is 4xx or 5xx, or is not able to be found.
 */
extern int
http_check_response_status(char *res)
{
	/* check NULL */
	if (res == NULL) {
		return -1;
	}

	/* pattern match "HTTP/X.Y ZYY " (X: 0 or 1) (Y: 0 ~ 9) (Z: 1 ~ 3) */
	if ( res[0] != 'H' || res[1] != 'T' || res[2] != 'T' || res[3] != 'P' ||
	     res[4] != '/' || ( res[5] != '1' && res[5] != '0' ) ||
	     res[6] != '.' || !isdigit(res[7]) || res[8] != ' ' ||
	     ( res[9] != '2' && res[9] != '3' && res[9] != '1' ) ||
	     !isdigit(res[10]) || !isdigit(res[11]) || res[12] != ' ' ) {
		return -1;
	}

	return 0;
}

/*!
 * check the HTTP method that is included in a HTTP Request header.
 * (RFC2616 http://www.ietf.org/rfc/rfc2616.txt)
 * @param  *req HTTP Request header
 * @return points to request-URI when HTTP method is valid.
 *         (It means that method is one of below.
 *         GET, HEAD, POST, PUT, DELETE, OPTIONS, TRACE, CONNECT,
 *         PROPFIND, PROPPATCH, COPY, MOVE, MKCOL, LOCK, UNLOCK)
 *         NULL when HTTP method is not valid.
 */
extern char *
http_check_request_method(char *req, size_t *length)
{
	char *uri = NULL;
	char *ptr = NULL;
	char *S = NULL;
	char *check_len =" HTTP/1." ;
	size_t len_count;
	size_t len = *length;
	int ret;

	/* check NULL */
	if (req == NULL) {
		return NULL;
	}

	/* check length size */
	if (len < 16)
	{
		return NULL;
	}

	/* request check */
	switch(req[0])
	{
	case 'G':
		/* GET method */
		if (req[1] != 'E' || req[2] != 'T' || req[3] != ' ') {
			return NULL;
		}
		uri = req + 4;
		break;
		
	case 'H':
		/* HEAD method */
		if (req[1] != 'E' || req[2] != 'A' || req[3] != 'D' || req[4] != ' ') {
			return NULL;
		}
		uri = req + 5;
		break;
		
	case 'P':
		switch(req[1])
		{
		case 'O':
			/* POST method */
			if (req[2] != 'S' || req[3] != 'T' || req[4] != ' ') {
				return NULL;
			}
			uri = req + 5;
			break;
			
		case 'U':
			/* PUT method */
			if (req[2] != 'T' || req[3] != ' ') {
				return NULL;
			}
			uri = req + 4;
			break;
			
		case 'R':
			if (req[2] != 'O' || req[3] != 'P') {
				return NULL;
			}
			switch(req[4])
			{
			case 'F':
				/* PROPFIND method (WebDAV) */
				if (req[5] != 'I' || req[6] != 'N' || req[7] != 'D' ||
				    req[8] != ' ') {
					return NULL;
				}
				uri = req + 9;
				break;
				
			case 'P':
				/* PROPPATCH method (WebDAV) */
				if (req[5] != 'A' || req[6] != 'T' || req[7] != 'C' || req[8] != 'H' ||
				    req[9] != ' ') {
					return NULL;
				}
				uri = req + 10;
				break;
				
			default:
				/* other PROP method */
				return NULL;
			}
			break;
			
		default:
			/* other P method */
			return NULL;
		}
		break;
		
	case 'O':
		/* OPTIONS method */
		if (req[1] != 'P' || req[2] != 'T' || req[3] != 'I' || req[4] != 'O' ||
		    req[5] != 'N' || req[6] != 'S' || req[7] != ' ') {
			return NULL;
		}
		uri = req + 8;
		break;
		
	case 'C':
		if (req[1] != 'O') {
			/* other C method */
			return NULL;
		}
		switch(req[2])
		{
		case 'N':
			/* CONNECT method */
			if (req[3] != 'N' || req[4] != 'E' || req[5] != 'C' || req[6] != 'T' ||
			    req[7] != ' ') {
				return NULL;
			}
			uri = req + 8;
			break;
			
		case 'P':
			/* COPY method (WebDAV) */
			if (req[3] != 'Y' || req[4] != ' ') {
				return NULL;
			}
			uri = req + 5;
			break;
			
		default:
			/* other CO method */
			return NULL;
		}
		break;
		
	case 'T':
		/* TRACE method */
		if (req[1] != 'R' || req[2] != 'A' || req[3] != 'C' || req[4] != 'E' ||
		    req[5] != ' ') {
			return NULL;
		}
		uri = req + 6;
		break;
		
	case 'D':
		/* DELETE method */
		if (req[1] != 'E' || req[2] != 'L' || req[3] != 'E' || req[4] != 'T' ||
		    req[5] != 'E' || req[6] != ' ') {
			return NULL;
		}
		uri = req + 7;
		break;
		
	case 'L':
		/* LOCK method (WebDAV) */
		if (req[1] != 'O' || req[2] != 'C' || req[3] != 'K' || req[4] != ' ') {
			return NULL;
		}
		uri = req + 5;
		break;
		
	case 'U':
		/* UNLOCK method */
		if (req[1] != 'N' || req[2] != 'L' || req[3] != 'O' || req[4] != 'C' ||
		    req[5] != 'K' || req[6] != ' ') {
			return NULL;
		}
		uri = req + 7;
		break;
		
	case 'M':
		switch(req[1])
		{
		case 'O':
			/* MOVE method (WebDAV) */
			if (req[2] != 'V' || req[3] != 'E' || req[4] != ' ') {
				return NULL;
			}
			uri = req + 5;
			break;
			
		case 'K':
			/* MKCOL method (WebDAV) */
			if (req[2] != 'C' || req[3] != 'O' || req[4] != 'L' || req[5] != ' ') {
				return NULL;
			}
			uri = req + 6;
			break;
			
		default:
			/* other M method */
			return NULL;
		}
		break;
			
	default:
		/* other method */
		return NULL;
	}

	/*substitute ptr for uri */
	ptr = uri;

	/* looks for the 2nd blank address */
	for(len_count = 0; *(ptr+len_count) != '\x0D' && *(ptr+len_count) != '\x0A' ; len_count++)
	{
		if( *(ptr+len_count) == ' ' )
		{
			S = ptr+len_count;
			break;
		}

		/* Up to the length of the request */
		if(len_count == (len))
		{
			return NULL;
		}
	}

	/* there is no blank */
	if(S == NULL)
	{
		return NULL;
	}

	/* string check */
	ret = strncmp(S, check_len, 8);

	/* string is not equal */
	if (ret != 0)
	{
		return NULL;
	}

	/* ver is 1.0 or 1.1 */
	if(S[8] !='0' && S[8]!='1')
	{
		return NULL;
	}

	/* end is changing line. */
	if(S[9] !='\x0D' || S[10]!='\x0A')
	{
		return NULL;
	}

	*length = len_count;

	return uri;
}

/*!
 * Encode IP address and port number.
 * @param  encoded  encoded strings
 * @param  s_addr   IP address (cf. inet_addr("xxx.xxx.xxx.xxx") => s_addr)
 * @param  sin_port port number
 * @return void
 */
extern void
http_encode_address(char *encoded, const u_long s_addr, const u_short sin_port)
{
	/* check null */
	if (encoded != NULL) {
		/* encoded address formed XXXXXXXXXXPPPPP */
		/* XXXXXXXXXX : IP address(10 bytes, number with left zero pad) */
		/* PPPPP      : port number(5 bytes, number with left zero pad) */

		sprintf(encoded, "%010lu%05u", s_addr, sin_port);
	}
}

/*!
 * Decode encoded strings of IP address and port number.
 * @param  encoded  encoded strings
 * @param  s_addr   decoded IP address (cf. inet_addr("xxx.xxx.xxx.xxx") => s_addr)
 * @param  sin_port decoded port number
 * @return void
 */
extern void
http_decode_address(char *encoded, u_long *s_addr, u_short *sin_port)
{
	char decode_buf[11];

	if (encoded != NULL && s_addr != NULL && sin_port != NULL) {
		/* get encoded ip address strings */
		memcpy(decode_buf, encoded, 10);
		decode_buf[10] = '\0';
		*s_addr = strtoul(decode_buf, NULL, 10);

		/* get encoded port number strings */
		memcpy(decode_buf, encoded + 10, 5);
		decode_buf[5] = '\0';
		*sin_port = atoi(decode_buf);
	}
}
 
/*!
 * Search a HTTP cookie field and check the existence of the cookie name.
 * @param  http_header HTTP request header
 * @param  cookie_name HTTP cookie name
 * @return char points to cookie value of found cookie name, or NULL.
 */
extern char*
http_search_header_cookie_value(char *http_header, char *cookie_name)
{
	int check;
	char *field_ptr;

	/* check NULL */
	if (http_header == NULL || cookie_name == NULL) {
		return NULL;
	}

	/* search cookie header field from HTTP request header */
	field_ptr = http_search_header_field(http_header, "Cookie");
	if (field_ptr == NULL) {
		return NULL;
	}

	while (1) {
		/* compare cookie field and cookie name */
		for (check = 0;
		     field_ptr[check] == cookie_name[check] && cookie_name[check] != '\0';
		     ++check);

		/* when header match the cookie name, next word of cookie name must be '=' */
		/* cf. CookieName'=' and CookieName'V'alue= */
		if (cookie_name[check] == '\0' && field_ptr[check] == '=') {
			/* matched cookie name completely! */
			++check;
			/* return pointer of searched cookie name's value */
			return &field_ptr[check];
		}

		/* skip until next cookie's value */
		for ( ;
		      field_ptr[check] != ';' && field_ptr[check] != ',';
		      ++check) {
			if (field_ptr[check] == '\r' || field_ptr[check] == '\0') {
				return NULL;
			}
		}

		/* skip a space, tab, etc. */
		for ( ++check;
		      !isgraph(field_ptr[check]) && field_ptr[check] != '\r' && field_ptr[check] != '\0';
		      ++check );

		/* cookie name was not found */
		if (field_ptr[check] == '\r' || field_ptr[check] == '\0') {
			return NULL;
		}

		field_ptr += check;
	}
}

/*!
 * Search a HTTP header field.
 * @param  http_header HTTP request/response header
 * @param  field_name  HTTP header field name
 * @return char points to field value of found header field, or NULL.
 */
extern char*
http_search_header_field(char *http_header, char *field_name)
{
	int check;
	char *word_ptr;

	/* check NULL */
	if (http_header == NULL || field_name == NULL) {
		return NULL;
	}

	/* skip a HTTP method line or HTTP response status line */
	word_ptr = http_skip_header_line(http_header);
	
	while (1) {
		/* word_ptr is always pointing to line top here */

		/* null check */
		if (word_ptr == NULL) {
			return NULL;
		}

		/* "\r\n\r" matched */
		if (*word_ptr == '\r') {
			return NULL;
		}

		/* compare header string and search name header */
		for (check = 0;
		     toupper(word_ptr[check]) == toupper(field_name[check]) && field_name[check] != '\0';
		     ++check);

		/* when header match the search name, next word of search name must be colon */
		/* cf. Accept':' and Accept'-'Language: */
		if (field_name[check] == '\0' && word_ptr[check] == ':') {
			/* matched search name header completely! */
			++check;

			/* skip a space, tab, etc. */
			for ( ;
			      !isgraph(word_ptr[check]) && word_ptr[check] != '\r' && word_ptr[check] != '\0';
			      ++check );

			/* return pointer of searched name header's value */
			return &word_ptr[check];
		}

		/* go next line */
		word_ptr = http_skip_header_line(word_ptr);
	}
}

/*!
 * Insert any header field to HTTP header
 * @param[in,out] header HTTP header strings
 * @param[in] offset_length offset length
 * @param[in] insert_field insert header field strings
 * @param[in,out] header_length all of HTTP header length
 * @retval 0  successfully insert header field
 * @retval -1 some errors occur.
 */
extern int
http_insert_field(char *header, int offset_length, char *insert_field, int header_length)
{
	char *copy_buffer;

	/* check args */
	if (header == NULL)
		return -1;
	if (insert_field == NULL)
		return -1;
	if (offset_length < 0)
		return -1;
	if (header_length < 0)
		return -1;
	if (header_length < offset_length)
		return -1;

	/* calloc buffer */
	copy_buffer = (char *) calloc(1, header_length - offset_length);
	if (copy_buffer == NULL)
		return -1;

	/* backup strings */
	memcpy(copy_buffer, header + offset_length, header_length - offset_length);

	/* insert field */
	memcpy(header + offset_length, insert_field, strlen(insert_field));

	/* append backup strings and terminate null*/
	memcpy(header + offset_length + strlen(insert_field), copy_buffer, header_length - offset_length);
	header[offset_length + strlen(insert_field) + header_length - offset_length] = '\0';

	/* free */
	free(copy_buffer);
	copy_buffer = NULL;

	return 0;
}

/*!
 * Skip current HTTP header line
 * @param  top_ptr
 * @return char    points to next line top. if there are no next, return NULL.
 */
extern char*
http_skip_header_line(char *line)
{
	char *top_ptr;

	/* check null */
	if (line == NULL) {
		return NULL;
	}

	top_ptr = (char *) line;
	
	/* skip until match '\r' or '\n' */
	for ( ; *top_ptr != '\0' && *top_ptr != '\r'; ++top_ptr) {
		if (*top_ptr == '\n') {
			/* "\n" matched */
			++top_ptr;
			return top_ptr;
		}
	}

	/* no next line */
	if (*top_ptr == '\0') {
		return NULL;
	}

	/* '\r' matched */
	/* then next word have to be '\n' */
	++top_ptr;
	if (*top_ptr != '\n') {
		return NULL;
	}

	/* "\r\n" matched */
	++top_ptr;

	return top_ptr;
}
