/*
 * @file  module.c
 * @brief the framework module of manage shared object 
 * @brief it enables dynamic loading of shared object 
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif /* _GNU_SOURCE */

#include <sys/param.h>
#include <sys/types.h>
#include <dlfcn.h>
#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include "logger_wrapper.h"
#include "l7vs_module.h"
#include "l7vs.h"

#define L7VS_MODULE_INITFN	      "init"

#if defined(LOGGER_PROCESS_VSD)
const LOG_CATEGORY_TAG loggerCategory = LOG_CAT_L7VSD_MODULE;
const LOG_CATEGORY_TAG loggerCategorySysMem = LOG_CAT_L7VSD_SYSTEM_MEMORY;
#else
const LOG_CATEGORY_TAG loggerCategory = LOG_CAT_L7VSADM_MODULE;
const LOG_CATEGORY_TAG loggerCategorySysMem = LOG_CAT_L7VSADM_COMMON;
#endif

char l7vs_module_path[MAXPATHLEN];

/*!
 * Module initialize function.
 * @param modpath [in] Module directory path.
 * @retval 0  Success.
 * @retval -1 Module directory path string is too long.
 */
int
l7vs_module_init(char *modpath)
{
	int return_value = 0;

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,1,
	            "in_function: int l7vs_module_init(void* modpath): modpath=\"%s\"", modpath);
	}
	/*------ DEBUG LOG END ------*/

	if (modpath == NULL) {
		strcpy(l7vs_module_path, L7VS_MODULE_PATH);
	} else {
		if (strlen(modpath) > sizeof(l7vs_module_path) - 1) {
			return_value = -1;
			goto init_out;
		}
		strcpy(l7vs_module_path, modpath);
	}

init_out:
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,2,
	            "out_function: int l7vs_module_init(void* modpath): return_value=%d", return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Module finalize function.
 */
void
l7vs_module_fini(void)
{
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,3,
	            "in_function: void l7vs_module_fini(void)");
	}
	/*------ DEBUG LOG END ------*/

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,4,
	            "out_function: void l7vs_module_fini(void)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Module load function.
 * @param modname [in] Module name.
 * @param type [in] Module type(protomod or sched).
 * @retern Loaded module struct.
 */
void *
l7vs_module_load(char *modname, char *type)
{
	void* h;
	void* modhandle;
	char* modpath;
	void* (*initf)(void *);
	void* return_value = NULL;

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,5,
	            "in_function: void* l7vs_module_load(char* modname, char* type): modname=\"%s\", type=\"%s\"",
		    modname, type);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (modname == NULL) {
		LOGGER_PUT_LOG_ERROR(loggerCategory,1,
		    "Arg(modname) is NULL pointer.");
		goto load_out;
	}
	if (type == NULL) {
		LOGGER_PUT_LOG_ERROR(loggerCategory,2,
		    "Arg(type) is NULL pointer.");
		goto load_out;
	}

	modpath = (char *) malloc(strlen(l7vs_module_path) + strlen(type) + strlen(modname) + 6);

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategorySysMem ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategorySysMem,11, "malloc: addr=%p, size=%ld",
		    modpath, (unsigned long int) strlen(l7vs_module_path) + strlen(type) + strlen(modname) + 6);
	}
	/*------ DEBUG LOG END ------*/

	if (modpath == NULL) {
		LOGGER_PUT_LOG_ERROR(loggerCategorySysMem,15,
		    "Could not allocate memory.");
		goto load_out;
	}
	sprintf(modpath, "%s/%s_%s.so", l7vs_module_path, type, modname);

	h = dlopen(modpath, RTLD_LAZY);
	if (h == NULL) {
		LOGGER_PUT_LOG_ERROR(loggerCategory,3,
		    "Could not open %s module: %s", modpath, dlerror());

		/*-------- DEBUG LOG --------*/
		if( LOG_LV_DEBUG == logger_get_log_level( loggerCategorySysMem ) ){
			LOGGER_PUT_LOG_DEBUG(loggerCategorySysMem,12, "free: %p", modpath);
		}
		/*------ DEBUG LOG END ------*/

		free(modpath);
		modpath = NULL;
		goto load_out;
	}

	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategorySysMem ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategorySysMem,13, "free: %p", modpath);
	}
	/*------ DEBUG LOG END ------*/

	free(modpath);
	modpath = NULL;

	*(void **) (&initf) = dlsym(h, L7VS_MODULE_INITFN);
	if (initf == NULL) {
		LOGGER_PUT_LOG_ERROR(loggerCategory,4,
		    "Could not find symbol %s: %s", L7VS_MODULE_INITFN, dlerror());
		dlclose(h);
		goto load_out;
	}

	modhandle = (*initf)(h);
	if (modhandle == NULL) {
		LOGGER_PUT_LOG_ERROR(loggerCategory,5,
		    "Module initialization failed.");
		dlclose(h);
		goto load_out;
	}

	return_value = modhandle;

load_out:
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,6,
	            "out_function: void* l7vs_module_load(char* modname, char* type): return_value=%p",
		    return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}

/*!
 * Module unload function.
 * @param handle [in] Module handle.
 */
void
l7vs_module_unload(void *handle)
{
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,7,
	            "in_function: void l7vs_module_unload(void* handle): handle=%p", handle);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (handle == NULL) {
		LOGGER_PUT_LOG_ERROR(loggerCategory,6,
		    "Arg(handle) is NULL pointer.");
		goto unload_out;
	}
	dlclose(handle);

unload_out:
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,8,
	            "out_function: void l7vs_module_unload(void* handle)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Register module to list function.
 * @param modlist [in,out] Registerd module list.
 * @param mod [in] Loaded module.
 */
void
l7vs_module_register(GList **modlist, void *mod)
{
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,9,
	            "in_function: void l7vs_module_register(GList** modlist, void* mod): "
		    "modlist=%p, mod=%p", modlist, mod);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (modlist == NULL) {
		LOGGER_PUT_LOG_ERROR(loggerCategory,7,
		    "Arg(modlist) is NULL pointer.");
		goto register_out;
	}
	if (mod == NULL) {
		LOGGER_PUT_LOG_ERROR(loggerCategory,8,
		    "Arg(mod) is NULL pointer.");
		goto register_out;
	}
	*modlist = g_list_append(*modlist, (gpointer)mod);

register_out:
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,10,
	            "out_function: void l7vs_module_register(GList** modlist, void* mod)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Remove module from list function.
 * @param modlist [in,out] Registerd module list.
 * @param mod [in] Loaded module.
 */
void
l7vs_module_remove(GList **modlist, void *mod)
{
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,11,
	            "in_function: void l7vs_module_remove(GList** modlist, void* mod): "
		    "modlist=%p, mod=%p", modlist, mod);
	}
	/*------ DEBUG LOG END ------*/

	/* check null */
	if (modlist == NULL) {
		LOGGER_PUT_LOG_ERROR(loggerCategory,9,
		    "Arg(modlist) is NULL pointer.");
		goto remove_out;
	}
	if (mod == NULL) {
		LOGGER_PUT_LOG_ERROR(loggerCategory,10,
		    "Arg(mod) is NULL pointer.");
		goto remove_out;
	}
	*modlist = g_list_remove(*modlist, mod);

remove_out:
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,12,
	            "out_function: void l7vs_module_remove(GList** modlist, void* mod)");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Lookup module from list function.
 * @param modname [in] Module name.
 * @param key [in] Lookup keyword.
 * @param cmp [in] Lookup function.
 * @retern Found module.
 */
void *
l7vs_module_lookup(GList *modlist, void *key, GCompareFunc cmp)
{
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,13,
	            "in_function: void* l7vs_module_lookup(GList* modlist, void* key, GCompareFunc cmp): "
		    "modlist=%p, key=%p, cmp=%p", modlist, key, cmp);
	}
	/*------ DEBUG LOG END ------*/

	GList* l;
	void*  return_value = NULL;

	/* check null */
	if (modlist == NULL) {
		goto lookup_out;
	}
	if (key == NULL) {
		LOGGER_PUT_LOG_ERROR(loggerCategory,12,
		    "Arg(key) is NULL pointer.");
		goto lookup_out;
	}
	if (cmp == NULL) {
		LOGGER_PUT_LOG_ERROR(loggerCategory,13,
		    "Arg(cmp) is NULL pointer.");
		goto lookup_out;
	}

	l = g_list_find_custom(modlist, (gpointer)key, cmp);
	if (l == NULL) {
		goto lookup_out;
	}
	return_value = (void *)l->data;

lookup_out:
	/*-------- DEBUG LOG --------*/
	if( LOG_LV_DEBUG == logger_get_log_level( loggerCategory ) ){
		LOGGER_PUT_LOG_DEBUG(loggerCategory,14,
	            "out_function: void* l7vs_module_lookup(GList* modlist, void* key, GCompareFunc cmp): "
		    "return_value=%p", return_value);
	}
	/*------ DEBUG LOG END ------*/

	return return_value;
}
