/*! @file
 *  @brief Replication structure program
 *
 *  @author NTT COMWARE
 *  @date
 *  @version
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <time.h>
#include <netdb.h>
#include <errno.h>
#include "logger_wrapper.h"
#include "parameter_wrapper.h"
#include "l7vs_replication.h"
#include "l7vs_iom.h"
#include "l7vs_iomuxlist.h"

#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <iostream>
#include <limits.h>

//---------- Initial set value

//! Initial value of Essential information set to SG file.


static struct l7vs_replication_info replication_info;
		
//! Initial value of Replication status
static struct l7vs_replication_state replication_state = {
	REPLICATION_OUT,
	0,
	0,
	0,
	0,
	NULL,
	NULL,
	0,
	NULL
};
//! iomux structure
static struct l7vs_iomux *l7vs_replication_iomux = NULL;

//! recv sock
struct  addressinfo{
        struct sockaddr_in      addr_in;
        struct sockaddr_in6     addr_in6;
};

struct  addressinfo addr;

//! Internal value structure
struct replication_internal_val
{
	int			socket;			//!< Opened socket file descriptor
	struct addrinfo		*address_info;		//!< Socket property internal_val
	struct  addressinfo	addr;
};


static struct replication_internal_val internal_val = {
		-1,
		NULL,
		addr
};

//! emun States Type string
static char replication_mode[6][24] = {
	"REPLICATION_OUT",
	"REPLICATION_SINGLE",
	"REPLICATION_MASTER",
	"REPLICATION_SLAVE",
	"REPLICATION_MASTER_STOP",
	"REPLICATION_SLAVE_STOP"
};

// Initialize network socket
static int l7vs_replication_send_sock_init();
// Initialize network socket
static int l7vs_replication_recv_sock_init();
// Finalize network socket
static void l7vs_replication_sock_fini();
// Set master mode to iomux
static int l7vs_replication_set_master();
// Set slave mode to iomux
static int l7vs_replication_set_slave();
// Obtain the memory the replication is done
static void *l7vs_replication_get_replication_address();
// Release replication memory 
static void l7vs_replication_release_replication_address();
// The memory the component uses 
static void *l7vs_replication_get_component_address();
// Release component memory
static void l7vs_replication_release_components_address();
// The memory with information on respect switch
static uint64_t *l7vs_replication_get_surface_info_address();
// Release memory of information on respect switch
static void l7vs_replication_release_surface_info_address();
// get value check
static int l7vs_replication_check_parameter();
// make serial
static unsigned long long l7vs_replication_make_serial();
// send
static int l7vs_replication_send_data(struct l7vs_replication_data *replication_data);
// recve
static int l7vs_replication_recv_data(struct l7vs_replication_data *replication_data);
// get ip from nic
static int getNICAddressInfo( const char* nicname, struct addressinfo* info );


//---------- Debug Function declaration
static void l7vs_replication_data_c_str(char *ret, struct l7vs_replication_data *replication_data);
static void l7vs_replication_component_c_str(char *ret, struct l7vs_replication_component *component_info, int i);
static void l7vs_replication_state_c_str(char *ret, struct l7vs_replication_state *replication_state);
static void l7vs_replication_info_c_str(char *ret, struct l7vs_replication_info *replication_info);
static void sockaddr_in_c_str(char *ret, struct sockaddr_in *addr_in);

/*! Initialize Replication
 *
 * @return 0/-1
 * @retval 0 Success
 * @retval -1 Error
 *
 */

int l7vs_replication_init()
{
	int	component_num = 0;
	int	ret = -1;
	int	ip_addr_ret = -1;
	int	service_name_ret = -1;
	int	nic_ret = -1;
	int	interval_ret = -1;

	// Check by continuous initialize.
	if ( REPLICATION_OUT != replication_state.service_status ) 
	{
		// Initialization has already been done.
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,1, "Initialization is a failure, because nitialization has already been done. mode : %s",
								replication_mode[(int)replication_state.service_status]);
		ret = -1;
		goto END;
	}

	memset(&replication_info, 0, sizeof(struct l7vs_replication_info));

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,1, "in_function : int l7vs_replication_init(void)");
	}
	// Debug log END

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG)
	{
		char replication_state_str[DEBUG_STR_LEN] = {0};
		l7vs_replication_state_c_str(replication_state_str,&replication_state);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,2,
					"set_l7vs_replication_state_struct : %s",
					replication_state_str);

		char replication_info_str[DEBUG_STR_LEN] = {0};
		l7vs_replication_info_c_str(replication_info_str,&replication_info);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,3,
					"set_l7vs_replication_info_struct : %s",
					replication_info_str);

		for ( int i=0; i<replication_info.component_num; i++)
		{
			char replication_component_str[DEBUG_STR_LEN] = {0};
			l7vs_replication_component_c_str(replication_component_str,replication_info.component_info,i);
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,4,
						"set_l7vs_replication_component_struct : %s",
						replication_component_str);
		}
	}
	// Debug log END

	// Check the Parameter exists
	ip_addr_ret = parameter_is_char_exist( PARAM_COMP_REPLICATION,"ip_addr");
	service_name_ret = parameter_is_char_exist( PARAM_COMP_REPLICATION, "service_name" );
	nic_ret = parameter_is_char_exist( PARAM_COMP_REPLICATION, "nic");
	interval_ret = parameter_is_int_exist( PARAM_COMP_REPLICATION, "interval" );

	// SG File not set
	if ( (0 == ip_addr_ret) && (0 == service_name_ret) && (0 == nic_ret) && (0 == interval_ret))
	{
		// Status Set
		replication_state.service_status = REPLICATION_SINGLE;
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,1, "Required item is not set in l7vs. ");
		ret = 0;
		goto END;
	}
	// IP Address exists
	if ( 0 == ip_addr_ret  )
	{
		// Status Set
		replication_state.service_status = REPLICATION_SINGLE;
		LOGGER_PUT_LOG_ERROR( LOG_CAT_L7VSD_REPLICATION,2, "IP Address is not set." );
		ret = -1;
		goto END;
	}
	// Port exists
	else if ( 0 == service_name_ret )
	{
		// Status Set
		replication_state.service_status = REPLICATION_SINGLE;
		LOGGER_PUT_LOG_ERROR( LOG_CAT_L7VSD_REPLICATION,3, "Port is not set." );
		ret = -1;
		goto END;
	}
	// NIC exists
	else if ( 0 == nic_ret )
	{
		// Status Set
		replication_state.service_status = REPLICATION_SINGLE;
		LOGGER_PUT_LOG_ERROR( LOG_CAT_L7VSD_REPLICATION,4, "NIC is not set." );
		ret = -1;
		goto END;
	}
	// Interval exists
	else if ( 0 == interval_ret )
	{
		// Status Set
		replication_state.service_status = REPLICATION_SINGLE;
		LOGGER_PUT_LOG_ERROR( LOG_CAT_L7VSD_REPLICATION,5, "Interval is not set." );
		ret = -1;
		goto END;
	}

	// Get IP Addressa
	strncpy(replication_info.ip_addr, parameter_get_char_value( PARAM_COMP_REPLICATION, "ip_addr" ),NI_MAXHOST);
	// Failed in the acquisition of IP
	if ( strcmp(replication_info.ip_addr,"") == 0)
	{
		LOGGER_PUT_LOG_ERROR( LOG_CAT_L7VSD_REPLICATION,6, "Could not get IP Address." );
		replication_state.service_status = REPLICATION_SINGLE;
		ret = -1;
		goto END;
	}

	// Get service_name (port)
	strncpy(replication_info.service_name,
		 parameter_get_char_value( PARAM_COMP_REPLICATION,
		 "service_name"), NI_MAXSERV);
	// Failed in the acquisition of Port
	if ( strcmp(replication_info.service_name,"") == 0)
	{
		LOGGER_PUT_LOG_ERROR( LOG_CAT_L7VSD_REPLICATION,7, "Could not get Port." );
		replication_state.service_status = REPLICATION_SINGLE;
		ret = -1;
		goto END;
	}

	// Get NIC
	strncpy(replication_info.nic, parameter_get_char_value( PARAM_COMP_REPLICATION, "nic" ),NIC_LENGTH);
	// Failed in the acquisition of NIC
	if ( strcmp(replication_info.nic,"") == 0)
	{
		LOGGER_PUT_LOG_ERROR( LOG_CAT_L7VSD_REPLICATION,8, "Could not get NIC." );
		replication_state.service_status = REPLICATION_SINGLE;
		ret = -1;
		goto END;
	}

	// Get interval
	replication_info.interval =  parameter_get_int_value( PARAM_COMP_REPLICATION, "interval" );

	// Set a starting component block number
	replication_info.component_num = 0;
	// Variable that sets ID
	char key_id[ID_LENGTH];
	char key_size[CMP_SIZE_LENGTH];

	// Conponent exists
	// Get Component infomation
	for ( int i=0; i<CMP_MAX; i++)
	{
		memset( &key_id, 0, ID_LENGTH );
		memset( &key_size, 0, CMP_SIZE_LENGTH);

		sprintf( key_id, "cmponent_id_%02d",i);
		sprintf( key_size, "cmponent_size_%02d",i);

		// ID and the Size exist
		if ( 0 == parameter_is_char_exist( PARAM_COMP_REPLICATION, key_id ) && 
					0 == parameter_is_int_exist( PARAM_COMP_REPLICATION, key_size ) )
		{
			break;
		}else{
			// Get Conmponent ID
			strcpy( replication_info.component_info[i].id,
				parameter_get_char_value(PARAM_COMP_REPLICATION,
				key_id));
			if ( 0 == strcmp(replication_info.component_info[i].id,""))
			{
				LOGGER_PUT_LOG_ERROR( LOG_CAT_L7VSD_REPLICATION,9,
									"Could not get  %s.",key_id );
				replication_state.service_status = REPLICATION_SINGLE;
				ret = -1;
				goto END;
			}
			
			// Get Block size of Component
			replication_info.component_info[i].block_size=parameter_get_int_value(PARAM_COMP_REPLICATION,
														key_size);

			// Count block head number
			if ( 0 == i)
			{
				replication_info.component_info[i].block_head = 0;
				replication_state.total_block += replication_info.component_info[i].block_size;
			}else{
				replication_info.component_info[i].block_head =  replication_info.component_info[i-1].block_head
										 + replication_info.component_info[i-1].block_size;
				replication_state.total_block += replication_info.component_info[i].block_size;
			}
			// Number of Component 
			component_num++;
		}
		replication_info.component_num = component_num;
	}


	// Check the Parameters value
	if ( 0 != l7vs_replication_check_parameter() )
	{
		// Status Set
		replication_state.service_status = REPLICATION_SINGLE;
		LOGGER_PUT_LOG_ERROR( LOG_CAT_L7VSD_REPLICATION,10, "Failed in parameter check." );
		ret = -1;
		goto SINGLE_END;
	}
	
	// The memory is 0 or less. 
	if ( 0 >= replication_state.total_block)
	{
		// Status Set
		replication_state.service_status = REPLICATION_SINGLE;
		ret = 0;
		goto SINGLE_END;
	}

	// The memory is larger than int. 
	if ( 100000 < replication_state.total_block)
	{
		// Status Set
		replication_state.service_status = REPLICATION_SINGLE;
		LOGGER_PUT_LOG_ERROR( LOG_CAT_L7VSD_REPLICATION,57, "Set number of blocks exceeds %d.",CMP_BLOCK_MAX );
		ret = -1;
		goto SINGLE_END;
	}

	// Get the Replication memory
	replication_state.replication_memory = l7vs_replication_get_replication_address();
	// Confirmation of Replication memory 
	if ( NULL == replication_state.replication_memory )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,30, "Replication memory is NULL.");
		// Status Set
		replication_state.service_status = REPLICATION_SINGLE;
		ret = -1;
		goto SINGLE_END;
			
	}
	// Get the Components memory
	replication_state.component_memory = l7vs_replication_get_component_address();
	// Confirmation of Components memory
	if ( NULL == replication_state.component_memory )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,31, "Components memory is NULL.");
		// free memory
		l7vs_replication_release_replication_address();
		// Status Set
		replication_state.service_status = REPLICATION_SINGLE;
		ret = -1;
		goto SINGLE_END;
	}

	// Memory for Switch Sarface Number
	replication_state.sarface_block_array_ptr = l7vs_replication_get_surface_info_address();
	if ( NULL == replication_state.sarface_block_array_ptr )
	{
		// free memory
		l7vs_replication_release_replication_address();
		l7vs_replication_release_components_address();
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,32, "Surface infomation memory is NULL.");
		// Status Set
		replication_state.service_status = REPLICATION_SINGLE;
		ret = -1;
		goto SINGLE_END;
	}

	// Get iomux
	l7vs_replication_iomux = l7vs_iomux_get_from_avail_list();
	if ( NULL == l7vs_replication_iomux )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM,2, "Can not get replication_iomux.");
		ret = -1;
		goto SINGLE_END;
	}

	// Status Set to Slave
	ret = l7vs_replication_set_slave();

	if (0 == ret)
	{
		replication_state.service_status = REPLICATION_SLAVE;
		goto END;
	}else{
		goto SINGLE_END;
	}

END:
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,2, "Initialized in %s mode.",
						replication_mode[(int)replication_state.service_status]);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG)
		{
			char replication_state_str[DEBUG_STR_LEN] = {0};
			l7vs_replication_state_c_str(replication_state_str,&replication_state);
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,5,
						"set_l7vs_replication_state_struct : %s",
						replication_state_str);

			char replication_info_str[DEBUG_STR_LEN] = {0};
			l7vs_replication_info_c_str(replication_info_str,&replication_info);
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,6,
						"set_l7vs_replication_info_struct : %s",
						replication_info_str);

			for ( int i=0; i<replication_info.component_num; i++)
			{
				char replication_component_str[DEBUG_STR_LEN] = {0};
				l7vs_replication_component_c_str(replication_component_str,replication_info.component_info,i);
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,7,
							"set_l7vs_replication_component_struct : %s",
							replication_component_str);
			}
		}
		// Debug log END
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,8,
							"out_function : int l7vs_replication_initialize(void)"
							"return = %d",ret);	
		}
		// Debug log END
		return ret;

SINGLE_END:
		if ( -1 != internal_val.socket)
		{
			close(internal_val.socket);
			internal_val.socket = -1;
		}

		if ( NULL != internal_val.address_info)
		{
			freeaddrinfo(internal_val.address_info);
			internal_val.address_info = NULL;
		}

		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,3, "Initialized in %s mode.",
					replication_mode[(int)replication_state.service_status]);

		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG)
		{
			char replication_state_str[DEBUG_STR_LEN] = {0};
			l7vs_replication_state_c_str(replication_state_str,&replication_state);
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,9,
						"set_l7vs_replication_state_struct : %s",
						replication_state_str);

			char replication_info_str[DEBUG_STR_LEN] = {0};
			l7vs_replication_info_c_str(replication_info_str,&replication_info);
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,10,
						"set_l7vs_replication_info_struct : %s",
						replication_info_str);

			for ( int i=0; i<replication_info.component_num; i++)
			{
				char replication_component_str[DEBUG_STR_LEN] = {0};
				l7vs_replication_component_c_str(replication_component_str,replication_info.component_info,i);
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,11,
							"set_l7vs_replication_component_struct : %s",
							replication_component_str);
			}
		}
			
		// Debug log END
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,12,
							"out_function : int l7vs_replication_initialize(void)"
							"return = %d",ret);	
		}
		// Debug log END

		return ret;
	
}

/*! Finalize Replication
 */
void l7vs_replication_fini()
{

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,13,
					"in_function : void l7vs_replication_fini(void)");
	}
	// Debug log END

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG)
	{
		char replication_state_str[DEBUG_STR_LEN] = {0};
		l7vs_replication_state_c_str(replication_state_str,&replication_state);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,14,
					"set_l7vs_replication_state_struct : %s",
					replication_state_str);

		char replication_info_str[DEBUG_STR_LEN] = {0};
		l7vs_replication_info_c_str(replication_info_str,&replication_info);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,15,
					"set_l7vs_replication_info_struct : %s",
					replication_info_str);

		for ( int i=0; i<replication_info.component_num; i++)
		{
			char replication_component_str[DEBUG_STR_LEN] = {0};
			l7vs_replication_component_c_str(replication_component_str,replication_info.component_info,i);
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,16,
						"set_l7vs_replication_component_struct : %s",
						replication_component_str);
		}
	}
	// Debug log END
	// Socket finalaize
	l7vs_replication_sock_fini();
	// Release replication memory
	l7vs_replication_release_replication_address();
	// Release component memory
	l7vs_replication_release_components_address();
	// Release sarface block memory
	l7vs_replication_release_surface_info_address();

	// reset of replication_state
	replication_state.send_time = 0;
	replication_state.last_send_block = 0;
	replication_state.last_recv_block = 0;
	replication_state.total_block = 0;
	replication_state.sarface_block_no = 0;

	// reset of replication_info
	memset(&replication_info, 0, sizeof(struct l7vs_replication_info));

	// imoux
	if ( NULL != l7vs_replication_iomux )
	{ 
		l7vs_iomux_put_to_avail_list(l7vs_replication_iomux);
		l7vs_replication_iomux = NULL;
	}

	// status change
	replication_state.service_status = REPLICATION_OUT;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG)
	{
		char replication_state_str[DEBUG_STR_LEN] = {0};
		l7vs_replication_state_c_str(replication_state_str,&replication_state);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,17,
					"set_l7vs_replication_state_struct : %s",
					replication_state_str);

		char replication_info_str[DEBUG_STR_LEN] = {0};
		l7vs_replication_info_c_str(replication_info_str,&replication_info);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,18,
					"set_l7vs_replication_info_struct : %s",
					replication_info_str);

		for ( int i=0; i<replication_info.component_num; i++)
		{
			char replication_component_str[DEBUG_STR_LEN] = {0};
			l7vs_replication_component_c_str(replication_component_str,replication_info.component_info,i);
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,19,
						"set_l7vs_replication_component_struct : %s",
						replication_component_str);
		}
	}
	// Debug log END

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,20, "out_function : l7vs_replication_fini(void)");
	}
	// Debug log END

}

/*! Switch Slave to Master
 */
extern void l7vs_replication_switch_to_master()
{
	int ret = -1;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,21,
						"extern void l7vs_replication_switch_to_master(void)");
	}
	// Debug log END

	switch (replication_state.service_status){
		case REPLICATION_SLAVE:
			// Set Mastre Mode
			ret = l7vs_replication_set_master();
			if ( 0 != ret )
			{
				replication_state.service_status = REPLICATION_SINGLE;

				if ( -1 != internal_val.socket)
				{
					close(internal_val.socket);
					internal_val.socket = -1;
				}
				if ( NULL != internal_val.address_info)
				{
					freeaddrinfo(internal_val.address_info);
					internal_val.address_info = NULL;
				}
				l7vs_replication_release_replication_address();
				l7vs_replication_release_components_address();
				l7vs_replication_release_surface_info_address();

				LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,54, "Switch to master NG. mode : %s",
								replication_mode[(int)replication_state.service_status]);
				// change iomux status
				l7vs_replication_iomux->status = iomux_replication_error;

				break;

			}else{

				// Copy from component area to replication area.
				memcpy(replication_state.replication_memory, replication_state.component_memory, replication_state.total_block*DATA_SIZE);
	
				// Set mode.
				replication_state.service_status = REPLICATION_MASTER;
	
				// change iomux status
				l7vs_replication_iomux->status = iomux_replication_send_wait;
				// Debug log
				if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,22,
									"l7vs_replication_iomux->status :"
									"befor : iomux_replication_recv_wait"
									"after : iomux_replication_send_wait");
				}
				// Debug log END
	
				// Debug log
				if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG)
				{
					char replication_state_str[DEBUG_STR_LEN] = {0};
					l7vs_replication_state_c_str(replication_state_str,&replication_state);
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,23,
								"set_l7vs_replication_state_struct : %s",
								replication_state_str);
	
					char replication_info_str[DEBUG_STR_LEN] = {0};
					l7vs_replication_info_c_str(replication_info_str,&replication_info);
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,24,
								"set_l7vs_replication_info_struct : %s",
								replication_info_str);
	
					for ( int i=0; i<replication_info.component_num; i++)
					{
						char replication_component_str[DEBUG_STR_LEN] = {0};
						l7vs_replication_component_c_str(replication_component_str,replication_info.component_info,i);
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,25,
									"set_l7vs_replication_component_struct : %s",
									replication_component_str);
					}
				}
				// Debug log END

				LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,5, "Switch to master OK. mode : %s",
								replication_mode[(int)replication_state.service_status]);
				break;
			}
		case REPLICATION_SLAVE_STOP:
			// Set Mastre Mode
			ret = l7vs_replication_set_master();
			if ( 0 != ret )
			{
				replication_state.service_status = REPLICATION_SINGLE;

				if ( -1 != internal_val.socket)
				{
					close(internal_val.socket);
					internal_val.socket = -1;
				}
				if ( NULL != internal_val.address_info)
				{
					freeaddrinfo(internal_val.address_info);
					internal_val.address_info = NULL;
				}
				l7vs_replication_release_replication_address();
				l7vs_replication_release_components_address();
				l7vs_replication_release_surface_info_address();

				LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,53, "Switch to master NG. mode : %s",
								replication_mode[(int)replication_state.service_status]);
				// change iomux status
				l7vs_replication_iomux->status = iomux_replication_error;

				break;
			}else{

				// Copy from component area to replication area.
				memcpy(replication_state.replication_memory, replication_state.component_memory, replication_state.total_block*DATA_SIZE);

				// Set mode.
				replication_state.service_status = REPLICATION_MASTER_STOP;
	
				// change iomux status
				l7vs_replication_iomux->status = iomux_replication_send_stop;
				// Debug log
				if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,26,
									"l7vs_replication_iomux->status :"
									"befor : iomux_replication_recv_stop)"
									"after : iomux_replication_send_stop");
				}
				// Debug log END
	
				// Debug log
				if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG)
				{
					char replication_state_str[DEBUG_STR_LEN] = {0};
					l7vs_replication_state_c_str(replication_state_str,&replication_state);
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,27,
								"set_l7vs_replication_state_struct : %s",
								replication_state_str);
	
					char replication_info_str[DEBUG_STR_LEN] = {0};
					l7vs_replication_info_c_str(replication_info_str,&replication_info);
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,28,
								"set_l7vs_replication_info_struct : %s",
								replication_info_str);
	
					for ( int i=0; i<replication_info.component_num; i++)
					{
						char replication_component_str[DEBUG_STR_LEN] = {0};
						l7vs_replication_component_c_str(replication_component_str,replication_info.component_info,i);
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,29,
									"set_l7vs_replication_component_struct : %s",
									replication_component_str);
					}
				}
				// Debug log END
	
				LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,7, "Switch to master OK. mode : %s",
								replication_mode[(int)replication_state.service_status]);
				break;
			}
		case REPLICATION_SINGLE:
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,42,"Starting by %s, doesn't shift to MASTER. ",
							replication_mode[(int)replication_state.service_status]);
			break;
		default:
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,2, "Can not switch to master. mode : %s",
									replication_mode[(int)replication_state.service_status]);
			break;
	}
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,30, "extern void l7vs_replication_switch_to_master(void)");
	}
}

/*! Set Master mode
*/
static int l7vs_replication_set_master()
{
	int sock_ret = -1;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,31, "in_function : static int l7vs_replication_set_master(void)");
	}	
	// Debug log END

	if ( REPLICATION_SLAVE != replication_state.service_status &&
						 REPLICATION_SLAVE_STOP != replication_state.service_status )
	{
		// Status Set
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,3, "Can not set to master. mode : %s",
								replication_mode[(int)replication_state.service_status]);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,32,
						"out_function : static int l7vs_replication_set_master(void)"
						"return = -1");
		}
		// Debug log END
		return -1;
	}

	// close socket and delete l7vs_iomux
	l7vs_replication_sock_fini();

	// make send socket
	sock_ret = l7vs_replication_send_sock_init();
	if ( -1 == sock_ret )
	{
		// Status Set
		replication_state.service_status = REPLICATION_SINGLE;
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,11, "Failed in the initialization of the send socket.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,33,
						"out_function : static int l7vs_replication_set_master(void)"
						"return = -1");
		}
		// Debug log END
		return -1;
	}

	// set of send socket : struct l7vs_iomux l7vs_replication_iomux
	// fd
	l7vs_replication_iomux->fd = internal_val.socket;
	// callback
	l7vs_replication_iomux->callback = l7vs_replication_send_callback;
	// status
	l7vs_replication_iomux->status = iomux_replication_out;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,34,
						"l7vs_replication_iomux->status :"
						"befor : Any execution mode"
						"after : iomux_replication_out");
	}
	// Debug log END

	// add : struct l7vs_iomux l7vs_replication_iomux
	l7vs_iomux_add(l7vs_replication_iomux, iom_write);

	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,8,"Initialization of send socket is success.");
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,35,
					"out_function : static int l7vs_replication_set_master(void)"
					"return = 0");
	}
	// Debug log END

	return 0;

}

/*! Switch Master to Slave
 */
extern void l7vs_replication_switch_to_slave()
{
	int ret = -1;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,36,
					"in_function : extern void l7vs_replication_switch_to_slave(void)");
	}
	// Debug log END

	switch (replication_state.service_status){
		case REPLICATION_MASTER:
			// Set Mastre Mode
			ret = l7vs_replication_set_slave();
			if ( 0 != ret )
			{
				replication_state.service_status = REPLICATION_SINGLE;

				if ( -1 != internal_val.socket)
				{
					close(internal_val.socket);
					internal_val.socket = -1;
				}
				if ( NULL != internal_val.address_info)
				{
					freeaddrinfo(internal_val.address_info);
					internal_val.address_info = NULL;
				}
				l7vs_replication_release_replication_address();
				l7vs_replication_release_components_address();
				l7vs_replication_release_surface_info_address();

				LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,55, "Switch to slave NG. mode : %s",
								replication_mode[(int)replication_state.service_status]);
				// change iomux status
				l7vs_replication_iomux->status = iomux_replication_error;

				break;
			}else{

				// initialize to replication area.
				memset(replication_state.replication_memory, 0, replication_state.total_block*DATA_SIZE);
	
				// Set mode.
				replication_state.service_status = REPLICATION_SLAVE;
	
				// change iomux status
				l7vs_replication_iomux->status = iomux_replication_recv_wait;
				// Debug log
				if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,37,
									"l7vs_replication_iomux->status :"
									"befor : iomux_replication_send_wait"
									"after : iomux_replication_recv_wait");
				}
				// Debug log END
		
				// Debug log
				if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG)
				{
					char replication_state_str[DEBUG_STR_LEN] = {0};
					l7vs_replication_state_c_str(replication_state_str,&replication_state);
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,38,
								"set_l7vs_replication_state_struct : %s",
								replication_state_str);
	
					char replication_info_str[DEBUG_STR_LEN] = {0};
					l7vs_replication_info_c_str(replication_info_str,&replication_info);
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,39,
								"set_l7vs_replication_info_struct : %s",
								replication_info_str);
	
					for ( int i=0; i<replication_info.component_num; i++)
					{
						char replication_component_str[DEBUG_STR_LEN] = {0};
						l7vs_replication_component_c_str(replication_component_str,replication_info.component_info,i);
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,40,
									"set_l7vs_replication_component_struct : %s",
									replication_component_str);
					}
				}
				// Debug log END
	
				LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,10, "Switch to slave OK. mode : %s",
								replication_mode[(int)replication_state.service_status]);
				break;
			}
		case REPLICATION_MASTER_STOP:
			// Set Mastre Mode
			ret = l7vs_replication_set_slave();
			if ( 0 != ret )
			{
				replication_state.service_status = REPLICATION_SINGLE;

				if ( -1 != internal_val.socket)
				{
					close(internal_val.socket);
					internal_val.socket = -1;
				}
				if ( NULL != internal_val.address_info)
				{
					freeaddrinfo(internal_val.address_info);
					internal_val.address_info = NULL;
				}
				l7vs_replication_release_replication_address();
				l7vs_replication_release_components_address();
				l7vs_replication_release_surface_info_address();

				LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,56, "Switch to slave NG. mode : %s",
								replication_mode[(int)replication_state.service_status]);
				// change iomux status
				l7vs_replication_iomux->status = iomux_replication_error;

				break;
			}else{

				// initialize to replication area.
				memset(replication_state.replication_memory, 0, replication_state.total_block*DATA_SIZE);
	
				// Set mode.
				replication_state.service_status = REPLICATION_SLAVE_STOP;
	
				// change iomux status
				l7vs_replication_iomux->status = iomux_replication_recv_stop;
				// Debug log
				if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,41,
									"l7vs_replication_iomux->status :"
									"befor : iomux_replication_send_stop"
									"after : iomux_replication_recv_stop");
				}
				// Debug log END
	
				// Debug log
				if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG)
				{
					char replication_state_str[DEBUG_STR_LEN] = {0};
					l7vs_replication_state_c_str(replication_state_str,&replication_state);
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,42,
								"set_l7vs_replication_state_struct : %s",
								replication_state_str);
	
					char replication_info_str[DEBUG_STR_LEN] = {0};
					l7vs_replication_info_c_str(replication_info_str,&replication_info);
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,43,
								"set_l7vs_replication_info_struct : %s",
								replication_info_str);
	
					for ( int i=0; i<replication_info.component_num; i++)
					{
						char replication_component_str[DEBUG_STR_LEN] = {0};
						l7vs_replication_component_c_str(replication_component_str,replication_info.component_info,i);
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,44,
									"set_l7vs_replication_component_struct : %s",
									replication_component_str);
					}
				}
				// Debug log END
	
				LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,12, "Switch to slave OK. mode : %s",
								replication_mode[(int)replication_state.service_status]);
				break;
			}
		case REPLICATION_SINGLE:
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,13, "Starting by %s , doesn't shift to SLAVE .",
							replication_mode[(int)replication_state.service_status]);
			break;
		default:
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,4, "Can not switch to slave. mode : %s",
							replication_mode[(int)replication_state.service_status]);
			break;
	}
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,45, "out_function : extern void l7vs_replication_switch_to_slave(void)");
	}
	// Debug log END
}	

/*! Set Slave mode
*/
static int l7vs_replication_set_slave()
{
	int sock_ret = -1;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,46, "int_function : static int l7vs_replication_set_slave(void)");
	}
	// Debug log END
	

	if ( REPLICATION_MASTER != replication_state.service_status &&
				REPLICATION_MASTER_STOP != replication_state.service_status &&
						REPLICATION_OUT != replication_state.service_status )
	{
		// Status Set
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,5, "Can not set to slave. mode : %s",
								replication_mode[(int)replication_state.service_status]);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,47,
						"out_function : static int l7vs_replication_set_slave(void)"
						"return = -1");
		}
		// Debug log END

		return -1;
	}

	// close socket and delete l7vs_iomux
	l7vs_replication_sock_fini();
	// make send socket
	sock_ret = l7vs_replication_recv_sock_init();
	if ( -1 == sock_ret )
	{
		// Status Set
		replication_state.service_status = REPLICATION_SINGLE;
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,12, "Failed in the initialization of the recv socket.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,48,
						"out_function : static int l7vs_replication_set_slave(void)"
						"return = -1");
		}
		// Debug log END
		return -1;
	}

	// set of send socket : struct l7vs_iomux l7vs_replication_iomux
	// fd
	l7vs_replication_iomux->fd = internal_val.socket;
	// callback
	l7vs_replication_iomux->callback = l7vs_replication_recv_callback;
	// status
	l7vs_replication_iomux->status = iomux_replication_out;
			// Debug log
			if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,49,
								"l7vs_replication_iomux->status : iomux_replication_out");
			}
			// Debug log END
	// add : struct l7vs_iomux l7vs_replication_iomux
	l7vs_iomux_add(l7vs_replication_iomux, iom_read);

	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,14,"Initialization of receive socket is success.");
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,50,
				"out_function : static int l7vs_replication_set_slave(void)"
				"return = 0");
	}
	// Debug log END
	return 0;

}

/*! Pay replication memory 
 *
 * @param component_id is the one to identify the component.
 * @param size of component use blocks
 * 
 * @return Replication memory address
 * @retval nonnull Replication memory address
 * @retval NULL Error
 */
extern void *l7vs_replication_pay_memory( char* component_id, unsigned int* size )
{
	char demand_id[ID_LENGTH];
	void *ret = NULL;
	int cmp_ret = -1;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,51,
					"extern void *l7vs_replication_pay_memory( char* component_id, unsigned int* size)"
					"component_id = %p"
					"size = %p",
					component_id,
					size);
	}
	// Debug log END


	// argument is illegal
	if ( NULL == component_id || NULL == size)
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,13, "Argument illegal. Component id or component size is NULL. ");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,52,
			"out_function : extern void *l7vs_replication_pay_memory( char* component_id, unsigned int* size)"
			"return = NULL");
		}	
		// Debug log END
		return NULL;
	}
	*size = 0;
	// Check replication mode.
	if ( REPLICATION_OUT == replication_state.service_status || 
						REPLICATION_SINGLE == replication_state.service_status) 
	{
		// Check mode that can use the replication.
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,15, 
					"Improper mode for Replication. mode : %s",
					replication_mode[(int)replication_state.service_status]);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,53,
			"out_function : extern void *l7vs_replication_pay_memory( char* component_id, unsigned int* size)"
			"return = NULL");
		}	
		// Debug log END
		return NULL;
	}

	if ( NULL == replication_state.component_memory )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,33, 
						"You can't get memory. Component memory is NULL. ");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,54,
			"out_function : extern void *l7vs_replication_pay_memory( char* component_id, unsigned int* size)"
			"return = NULL");
		}	
		// Debug log END
		return NULL;
	}

	// memory initialize
	memset( demand_id, 0, ID_LENGTH );

	// get Demand ID
	strncpy( demand_id, component_id, ID_LENGTH );

	// search Component ID
	for ( int i = 0; i < replication_info.component_num; i++ )
	{
		// Demand ID is compared with Component ID
		cmp_ret = strncmp(demand_id, replication_info.component_info[i].id, ID_LENGTH);
		if ( 0 == cmp_ret )
		{

			// size check
			if ( 0 == replication_info.component_info[i].block_size )
			{
				LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,16, "Component block size is 0.");
				// Debug log
				if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,191,
					"out_function : extern void *l7vs_replication_pay_memory( char* component_id, unsigned int* size)"
					"return = NULL");
				}	
				// Debug log END
				return NULL;
			}

			// block_head check
			if ( replication_info.component_info[i].block_head < 0 || 
				replication_info.component_info[i].block_head > replication_state.total_block)
			{
				LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,14, 
					"Too many component block. Max is %d . ",replication_state.total_block);
				// Debug log
				if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,55,
					"out_function : extern void *l7vs_replication_pay_memory( char* component_id, unsigned int* size)"
					"return = NULL");
				}	
				// Debug log END
				return NULL;
			}
			// Pay memory address
			ret = (char *)replication_state.component_memory 
					+ replication_info.component_info[i].block_head * DATA_SIZE;
			// Nnumber of blocks of ID was returned. 
			*size = replication_info.component_info[i].block_size;

			// Debug log
			if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,56,
					"size data : %u",replication_info.component_info[i].block_size);
			}
			// Debug log END

			// LOG INFO
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,16, 
					"Component Info ID : \"%s\". Block size : %d . Head Block No : %d/  Pay memory : %p ",
					replication_info.component_info[i].id, 
					replication_info.component_info[i].block_size,
					replication_info.component_info[i].block_head,
					(char *)ret);
			// Debug log
			if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,57,
					"out_function : extern void *l7vs_replication_pay_memory( char* component_id, unsigned int* size)"
					"return = %p",
					ret);
			}	
			// Debug log END
			return ret;
		}
	}
	LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,15, 
					"Unknown component ID. Component ID : %s",demand_id);
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,58,
		"out_function : extern void *l7vs_replication_pay_memory( char* component_id, unsigned int* size)"
		"return = NULL");
	}	
		// Debug log END
	return NULL;
}



/*! Replication Callback
 *
 * @param sturuct l7vs_iomux *iom
 * 
 * @return 0/1
 *
 */
extern int l7vs_replication_send_callback( struct l7vs_iomux *iom )
{
	struct l7vs_replication_data replication_data;
	int send_ret = -1;

	// Debug log
//	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
//		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,59,
//					"in_function : extern int l7vs_replication_send_callback( struct l7vs_iomux *iom )"
//					"iomux = %p",iom);
//	}
	// Debug log END

	// change to iom_write
	l7vs_iomux_mod(iom, iom_write);

	// Check by continuous initialize.
	if ( REPLICATION_MASTER != replication_state.service_status &&
		REPLICATION_MASTER_STOP != replication_state.service_status ) 
	{
		// Initialization has already been done.
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,16,
					"Can not send_callback. Mode is difficult.  mode : %s",
					replication_mode[(int)replication_state.service_status]);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,60,
						"out_function : extern int l7vs_replication_send_callback( struct l7vs_iomux *iom )"
						"return = -1" );
		}
		// Debug log END
		return -1;
	}else if ( REPLICATION_MASTER_STOP == replication_state.service_status ) {
		// Debug log
//		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
//			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,61,
//						"Can not send Replication data, because mode is MASTER_STOP..");
//			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,62,
//						"out_function : extern int l7vs_replication_send_callback( struct l7vs_iomux *iom )"
//						"return = 0" );
//		}
		// Debug log END

		return 0;
	}

	// iomux is NULL
	if ( (void *)NULL ==iom )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,34, "Send Argument(struct iomux) is NULL." );
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,63,
						"out_function : extern int l7vs_replication_send_callback( struct l7vs_iomux *iom )"
						"return = -1" );
		}
		// Debug log END
		return -1;
	}

	// Replication memory is NULL
	if ( NULL == replication_state.replication_memory )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,35, "Replication memory is NULL." );
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,64,
						"out_function : extern int l7vs_replication_send_callback( struct l7vs_iomux *iom )"
						"return = -1" );
		}
		// Debug log END
		return -1;
	}

	// Component memory is NULL
	if ( NULL == replication_state.component_memory )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,36, "Component memory is NULL." );
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,65,
						"out_function : extern int l7vs_replication_send_callback( struct l7vs_iomux *iom )"
						"return = -1" );
		}
		// Debug log END
		return -1;
	}

	// 
	if ( 0 != l7vs_replication_checkinterval() )
	{
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_SYSTEM_MEMORY,1, "Send data wait ...." );
		}
		// Debug log END

		return 0;
	}

	// Temporary preservation sturuct initialize
	memset( &replication_data, 0, sizeof(struct l7vs_replication_data) );
	send_ret = l7vs_replication_send_data(&replication_data);

	// change iomux status
	l7vs_replication_iomux->status = iomux_replication_send_wait;
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,66,
					"l7vs_replication_iomux->status :"
					"befor : iomux_replication_sending"
					"after : iomux_replication_send_wait");
	}

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		char replication_data_str[DEBUG_STR_LEN] = {0};
		l7vs_replication_data_c_str(replication_data_str,&replication_data);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,67,
						"replication_data_struct : %s",
						replication_data_str);
	}
	// Debug log END

	if ( 0 != send_ret )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM,3,
					"Send data is Failed." );
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,68,
						"out_function : extern int l7vs_replication_send_callback( struct l7vs_iomux *iom )"
						"return = -1" );
		}
		// Debug log END
		return -1;
	}

	// set last send block number
	if ( replication_state.last_send_block < replication_state.total_block-1 )
	{
		replication_state.last_send_block += 1;
	}else if(replication_state.last_send_block == replication_state.total_block-1){
		replication_state.last_send_block = 0;
	}else{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,17,
					"Last send block number is illegal.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,69,
						"out_function : extern int l7vs_replication_send_callback( struct l7vs_iomux *iom )"
						"return = -1" );
		}
		// Debug log END
		return -1;
	}

	// sarface block number is change
	if(replication_state.total_block == replication_state.last_send_block + 1 )
	{
		// Synchronization is executed. 
		memcpy(replication_state.replication_memory,
			replication_state.component_memory,
			replication_state.total_block*DATA_SIZE );

		// serial initialize
		replication_state.sarface_block_no = 0;

		// make new serial
		replication_state.sarface_block_no = (uint64_t)l7vs_replication_make_serial();
		if ( 0 == replication_state.sarface_block_no )
		{
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,18,
						"Could not get serial number .");
			// Debug log
			if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,70,
							"out_function : extern int l7vs_replication_send_callback( struct l7vs_iomux *iom )"
							"return = -1" );
			}
			// Debug log END
			return -1;
		}
//		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
//			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION, 0,
//						"Copy Success from component memory to replication memory." );
//		}
	}
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,71,
					"out_function : extern int l7vs_replication_send_callback( struct l7vs_iomux *iom )"
					"return = 0" );
	}
	// Debug log END

	return 0;
}

/*! Replication Callback
 *
 * @param sturuct l7vs_iomux *iom
 * 
 * @return 0/1
 *
 */
extern int l7vs_replication_recv_callback( struct l7vs_iomux *iom )
{
	struct l7vs_replication_data replication_data;
	int recv_ret = -1;
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,72, 
					"in_function : extern int l7vs_replication_recv_callback( struct l7vs_iomux *iom )"
					"iom = %p",iom);
	}
	// Debug log END

	// change to iom_read
	l7vs_iomux_mod(iom, iom_read);

	// Check by continuous initialize.
	if ( REPLICATION_SLAVE != replication_state.service_status &&
		REPLICATION_SLAVE_STOP != replication_state.service_status ) 
	{
		// Initialization has already been done.
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,19,
					"Can not recv_callback. Mode is difficult.  mode : %s",
					replication_mode[(int)replication_state.service_status]);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,73, 
						"out_function : extern int l7vs_replication_recv_callback( struct l7vs_iomux *iom )"
						"return = -1");
		}
		// Debug log END
		return -1;
	}else if ( REPLICATION_SLAVE_STOP == replication_state.service_status ) {
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,17,
					"Can not receive Replication data, because mode is SLAVE_STOP.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,74, 
						"out_function : extern int l7vs_replication_recv_callback( struct l7vs_iomux *iom )"
						"return = 0");
		}
		// Debug log END
		return 0;
	}

	// iomux is NULL
	if ( (void *)NULL ==iom )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,37, "Recv Argument(struct iomux) is NULL." );
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,75, 
						"out_function : extern int l7vs_replication_recv_callback( struct l7vs_iomux *iom )"
						"return = -1");
		}
		// Debug log END
		return -1;
	}

	// Replication memory is NULL
	if ( NULL == replication_state.replication_memory )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,38, "Replication memory is NULL." );
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,76, 
						"out_function : extern int l7vs_replication_recv_callback( struct l7vs_iomux *iom )"
						"return = -1");
		}
		// Debug log END
		return -1;
	}

	// Component memory is NULL
	if ( NULL == replication_state.component_memory )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,39, "Component memory is NULL." );
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,77, 
						"out_function : extern int l7vs_replication_recv_callback( struct l7vs_iomux *iom )"
						"return = -1");
		}
		// Debug log END
		return -1;
	}

	// Sarface block array memory is NULL
	if ( NULL == replication_state.sarface_block_array_ptr)
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,20, "sarface block array pointer is NULL.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,78, 
						"out_function : extern int l7vs_replication_recv_callback( struct l7vs_iomux *iom )"
						"return = -1");
		}
		// Debug log END
		return -1;
	}

	// Temporary preservation sturuct initialize
	memset( &replication_data, 0, sizeof(struct l7vs_replication_data) );
	recv_ret = l7vs_replication_recv_data(&replication_data);

	// change iomux status
	l7vs_replication_iomux->status = iomux_replication_recv_wait;
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,79,
					"l7vs_replication_iomux->status :"
					"befor : iomux_replication_receving"
					"after : iomux_replication_recv_wait");
	}
	// Debug log END
	
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		char replication_data_str[DEBUG_STR_LEN] = {0};
		l7vs_replication_data_c_str(replication_data_str,&replication_data);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,80,
						"replication_data_struct : %s",
						replication_data_str);
	}
	// Debug log END

	if ( 0 != recv_ret )
	{
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_SYSTEM,3,
					"Failed in the reception processing of data because of illegal receive data." );
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,81, 
						"out_function : extern int l7vs_replication_recv_callback( struct l7vs_iomux *iom )"
						"return = -1");
		}
		// Debug log END
		return -1;
	}

	// set sarface block
	replication_state.sarface_block_array_ptr[replication_data.block_num] = replication_data.serial;

	// set last send block number
	if ( replication_state.last_recv_block < replication_state.total_block-1 )
	{
		replication_state.last_recv_block += 1;
	}else if(replication_state.last_recv_block == replication_state.total_block-1){
		replication_state.last_recv_block = 0;
	}else{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,21,
					"Last send block number is illegal.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,83, 
						"out_function : extern int l7vs_replication_recv_callback( struct l7vs_iomux *iom )"
						"return = -1");
		}
		// Debug log END
		return -1;
	}

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,84, 
					"out_function : extern int l7vs_replication_recv_callback( struct l7vs_iomux *iom )"
					"return = 0");
	}
	// Debug log END
	return 0;

	
}

/*! Replication Dump
 *
 */
extern void l7vs_replication_dump_memory()
{
	int size;
	char *p;
	char *head;
	int h = 0;
	int i = 0;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,85, "in_function : extern void l7vs_replication_dump_memory(void)");
	}
	// Debug log END

	if ( ( REPLICATION_SINGLE == replication_state.service_status ) || 
				( REPLICATION_OUT == replication_state.service_status ) )
	{
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,6, 
				"Replication memory dump failure. mode : %s",
				replication_mode[(int)replication_state.service_status]);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,86, "out_function : extern void l7vs_replication_dump_memory(void)");
		}
		// Debug log END
		return;
	}

	// Replication memory is NULL.
	if ( NULL == replication_state.replication_memory)
	{
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_SYSTEM,1, 
				"Replication memory is NULL.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,87, "out_function : extern void l7vs_replication_dump_memory(void)");
		}
		// Debug log END
		return;
	}
	
	// Dump size
	size = 480 * replication_state.total_block;

	if ( 0 == size )
	{
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,7, 
				"Can not get Replication memory");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,88, "out_function : extern void l7vs_replication_dump_memory(void)");
		}
		// Debug log END
		return;
		
	}

	// Memory Dump
	p = (char *)replication_state.replication_memory;

	// Output mode
	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,18,"Replication Dump Start ----------------------------");
	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,19,"Mode is [ %s ].",
					replication_mode[(int)replication_state.service_status]);
	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,20,"Total Block is [ %u ]",
					replication_state.total_block);

	// Converts into the binary, and writes it to the file. 
	for ( h = 0; h < size / DATA_SIZE ; h++ )
	{
	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,21,"Block Number [ %d ]",h);
	
		for ( i = 0; i < DATA_SIZE / LOG_DATA_WIDTH ; i++ )
		{
			head = p + h*DATA_SIZE + i*LOG_DATA_WIDTH;
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,22,
						"%02hhX %02hhX %02hhX %02hhX  %02hhX %02hhX %02hhX %02hhX  "
						"%02hhX %02hhX %02hhX %02hhX  %02hhX %02hhX %02hhX %02hhX"
						, *head, *(head+1), *(head+2), *(head+3)
						, *(head+4), *(head+5), *(head+6), *(head+7)
						, *(head+8), *(head+9), *(head+10), *(head+11)
						, *(head+12), *(head+13), *(head+14), *(head+15));

		}
	}
	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,23,"Replication Dump End ------------------------------");


	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){

		char	*q;
		char	*head_debug;
		int	j=0;
		int	k=0;

		// Component Memory Dump
		q = (char *)replication_state.component_memory;

		// Output mode
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,89,"Component Dump Satrt ----------------------------");
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,90,"Mode is [ %s ].",
						replication_mode[(int)replication_state.service_status]);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,91,"Total Block is [ %u ]",
						replication_state.total_block);
	
		// Converts into the binary, and writes it to the file. 
		for ( j = 0; j < size / DATA_SIZE ; j++ )
		{
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,92,"Block Number [ %d ]",j);
		
			for ( k = 0; k < DATA_SIZE / LOG_DATA_WIDTH ; k++ )
			{
				head_debug = q + j*DATA_SIZE + k*LOG_DATA_WIDTH;
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,93,
							"%02hhX %02hhX %02hhX %02hhX  %02hhX %02hhX %02hhX %02hhX  "
							"%02hhX %02hhX %02hhX %02hhX  %02hhX %02hhX %02hhX %02hhX"
							, *head_debug, *(head_debug+1), *(head_debug+2), *(head_debug+3)
							, *(head_debug+4), *(head_debug+5), *(head_debug+6), *(head_debug+7)
							, *(head_debug+8), *(head_debug+9), *(head_debug+10), *(head_debug+11)
							, *(head_debug+12), *(head_debug+13), *(head_debug+14), *(head_debug+15));
	
			}
		}
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,94,"Component Dump End ------------------------------");
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,95, "out_function : extern void l7vs_replication_dump_memory(void)");
	}
	// Debug log END
	return;
}


/*! Chenge Status isActive
 */
extern void l7vs_replication_start()
{

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,96, "in_function : extern void l7vs_replication_start(void)");
	}
	// Debug log END

	switch (replication_state.service_status){
		case REPLICATION_MASTER_STOP:
			replication_state.service_status = REPLICATION_MASTER;
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,24, "Replication start. mode : %s",
					replication_mode[(int)replication_state.service_status]);
			break;
		case REPLICATION_SLAVE_STOP:
			replication_state.service_status = REPLICATION_SLAVE;
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,25, "Replication start. mode : %s",
					replication_mode[(int)replication_state.service_status]);
			break;
		case REPLICATION_MASTER:
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,8, "Could not MASTER start, because already start. mode : %s",
					replication_mode[(int)replication_state.service_status]);
			break;
		case REPLICATION_SLAVE:
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,9, "Could not SALVE start, because already start. mode : %s",
					replication_mode[(int)replication_state.service_status]);
			break;
		default:
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,10, "Could not start, because  mode is %s.",
					replication_mode[(int)replication_state.service_status]);
			break;
	}
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,97, "out_function : extern void l7vs_replication_start(void)");
	}
	// Debug log END
			
}

/*! Chenge Status isStop
 */
extern void l7vs_replication_stop()
{
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,98, "in_function : extern void l7vs_replication_start(void)");
	}
	// Debug log END
	switch (replication_state.service_status){
		case REPLICATION_MASTER:
			replication_state.service_status = REPLICATION_MASTER_STOP;
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,26, "Replication stop. mode : %s",
							replication_mode[(int)replication_state.service_status]);
			break;
		case REPLICATION_SLAVE:
			replication_state.service_status = REPLICATION_SLAVE_STOP;
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,27, "Replication stop. mode : %s",
							replication_mode[(int)replication_state.service_status]);
			break;
		case REPLICATION_MASTER_STOP:
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,11, "Could not MASTER stop, because already stop. mode : %s",
							replication_mode[(int)replication_state.service_status]);
			break;
		case REPLICATION_SLAVE_STOP:
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,12, "Could not SLAVE stop, because already stop. mode : %s",
							replication_mode[(int)replication_state.service_status]);
			break;
		default:
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,17, "Could not stop, because  mode is %s",
							replication_mode[(int)replication_state.service_status]);
			break;

	}
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,99, "out_function : extern void l7vs_replication_start(void)");
	}
	// Debug log END
}

/*! Compulsion reproduction execution
 */
extern void l7vs_replication_compulsorily()
{
	struct	l7vs_replication_data replication_data;
	int	send_ret = -1;
	int	interval_ret = -1;
	int	ms_time = 0;
	struct timespec	time;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,100, "in_function : extern void l7vs_replication_compulsorily(void)");
	}
	// Debug log END

	// Check by continuous initialize.
	if ( REPLICATION_MASTER != replication_state.service_status &&
		REPLICATION_MASTER_STOP != replication_state.service_status ) 
	{
		// Initialization has already been done.
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,23,
					"Could not compulsion replication. Mode is different. mode : %s",
					replication_mode[(int)replication_state.service_status]);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,101, "out_function : extern void l7vs_replication_compulsorily(void)");
		}
		// Debug log END
		return;
	}else if ( REPLICATION_MASTER_STOP == replication_state.service_status ) {
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,28,
					"Can not replication compulsorily, because mode is %s .",
						replication_mode[(int)replication_state.service_status]);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,102, "out_function : extern void l7vs_replication_compulsorily(void)");
		}
		// Debug log END
		return;
	}

	// Replication memory is NULL
	if ( NULL == replication_state.replication_memory )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,40, "Replication memory is NULL." );
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,103, "out_function : extern void l7vs_replication_compulsorily(void)");
		}
		// Debug log END
		return;
	}

	// Component memory is NULL
	if ( NULL == replication_state.component_memory )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,41, "Component memory is NULL." );
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,104, "out_function : extern void l7vs_replication_compulsorily(void)");
		}
		// Debug log END
		return;
	}

	// set send interval
	time.tv_sec = 0;
	time.tv_nsec = 0;
	interval_ret = parameter_is_int_exist( PARAM_COMP_REPLICATION, "compulsorily_interval" );

	if ( 0 == interval_ret )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION, 51, "Could not get interval value for replication compulsorily.");
		return;
	}

	ms_time = parameter_get_int_value( PARAM_COMP_REPLICATION, "compulsorily_interval" );

	if ( ms_time < MIN_COMPULSORILRY_INTERVAL || MAX_COMPULSORILRY_INTERVAL < ms_time )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION, 52, "Invalid compulsorily interval.");
		return;
	}

	time.tv_nsec =  (long)( ms_time * 1000000 );

	// set last send block is c maximum block
	replication_state.last_send_block = replication_state.total_block-1;

	for(unsigned int i = 0; i < replication_state.total_block; i++)
	{
		// set compulsorily interval.
		nanosleep( &time, NULL);

		// Temporary preservation sturuct initialize
		memset( &replication_data, 0, sizeof(struct l7vs_replication_data) );
		send_ret = l7vs_replication_send_data(&replication_data);

		// change iomux status
		l7vs_replication_iomux->status = iomux_replication_send_wait;
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,105,
						"l7vs_replication_iomux->status :"
						"befor : iomux_replication_sending"
						"after : iomux_replication_send_wait");
		}
		// Debug log END

		if ( 0 != send_ret )
		{
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM,5,
						"Send data is Failed." );
			// Debug log
			if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,106, "out_function : extern void l7vs_replication_compulsorily(void)");
			}
			// Debug log END
			return;
		}

		// set last send block number
		if ( replication_state.last_send_block < replication_state.total_block-1 )
		{
			replication_state.last_send_block += 1;
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,29,
						"Data sending succeeded. Send block number : %u Version : %llu"
						,replication_state.last_send_block
						,(unsigned long long)replication_state.sarface_block_no );
		}else if(replication_state.last_send_block == replication_state.total_block-1){
			replication_state.last_send_block = 0;
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,30,
						"Data sending succeeded. Send block number : %u Version : %llu"
						,replication_state.last_send_block
						,(unsigned long long)replication_state.sarface_block_no );
		}else{
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,24,
						"Last send block number is illegal.");
			// Debug log
			if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,107, "out_function : extern void l7vs_replication_compulsorily(void)");
			}
			// Debug log END
			return;
		}

		// sarface block number is change
		if(replication_state.total_block == replication_state.last_send_block + 1 )
		{
			// Synchronization is executed. 
			memcpy(replication_state.replication_memory,
				replication_state.component_memory,
				replication_state.total_block*DATA_SIZE );

			// serial initialize
			replication_state.sarface_block_no = 0;

			// make new serial
			replication_state.sarface_block_no = (uint64_t)l7vs_replication_make_serial();
			if ( 0 == replication_state.sarface_block_no )
			{
				LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,25,
							"Could not get serial number.");
				// Debug log
				if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,108, "out_function : extern void l7vs_replication_compulsorily(void)");
				}
				// Debug log END
				return;
			}
			if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,109,
							"Copy Success from component memory to replication memory." );
			}
		}
	}
	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,31, "Replication compulsorily is success." );
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,110, "out_function : extern void l7vs_replication_compulsorily(void)");
	}
	// Debug log END
	return;

}

/*! Interval Re-setting
 */
extern void l7vs_replication_reset()
{
	int ret;
	unsigned short value = 0;
	
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,111, "in_function : extern void l7vs_replication_reset(void)");
	}
	// Debug log END

	// Check Parameter exists
	ret = parameter_is_int_exist(PARAM_COMP_REPLICATION,"interval");
	if ( 0 == ret )
	{
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,13, "Not chage re-setting value.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,112, "out_function : extern void l7vs_replication_reset(void)");
		}
		// Debug log END
		return;
	}

	// Get interval
	value =  parameter_get_int_value( PARAM_COMP_REPLICATION, "interval" );
	
	// Check interval
	if ( value < MIN_INTERVAL || MAX_INTERVAL < value ){
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,14, "Invalid Interval value. value : %d",(int)value);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,113, "out_function : extern void l7vs_replication_reset(void)");
		}
		// Debug log END
		return;
	}
	//set interval
	replication_info.interval =  value;
}

/*! Get Status
 *
 * @return REPLICATION_MODE_TAG enumration
 *
 */
extern enum REPLICATION_MODE_TAG l7vs_replication_get_status()
{
	
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,114, "function : extern enum REPLICATION_MODE_TAG l7vs_replication_get_status(void)");
	}
	// Debug log END
	return replication_state.service_status;
};


/*! Send Interval Check
 *
 * @return 0/-1
 * @retval 0 Send data
 * @retval -1 Not send data
 */
extern int l7vs_replication_checkinterval()
{
	struct timeval tv;
	struct timezone tz;
	unsigned long long sending_time = 0;
	unsigned long long last_send_time = 0;
	unsigned long long diff_time = 0;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,115, "in_function : extern int l7vs_replication_checkinterval(void)");
	}
	// Debug log END

	// last time
	last_send_time = replication_state.send_time;

	// now time
	gettimeofday(&tv,&tz);
	sending_time = (unsigned long long)(tv.tv_sec*1000000+tv.tv_usec);


	// Last send time is 0.
	if ( 0 == last_send_time )
	{
		replication_state.send_time = sending_time;
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,32,
			" Last send time is 0. ");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,116,
						"out_function : extern int l7vs_replication_checkinterval(void)"
						"return = 0");
		}
		// Debug log END
		return 0;
	}
	//Last send time is illegal. 
	if ( last_send_time > sending_time)
	{
		replication_state.send_time = sending_time;
		LOGGER_PUT_LOG_ERROR (LOG_CAT_L7VSD_REPLICATION,26,
			" Last send time illegal. ");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,117,
						"out_function : extern int l7vs_replication_checkinterval(void)"
						"return = -1");
		}
		// Debug log END
		return -1;
	}	
	// interval
	diff_time = sending_time - last_send_time;
	// exit

	if ( diff_time >= replication_info.interval)
	{
		replication_state.send_time = sending_time;
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,118," Send time is arrived ");
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,119,
						"out_function : extern int l7vs_replication_checkinterval(void)"
						"return = 0");
		}
		// Debug log END
		return 0;
	}else{

		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,120," Send time is not arrived. ");
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,121,
						"out_function : extern int l7vs_replication_checkinterval(void)"
						"return = -1");
		}
		// Debug log END
		return -1;
	}
}

/*! Parameter Check
 *
 * @return 0/-1
 * @retval 0 Success
 * @retval -1 Error 
 */
static int l7vs_replication_check_parameter()
{
	int ret = -1;
	int info_ret = -1;
	int cmp_ret = -1;
	struct addrinfo hints;
	size_t sum=0;
	
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,122,
					"in_function : static int l7vs_replication_check_parameter(void)");
	}
	// Debug log END

	// initialize structure
	memset(&hints, 0, sizeof(struct addrinfo));


	// set address hints
	hints.ai_family = AF_UNSPEC;     // permit any protocol (IPv4, IPv6, etc.)
	hints.ai_socktype = SOCK_DGRAM;  // UDP
	// IP NULL check
	if ( 0 == strcmp(replication_info.ip_addr, "") )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,27,"IP is unsetting.");
		ret = -1;
		goto END;
	}
	// Port NULL check
	if ( 0 == strcmp(replication_info.service_name,"") )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,28,"Port is unsetting.");
		ret = -1;
		goto END;
	}
	// Whether IP and the port are effective is confirmed.
	info_ret = getaddrinfo(replication_info.ip_addr,replication_info.service_name,&hints,&internal_val.address_info);
	if ( 0 != info_ret )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,27, "Failed to get IP or Service Name.(%d)",ret);
		ret = -1;
		goto END;
	}
	struct sockaddr_in *sin;
	sin = (sockaddr_in *)internal_val.address_info->ai_addr;
	
	// Port check
	if ( sin->sin_port == htons(0)) 
	{
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_REPLICATION,15, "You can't specify port number 0.")
		ret = -1;
		goto END;
	}
	// NIC check
	if ( 0 == strcmp( replication_info.nic, "") )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,29,"NIC is unsetting.");
		ret = -1;
		goto END;
	}
	// Interval check
	if ((MIN_INTERVAL>replication_info.interval) || (MAX_INTERVAL<replication_info.interval)) 
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,30,"Invalid Interval value");
		ret = -1;
		goto END;
	}
	// Components ID check
	// Components Size check
	if (0 == replication_info.component_num)
	{
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,33,"Can not get component, because component is 0.");
		ret = 0;
		goto END;
	}
	for ( int i=0; i < replication_info.component_num; i++ )
	{
		sum += replication_info.component_info[i].block_size ;
		for ( int j=i+1; j<replication_info.component_num; j++)
		{
			cmp_ret = strcmp(replication_info.component_info[j].id,replication_info.component_info[i].id);
			if (0 == cmp_ret)
			{
				LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,31,
						"Too many component block. Max is %d . ",replication_state.total_block);
				ret = -1;
				goto END;
			}
		}
	}
	if ( sum > CMP_BLOCK_MAX )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,32,"Total component size is too large.");
		ret = -1;
		goto END;
	}
	ret = 0;
END:
	if ( NULL != internal_val.address_info )
	{
		freeaddrinfo(internal_val.address_info);
		internal_val.address_info = NULL;
	}

	if ( 0 == ret )
	{
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,34,"Parameter Check OK.");
	}else{
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,35,"Parameter Check NG.");
	}
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,123,
					"out_function : static int l7vs_replication_check_parameter(void)"
					"return = %d",ret);
	}
	// Debug log END
	return ret;
};


/*! Get Replication Memory
 *
 * @param total_block The entire block that the component uses. 
 *
 * @return memory Replication memory
 * @retval memory memory get Success
 * @retval NULL Error
 *
 */
static void *l7vs_replication_get_replication_address()
{
	unsigned int total_block = replication_state.total_block;
	void *memory = NULL;

	
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,124,
					"in_function : static void *l7vs_replication_get_replication_address(void)");
	}
	// Debug log END

	// Check by continuous initialize.
	if ( REPLICATION_OUT != replication_state.service_status ) 
	{
		// Initialization has already been done.
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,33, "Already got replication memory.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,125,
						"out_function : static void *l7vs_replication_get_replication_address(void)"
						"return = NULL");
		}
		// Debug log END
		return NULL;
	}

	// Check Total Block
	if ( 0 == total_block || total_block > CMP_BLOCK_MAX)
	{
		// Check Total Block
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,34, "Invalid total component blocks.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,126,
						"out_function : static void *l7vs_replication_get_replication_address(void)"
						"return = NULL");
		}
		// Debug log END
		return NULL;
	}
	// Get replication memory
	memory = malloc( total_block*DATA_SIZE );

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,40, "malloc : memory=%p: size=%d",memory,total_block*DATA_SIZE);
	}
	// Debug log END

	// malloc Error
	if ( (void *)NULL == memory )
	{
		LOGGER_PUT_LOG_FATAL(LOG_CAT_L7VSD_SYSTEM_MEMORY,1,"Replication memory is Malloc Error.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,127,
						"out_function : static void *l7vs_replication_get_replication_address(void)"
						"return = NULL");
		}
		// Debug log END
		return NULL;
	}
	memset(memory,0,total_block*DATA_SIZE);

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,128,
					"out_function : static void *l7vs_replication_get_replication_address(void)"
					"return = %p",memory);
	}
	// Debug log END
	return memory;
}


/*! Get Component Memory
 *
 * @return memory Component memory
 * @retval memory memory get Success
 * @retval NULL Error
 *
 */
static void *l7vs_replication_get_component_address()
{
	unsigned int total_block ;
	void *memory = NULL ;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,129, "in_function : static void *l7vs_replication_get_component_address(void)");
	}
	// Debug log END

	total_block = replication_state.total_block;
	// Check by continuous initialize.
	if ( REPLICATION_OUT != replication_state.service_status ) 
	{
		// Initialization has already been done.
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,35, "Already got component memory.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,130, 
			"out_function : static void *l7vs_replication_get_component_address(void)"
			"return = NULL");
		}
		// Debug log END
		return NULL;
	}
	// Check Total Block
	if ( 0 == total_block || total_block > CMP_BLOCK_MAX)
	{
		// Check Total Block
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,36, "Invalid total component blocks.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,131, 
			"out_function : static void *l7vs_replication_get_component_address(void)"
			"return = NULL");
		}
		// Debug log END
		return NULL;
	}
	// Get component memory
	memory = malloc( total_block*DATA_SIZE );

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,41, "malloc : memory=%p: size=%d",memory,total_block*DATA_SIZE);
	}
	// Debug log END

	// malloc Error
	if ( (void *)NULL == memory )
	{
		LOGGER_PUT_LOG_FATAL(LOG_CAT_L7VSD_SYSTEM_MEMORY,2,"Component memory is Malloc Error.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,132, 
			"out_function : static void *l7vs_replication_get_component_address(void)"
			"return = NULL");
		}
		// Debug log END
		return NULL;
	}
	memset(memory,0,total_block*DATA_SIZE);

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,133, 
		"out_function : static void *l7vs_replication_get_component_address(void)"
		"return = %p",memory);
	}
	// Debug log END
	return memory;
}


/*! Get Sarface Number Memory 
 *
 * @return memory Component memory
 * @retval memory memory get Success
 * @retval NULL Error
 *
 */
static uint64_t *l7vs_replication_get_surface_info_address()
{
	unsigned int total_block;
	uint64_t *memory = NULL;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,134, 
				"in_function : static uint64_t *l7vs_replication_get_surface_info_address(void)");
	}
	// Debug log END
	total_block = replication_state.total_block;
	// Check by continuous initialize.
	if ( REPLICATION_OUT != replication_state.service_status ) 
	{
		// Initialization has already been done.
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,37, "Already got surface info memory.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,135, 
			"out_function : static uint64_t *l7vs_replication_get_surface_info_address(void)"
			"return = NULL");
		}
		// Debug log END
		return NULL;
	}
	// Check Total Block
	if ( 0 == total_block || total_block > CMP_BLOCK_MAX)
	{
		// Check Total Block
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,38, "Invalid total component blocks.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,136, 
			"out_function : static uint64_t *l7vs_replication_get_surface_info_address(void)"
			"return = NULL");
		}
		// Debug log END
		return NULL;
	}
	// Get memory
	memory = (uint64_t*)malloc( total_block*sizeof(uint64_t) );

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,42, "malloc : memory=%p: size=%lu",memory,total_block*sizeof(uint64_t));
	}
	// Debug log END

	// malloc Error
	if ( (uint64_t *)NULL == memory )
	{
		LOGGER_PUT_LOG_FATAL(LOG_CAT_L7VSD_SYSTEM_MEMORY,3,"Surface info address is Malloc Error.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,137, 
			"out_function : static uint64_t *l7vs_replication_get_surface_info_address(void)"
			"return = NULL");
		}
		// Debug log END
		return NULL;
	}
	memset(memory,0,total_block*sizeof(uint64_t));
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,138, 
		"out_function : static uint64_t *l7vs_replication_get_surface_info_address(void)"
		"return = %p",memory);
	}
	// Debug log END
	return memory;
}

/*! Initialize send network socket
 *
 * @param *fd file descriptor
 * 
 * @return 0/-1
 *  @retval  0 Success
 *  @retval -1 Error
 */
static int l7vs_replication_send_sock_init( )
{
	int info_ret;
	int sockopt_ret;
	int errsv;
	int ret = -1;
	struct addrinfo hints;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,139, 
		"in_function : static int l7vs_replication_send_sock_init(void)");
	}
	// Debug log END

	// Check by continuous initialize.
	if ( REPLICATION_SINGLE == replication_state.service_status ||
		REPLICATION_MASTER == replication_state.service_status ||
		REPLICATION_MASTER_STOP == replication_state.service_status )
	{
		// Initialization has already been done.
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,39,
					"send-socket initialization has already been done. mode : %s",
					replication_mode[(int)replication_state.service_status]);
		ret = -1;
		goto END;
	}
	// initialize structure
	memset(&hints, 0, sizeof(struct addrinfo));

	// set address hints
	hints.ai_family = AF_UNSPEC;     // permit any protocol (IPv4, IPv6, etc.)
	hints.ai_socktype = SOCK_DGRAM;  // UDP
	hints.ai_flags = AI_ALL; 

	//------ Make send socket -------//
	// Whether IP and the port are effective is confirmed.
	info_ret = getaddrinfo(replication_info.ip_addr,
				replication_info.service_name,
				&hints,
				&internal_val.address_info);
	if ( 0 != info_ret )
        {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,40, "Failed to get IP or Port.");
		ret = -1;
		goto END;
	}

	internal_val.socket = socket( internal_val.address_info->ai_family,
					internal_val.address_info->ai_socktype,
					internal_val.address_info->ai_protocol );
	// if cannot make a socket
	if ( internal_val.socket < 3 ) 
	{
		errsv = errno;
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,28,
					"Make send socket error : %s ai_family:%d ai_socktype:%d ai_protocol:%d",
					strerror(errsv),internal_val.address_info->ai_family,
					internal_val.address_info->ai_socktype,
					internal_val.address_info->ai_protocol);
		ret = -1;
		goto END;
	}
	// set a NIC
	 sockopt_ret = setsockopt(internal_val.socket, SOL_SOCKET, SO_BINDTODEVICE,
						replication_info.nic, strlen(replication_info.nic) + 1);
	if ( -1 == sockopt_ret ) 
	{
		// get error no 
		errsv = errno;
		// cannot set a NIC.  Close a Socket.
		close(internal_val.socket);
		internal_val.socket = -1;
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,29,
					"Fails in the option operation of send socket : %s File descriptor : %d NIC : %s",
					strerror(errsv),internal_val.socket,replication_info.nic);
		ret = -1;
		goto END;
	}
	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_SYSTEM_SOCKET,1,"send-socket is make success.");
	ret = 0;
END:

	if ( 0 == ret )
	{
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,36,"send-soket initialization is  succeess.");
	}else{
		if ( -1 != internal_val.socket )	
		{
			close(internal_val.socket);
			internal_val.socket = -1;
		}
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,37,"send-soket initialization is failure.");
	}
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,140, 
		"out_function : static int l7vs_replication_send_sock_init(void)"
		"ret = %d",ret);
	}
	// Debug log END
	return ret;
}


/*! Initialize recv network socket
 *
 * @param *fd file descriptor
 * 
 * @return 0/-1
 *  @retval  0 Success
 *  @retval -1 Error
 */
static int l7vs_replication_recv_sock_init( )
{
	int	getip_ret;
	int	bind_ret;
	int	errsv;
	int	ret = -1;
	struct	servent 	*servent_info;
	

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,141, 
		"in_function : static int l7vs_replication_recv_sock_init(void)");
	}
	// Debug log END

	// Check by continuous initialize.
	if ( REPLICATION_SINGLE == replication_state.service_status ||
		REPLICATION_SLAVE == replication_state.service_status ||
		REPLICATION_SLAVE_STOP == replication_state.service_status )
	{
		// Initialization has already been done.
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,41,
					"recv-socket initialization has already been done. mode : %s",
					replication_mode[(int)replication_state.service_status]);
		ret = -1;
		goto END;
	}

	// get Name for NetworkDeviceName ( IPv4 or IPv6 )
	// address info structs initialize
	memset( &internal_val.addr, 0, sizeof(struct  addressinfo) );

	// get ip address from nic
	getip_ret = getNICAddressInfo(replication_info.nic, &addr);
	if ( 0 == getip_ret )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,31,"You can not get IP address from nic.");
		ret = -1;
		goto END;
	}


	servent_info = getservbyname(replication_info.service_name,"udp");
	if( !servent_info )
	{
		internal_val.addr.addr_in.sin_port = htons( atoi(replication_info.service_name ) );
	}
	else{
		internal_val.addr.addr_in.sin_port = servent_info->s_port ;
	}

	if ( 0 == internal_val.addr.addr_in.sin_port )
	{
		LOGGER_PUT_LOG_ERROR( LOG_CAT_L7VSD_SYSTEM_SOCKET,32,"You can not get port address from %s.",replication_info.service_name );
		ret = -1;
		goto END;
	}


	//make recv socket
	internal_val.socket = socket( PF_INET,
					SOCK_DGRAM,
					IPPROTO_UDP );

	// if cannot make a socket
	if ( internal_val.socket < 3 ) 
	{
		errsv = errno;
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,32,
					"Make recv socket error : %s ai_family:%d ai_socktype:%d ai_protocol:%d",
					strerror(errsv),
					PF_INET,
					SOCK_DGRAM,
					IPPROTO_UDP);
		ret = -1;
		goto END;
	}

	bind_ret = bind(internal_val.socket,(sockaddr *)&internal_val.addr.addr_in, sizeof(internal_val.addr.addr_in));

	// get error no 
	errsv = errno;
	// Error
	if ( bind_ret != 0 )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,33,
					 "Bind error : %s File descriptor : %d ai_addrlen : %d",
					strerror(errsv),
					internal_val.socket,
					(int)sizeof(struct sockaddr));
		//! Close a Socket.
		ret = -1;
		goto END;
	}
	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_SYSTEM_SOCKET,2,"recv-socket is make success.");
	ret = 0;
END:

	if ( 0 == ret )
	{
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,38,"recv-socket initialization is  succeess.");
	}else{
		if ( -1 != internal_val.socket )	
		{
			close(internal_val.socket);
			internal_val.socket = -1;
		}
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,39,"recv-socket initialization is failure..");
	}
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,142, 
		"out_function : static int l7vs_replication_recv_sock_init(void)"
		"ret = %d",ret);
	}
	// Debug log END
	return ret;
}

/*! Finalize network socket
 */
static void l7vs_replication_sock_fini()
{
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,143, 
		"in_function : static void l7vs_replication_sock_fini(void)");
	}
	// Debug log END

	// close a send socket.
	if ( -1 != internal_val.socket )
	{
		close(internal_val.socket);
		internal_val.socket = -1;
	}
	// free address info
	if ( NULL != internal_val.address_info)
	{
		freeaddrinfo(internal_val.address_info);
		internal_val.address_info = NULL;
	}

	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,40, "Socket(send-socket,recv-socket) is finalize.");

	if ( NULL != l7vs_replication_iomux) 
	{
		// replication fini	
		l7vs_replication_iomux->status = iomux_replication_destroy;
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,144,
						"l7vs_replication_iomux->status : iomux_replication_destroy");
		}
		// Debug log END

		// delete l7vs_iomux
		if ( -1 != l7vs_replication_iomux->fd ) {
			l7vs_iomux_remove(l7vs_replication_iomux);
			l7vs_replication_iomux->fd = -1;
		}
	}
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,145, 
		"out_function : static void l7vs_replication_sock_fini(void)");
	}
	// Debug log END
	return;
}

/*! Make serial number
 *
 *  @return Serial number
 *  @retval nonzero Serial number
 *  @retval 0 Error
 */
static unsigned long long l7vs_replication_make_serial()
{
	unsigned long long int serial_num;
	struct timespec current_time;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,146, 
		"in_function : static void l7vs_replication_sock_fini(void)");
	}
	// Debug log END

	// get time by clock_gettime
	if ( clock_gettime(CLOCK_REALTIME, &current_time) == -1 )
	{
		// failre.
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM,6, "You failed to get of time.");
		serial_num = 0;
	} else {
		// make a serial succeeds.
		serial_num = (unsigned long long int) current_time.tv_sec * 1000000 +
					(unsigned long long int) current_time.tv_nsec / 1000;
	}
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,147, 
		"in_function : static void l7vs_replication_sock_fini(void)"
		"return = %llu",serial_num);
	}
	// Debug log END

        return serial_num;
}

/*! Send transfer data to standby server
 *
 *  @param data Points to input data from external program. This will be send to standby server.
 *
 *  @return 0/-1
 *  @retval 0 Success
 *  @retval -1 Error
 */
static int l7vs_replication_send_data(struct l7vs_replication_data *replication_data)
{
	char *send_memory;
	int send_byte;
	int errsv;
	struct l7vs_replication_data send_data;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,148, 
		"in_function : static int l7vs_replication_send_data(struct l7vs_replication_data *replication_data)"
		"replication_data = %p",replication_data);
	}
	// Debug log END


	// Check Argument is NULL. 
	if ( NULL == replication_data )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,42, "Argument is NULL.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,149, 
			"out_function : static int l7vs_replication_send_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}	

	// mode check 
	if ( REPLICATION_MASTER != replication_state.service_status)
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,43, "Can not send replication data. Mode is difficult.  mode : %s",
						replication_mode[(int)replication_state.service_status]);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,150, 
			"out_function : static int l7vs_replication_send_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}
	// Check to replication memory is not  NULL
	if ( NULL == replication_state.replication_memory )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,42, "Replication memory is NULL.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,151, 
			"out_function : static int l7vs_replication_send_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}

	// Check to iomux memory is not NULL
	if ( NULL == l7vs_replication_iomux)
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,43, "Initialize is not yet. Iomux is NULL.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,152, 
			"out_function : static int l7vs_replication_send_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}

	// internal_val.address_info is NULL
	if ( NULL == internal_val.address_info)
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,44, "Initialize is not yet. address info is NULL.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,153, 
			"out_function : static int l7vs_replication_send_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}


	// make replication data struct
	//initialize
	memset(replication_data, 0, sizeof(struct l7vs_replication_data));
	// Set replication id
	replication_data->id = REPLICATION_ID;
	// set block_num (replication_state.last_send_block + 1) and Range check of memory
	if ( replication_state.last_send_block < replication_state.total_block -1)
	{
		replication_data->block_num = replication_state.last_send_block + 1;
	} else if ( replication_state.last_send_block == replication_state.total_block -1){
		replication_data->block_num = 0;
	}else{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,44, "Recv block number is too large.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,154, 
			"out_function : static int l7vs_replication_send_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}
		
	// set serial
	replication_data->serial = replication_state.sarface_block_no;
	if ( 0 == replication_data->serial && 0 == replication_data->block_num)
	{
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_REPLICATION,41, "Serial number is 0, first send processing. ");
	}

	// set data size (sizeof(replication_data))
	replication_data->size = sizeof(struct l7vs_replication_data);

	if ( replication_data->size > SEND_DATA_SIZE)
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,45, "Recv block data size is too large.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,155, 
			"out_function : static int l7vs_replication_send_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}

	// change iomux status
	l7vs_replication_iomux->status = iomux_replication_sending;
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,156,
					"l7vs_replication_iomux->status :"
					"befor : iomux_replication_send_wait"
					"after : iomux_replication_sending");
	}
	// Debug log END

	// set send data
	memset(&send_data,0,sizeof(struct l7vs_replication_data));
	// change to network byteorder
	send_data.id = htons(replication_data->id);
	send_data.serial = replication_data->serial;
	send_data.block_num = htonl(replication_data->block_num);
	send_data.size = (uint64_t)replication_data->size;

	// set replication data (1 block)
	send_memory = (char *)replication_state.replication_memory + DATA_SIZE*replication_data->block_num;
	memcpy( &send_data.data, send_memory, DATA_SIZE );

	// send to data
	send_byte = sendto(internal_val.socket, &send_data, (int)replication_data->size, 0,
					internal_val.address_info->ai_addr, internal_val.address_info->ai_addrlen);
	errsv = errno;	

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		char replication_data_str[DEBUG_STR_LEN] = {0};
		l7vs_replication_data_c_str(replication_data_str,replication_data);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,157,
						"replication_data_struct : %s",
						replication_data_str);
	}

	// Debug log END
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG)
	{
		char	*head;
		int	h=0;

		for ( h = 0; h < (DATA_SIZE / LOG_DATA_WIDTH) ; h++ )
		{
			head = (char *)&send_data.data + h*LOG_DATA_WIDTH;
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,158,
						"%02hhX %02hhX %02hhX %02hhX  %02hhX %02hhX %02hhX %02hhX  "
						"%02hhX %02hhX %02hhX %02hhX  %02hhX %02hhX %02hhX %02hhX"
						, *head, *(head+1), *(head+2), *(head+3)
						, *(head+4), *(head+5), *(head+6), *(head+7)
						, *(head+8), *(head+9), *(head+10), *(head+11)
						, *(head+12), *(head+13), *(head+14), *(head+15));
		}
	}
	// Debug log END
	if (-1 ==  send_byte)	
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM,7, 
					"Data send error : %s  fd : %d size : %d ai_addr : %p addrlen : %d",
					strerror(errsv),
					internal_val.socket,
					(int)replication_data->size,
					internal_val.address_info->ai_addr,
					internal_val.address_info->ai_addrlen);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,159, 
			"out_function : static int l7vs_replication_send_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,160,"Sent data is success. %d", send_byte); 
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,161, 
		"out_function : static int l7vs_replication_send_data(struct l7vs_replication_data *replication_data)"
		"return = 0");
	}
	// Debug log END
        return 0;
}
/*! Receive transfer data from active server
 *
 *  @param  recv_data Points to output data from external program.
 *
 *  @return 0/-1 
 *  @retval 0 Success
 *  @retval -1 Error
 */
static int l7vs_replication_recv_data(struct l7vs_replication_data *replication_data)
{
	char	*recv_memory;
	int	recv_byte;
	int	errsv = 0;
	socklen_t fromlen;
	struct l7vs_replication_data recv_data;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		if ( NULL == replication_data )
		{
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,162, 
			"in_function : static int l7vs_replication_recv_data(struct l7vs_replication_data *replication_data)"
			"replication_data = NULL");
			
		}else{
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,163, 
			"in_function : static int l7vs_replication_recv_data(struct l7vs_replication_data *replication_data)"
			"replication_data = %p",replication_data);
		}
	}
	// Debug log END

	// Check Argument is NULL. 
	if ( NULL == replication_data )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,46, "Argument is NULL.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,164, 
			"out_function : static int l7vs_replication_recv_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}	

	// mode check 
	if ( REPLICATION_SLAVE != replication_state.service_status)
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,47, "Can not receive replication data. Mode is difficult.  mode : %s",
						replication_mode[(int)replication_state.service_status]);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,165, 
			"out_function : static int l7vs_replication_recv_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}

	// Check to replication memory is not  NULL
	if ( NULL == replication_state.replication_memory )
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,45, "Replication memory is NULL.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,166, 
			"out_function : static int l7vs_replication_recv_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}

	// Check to iomux memory is not NULL
	if ( NULL == l7vs_replication_iomux)
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,46, "Initialize is not yet. Iomux is NULL.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,167, 
			"out_function : static int l7vs_replication_recv_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}

	fromlen = sizeof(struct sockaddr);

	// change iomux status
	l7vs_replication_iomux->status = iomux_replication_receving;
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,168,
					"l7vs_replication_iomux->status :"
					"before : iomux_replication_recv_wait :"
					"after : iomux_replication_receving");
	}
	// Debug log END

	//initialize
	memset(&recv_data, 0, sizeof(struct l7vs_replication_data));
	memset(replication_data, 0, sizeof(struct l7vs_replication_data));

	// recv data
	recv_byte = recvfrom( internal_val.socket, &recv_data,
				sizeof(struct l7vs_replication_data),
				MSG_DONTWAIT,
				(sockaddr *)&internal_val.addr.addr_in,
				&fromlen );
	errsv = errno;

	// change to host byteorder
	replication_data->id = ntohs(recv_data.id);
	replication_data->serial = recv_data.serial;
	replication_data->block_num = ntohl(recv_data.block_num);
	replication_data->size = (uint64_t) recv_data.size;

	memcpy(&replication_data->data, &recv_data.data, DATA_SIZE);

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG)
	{
		char	*head;
		int	h;

		for ( h = 0; h < (DATA_SIZE / LOG_DATA_WIDTH) ; h++ )
		{
			head = (char *)&recv_data.data + h*LOG_DATA_WIDTH;
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,169,
						"%02hhX %02hhX %02hhX %02hhX  %02hhX %02hhX %02hhX %02hhX  "
						"%02hhX %02hhX %02hhX %02hhX  %02hhX %02hhX %02hhX %02hhX"
						, *head, *(head+1), *(head+2), *(head+3)
						, *(head+4), *(head+5), *(head+6), *(head+7)
						, *(head+8), *(head+9), *(head+10), *(head+11)
						, *(head+12), *(head+13), *(head+14), *(head+15));
		}
	}
	// Debug log END
	if (-1 ==  recv_byte)	
	{
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_SYSTEM,2, 
					"Reception processing is error : %s  fd : %d data : %p size : %d ai_addr : %p addrlen : %d",
					strerror(errsv),
					internal_val.socket,
					(void *)&recv_data,
					(int)recv_data.size,
					&internal_val.addr.addr_in,
					fromlen);
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,170, 
			"out_function : static int l7vs_replication_recv_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,171,"Recv data is success. %u", recv_byte); 
	}
	// Check replication ID
	if (replication_data->id != REPLICATION_ID)
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,48, "Get invalid data.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,172, 
			"out_function : static int l7vs_replication_recv_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}

	// block number is over

	if (replication_data->block_num > replication_state.total_block)
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,49, "Recv block number is too large.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,173, 
			"out_function : static int l7vs_replication_recv_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}

	// Comparison of serial numbers
	if ( replication_data->serial < replication_state.sarface_block_array_ptr[replication_data->block_num])
	{
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_REPLICATION,50, "Recv replication data is too old.");
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,174, 
			"out_function : static int l7vs_replication_recv_data(struct l7vs_replication_data *replication_data)"
			"return = -1");
		}
		// Debug log END
		return -1;
	}else{
		// Substitution of version
		replication_state.sarface_block_array_ptr[replication_data->block_num] = replication_data->serial;
	}

	// set recv data
	recv_memory = (char *)replication_state.replication_memory + DATA_SIZE*replication_data->block_num;

	// received data. 
	memcpy(recv_memory, &replication_data->data, DATA_SIZE);

	// set sarface block
	replication_state.sarface_block_array_ptr[replication_data->block_num] = replication_data->serial;

	// Sarface numbers are compared.
	for ( unsigned int i = 0; i < replication_state.total_block-1; i++ )
	{
		if ( replication_state.sarface_block_array_ptr[i] != replication_state.sarface_block_array_ptr[i+1])
		{
			break;
		}
		if ( i == replication_state.total_block-2)
		{
			// Synchronization is executed.
			memcpy(replication_state.component_memory,
				replication_state.replication_memory,
				replication_state.total_block*DATA_SIZE );
			if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,82, "Copy Success from replication memory to component memory.");
			}
		}
	}

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,175, 
		"out_function : static int l7vs_replication_recv_data(struct l7vs_replication_data *replication_data)"
		"return = 0");
	}
	// Debug log END
        return 0;
}

/*! Release Replication Memory
 */
static void l7vs_replication_release_replication_address()
{
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,176,
				"in_function : static void l7vs_replication_release_replication_address(void)");
	}
	// Debug log END

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,43,
					"free : replication_state.replication_memory=%p",
					replication_state.replication_memory);
	}
	// Debug log END
	if (NULL != replication_state.replication_memory )
	{
		free(replication_state.replication_memory);
	}
	replication_state.replication_memory=NULL;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,177,
				"out_function : static void l7vs_replication_release_replication_address(void)");
	}
	// Debug log END
}

/*!
 * Release Components Memory
 */
static void l7vs_replication_release_components_address()
{
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,178,
				"in_function : static void l7vs_replication_release_components_address(void)");
	}
	// Debug log END

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,44,
					"free : replication_state.component_memory=%p",
					replication_state.component_memory);
	}
	// Debug log END
	if ( NULL != replication_state.component_memory)
	{
		free(replication_state.component_memory);
	}
	replication_state.component_memory=NULL;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,179,
				"out_function : static void l7vs_replication_release_components_address(void)");
	}
	// Debug log END
}

/*! Release Sarface Memory
 */
static void l7vs_replication_release_surface_info_address()
{
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,180,
				"in_function : static void l7vs_replication_release_surface_info_address(void)");
	}

	// Debug log END

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,45,
					"free : replication_state.sarface_block_array_ptr=%p",
					replication_state.sarface_block_array_ptr);
	}
	// Debug log END
	if ( NULL != replication_state.sarface_block_array_ptr )
	{
		free(replication_state.sarface_block_array_ptr);
	}
	replication_state.sarface_block_array_ptr=NULL;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,181,
				"out_function : static void l7vs_replication_release_surface_info_address(void)");
	}
	// Debug log END

}

/*
 * get Name for NetworkDeviceName 
 * @param[IN]	const char*		NetworkDeviceName
 * @param[OUT] 	struct addressinfo*	address info struct
 * @return 	0 is fail / not 0 is success.
 */
static int getNICAddressInfo( const char* nicname, struct addressinfo* info )
{
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,182,
				"in_function : static int getNICAddressInfo( const char* nicname, struct addressinfo* info )"
				"nicname = %p"
				"info = %p",
				nicname,
				info);
	}
	// Debug log END

	//NIC NAME Check
	if( !nicname || !info )
	{
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,183,
					"out_function : static int getNICAddressInfo( const char* nicname, struct addressinfo* info )"
					"return = 0");
		}
		// Debug log END
		return 0;
	}

	//Networkdevice struct define
	struct ifreq ifr;
	memset( &ifr, 0, sizeof( struct ifreq ) );

	//create socket
	int fd  = socket( AF_INET, SOCK_DGRAM, 0 );
	if( fd < 0 )
	{
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,184,
					"out_function : static int getNICAddressInfo( const char* nicname, struct addressinfo* info )"
					"return = 0");
		}
		// Debug log END
		return 0;
	}

	//get networkdevice struct for IPv4
	strncpy( ifr.ifr_name, nicname, IFNAMSIZ-1 );
	ifr.ifr_addr.sa_family = AF_INET;

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM,1,
				"Use ioctl() :"
				"in_function : static int getNICAddressInfo( const char* nicname, struct addressinfo* info )"
				"nicname = %p"
				"info = %p",
				nicname,
				info);
	}
	// Debug log END

	if( ioctl( fd, SIOCGIFADDR, &ifr ) < 0 )
	{
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,185,
					"out_function : static int getNICAddressInfo( const char* nicname, struct addressinfo* info )"
					"return = 0");
		}
		// Debug log END
		return 0;
	}
	memcpy( &(info->addr_in), &(ifr.ifr_addr), sizeof( struct sockaddr_in ) );
	
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		char sockaddr_in_str[DEBUG_STR_LEN] = {0};
		sockaddr_in_c_str(sockaddr_in_str,&(info->addr_in));
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,186,
						"sockaddr_in_struct : %s",
						sockaddr_in_str);
	}
	// Debug log END


	//get networkdevice struct for IPv6
	memset( &ifr, 0, sizeof( struct ifreq ) );
	strncpy( ifr.ifr_name, nicname, IFNAMSIZ-1 );
	ifr.ifr_addr.sa_family = AF_INET6;
	if( ioctl( fd, SIOCGIFADDR, &ifr ) < 0 )
	{
		// Debug log
		if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,187,
					"out_function : static int getNICAddressInfo( const char* nicname, struct addressinfo* info )"
					"return = 0");
		}
		// Debug log END
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		char sockaddr_in_str[DEBUG_STR_LEN] = {0};
		sockaddr_in_c_str(sockaddr_in_str,&(info->addr_in));
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,188,
						"sockaddr_in_struct : %s",
						sockaddr_in_str);
	}
	// Debug log END
		return 0;
	}
	memcpy( &(info->addr_in6 ), &(ifr.ifr_addr), sizeof( struct sockaddr_in6 ) );

	close( fd );

	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,189,
				"out_function : static int getNICAddressInfo( const char* nicname, struct addressinfo* info )"
				"return = 1");
	}
	// Debug log END
	// Debug log
	if (logger_get_log_level(LOG_CAT_L7VSD_REPLICATION) == LOG_LV_DEBUG){
		char sockaddr_in_str[DEBUG_STR_LEN] = {0};
		sockaddr_in_c_str(sockaddr_in_str,&(info->addr_in));
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REPLICATION,190,
						"sockaddr_in_struct : %s",
						sockaddr_in_str);
	}
	// Debug log END
	return 1;
}


/*!
 * Extract struct l7vs_protomod to strings for debug log.
 * @param  string
 * @return l7vs_replication_data
 */

static void l7vs_replication_data_c_str(char *ret, struct l7vs_replication_data *replication_data)
{
	if ( !replication_data ){
		snprintf(ret, DEBUG_STR_LEN, "(nil)");
		return;
	}	
	snprintf( ret, DEBUG_STR_LEN,
			"replication_data = %p:"
			"id = %u:"
			"serial = %llu:"
			"block_num = %u:"
			"size = %llu:"
			"data = %s:",
			replication_data,
			replication_data->id,
			replication_data->serial,
			replication_data->block_num,
			(long long int)replication_data->size,
			replication_data->data);
	printf("%s",ret);
	return;
}

static void l7vs_replication_component_c_str(char *ret, struct l7vs_replication_component *component_info, int i)
{
	if ( !component_info )
	{
		snprintf(ret, DEBUG_STR_LEN, "(nil)");
		return;
	}
	snprintf( ret, DEBUG_STR_LEN,
			"replication_component = %p:"
			"id[%d] = %s:"
			"block_head[%d] = %u:"
			"block_size[%d] = %u:",
			component_info,
			i,component_info[i].id,
			i,component_info[i].block_head,
			i,component_info[i].block_size);
	return;
}

static void l7vs_replication_state_c_str(char *ret, struct l7vs_replication_state *replication_state)
{
	if ( !replication_state )
	{
		snprintf(ret, DEBUG_STR_LEN, "(nil)");
		return;
	}

	snprintf( ret, DEBUG_STR_LEN,
			"replication_state = %p:"
			"service_status = %s:"
			"send_time = %llu:"
			"last_send_block = %u:"
			"last_recv_block = %u:"
			"total_block = %d:"
			"replication_memory = %p:"
			"component_memory = %p:"
			"sarface_block_no = %llu:"
			"sarface_block_array_ptr = %p:",
			replication_state,
			replication_mode[(int)replication_state->service_status],
			replication_state->send_time,
			replication_state->last_send_block,
			replication_state->last_recv_block,
			replication_state->total_block,
			replication_state->replication_memory,
			replication_state->component_memory,
			(long long int)replication_state->sarface_block_no,
			replication_state->sarface_block_array_ptr);
	return;
}

static void l7vs_replication_info_c_str(char *ret, struct l7vs_replication_info *replication_info)
{
	if ( !replication_info )
	{
		snprintf(ret, DEBUG_STR_LEN, "(nil)");
		return;
	}

	snprintf( ret, DEBUG_STR_LEN,
			"replication_info = %p:"
			"ip_addr = %s:"
			"service_name = %s:"
			"nic = %s:"
			"interval = %u:"
			"component_num = %d:"
			"component_info = %p:",
			replication_info,
			replication_info->ip_addr,
			replication_info->service_name,
			replication_info->nic,
			replication_info->interval,
			replication_info->component_num,
			replication_info->component_info);
}

static void sockaddr_in_c_str(char *ret, struct sockaddr_in *addr_in)
{
	struct sockaddr_in p;

	memset(&p,0,sizeof(struct sockaddr_in));

	if ( !addr_in )
	{
		snprintf(ret, DEBUG_STR_LEN, "(nil)");
		return;
	}

	memcpy(&p,addr_in,sizeof(struct sockaddr_in));
	snprintf( ret, DEBUG_STR_LEN,
			"addr_in = %p:"	
			"sin_family = %d:"
			"sin_port = %u:"
			"in_addr.sin_addr = %llu:"
			"sin_zero = %s:",
			addr_in,
			addr_in->sin_family,
			addr_in->sin_port,
			(long long int)p.sin_addr.s_addr,
			addr_in->sin_zero);
	return;

}

