# mkxml.sh
# -----------------------------------------------------------------------------
#
# XML Manifest Generator Tool (Bourne Shell Script) for mingw-get
#
# Source this script from any package building tool-kit, to create a first-cut
# (tentative) mingw-get conformant package installation manifest.  Assumes that
# the following variables are defined, and passed, by the invoking process:--
#
#   $PROJECT		(optional) project name [default=MinGW]
#   $PROJECT_HOME	(optional) project web site [default=www.mingw.org]
#
#   $PACKAGE		(required) the name of the package
#   $VERSION		(required) the package version (specified upstream)
#   $RELEASE		(required) MinGW project specified release number
#   $ARCH		(optional) designated subsystem [default=mingw32]
#   $TITLE		(optional) title text to be associated with...
#   $DESCRIPTION	(optional) mingw-get catalogue description
#   $COMPONENTS 	(optional) space separated list of component tags
#   $ALIASES		(optional) space separated list of alias names
#
# In addition to the foregoing, optional mingw-get package group affiliations
# may be specified, by passing the following:--
#
#   $AFFILIATES 	(optional) space separated list of affiliate indices
#   $AFFILIATE_n	(optional) name of affiliate group for index entry "n"
#				   in $AFFILIATES; (repeat for each value of
#				   "n", where "n" matches the index value).
#
# -----------------------------------------------------------------------------
#
# $Id: mkxml.sh,v 969b29177aef 2014/01/20 16:13:39 keithmarshall $
#
# Written by Keith Marshall <keithmarshall@users.sourceforge.net>
# Copyright (C) 2011, 2013, MinGW.org Project
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# -----------------------------------------------------------------------------
#
  : ${MINGW_GET_PACKAGE_ALIASES="$ALIASES"}
  : ${MINGW_GET_SUBSYSTEM_NAME="${ARCH-"mingw32"}"}
  : ${MINGW_GET_PACKAGE_NAME="$MINGW_GET_SUBSYSTEM_NAME-$PACKAGE"}
  : ${DESCRIPTION+"${MINGW_GET_PACKAGE_DESCRIPTION="$DESCRIPTION"}"}
  : ${TITLE+"${MINGW_GET_PACKAGE_TITLE="$TITLE"}"}

  MINGW_GET_TARNAME=$PACKAGE-$VERSION-$RELEASE-$MINGW_GET_SUBSYSTEM_NAME
  MINGW_GET_TARNAME_TEMPLATE=$PACKAGE-%-$MINGW_GET_SUBSYSTEM_NAME-%

# We want to use groff, in nroff emulation mode, to control layout.
#
  nroff(){ groff -Tascii; }

# If packages are compressed, attempt to deduce the appropriate extension.
#
  case $COMPRESS_CMD in
    gzip)	TAREXT="tar.gz" ;;
    bzip2)	TAREXT="tar.bz2" ;;
    lzma)	TAREXT="tar.lzma" ;;
    xz) 	TAREXT="tar.xz" ;;
  esac

# Generate the initial section of the XML catalogue file, as far as the
# opening tag for the primary "package" specification element.
#
  cat <<-EOF | nroff
	.ll 256
	<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
	.br
	<software-distribution project="${PROJECT-"MinGW"}"
	home="${PROJECT_HOME-"http://www.mingw.org"}" issue="@YYYYMMDDNN@">
	.sp
	.in 2
	<!-- File: ${MINGW_GET_PACKAGE_NAME}.xml
	${MINGW_GET_PACKAGE_TITLE+"~~ $MINGW_GET_PACKAGE_TITLE "}-->
	.sp
	<package-collection subsystem="${ARCH="mingw32"}">
	.in +2
	<download-host uri=\\c
	"${MINGW_GET_URI-"http://prdownloads.sourceforge.net/mingw/%F?download"}"
	/>
	.sp
	<package name="${MINGW_GET_PACKAGE_NAME}"\\c
	.if !?$MINGW_GET_PACKAGE_ALIASES?? \\& alias="$MINGW_GET_PACKAGE_ALIASES"\\c
	>
	.sp
	.pl 0
	EOF

# Add entries for all specified mingw-get package group affiliations,
# (if any).
#
  test "x$AFFILIATES" = x || {
    for key in $AFFILIATES
    do eval 'affiliate_reference=$'AFFILIATE_$key
       cat <<-EOF | nroff
	.pl 1
	.po 6
	.ll 256
	<affiliate group="$affiliate_reference" />
	EOF
    done
    echo
  }

# If a package description has been provided, format it into an XML
# "description" element.
#
  test -n "$MINGW_GET_PACKAGE_DESCRIPTION" && {
    cat <<-EOF | nroff
	.pl 99999
	.po 6
	.ll 256
	.hy 0
	.nf
	.de para
	.in -2
	.ie \\\\n[para-break] </paragraph>
	.el .nr para-break 1
	.br
	<paragraph>
	.in
	..
	.blm para
	.in 0
	<description lang="en" title="$MINGW_GET_PACKAGE_TITLE">
	.ll 72
	.in 4
	$MINGW_GET_PACKAGE_DESCRIPTION
	.if \\n[para-break] \\{\\
	.   in -2
	.   nop </paragraph>
	.   \\}
	.ll 256
	.in 0
	</description>
	.pl 0
	EOF
    echo
  }

# Add reference entries for the source and licence packages.
#
  for component in src $COMPONENTS
    do case $component in
         src) tag=source ;;
         lic) tag=licence ;;
         *)   tag=void ;;
       esac
       test $tag = void || cat <<-EOF | nroff
	.pl 1
	.ll 256
	.po 6
	<$tag tarname="$MINGW_GET_TARNAME_TEMPLATE-$component.tar${TAREXT+".%"}" />
	EOF
    done

# Add "component"/"release" entries for each specified component package.
# (Note that this does NOT specify dependencies; the addition of "requires")
# (specifications is left as a manual exercise, for the package maintainer.)
#
  for component in $COMPONENTS
    do echo
       cat <<-EOF | nroff
	.pl 1
	.ll 256
	.po 6
	<component class="`IFS=-;set -- $component;echo $1`">
	.in +2
	<release tarname="$MINGW_GET_TARNAME-$component.${TAREXT-"tar"}" />
	.in
	</component>
	EOF
    done

# Close out the "package" element, and all of its ancestors,
# to complete the XML file.
#
  cat <<-EOF | nroff
	.ll 256
	.nf
	.ti 4
	</package>
	.sp
	.ti 2
	</package-collection>
	</software-distribution>
	.fi
	<!-- vim: set nocompatible expandtab fileformat=unix
	textwidth=82 tabstop=2 shiftwidth=2: -->
	.pl 0
	EOF
#
# -----------------------------------------------------------------------------
# $RCSfile: mkxml.sh,v $: end of file
