--[[
--@brief  This mod lets a definition of the global variable become a clear statement or a fixed number.
--@date   Mon,19 May,2014 - Fri,23 May,2014
--@date   Sat,25 Apr,2020
--@date   Tue,28 Apr,2020
--@date   Sun,06 Mar,2022
--@date   Wed,09 Mar,2022
--@author Copyright(C)2014 G-HAL

-- References and Citations;
-- strict.lua
-- checks uses of undeclared global variables
-- All global variables must be 'declared' through a regular assignment
-- (even assigning nil will do) in a main chunk before being used
-- anywhere or assigned to inside a function.
--]]
--[[
Copyright(C)2014 G-HAL
Copyright(C)1994-2014 Lua.org, PUC-Rio.

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
--]]

local enable_debug = false;
local jit_check = jit;
local mt_declared = {};
local mt_table = {};

local mt_type_none = 0;
local mt_type_func_Lua = 1;
local mt_type_func_main = 2;
local mt_type_var_global = 3;
local mt_type_var_global_nil = 4;
local mt_type_var_const = 5;
local mt_type_var_const_nil = 6;
local mt_type_c = 7;


-- function to define global variable
function define_global_array_index(o, n, v)
  if jit_check then return rawset(o, n, v); end;

  local k = tostring(o);
  local msg = k;
  if (o == _G) then msg = "_G"; end;

  if enable_debug then print("define_global_array_index;"..msg..";"..tostring(n)); end;
  local t = mt_declared[k][n];
  if t then
    if (t == mt_type_func_Lua) then
      error(msg..": assign to redeclared global function '"..tostring(n).."' to global variable "..tostring(v), 2);
    end;
    if (t == mt_type_func_main) then
      error(msg..": assign to redeclared function '"..tostring(n).."' to global variable "..tostring(v), 2);
    end;
    if (t == mt_type_var_global) or (t == mt_type_var_global_nil) then
      error(msg..": assign to redeclared global variable '"..tostring(n).."' to global variable "..tostring(v), 2);
    end;
    if (t == mt_type_var_const) or (t == mt_type_var_const_nil) then
      error(msg..": assign to redeclared constant variable '"..tostring(n).."' to global variable "..tostring(v), 2);
    end;
    error(msg..": assign to redeclared '"..tostring(n).."' to global variable "..tostring(v), 2);
  end;

  if (nil == v) then
    mt_declared[k][n] = mt_type_var_global_nil;
    return rawset(mt_table[k], n, 0);
  end;
  mt_declared[k][n] = mt_type_var_global;
  return rawset(mt_table[k], n, v);
end;
function define_global(n, v)
  if jit_check then return rawset(_G, n, v); end;
  if enable_debug then print("define_global;"); end;
  return define_global_array_index(_G, n, v);
end;

-- function to define constant variable
function define_const_array_index(o, n, v)
  if jit_check then return rawset(o, n, v); end;

  local k = tostring(o);
  local msg = k;
  if (o == _G) then msg = "_G"; end;

  if enable_debug then print("define_const_array_index;"..msg..";"..tostring(n)); end;
  local t = mt_declared[k][n];
  if t then
    if (t == mt_type_func_Lua) then
      error(msg..": assign to redeclared global function '"..tostring(n).."' to constant variable "..tostring(v), 2);
    end;
    if (t == mt_type_func_main) then
      error(msg..": assign to redeclared function '"..tostring(n).."' to constant variable "..tostring(v), 2);
    end;
    if (t == mt_type_var_global) or (t == mt_type_var_global_nil) then
      error(msg..": assign to redeclared global variable '"..tostring(n).."' to constant variable "..tostring(v), 2);
    end;
    if (t == mt_type_var_const) or (t == mt_type_var_const_nil) then
      error(msg..": assign to redeclared constant variable '"..tostring(n).."' to constant variable "..tostring(v), 2);
    end;
    error(msg..": assign to redeclared '"..tostring(n).."' to constant variable "..tostring(v), 2);
  end;

  if (nil == v) then
    mt_declared[k][n] = mt_type_var_const_nil;
    return rawset(mt_table[k], n, 0);
  end;
  mt_declared[k][n] = mt_type_var_const;
  return rawset(mt_table[k], n, v);
end;
function define_const(n, v)
  if jit_check then return rawset(_G, n, v); end;
  if enable_debug then print("define_const;"); end;
  return define_const_array_index(_G, n, v);
end;


local function init(obj)
  if jit_check then return; end;

  local key = tostring(obj);
  if enable_debug then print("init;"..key); end;
  mt_declared[key] = {};
  mt_table[key] = {};

  local mt = getmetatable(obj);
  if (nil == mt) then
    mt = {};
    setmetatable(obj, mt);
  end;

  local function what ()
    local d = debug.getinfo(3, "S");
    return d and d.what or "C";
  end;

  -- declaration
  mt.__newindex = function (o, n, v)
    local k = tostring(o);
    local msg = k;
    if (o == _G) then msg = "_G"; end;

    if enable_debug then print("__newindex;"..msg..";"..tostring(n)); end;
    local t = mt_declared[k][n];
    if t then
      if (t == mt_type_func_Lua) then
        error(msg..": assign to change global function '"..tostring(n).."' to "..tostring(v), 2);
      end;
      if (t == mt_type_func_main) then
        error(msg..": assign to change function '"..tostring(n).."' to "..tostring(v), 2);
      end;
      if (t == mt_type_var_global) or (t == mt_type_var_global_nil) then
        if (nil == v) then
          mt_declared[k][n] = mt_type_var_global_nil;
          return rawset(mt_table[k], n, 0);
        end;
        mt_declared[k][n] = mt_type_var_global;
        return rawset(mt_table[k], n, v);
      end;
      if (t == mt_type_var_const) or (t == mt_type_var_const_nil) then
        error(msg..": assign to change constant variable '"..tostring(n).."' to "..tostring(v), 2);
      end;
      return rawset(mt_table[k], n, v);
    end;

    local w = what();
    if ("C" == w) then
      mt_declared[k][n] = mt_type_c;
      return rawset(mt_table[k], n, v);
    end;
    if ("function" == type(v)) then
      if ("Lua" == w) then
        mt_declared[k][n] = mt_type_func_Lua;
        return rawset(mt_table[k], n, v);
      end;
      if ("main" == w) then
        mt_declared[k][n] = mt_type_func_main;
        return rawset(mt_table[k], n, v);
      end;
      error(msg..": assign to undeclared function '"..tostring(n).."' to "..tostring(v), 2);
    end;
    error(msg..": assign to undeclared variable '"..tostring(n).."' to "..tostring(v), 2);
  end;

  -- reference
  mt.__index = function (o, n)
    local k = tostring(o);
    local msg = k;
    if (o == _G) then msg = "_G"; end;

    if enable_debug then print("__index;"..msg..";"..tostring(n)); end;
    local t = mt_declared[k][n];
    if t then
      if (t == mt_type_var_global_nil) or (t == mt_type_var_const_nil) then
        return nil;
      end;
      return rawget(mt_table[k], n);
    end;

    if ("C" == what()) then
      return rawget(mt_table[k], n);
    end;
    error(msg..": global variable '"..tostring(n).."' is not declared", 2);
  end;

  -- iterator
  mt.__pairs = function(o)
    local k = tostring(o);
    local function stateless_iter(tbl, k)
      local v;
      k, v = next(mt_table[tbl], k);
      local t = mt_declared[tbl][k];
      if (t == mt_type_var_global_nil) or (t == mt_type_var_const_nil) then
        return k, nil;
      end;
      if (nil ~= v) then
        return k, v;
      end;
      return nil;
    end;
    return stateless_iter, k, nil;
  end;

  -- iterator
  mt.__ipairs = function(o)
    local k = tostring(o);
    local function stateless_iter(tbl, i)
      i = i + 1;
      local v = mt_table[tbl][i];
      local t = mt_declared[tbl][i];
      if (t == mt_type_var_global_nil) or (t == mt_type_var_const_nil) then
        return i, nil;
      end;
      if (nil ~= v) then
        return i, v;
      end;
      return nil;
    end;
    return stateless_iter, k, 0;
  end;
end;

function define_global_array(o)
  if jit_check then return rawset(_G, o, {}); end;

  if enable_debug then print("define_global_array;"..tostring(o)); end;
  local ret = define_global(o, {});
  init(_G[o]);
  return;
end;


init(_G);

-- [ End of File ]
