#!/usr/local/bin/luac53 -p --
--[[
--@brief  Quasi Robot Language; suspend
--@date   Sun,31 Oct,2021
--@date   Wed,03 Nov,2021
--@date   Sun,07 Nov,2021
--@date   Sun,06 Mar,2022
--@date   Fri,11 Mar,2022
--@author Copyright(C)2021-2022 G-HAL
--]]



local M = {};

define_const("c_suspend_separator", '\t');

define_const("c_suspend_var_class_tag", ':');
define_const("c_suspend_var_class_QRL_variable", 'QRL_variable');

define_const("c_suspend_var_QRL_systemvariable", 'g_QRL_SystemVariable');
define_const("c_suspend_var_QRL_uservariable", 'g_QRL_UserVariable');

define_const("c_suspend_var_type_nil", 'nil');
define_const("c_suspend_var_type_string", 'string');
define_const("c_suspend_var_type_number", 'number');
define_const("c_suspend_var_type_boolean", 'boolean');
define_const("c_suspend_var_type_boolean_true", 'true');
define_const("c_suspend_var_type_boolean_false", 'false');
define_const("c_suspend_var_type_table", 'table');
define_const("c_suspend_var_type_function", 'function');
define_const("c_suspend_var_type_thread", 'thread');
define_const("c_suspend_var_type_userdata", 'userdata');
define_const("c_suspend_var_type_empty", 'empty');


M.Hibernate = function(fname)
  -- print('qrl_chip_hibernate ' .. fname );

  local f = io.open(fname, "w");

  local function w_v(v)
    local t = type(v);
    f:write(t);
    f:write(c_suspend_separator);
    if (c_suspend_var_type_nil == t) then
      f:write(c_suspend_var_type_nil);
    elseif (c_suspend_var_type_string == t) then
      f:write(v);
    elseif (c_suspend_var_type_number == t) then
      f:write(v);
    elseif (c_suspend_var_type_boolean == t) then
      if v then
        f:write(c_suspend_var_type_boolean_true);
      else
        f:write(c_suspend_var_type_boolean_false);
      end;
    elseif (c_suspend_var_type_table == t) then
      local s = tostring(v);
      f:write(string.sub(s,(string.find(s,' ')+1)));
    elseif (c_suspend_var_type_function == t) then
      local s = tostring(v);
      f:write(string.sub(s,(string.find(s,' ')+1)));
    elseif (c_suspend_var_type_thread == t) then
      local s = tostring(v);
      f:write(string.sub(s,(string.find(s,' ')+1)));
    elseif (c_suspend_var_type_userdata == t) then
      f:write(Mek_GUID(v));
    else
      f:write(v);
    end;
  end;

  local function w_var(k, v)
    f:write(c_suspend_var_class_QRL_variable .. c_suspend_var_class_tag .. c_suspend_separator .. c_suspend_var_QRL_systemvariable .. '[' .. k .. ']' .. c_suspend_separator);
    w_v(v);
    f:write('\n');
    return;
  end;

  local function w_array(k, v)
    if (nil == next(v)) then
      f:write(c_suspend_var_class_QRL_variable .. c_suspend_var_class_tag .. c_suspend_separator .. c_suspend_var_QRL_systemvariable .. '[' .. k .. ']' .. '[' .. '' .. ']' .. c_suspend_separator .. c_suspend_var_type_empty .. '\n');
    else
      for key2, value2 in pairs(v) do
        f:write(c_suspend_var_class_QRL_variable .. c_suspend_var_class_tag .. c_suspend_separator .. c_suspend_var_QRL_systemvariable .. '[' .. k .. ']' .. '[' .. key2 .. ']' .. c_suspend_separator);
        w_v(value2);
        f:write('\n');
      end;
    end;
    return;
  end;

  local function w_g_QRL_SystemVariable(k, v)
    if (c_suspend_var_type_table == type(v)) then
      w_array(k, v);
    else
      w_var(k, v);
    end;
    return;
  end;

  local function w_g_QRL_UserVariable(k, t, v)
    f:write(c_suspend_var_class_QRL_variable .. c_suspend_var_class_tag .. c_suspend_separator .. c_suspend_var_QRL_uservariable .. '[' .. k .. ']' .. c_suspend_separator .. t .. c_suspend_separator);
    w_v(v);
    f:write('\n');
    return;
  end;


  for key, value in pairs(g_QRL_SystemVariable) do
    w_g_QRL_SystemVariable(key, value);
  end;

  if (nil == next(g_QRL_UserVariable_Type)) then
    f:write(c_suspend_var_class_QRL_variable .. c_suspend_var_class_tag .. c_suspend_separator .. c_suspend_var_QRL_uservariable .. '[' .. '' .. ']' .. c_suspend_separator .. c_suspend_var_type_empty .. '\n');
  else
    for key, value in pairs(g_QRL_UserVariable_Type) do
      w_g_QRL_UserVariable(key, value, g_QRL_UserVariable_Contents[key]);
    end;
  end;

  f:close();

  return;
end;


M.Resume = function(GB, fname)
  -- print('qrl_chip_resume ' .. fname );

  local f = io.open(fname, "r");
  if (nil == f) then
    return;
  end;

  local function split_by_tab(str)
    if (nil == str) then
      return nil, nil;
    end;
    local ptr = string.find(str, c_suspend_separator);
    if ((nil == ptr) or (ptr < 2)) then
      return nil, nil;
    end;
    local before = string.sub(str, 0, (ptr - 1));
    local after = string.sub(str, (ptr + 1));
    return before, after;
  end;

  for line in f:lines() do
    do
      -- Divide the line.
      local tmp_str = line;

      local class;
      class, tmp_str = split_by_tab(tmp_str);
      local class_len = string.len(class);
      if ((nil == class) or (c_suspend_var_class_tag ~= string.sub(class, class_len))) then
        print('ERROR: suspend.lua: Resume(): Illigal classification. ' .. line);
        break;
      end;
      class = string.sub(class, 0, (class_len - 1));

      local var_name;
      var_name, tmp_str = split_by_tab(tmp_str);
      if (nil == var_name) then
        print('ERROR: suspend.lua: Resume(): Illigal variable name. ' .. line);
        break;
      end;
      local var_name_index_s, var_name_index_e, var_name_index2 = string.find(var_name, '%[[_%w]+%]%[([_%w]*)%]');
      local var_name_index = nil;
      if (nil ~= var_name_index_s) then
        local dummy1, dummy2;
        dummy1, dummy2, var_name_index = string.find(var_name, '%[([_%w]+)%]%[[_%w]*%]');
        var_name = string.sub(var_name, 0, (var_name_index_s - 1));
      else
        var_name_index_s, var_name_index_e, var_name_index = string.find(var_name, '%[([_%w]*)%]');
        if (nil ~= var_name_index_s) then
          var_name = string.sub(var_name, 0, (var_name_index_s - 1));
        else
          print('ERROR: suspend.lua: Resume(): Illigal index name. ' .. line);
          break;
        end;
      end;
      if (nil == var_name_index) then
        print('ERROR: suspend.lua: Resume(): An index name was not found. ' .. line);
        break;
      end;

      local QRL_var_type = nil;
      if (c_suspend_var_class_QRL_variable == class) and (c_suspend_var_QRL_uservariable == var_name) then
        if (c_suspend_var_type_empty == tmp_str) then
          QRL_var_type = -1;
        else
          local QRL_var_type_raw;
          QRL_var_type_raw, tmp_str = split_by_tab(tmp_str);
          if (nil == QRL_var_type_raw) then
            print('ERROR: suspend.lua: Resume(): QRL variable type was not found. ' .. line);
            break;
          end;
          QRL_var_type = tonumber(QRL_var_type_raw);
        end;
      end;

      local var_type, tmp2_str = split_by_tab(tmp_str);
      if (nil == var_type) then
        var_type = tmp_str;
      end;

      local var_contents = tmp2_str;

      -- Check the line.
      if (c_suspend_var_class_QRL_variable == class) then
        if (nil == var_name) then
          print('ERROR: suspend.lua: Resume(): "' .. c_suspend_var_class_QRL_variable .. '" must have a name. ' .. line);
          break;
        end;
        if (c_suspend_var_QRL_systemvariable == var_name) then
          if (nil == var_name_index) then
            print('ERROR: suspend.lua: Resume(): "' .. c_suspend_var_QRL_systemvariable .. '" must have an index. ' .. line);
            break;
          end;
        elseif (c_suspend_var_QRL_uservariable == var_name) then
          if (nil ~= var_name_index2) then
            print('ERROR: suspend.lua: Resume(): "' .. c_suspend_var_QRL_uservariable .. '" cannot have indexes. ' .. line);
            break;
          end;
        else
          print('ERROR: suspend.lua: Resume(): A variable name "' .. var_name .. '" . ' .. line);
          break;
        end;
      else
        print('ERROR: suspend.lua: Resume(): A class "' .. class .. '" is not supported. ' .. line);
        break;
      end;

      -- Convert the contents.
      if (c_suspend_var_type_nil == var_type) then
        var_contents = nil;
      elseif (c_suspend_var_type_string == var_type) then
        var_contents = var_contents;
      elseif (c_suspend_var_type_number == var_type) then
        var_contents = tonumber(var_contents);
      elseif (c_suspend_var_type_boolean == var_type) then
        if (c_suspend_var_type_boolean_true == var_contents) then
          var_contents = true;
        elseif (c_suspend_var_type_boolean_false == var_contents) then
          var_contents = false;
        else
          print('ERROR: suspend.lua: Resume(): "' .. c_suspend_var_type_boolean .. '" must be true or false. ' .. line);
          break;
        end;
      elseif (c_suspend_var_type_table == var_type) then
        print('ERROR: suspend.lua: Resume(): "' .. c_suspend_var_type_table .. '" is not supported. ' .. line);
        break;
      elseif (c_suspend_var_type_function == var_type) then
        print('ERROR: suspend.lua: Resume(): "' .. c_suspend_var_type_function .. '" is not supported. ' .. line);
        break;
      elseif (c_suspend_var_type_thread == var_type) then
        print('ERROR: suspend.lua: Resume(): "' .. c_suspend_var_type_thread .. '" is not supported. ' .. line);
        break;
      elseif (c_suspend_var_type_userdata == var_type) then
        var_contents = GUID_to_Mek( GB , var_contents);
      elseif (c_suspend_var_type_empty == var_type) then
        var_contents = nil;
      else
        print('ERROR: suspend.lua: Resume(): "' .. var_type .. '" is not supported. ' .. line);
        break;
      end;

      -- Set the variable.
      if (c_suspend_var_class_QRL_variable == class) then
        if (c_suspend_var_QRL_systemvariable == var_name) then
          if (c_suspend_var_type_empty == var_type) then
            g_QRL_SystemVariable[var_name_index] = {};
          else
            if (nil == var_name_index2) then
              g_QRL_SystemVariable[var_name_index] = var_contents;
            else
              g_QRL_SystemVariable[var_name_index][var_name_index2] = var_contents;
            end;
          end;
        elseif (c_suspend_var_QRL_uservariable == var_name) then
          if (c_suspend_var_type_empty == var_type) then
            g_QRL_UserVariable_Type = {};
            g_QRL_UserVariable_Contents ={};
          else
            g_QRL_UserVariable_Type[var_name_index] = QRL_var_type;
            g_QRL_UserVariable_Contents[var_name_index] = var_contents;
          end;
        end;
      end;
    end;
  end;

  f:close();

  return;
end;


M.Destroy = function(fname)
  -- print('qrl_chip_destroy ' .. fname );
  os.remove(fname);
  return;
end;

return M;

-- [ End of File ]
