#!/usr/local/bin/luac53 -p --
--[[
--@brief  Quasi Robot Language; A*
--@date   Sun,27 Feb,2022
--@date   Fri,11 Mar,2022
--@author Copyright(C)2022 G-HAL
--]]



local M = {};

local enable_debugger = true;
define_const("AMax", 8);
local action_list = {};


local binaryheap = require "LuaData/binaryheap/binaryheap";

local function Cell(d, x, y)
  return string.format("%d_%d_%d", d, x, y);
end;

local function Coordinates(cell)
  local d, x, y = cell:match("(%d+)_(%d+)_(%d+)");
  return tonumber(d), tonumber(x), tonumber(y);
end;



M.Init = function()
  action_list = { pos = {}, action = {} };
  return;
end;

M.Search = function(GB, Mek, target_mek)
  local bh = binaryheap.minUnique();
  local openlist = { count = {}, before_pos = {}, action = {}, cost = {} };
  local closelist = { count = {}, before_pos = {}, action = {}, cost = {} };
  action_list = { pos = {}, action = {} };

  -- Make Start Node
  local x_s, y_s, z_s = GearCurrentLocation(Mek);
  local d_s = Mek_NAttValue_Location(Mek, NAS_D);
  local pos_s = Cell(d_s, x_s, y_s);
  local beforepos_s = "";
  local cost_s = 0;

  -- Make Goal Node
  local x_g, y_g, z_g = GearCurrentLocation(target_mek);
  local d_g = Mek_NAttValue_Location(target_mek, NAS_D);
  local pos_g = Cell(d_g, x_g, y_g);

  -- Add Start Node to openlist
  bh:insert(cost_s, pos_s);
  openlist.count[pos_s] = 0;
  openlist.before_pos[pos_s] = beforepos_s;
  openlist.action[pos_s] = NAV_Stop;
  openlist.cost[pos_s] = cost_s;

  -- Check openlist
  local pos_result = 0;
  while bh:peek() do

    -- Get the Top of openlist
    local pos_n, f_n = bh:pop();
    local count = openlist.count[pos_n] + 1;
    local beforepos_n = openlist.before_pos[pos_n];
    local action_n = openlist.action[pos_n];
    local g_n = openlist.cost[pos_n];
    local d_n, x_n, y_n = Coordinates(pos_n);
    openlist.count[pos_n] = nil;
    openlist.before_pos[pos_n] = nil;
    openlist.action[pos_n] = nil;
    openlist.cost[pos_n] = nil;

    -- Check Goal
    if (x_g == x_n) and (y_g == y_n) then
      closelist.count[pos_n] = count;
      closelist.before_pos[pos_n] = beforepos_n;
      closelist.action[pos_n] = action_n;
      closelist.cost[pos_n] = g_n;
      pos_result = pos_n;
      break;
    else
      if closelist.cost[pos_n] then
        PrintErrorMessage(string.format("A*.Search:***BUG***: dup: %s", pos_n));
        if (g_n < closelist.cost[pos_n]) then
          closelist.count[pos_n] = count;
          closelist.before_pos[pos_n] = beforepos_n;
          closelist.action[pos_n] = action_n;
          closelist.cost[pos_n] = g_n;
        else
        end;
      else
        closelist.count[pos_n] = count;
        closelist.before_pos[pos_n] = beforepos_n;
        closelist.action[pos_n] = action_n;
        closelist.cost[pos_n] = g_n;
      end;
    end;

    -- Check Neighbors
    local function check(action_m, d_m, x_m, y_m)
      --print(string.format("check: %d, %d, %d, %d", action_m, d_m, x_m, y_m));
      if (d_m < 0) then
        d_m = (AMax - 1);
      end;
      if (AMax <= d_m) then
        d_m = 0;
      end;
      if OnTheMap_XY(x_m, y_m) then
        local function cost()
          -- Check if it is movable or not.
          if ((x_g ~= x_m) or (y_g ~= y_m)) and IsBlocked_for_General(Mek, GB, x_m, y_m) then
            return -1;
          end;

          -- Calculate the Cost of Action
          local switch = {
            default         = function() assert(false, string.format("A*.Search:***BUG***: %s", pos_m)); return 0; end;
            [NAV_NormSpeed] = function() return 1; end;
            [NAV_TurnLeft]  = function() return 2; end;
            [NAV_TurnRight] = function() return 2; end;
            [NAV_Reverse]   = function() return 3; end;
          };
          local action_cost = (switch[action_m] or switch.default)();

          -- Calculate the Cost of Penalty
          -- ***BUG***
          local penalty_cost = 0;
          if (0 < action_n) and (action_n ~= action_m) then
            penalty_cost = penalty_cost + 1;
          end;

          return action_cost + penalty_cost;
        end;
        local function heuristic()
          local diff_x = (x_g - x_m);
          local diff_y = (y_g - y_m);
          local h = math.floor(math.sqrt(diff_x * diff_x + diff_y * diff_y));
          return h;
        end;

        local c = cost();
        if (c < 0) then
          return;
        end;
        local g_m = g_n + c;
        local f_m = g_m + heuristic();

        local pos_m = Cell(d_m, x_m, y_m);
        local flag_o = openlist.cost[pos_m];
        local flag_c = closelist.cost[pos_m];
        if (not flag_o) and (not flag_c) then
          --print(string.format("insert: %s", pos_m));
          bh:insert(f_m, pos_m);
          openlist.count[pos_m] = count;
          openlist.before_pos[pos_m] = pos_n;
          openlist.action[pos_m] = action_m;
          openlist.cost[pos_m] = g_m;
        elseif flag_o then
          if (f_m < flag_o) then
            if enable_debugger then
              print(string.format("flag_o: %s, f_m: %d, f_o: %d", pos_m, f_m, flag_o));
            end;
            bh:update(pos_m, f_m);
            openlist.count[pos_m] = count;
            openlist.before_pos[pos_m] = pos_n;
            openlist.action[pos_m] = action_m;
            openlist.cost[pos_m] = g_m;
          end;
        elseif flag_c then
          if (f_m < flag_c) then
            if enable_debugger then
              print(string.format("flag_c: %s, f_m: %d, f_o: %d", pos_m, f_m, flag_c));
            end;
            bh:insert(f_m, pos_m);
            openlist.count[pos_m] = count;
            openlist.before_pos[pos_m] = pos_n;
            openlist.action[pos_m] = action_m;
            openlist.cost[pos_m] = g_m;
            closelist.count[pos_n] = count;
            closelist.before_pos[pos_m] = nil;
            closelist.action[pos_m] = nil;
            closelist.cost[pos_m] = nil;
          end;
        else
          assert(false, string.format("A*.Search:***BUG***: %s", pos_m));
        end;
      end;
      return;
    end;
    check(NAV_NormSpeed, d_n, (x_n + AngDir_X[d_n + 1]), (y_n + AngDir_Y[d_n + 1]));
    check(NAV_TurnLeft,  (d_n - 1), x_n, y_n);
    check(NAV_TurnRight, (d_n + 1), x_n, y_n);
    check(NAV_Reverse,   d_n, (x_n - AngDir_X[d_n + 1]), (y_n - AngDir_Y[d_n + 1]));
  end;
  if not pos_result then
    return 0;
  end;

  -- Get the Result
  local pos = pos_result;
  local i;
  for i = closelist.count[pos], 1, -1 do
    action_list.pos[i] = pos;
    action_list.action[i] = closelist.action[pos];
    pos = closelist.before_pos[pos];
  end;

  -- Debug Output
  if enable_debugger then
    print(string.format("start: %s, goal: %s", pos_s, pos_g));
    local d, x, y;
    for d = 0, (AMax - 1) do
      print(string.format("d:%d", d));
      for y = 1, YMax do
        local row = {};
        for x = 1, XMax do
          local cell = Cell(d, x, y);
          local cost = closelist.cost[cell];
          if cost then
            table.insert(row, cost);
          else
            table.insert(row, "*");
          end;
        end;
        print(table.concat(row, " "));
      end;
    end;

    local k, v;
    for k, v in pairs(openlist.cost) do
      print(string.format("openlist[%d]: key: %s, value: %d, action: %d, before: %s", openlist.count[k], k, v, openlist.action[k], openlist.before_pos[k]));
    end;
    for k, v in pairs(closelist.cost) do
      print(string.format("closelist[%d]: key: %s, value: %d, action: %d, before: %s", closelist.count[k], k, v, closelist.action[k], closelist.before_pos[k]));
    end;

    for i = 1, #action_list.action do
      print(string.format("action_list[%d]: pos: %s, action: %d", i, action_list.pos[i], action_list.action[i]));
    end;

    print("");
  end;

  return (#action_list.action - 1);
end;

M.Exec = function()
  if (2 <= #action_list.action) then
    g_QRL_SystemVariable["ExecQueue_Type"]            = c_ExecQueue_Type_PrepAction;
    g_QRL_SystemVariable["ExecQueue_Type_PrepAction"] = action_list.action[2];
  else
    PrintErrorMessage('A*.Exec:***BUG***:Illigal function call.');
  end;
  return;
end;

return M;

-- [ End of File ]
