/*
	vfdutil.c

	Virtual FD utility functions
	Copyright (C) 2003 Kenji Kato
*/

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <stdio.h>
#include <crtdbg.h>

#include "vfdctl.h"
#include "vfdutil.h"

//
//	Ensure Virtual FD image is unmounted
//
DWORD EnsureUnmount(
	ENSURE_CALLBACK retry_cb,
	ENSURE_CALLBACK cont_cb,
	DWORD param)
{
	char	drive_letter = '\0';
	int		i;
	DWORD	ret;

	//	get current drive letter
	if ((ret = VfdGetDriveLetter(&drive_letter)) != ERROR_SUCCESS) {
		DEBUG_TRACE1("EnsureUnmount: VfdGetDriveLetter - %s\n", ErrMsg(ret));
	}

	//	remove current drive letter
	if (isalpha(drive_letter)) {
		if ((ret = VfdDelDriveLetter(drive_letter)) != ERROR_SUCCESS) {
			DEBUG_TRACE1("EnsureUnmount: VfdDelDriveLetter - %s\n", ErrMsg(ret));
			drive_letter = '\0';
		}
	}

	//	Unmount the drive
unmount_retry:
	SetCursor(LoadCursor(NULL, IDC_WAIT));

	i = 0;

	for(;;) {
		if ((ret = VfdUmount()) != ERROR_ACCESS_DENIED || ++i > 10) {
			break;
		}

		Sleep(500);			//	retry after 0.5 sec.
	}

	SetCursor(LoadCursor(NULL, IDC_ARROW));

	//	retry ?
	if (ret == ERROR_ACCESS_DENIED && retry_cb && (*retry_cb)(param)) {
		goto unmount_retry;
	}

	if (ret == ERROR_SUCCESS || ret == ERROR_NOT_READY) {
		//	the drive is empty
		return ret;
	}

	//	continue ?
	if (cont_cb && (*cont_cb)(param ? param : ret)) {
		return ERROR_NOT_READY;
	}

	// No -- restore the drive letter
	if (isalpha(drive_letter)) {
		VfdSetDriveLetter(drive_letter);
	}

	return ret;
}

//
// choose first available drive letter
//
char ChooseDriveLetter()
{
	DWORD	logical_drives = GetLogicalDrives();
	char	drive_letter = 'A';

	if (logical_drives == 0) {
		return '\0';
	}

	while (logical_drives & 0x1) {
		logical_drives >>= 1;
		drive_letter++;
	}

	if (drive_letter > 'Z') {
		return '\0';
	}

	return drive_letter;
}

#ifdef _DEBUG
//
//	Get system error message string
//
const char *ErrMsg(DWORD err)
{
	static char msg[256];

	if (!VfdErrorMessage(err, msg, sizeof(msg))) {
		sprintf(msg, "Unknown error %lu (0x%08x)\n", err, err);
	}

	return msg;
}

//
//	Format and output debug string
//
void DebugTrace(LPCTSTR file, int line, LPCTSTR format, ...)
{
	char buf[512];
	int len;
	va_list args;

	len = sprintf(buf, "%s(%lu) : ", file, line);

	va_start(args, format);

	vsprintf(buf + len, format, args);

	OutputDebugString(buf);
}
#endif	// _DEBUG

/* End Of File */
