package shooting;
 
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
 
import org.lwjgl.input.Controller;
 
public abstract class GameController {
    /**
     *  論理キーを表す enum
     *  ゲームコントローラーが持つ○ボタン（決定ボタン）や×ボタン（キャンセルボタン）などに該当する
     */
    public enum Input {
        UP,
        DOWN,
        RIGHT,
        LEFT,
         
        CIRCLE,
        CROSS,
        SQUARE,
        TRIANGLE,
         
        SELECT,
        START,
        R1,R2,
        L1,L2;
    }
    private final Map<Integer, Input> keyConfigs;
    private final Map<Input, Boolean> keyStatus = new HashMap<Input, Boolean>();
 
    private Input               keyEvent;
     
    public GameController(Map<Integer, Input> keyConfigs) {
        this.keyConfigs = new HashMap<Integer, Input>();
 
        //  デフォルトのキーコンフィグとして、引数　keyConfigs　の内容を設定する
        for (int keyCode: keyConfigs.keySet()) {
            setKeyConfig(keyCode, keyConfigs.get(keyCode));
        }
    }
 
    /**
     *  キーの入力状態をチェックし、状態を更新する
     */
    public abstract void update();
 
    /**
     *  引数に渡されたジョイスティックがこのゲームコントローラーのものなら true を、そうでなければ false を返す
     */
    public abstract boolean isOwner(Controller controller);
     
    /**
     *  このゲームコントローラーの名前を返す
     */
    public abstract String getName();
 
    /**
     *  引数に渡されたキーコードに割り当てられた論理キーを返す
     */
    protected Input getInput(int keyCode) {
        return keyConfigs.get(keyCode);
    }
 
    /**
     *  論理キーに割り当てられているすべてのキーコードの Set を返す
     */
    protected Set<Integer> keyCodes() {
        return keyConfigs.keySet();
    }
     
    /**
     *  引数に渡されたキーコードに、論理キーを割り当てる
     */
    public void setKeyConfig(int keyCode, Input input) {
        keyConfigs.put(keyCode, input);
    }
     
    /**
     *  引数に渡された論理キーの入力状態を設定する
     */
    protected void setKeyStatus(Input input, boolean isPressing) {
        //  論理キーが今まで押されていて、かつ離された場合、論理キーの入力イベントが完了したものとして保持しておく
        //  ただし、保持できる入力イベントは、最初の 1 つのみとする
        if ((isPressing(input)) && (!isPressing) && (this.keyEvent == null)) {
            this.keyEvent = input;
        }
         
        //  論理キーの入力状態を設定する
        keyStatus.put(input, isPressing);
    }
 
    /**
     *  引数に渡された論理キーの入力状態を返す
     */
    public boolean isPressing(Input input) {
        //  現在押されていれば true を、そうでなければ false を返す
        Boolean isPressing = keyStatus.get(input);
         
        if (isPressing == null) {
            return false;
        } else {
            return isPressing.booleanValue();
        }
    }
 
    /**
     *  論理キーの入力イベントを返す
     *  入力イベントとは、キーが押された後に離された時点で発生する。
     *  また、入力イベントは最初に発生したものを１つだけ保するものとする。
     */
    public Input getKeyEvent() {
        Input   keyEvent = this.keyEvent;
 
        //  論理キーの入力イベントは、取得したら空にする
        this.keyEvent = null;
         
        return keyEvent;
    }

}