/*
 * PMX bone information handler
 *
 * License : The MIT License
 * Copyright(c) 2015 MikuToga Partners
 */

package jp.sfjp.mikutoga.pmx.parser;

import jp.sfjp.mikutoga.bin.parser.LoopHandler;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.bin.parser.ParseStage;

/**
 * PMXモデルの各種ボーン情報の通知用ハンドラ。
 * ボーン定義の出現順と、0から始まるボーンIDは対応する。
 */
public interface PmxBoneHandler extends LoopHandler {

    /** ボーン定義抽出ループ。 */
    ParseStage BONE_LIST = new ParseStage();

    /** IKチェーンリスト抽出ループ。 */
    ParseStage IKCHAIN_LIST = new ParseStage();

    /**
     * ボーン名前情報の通知を受け取る. ボーン最初の要素.
     * <p>{@link #BONE_LIST}ループの構成要素.</p>
     * @param name ボーン名
     * @param nameE name of the bone
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxBoneInfo(String name, String nameE)
            throws MmdFormatException;

    /**
     * ボーン位置情報の通知を受け取る.
     * <p>{@link #BONE_LIST}ループの構成要素.</p>
     * @param xPos X座標
     * @param yPos Y座標
     * @param zPos Z座標
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxBonePosition(float xPos, float yPos, float zPos)
            throws MmdFormatException;

    /**
     * ボーン構造情報の通知を受け取る.
     * <p>{@link #BONE_LIST}ループの構成要素.</p>
     * @param parentId 親(前)ボーンID. 無い場合は-1.
     * @param depth 変形深さ
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxBoneStructure(int parentId, int depth)
            throws MmdFormatException;

    /**
     * ボーン種類の情報の通知を受け取る. 
     * <p>{@link #BONE_LIST}ループの構成要素.</p>
     * @param flags フラグが収められてる16 bit. {@link jp.sfjp.mikutoga.pmx.BoneFlags BoneFlags} 参照.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    public void pmxBoneFlags(short flags)
            throws MmdFormatException;

    /**
     * ボーンの矢印先座標の情報の通知を受け取る. 
     * <p>{@link #BONE_LIST}ループの構成要素. 無い場合は呼ばれない.</p>
     * @param offX 矢先相対X
     * @param offY 矢先相対Y
     * @param offZ 矢先相対Z
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    public void pmxBoneOffset(float offX, float offY, float offZ)
            throws MmdFormatException;
    
    /**
     * ボーン矢印接続情報の通知を受け取る.
     * <p>{@link #BONE_LIST}ループの構成要素. 無い場合は呼ばれない.</p>
     * @param arrowId 矢先(PMDでは子)ボーンID. 末端の場合は-1.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    public void pmxBoneArrowhead(int arrowId)
            throws MmdFormatException;

    /**
     * ボーン連動情報の通知を受け取る.
     * <p>{@link #BONE_LIST}ループの構成要素. 無い場合は呼ばれない.
     * 移動連動か回転連動可は{@link #pmxBoneFlags(short)}で確認する.</p>
     * @param linkParent 連動親ボーンID. 無い場合は-1.
     * @param ratio 連動比. 負を指定することも可能.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    public void pmxBoneLink(int linkParent, float ratio)
            throws MmdFormatException;

    /**
     * ボーン制限回転軸情報の通知を受け取る.
     * <p>{@link #BONE_LIST}ループの構成要素. 無い場合は呼ばれない.</p>
     * @param x 軸の絶対x軸座標.
     * @param y 軸の絶対y軸座標.
     * @param z 軸の絶対z軸座標.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    public void pmxBoneRotateAxe(float x, float y, float z)
            throws MmdFormatException;

    /**
     * ボーンローカル座標情報の通知を受け取る.
     * <p>{@link #BONE_LIST}ループの構成要素. 無い場合は呼ばれない.</p>
     * @param xx ローカルx軸の絶対x成分.
     * @param xy ローカルx軸の絶対y成分.
     * @param xz ローカルx軸の絶対z成分.
     * @param zx ローカルz軸の絶対x成分.
     * @param zy ローカルz軸の絶対y成分.
     * @param zz ローカルz軸の絶対z成分.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    public void pmxBoneLocalAxis(float xx, float xy, float xz, float zx, float zy, float zz)
            throws MmdFormatException;

    /**
     * ボーン外部親情報の通知を受け取る.
     * <p>{@link #BONE_LIST}ループの構成要素. 無い場合は呼ばれない.</p>
     * @param extraParent 外部親のID.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    public void pmxBoneExtraParent(int extraParent)
            throws MmdFormatException;

    /**
     * IKボーン情報の通知を受け取る.
     * <p>{@link #BONE_LIST}ループの構成要素.</p>
     * @param targetId IKボーンが接近しようとするIK接続先ボーンID
     * @param depth 再帰演算の深さ
     * @param weight 制限角度強度
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxBoneIKInfo(int targetId, int depth, float weight)
            throws MmdFormatException;

    /**
     * IKチェイン要素の通知を受け取る.
     * <p>{@link #BONE_LIST}ループの下位{@link #IKCHAIN_LIST}ループの構成要素.</p>
     * @param childId IK影響下ボーンID
     * @param limit_rotation 回転角制御 minx, miny, minz, maxx, maxy, maxz.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    public void pmxIKChainInfo(int childId, float[] limit_rotation)
            throws MmdFormatException;
}
