/*
 * PMX morph information handler
 *
 * License : The MIT License
 * Copyright(c) 2010 MikuToga Partners
 */

package jp.sfjp.mikutoga.pmx.parser;

import jp.sfjp.mikutoga.bin.parser.LoopHandler;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.bin.parser.ParseStage;

/**
 * PMXモデルのモーフ情報の通知用ハンドラ。
 */
public interface PmxMorphHandler extends LoopHandler {

    /** モーフ抽出ループ。 */
    ParseStage MORPH_LIST = new ParseStage();

    /** モーフ要素抽出ループ。 */
    ParseStage MORPH_ELEMENT_LIST = new ParseStage();


    /**
     * モーフ情報の通知を受け取る.
     * <p>{@link #MORPH_LIST}ループの構成要素</p>
     * @param morphName モーフ名.
     * @param morphNameE name of the morph.
     * @param panelType モーフ操作パネルをどこに表示するか.
     * <ul>
     * <li>0:システム予約</li>
     * <li>1:まゆ</li>
     * <li>2:目</li>
     * <li>3:リップ</li>
     * <li>4:その他</li>
     * </ul>
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMorphInfo(String morphName, String morphNameE, byte panelType)
            throws MmdFormatException;

    /**
     * モーフ変形量の種類・要素数を受け取る.
     * <p>{@link #MORPH_LIST}ループの構成要素.</p>
     * @param morphType モーフの種類.<ul>
     * <li>0:グループ</li>
     * <li>1:頂点</li>
     * <li>2:ボーン</li>
     * <li>3:UV</li>
     * <li>4:追加UV1</li>
     * <li>5:追加UV2</li>
     * <li>6:追加UV3</li>
     * <li>7:追加UV4</li>
     * <li>8:材質</li>
     * </ul>
     * @param numOfOffsets オフセット要素の数.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMorphElementListHeader(byte morphType, int numOfOffsets)
            throws MmdFormatException;

    /**
     * 頂点モーフ変形の通知を受け取る.
     * <p>{@link #MORPH_LIST}ループの下位{@link #MORPHELEMENT_LIST}の構成要素.</p>
     * @param vertexId 頂点ID
     * @param xPos X軸変位
     * @param yPos Y軸変位
     * @param zPos Z軸変位
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMorphVertexInfo(int vertexId,
                               float xPos, float yPos, float zPos)
            throws MmdFormatException;

    /**
     * UVモーフ変形の通知を受け取る.
     * <p>{@link #MORPH_LIST}ループの下位{@link #MORPHELEMENT_LIST}の構成要素.</p>
     * @param vertexId 頂点ID
     * @param uPos U軸変位
     * @param vPos V軸変位
     * @param zPos Z軸変位. (無駄だが記録してある.)
     * @param wPos w軸変位. (無駄だが記録してある.)
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMorphUVInfo(int vertexId,
                               float uPos, float vPos, float zPos, float wPos)
            throws MmdFormatException;

    /**
     * ボーンモーフ変形の通知を受け取る.
     * <p>{@link #MORPH_LIST}ループの下位{@link #MORPHELEMENT_LIST}の構成要素.</p>
     * @param vertexId ボーンID
     * @param xPos X軸変位.
     * @param yPos Y軸変位.
     * @param zPos Z軸変位.
     * @param qx クオータニオンx軸.
     * @param qy クオータニオンy軸.
     * @param qz クオータニオンz軸.
     * @param qw クオータニオンw軸.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMorphBoneInfo(int vertexId,
            float xPos, float yPos, float zPos,
            float qx, float qy, float qz, float qw)
            throws MmdFormatException;

    /**
     * 材質モーフ変形の通知を受け取る.
     * <p>{@link #MORPH_LIST}ループの下位{@link #MORPHELEMENT_LIST}の構成要素.</p>
     * @param materialId 材質ID. -1:全材質対象.
     * @param type 0:乗算式か, 1:加算式か.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMorphMaterialInfo(int materialId, byte type)
            throws MmdFormatException;

    /**
     * 材質モーフ変形の拡散光成分の通知を受け取る.
     * <p>{@link #MORPH_LIST}ループの下位{@link #MORPHELEMENT_LIST}の構成要素.</p>
     * @param red 0.0～1.0の範囲の赤成分
     * @param green 0.0～1.0の範囲の緑成分
     * @param blue 0.0～1.0の範囲の青成分
     * @param alpha 0.0(透明)～1.0(不透明)のアルファ値。
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMorphMaterialDiffuse(float red, float green, float blue,
                               float alpha)
            throws MmdFormatException;

    /**
     * 材質モーフ変形の反射光成分の通知を受け取る.
     * <p>{@link #MORPH_LIST}ループの下位{@link #MORPHELEMENT_LIST}の構成要素.</p>
     * @param red 0.0～1.0の範囲の赤成分
     * @param green 0.0～1.0の範囲の緑成分
     * @param blue 0.0～1.0の範囲の青成分
     * @param shininess 光沢強度(1～15ぐらい)
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMorphMaterialSpecular(float red, float green, float blue,
                                float shininess)
            throws MmdFormatException;

    /**
     * 材質モーフ変形の環境色成分の通知を受け取る.
     * <p>{@link #MORPH_LIST}ループの下位{@link #MORPHELEMENT_LIST}の構成要素.</p>
     * @param red 0.0～1.0の範囲の赤成分
     * @param green 0.0～1.0の範囲の緑成分
     * @param blue 0.0～1.0の範囲の青成分
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMorphMaterialAmbient(float red, float green, float blue)
            throws MmdFormatException;

    /**
     * 材質モーフ変形のエッジ成分の通知を受け取る.
     * <p>{@link #MORPH_LIST}ループの下位{@link #MORPHELEMENT_LIST}の構成要素.</p>
     * @param red 0.0～1.0の範囲の赤成分
     * @param green 0.0～1.0の範囲の緑成分
     * @param blue 0.0～1.0の範囲の青成分
     * @param alpha 0.0(透明)～1.0(不透明)のアルファ値。
     * @param thick 0.0～2.0くらいの厚さ
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    public void pmxMorphMaterialEdges(float red, float green, float blue,
            float alpha, float thick)
            throws MmdFormatException;

    /**
     * 材質モーフ変形のテキスチャーフィルター成分の通知を受け取る.
     * <p>{@link #MORPH_LIST}ループの下位{@link #MORPHELEMENT_LIST}の構成要素.</p>
     * @param red 0.0～1.0の範囲の赤成分
     * @param green 0.0～1.0の範囲の緑成分
     * @param blue 0.0～1.0の範囲の青成分
     * @param alpha 0.0(透明)～1.0(不透明)のアルファ値。
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMorphMaterialTexture(float red, float green, float blue, float alpha)
            throws MmdFormatException;

    /**
     * 材質モーフ変形のスフィアマップフィルター成分の通知を受け取る.
     * <p>{@link #MORPH_LIST}ループの下位{@link #MORPHELEMENT_LIST}の構成要素.</p>
     * @param red 0.0～1.0の範囲の赤成分
     * @param green 0.0～1.0の範囲の緑成分
     * @param blue 0.0～1.0の範囲の青成分
     * @param alpha 0.0(透明)～1.0(不透明)のアルファ値。
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMorphMaterialSphere(float red, float green, float blue, float alpha)
            throws MmdFormatException;

    /**
     * 材質モーフ変形のトゥーンフィルター成分の通知を受け取る.
     * <p>{@link #MORPH_LIST}ループの下位{@link #MORPHELEMENT_LIST}の構成要素.</p>
     * @param red 0.0～1.0の範囲の赤成分
     * @param green 0.0～1.0の範囲の緑成分
     * @param blue 0.0～1.0の範囲の青成分
     * @param alpha 0.0(透明)～1.0(不透明)のアルファ値。
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMorphMaterialToon(float red, float green, float blue, float alpha)
            throws MmdFormatException;

    /**
     * グループモーフ変形の通知を受け取る.
     * <p>{@link #MORPH_LIST}ループの下位{@link #MORPHELEMENT_LIST}の構成要素.</p>
     * @param morphId 子モーフのID.
     * @param ratio 0.0～1.0の範囲の比率.
     * 仕様書には乗算と書かれているが、実際は加算なんじゃないかな.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMorphGroupInfo(int morphId,float ratio)
            throws MmdFormatException;
}
