/*
 * pmx file parser
 *
 * License : The MIT License
 * Copyright(c) 2015 MikuToga Partners
 */

package jp.sfjp.mikutoga.pmx.parser;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Arrays;
import jp.sfjp.mikutoga.bin.parser.CommonParser;
import jp.sfjp.mikutoga.bin.parser.MmdEofException;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.bin.parser.TextDecoder;
import jp.sfjp.mikutoga.pmx.BoneFlags;

/**
 * PMXモデルファイルのパーサ基本部. PMX version 2.0 相当.
 */
public class PmxParserBase extends CommonParser {

    /**
     * PMXで用いられる文字エンコーディングス.
     * 0:UTF-16, 1:UTF-8 .
     */
    public static final Charset [] encodes = new Charset[] {
        Charset.forName("UTF-16LE"),
        Charset.forName("UTF-8")
    };

    /** 改行文字列 CR。 */
    protected static final String CR = "\r";       // 0x0d
    /** 改行文字列 LF。 */
    protected static final String LF = "\n";       // 0x0a
    /** 改行文字列 CRLF。 */
    protected static final String CRLF = CR + LF;  // 0x0d, 0x0a

//    private static final int HEADER_LENGTH = 4;
    private static final byte[] MAGIC_BYTES = {
        (byte)0x50, (byte)0x4d, (byte)0x58, (byte)0x20 // "PMX "
    };

    static{
    }

    private PmxBasicHandler basicHandler =       PmxUnifiedHandler.EMPTY;
    private PmxVertexHandler vertexHandler =     PmxUnifiedHandler.EMPTY;
    private PmxFaceHandler faceHandler =         PmxUnifiedHandler.EMPTY;
    private PmxMaterialHandler materialHandler = PmxUnifiedHandler.EMPTY;
    private PmxBoneHandler boneHandler =         PmxUnifiedHandler.EMPTY;
    private PmxMorphHandler morphHandler =       PmxUnifiedHandler.EMPTY;
/*    private PmxMenuHandler menuHandler =         PmxUnifiedHandler.EMPTY;
    private PmxRigidHandler rigidHandler =       PmxUnifiedHandler.EMPTY;
    private PmxJointHandler jointHandler =       PmxUnifiedHandler.EMPTY;*/

    /**
     * kind of encode
     */
    private byte encode;
    private byte num_uv;
    private byte size_vertex;
    private byte size_texture;
    private byte size_material;
    private byte size_bone;
    private byte size_morph;
    private byte size_rigid;

    private TextDecoder decoder;
    private int vertexCount      = -1;
    private int faceCount      = -1;
    private int textureCount      = -1;
    private int materialCount      = -1;
    private int boneCount      = -1;
    private int morphCount     = -1;
    private int boneGroupCount = -1;
    private int rigidCount     = -1;


    /**
     * コンストラクタ。
     * @param source 入力ソース
     */
    public PmxParserBase(InputStream source){
        super(source);
    }


    /**
     * 文字列の最後がLF(0x0a)の場合削除する。
     * <p>ボーングループ名対策。
     * @param name 文字列
     * @return 末尾LFが削除された文字列
     */
    public static String chopLastLF(String name){
        String result;

        if(name.endsWith(LF)){
            result = name.substring(0, name.length() - 1);
        }else{
            result = name;
        }

        return result;
    }

    /**
     * 基本情報通知ハンドラを登録する。
     * @param handler ハンドラ
     */
    public void setBasicHandler(PmxBasicHandler handler){
        if(handler == null){
            this.basicHandler = PmxUnifiedHandler.EMPTY;
        }else{
            this.basicHandler = handler;
        }
    }

    /**
     * 頂点ハンドラを登録する。
     * @param handler ハンドラ
     */
    public void setVertexHandler(PmxVertexHandler handler){
        if(handler == null){
            this.vertexHandler = PmxUnifiedHandler.EMPTY;
        }else{
            this.vertexHandler = handler;
        }
    }

    /**
     * 面ハンドラを登録する。
     * @param handler ハンドラ
     */
    public void setFaceHandler(PmxFaceHandler handler){
        if(handler == null){
            this.faceHandler = PmxUnifiedHandler.EMPTY;
        }else{
            this.faceHandler = handler;
        }
    }

    /**
     * 材質情報通知ハンドラを登録する。
     * @param handler ハンドラ
     */
    public void setMaterialHandler(PmxMaterialHandler handler){
        if(handler == null){
            this.materialHandler = PmxUnifiedHandler.EMPTY;
        }else{
            this.materialHandler = handler;
        }
    }

    /**
     * ボーン情報通知ハンドラを登録する。
     * @param handler ハンドラ
     */
    public void setBoneHandler(PmxBoneHandler handler){
        if(handler == null){
            this.boneHandler = PmxUnifiedHandler.EMPTY;
        }else{
            this.boneHandler = handler;
        }
    }

    /**
     * モーフ情報通知ハンドラを登録する。
     * @param handler ハンドラ
     */
    public void setMorphHandler(PmxMorphHandler handler){
        if(handler == null){
            this.morphHandler = PmxUnifiedHandler.EMPTY;
        }else{
            this.morphHandler = handler;
        }
        return;
    }

    /**
     * パースによって得られたボーン数を返す。
     * @return ボーン数
     */
    protected int getBoneCount(){
        return this.boneCount;
    }

    /**
     * パースによって得られたモーフ数を返す。
     * @return モーフ数
     */
    protected int getMorphCount(){
        return this.morphCount;
    }

    /**
     * パースによって得られたボーングループ数を返す。
     * @return ボーングループ数
     */
    protected int getBoneGroupCount(){
        return this.boneGroupCount;
    }

    /**
     * バイト長指定の可変長文字列を読み込む.
     * 入力バイト列はUTF-16,UTF-8エンコーディングとして解釈される.
     * IO入力は指定バイト数だけ読み進められる.
     * @return デコードされた文字列
     * @throws IOException IOエラー
     * @throws MmdEofException 読み込む途中でストリーム終端に達した.
     * @throws MmdFormatException 不正な文字エンコーディングが検出された.
     */
    protected String parsePmxText()
            throws IOException,
                   MmdEofException,
                   MmdFormatException {
        int byteLen=parseLeInt();
//        System.err.println("position:"+getPosition()+" text length:"+byteLen);
        String result = parseString(this.decoder, byteLen);
        return result;
    }

    /**
     * 可変長IDを読み込む.
     * <p>{@link #size_bone} 指定長のID番号を読み込む.
     * @param size 長さ byte.
     * @return デコードされた番号
     * @throws IOException IOエラー
     * @throws MmdEofException 読み込む途中でストリーム終端に達した。
     */
    protected int parsePmxId(byte size)
            throws IOException,
                   MmdEofException {
        int id=-1;
        switch(size){
            case 1:
                id=parseByte();
                break;
                
            case 2:
                id=parseLeShort();
                break;

            case 4:
                id=parseLeInt();
                break;
        }
        return id;
    }

    /**
     * 可変長ID(符号なし)を読み込む.
     * <p>{@link #size_bone} 指定長のID番号を読み込む.
     * @param size 長さ byte.
     * @return デコードされた番号
     * @throws IOException IOエラー
     * @throws MmdEofException 読み込む途中でストリーム終端に達した。
     */
    protected int parsePmxUId(byte size)
            throws IOException,
                   MmdEofException {
        int id=-1;
        switch(size){
            case 1:
                id=parseUByteAsInt();
                break;
                
            case 2:
                id=parseLeUShortAsInt();
                break;

            case 4:
                id=parseLeInt();
                break;
        }
        return id;
    }
    
    /**
     * PMXファイルのパースを開始する。
     * @throws IOException IOエラー
     * @throws MmdFormatException フォーマットエラー
     */
    public void parsePmx()
            throws IOException, MmdFormatException {
        this.basicHandler.pmxParseStart();

        parseBody();

        boolean hasMoreData = hasMore();
        this.basicHandler.pmxParseEnd(hasMoreData);

    }

    /**
     * PMXファイル本体のパースを開始する。
     * パーサを拡張する場合はこのメソッドをオーバーライドする。
     * @throws IOException IOエラー
     * @throws MmdFormatException フォーマットエラー
     */
    protected void parseBody() throws IOException, MmdFormatException{
        parsePmxHeader();

        parseVertexList();
        parseFaceList();
        parseMaterialList();
        parseBoneList();
//        parseIKList();
        parseMorphList();
/*        parseMorphOrderList();
        parseBoneGroupList();
        parseGroupedBoneList();*/
    }

    /**
     * PMXファイルヘッダ部のパースと通知。
     * @throws IOException IOエラー
     * @throws MmdFormatException フォーマットエラー
     */
    private void parsePmxHeader() throws IOException, MmdFormatException{
        byte[] header = new byte[MAGIC_BYTES.length];
        parseByteArray(header);

        if( ! Arrays.equals(header, MAGIC_BYTES) ){
            throw new MmdFormatException("unknown PMX-header type");
        }
        float version=parseLeFloat();
        int n=parseByte();
        byte [] barray=new byte[n];
        parseByteArray(barray);
        encode=barray[0];
        decoder = new TextDecoder(encodes[encode]);
        num_uv=barray[1];
        this.basicHandler.pmxHeaderInfo(version,encode,num_uv);
        
        size_vertex=barray[2];
        size_texture=barray[3];
        size_material=barray[4];
        size_bone=barray[5];
        size_morph=barray[6];
        size_rigid=barray[7];
        
        String modelName = parsePmxText();
        String modelNameE = parsePmxText();
        this.basicHandler.pmxModelName(modelName, modelNameE);

        String description = parsePmxText();
        String descriptionE = parsePmxText();
        description = description.replace(CRLF, LF);
        descriptionE = descriptionE.replace(CRLF, LF);
        this.basicHandler.pmxModelDescription(description, descriptionE);
    }

    /**
     * 頂点情報のパースと通知。
     * @throws IOException IOエラー
     * @throws MmdFormatException フォーマットエラー
     */
    private void parseVertexList() throws IOException, MmdFormatException{
        int vertexNum = parseLeInt();

        this.vertexHandler.loopStart(PmxVertexHandler.VERTEX_LIST,vertexNum);

        for(int ct = 0; ct < vertexNum; ct++){
            float xPos = parseLeFloat();
            float yPos = parseLeFloat();
            float zPos = parseLeFloat();
            this.vertexHandler.pmxVertexPosition(xPos, yPos, zPos);

            float xVec = parseLeFloat();
            float yVec = parseLeFloat();
            float zVec = parseLeFloat();
            this.vertexHandler.pmxVertexNormal(xVec, yVec, zVec);

            float uVal = parseLeFloat();
            float vVal = parseLeFloat();
            this.vertexHandler.pmxVertexUV(uVal, vVal);

            for(byte i=0;i<num_uv;i++){
                float x=parseLeFloat();
                float y=parseLeFloat();
                float z=parseLeFloat();
                float w=parseLeFloat();
                this.vertexHandler.pmxVertexExUV(i,x,y,z,w);
            }
 
            byte bdef=parseByte();
            int []boneIds;
            float []weights;
            if(bdef<3){
                boneIds=new int[4>>(2-bdef)];
            }else if(bdef==3) {
                boneIds=new int[2];
            }else {
                boneIds=new int[2];
            }
            weights=new float[boneIds.length];

            for(byte i=0;i<boneIds.length;i++){
                boneIds[i]=parsePmxId(this.size_bone);
            }
            if(bdef==0){
                weights[0]=1.0f;
            } else if(bdef==2){
                for(byte i=0;i<4;i++){
                    weights[i]=parseLeFloat();
                }
            } else {
                weights[0]=parseLeFloat();
                weights[1]=1.0f-weights[0];
            }

            float [] sdef=null;
            if(bdef==3){
                sdef=new float[9];
                for(int i=0;i<9;i++){
                    sdef[i]=parseLeFloat();
                }
            }
            this.vertexHandler.pmxVertexWeight(boneIds, weights, sdef);

            float edge = parseLeFloat();
            this.vertexHandler.pmxVertexEdge(edge);

            this.vertexHandler.loopNext(PmxVertexHandler.VERTEX_LIST);
        }
        this.vertexHandler.loopEnd(PmxVertexHandler.VERTEX_LIST);
    }

    /**
     * 面情報のパースと通知.
     * @throws IOException IOエラー
     * @throws MmdFormatException フォーマットエラー
     */
    private void parseFaceList() throws IOException, MmdFormatException{
        int faceNum = parseLeInt();
        this.faceCount=faceNum/3;
        if(this.faceCount*3 - faceNum != 0) throw new MmdFormatException();
        
        this.faceHandler.loopStart(PmxFaceHandler.FACE_LIST, this.faceCount);

        for(int ct = 0; ct < this.faceCount; ct++){
            int vertexId1 = parsePmxUId(size_vertex);
            int vertexId2 = parsePmxUId(size_vertex);
            int vertexId3 = parsePmxUId(size_vertex);
            this.faceHandler.pmxFaceTriangle(vertexId1,
                    vertexId2,
                    vertexId3);
            this.faceHandler.loopNext(PmxFaceHandler.FACE_LIST);
        }
        this.faceHandler.loopEnd(PmxFaceHandler.FACE_LIST);
    }

    /**
     * 材質情報のパースと通知。
     * @throws IOException IOエラー
     * @throws MmdFormatException フォーマットエラー
     */
    private void parseMaterialList() throws IOException, MmdFormatException{
        int textureNum = parseLeInt();
        this.materialHandler.loopStart(PmxMaterialHandler.TEXTURE_LIST,
                                       textureNum );
        
        for(int ct = 0; ct < textureNum; ct++){
            String textureFile = parsePmxText();
            this.materialHandler.pmxTexture(textureFile);
        }
        this.materialHandler.loopEnd(PmxMaterialHandler.TEXTURE_LIST);
        
        int materialNum = parseLeInt();
        this.materialHandler.loopStart(PmxMaterialHandler.MATERIAL_LIST,
                                       materialNum );

        for(int ct = 0; ct < materialNum; ct++){
            String materialName = parsePmxText();
            String materialNameE = parsePmxText();
            this.materialHandler.pmxMaterialNames(materialName,materialNameE);        

            parseColor();
            byte flags=parseByte();
            float red=parseLeFloat();
            float green=parseLeFloat();
            float blue=parseLeFloat();
            float alpha=parseLeFloat();
            float thick=parseLeFloat();
            this.materialHandler.pmxMaterialEdges(flags,red,green,blue,alpha,thick);

            int texture = parsePmxId(this.size_texture);
            int sphere = parsePmxId(this.size_texture);
            byte mode=parseByte();
            this.materialHandler.pmxMaterialTexture(texture, sphere, mode);

            byte sharedToon = parseByte();
            int toon;
            if(sharedToon == 0){
                toon = parsePmxId(this.size_texture);
            }else {
                toon = parseByte();
            }
            this.materialHandler.pmxMaterialToon(sharedToon, toon);

            String script = parsePmxText();
            int faceCount = parseLeInt();
            this.materialHandler.pmxMaterialInfo(script, faceCount);

            this.materialHandler.loopNext(PmxMaterialHandler.MATERIAL_LIST);
        }

        this.materialHandler.loopEnd(PmxMaterialHandler.MATERIAL_LIST);
    }

    /**
     * 色情報のパースと通知。
     * @throws IOException IOエラー
     * @throws MmdFormatException フォーマットエラー
     */
    private void parseColor() throws IOException, MmdFormatException{
        float red;
        float green;
        float blue;

        red   = parseLeFloat();
        green = parseLeFloat();
        blue  = parseLeFloat();
        float alpha = parseLeFloat();

        this.materialHandler.pmxMaterialDiffuse(red, green, blue, alpha);

        red   = parseLeFloat();
        green = parseLeFloat();
        blue  = parseLeFloat();
        float shininess = parseLeFloat();

        this.materialHandler.pmxMaterialSpecular(red, green, blue,
                                                 shininess);

        red   = parseLeFloat();
        green = parseLeFloat();
        blue  = parseLeFloat();

        this.materialHandler.pmxMaterialAmbient(red, green, blue);
    }

    /**
     * ボーン情報のパースと通知。
     * @throws IOException IOエラー
     * @throws MmdFormatException フォーマットエラー
     */
    private void parseBoneList() throws IOException, MmdFormatException{
        this.boneCount = parseLeInt();
        this.boneHandler.loopStart(PmxBoneHandler.BONE_LIST, this.boneCount);

        for(int ct = 0; ct < this.boneCount; ct++){
            String boneName = parsePmxText();
            String boneNameE = parsePmxText();
            this.boneHandler.pmxBoneInfo(boneName, boneNameE);

            float xPos = parseLeFloat();
            float yPos = parseLeFloat();
            float zPos = parseLeFloat();

            this.boneHandler.pmxBonePosition(xPos, yPos, zPos);

            int parentId  = parsePmxId(this.size_bone);
            int depth = parseLeInt();
            this.boneHandler.pmxBoneStructure(parentId,depth);

            short flags = parseLeShort();
            this.boneHandler.pmxBoneFlags(flags);

            if(!BoneFlags.OFFSET.check(flags)){
                float offX=parseLeFloat();
                float offY=parseLeFloat();
                float offZ=parseLeFloat();
                this.boneHandler.pmxBoneOffset(offX,offY,offZ);
            }else{
                int arrowId    = parsePmxId(this.size_bone);
                this.boneHandler.pmxBoneArrowhead(arrowId);
            }

            if(BoneFlags.ROTATE_LINK.check(flags)
                    || BoneFlags.MOVE_LINK.check(flags)){
                int linkParent = parsePmxId(this.size_bone);
                float r = parseLeFloat();
                this.boneHandler.pmxBoneLink(linkParent,r);
            }

            if(BoneFlags.AXIS_ROTATE.check(flags)){
                float x=parseLeFloat();
                float y=parseLeFloat();
                float z=parseLeFloat();
                this.boneHandler.pmxBoneRotateAxe(x,y,z);
            }

            if(BoneFlags.LOCAL_AXIS.check(flags)){
                float xx=parseLeFloat();
                float xy=parseLeFloat();
                float xz=parseLeFloat();
                float zx=parseLeFloat();
                float zy=parseLeFloat();
                float zz=parseLeFloat();
                this.boneHandler.pmxBoneLocalAxis(xx,xy,xz,zx,zy,zz);
            }

            if(BoneFlags.EXTRA.check(flags)){
                int extraParent = parseLeInt();
                this.boneHandler.pmxBoneExtraParent(extraParent);
            }

            if(BoneFlags.IK.check(flags)){
                int targetId=parsePmxId(this.size_bone);
                int trial=parseLeInt();
                float radian=parseLeFloat();
                this.boneHandler.pmxBoneIKInfo(targetId,trial,radian);
                int bones=parseLeInt();
                this.boneHandler.loopStart(PmxBoneHandler.IKCHAIN_LIST,bones);
                for(int i=0;i<bones;i++){
                    int ikElement=parsePmxId(this.size_bone);
                    byte limit=parseByte();
                    float [] limit_rotation=null;
                    if(limit>0){
                        limit_rotation=new float[6];
                        limit_rotation[0]=parseLeFloat();
                        limit_rotation[1]=parseLeFloat();
                        limit_rotation[2]=parseLeFloat();
                        limit_rotation[3]=parseLeFloat();
                        limit_rotation[4]=parseLeFloat();
                        limit_rotation[5]=parseLeFloat();
                    }
                    this.boneHandler.pmxIKChainInfo(ikElement,limit_rotation);
                    this.boneHandler.loopNext(PmxBoneHandler.IKCHAIN_LIST);
                }
                this.boneHandler.loopEnd(PmxBoneHandler.IKCHAIN_LIST);
            }

            this.boneHandler.loopNext(PmxBoneHandler.BONE_LIST);
        }

        this.boneHandler.loopEnd(PmxBoneHandler.BONE_LIST);
    }

    /**
     * モーフ情報のパースと通知。
     * @throws IOException IOエラー
     * @throws MmdFormatException フォーマットエラー
     */
    private void parseMorphList() throws IOException, MmdFormatException{
        this.morphCount = parseLeInt();
        this.morphHandler.loopStart(PmxMorphHandler.MORPH_LIST,
                                    this.morphCount);

        for(int ct = 0; ct < this.morphCount; ct++){
            String name = parsePmxText();
            String nameE = parsePmxText();
            byte panelType  = parseByte();
            this.morphHandler.pmxMorphInfo(name, nameE, panelType);

            byte morphType  = parseByte();
            int numOfOffsets = parseLeInt();
            this.morphHandler.pmxMorphElementListHeader(morphType, numOfOffsets);

            parseMorphElementList(morphType,numOfOffsets);

            this.morphHandler.loopNext(PmxMorphHandler.MORPH_LIST);
        }

        this.morphHandler.loopEnd(PmxMorphHandler.MORPH_LIST);

    }

    /**
     * モーフ変形要素リストのパースと通知。
     * @param elementCount 要素数
     * @throws IOException IOエラー
     * @throws MmdFormatException フォーマットエラー
     */
    private void parseMorphElementList(byte morphType, int elementCount)
            throws IOException, MmdFormatException{
        this.morphHandler.loopStart(PmxMorphHandler.MORPH_ELEMENT_LIST,
                                    elementCount);

        int id;
        for(int ct = 0; ct < elementCount; ct++){
            if(morphType==0){// group
                id=parsePmxId(size_morph);
                float ratio=parseLeFloat();
                this.morphHandler.pmxMorphGroupInfo(id, ratio);
            } else if(morphType==2){ // bone
                id=parsePmxId(size_bone);
                float xPos = parseLeFloat();
                float yPos = parseLeFloat();
                float zPos = parseLeFloat();
                float qx = parseLeFloat();
                float qy = parseLeFloat();
                float qz = parseLeFloat();
                float qw = parseLeFloat();
                this.morphHandler.pmxMorphBoneInfo(id, xPos, yPos, zPos,
                        qx, qy, qz, qw);
            } else if(morphType<8){ // vertex, UV
                id=parsePmxUId(size_vertex);
                if(morphType==1){
                    float xPos = parseLeFloat();
                    float yPos = parseLeFloat();
                    float zPos = parseLeFloat();
                    this.morphHandler.pmxMorphVertexInfo(id, xPos, yPos, zPos);
                } else { // UV
                    float u = parseLeFloat();
                    float v = parseLeFloat();
                    float z = parseLeFloat();
                    float w = parseLeFloat();
                    this.morphHandler.pmxMorphUVInfo(id, u, v, z, w);
                }
            } else if(morphType==8){ // material
                id=parsePmxId(size_material);
                byte type=parseByte();
                this.morphHandler.pmxMorphMaterialInfo(id, type);

                float r = parseLeFloat();
                float g = parseLeFloat();
                float b = parseLeFloat();
                float a = parseLeFloat();
                this.morphHandler.pmxMorphMaterialDiffuse(r, g, b, a);

                r = parseLeFloat();
                g = parseLeFloat();
                b = parseLeFloat();
                a = parseLeFloat();
                this.morphHandler.pmxMorphMaterialSpecular(r, g, b, a);

                r = parseLeFloat();
                g = parseLeFloat();
                b = parseLeFloat();
                this.morphHandler.pmxMorphMaterialAmbient(r, g, b);

                r = parseLeFloat();
                g = parseLeFloat();
                b = parseLeFloat();
                a = parseLeFloat();
                float thickness = parseLeFloat();
                this.morphHandler.pmxMorphMaterialEdges(r, g, b, a, thickness);

                r = parseLeFloat();
                g = parseLeFloat();
                b = parseLeFloat();
                a = parseLeFloat();
                this.morphHandler.pmxMorphMaterialTexture(r, g, b, a);

                r = parseLeFloat();
                g = parseLeFloat();
                b = parseLeFloat();
                a = parseLeFloat();
                this.morphHandler.pmxMorphMaterialSphere(r, g, b, a);

                r = parseLeFloat();
                g = parseLeFloat();
                b = parseLeFloat();
                a = parseLeFloat();
                this.morphHandler.pmxMorphMaterialToon(r, g, b, a);
            }
            this.morphHandler.loopNext(PmxMorphHandler.MORPH_ELEMENT_LIST);
        }
        this.morphHandler.loopEnd(PmxMorphHandler.MORPH_ELEMENT_LIST);
    }

    /**
     * モーフGUI表示順のパースと通知。
     * @throws IOException IOエラー
     * @throws MmdFormatException フォーマットエラー
     */
/*    private void parseMorphOrderList()
            throws IOException, MmdFormatException{
        int morphOrderCount = parseUByteAsInt();

        this.morphHandler.loopStart(PmdMorphHandler.MORPHORDER_LIST,
                                    morphOrderCount );

        for(int ct = 0; ct < morphOrderCount; ct++){
            int morphId = parseLeUShortAsInt();
            this.morphHandler.pmdMorphOrderInfo(morphId);

            this.morphHandler.loopNext(PmdMorphHandler.MORPHORDER_LIST);
        }

        this.morphHandler.loopEnd(PmdMorphHandler.MORPHORDER_LIST);

        return;
    }*/

    /**
     * ボーングループ名のパースと通知。
     * @throws IOException IOエラー
     * @throws MmdFormatException フォーマットエラー
     */
/*    private void parseBoneGroupList()
            throws IOException, MmdFormatException{
        this.boneGroupCount = parseUByteAsInt();

        this.boneHandler.loopStart(PmdBoneHandler.BONEGROUP_LIST,
                                   this.boneGroupCount);

        for(int ct = 0; ct < this.boneGroupCount; ct++){
            String groupName =
                    parsePmdText(PmdConst.MAXBYTES_BONEGROUPNAME);
            groupName = chopLastLF(groupName);
            this.boneHandler.pmdBoneGroupInfo(groupName);

            this.boneHandler.loopNext(PmdBoneHandler.BONEGROUP_LIST);
        }

        this.boneHandler.loopEnd(PmdBoneHandler.BONEGROUP_LIST);

    }*/

    /**
     * ボーングループ内訳のパースと通知。
     * @throws IOException IOエラー
     * @throws MmdFormatException フォーマットエラー
     */
/*    private void parseGroupedBoneList()
            throws IOException, MmdFormatException{
        int groupedBoneCount = parseLeInt();

        this.boneHandler.loopStart(PmdBoneHandler.GROUPEDBONE_LIST,
                                   groupedBoneCount);

        for(int ct = 0; ct < groupedBoneCount; ct++){
            int boneId  = parseLeUShortAsInt();
            int groupId = parseUByteAsInt();
            this.boneHandler.pmdGroupedBoneInfo(boneId, groupId);

            this.boneHandler.loopNext(PmdBoneHandler.GROUPEDBONE_LIST);
        }

        this.boneHandler.loopEnd(PmdBoneHandler.GROUPEDBONE_LIST);

    }*/

}
