package jp.sourceforge.masasa.architecture.framework.validation;

import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.masasa.architecture.framework.validation.annotation.CheckCreditCard;
import junit.framework.TestCase;

import org.apache.commons.validator.CreditCardValidator;
import org.junit.Test;

/**
 * @see org.apache.commons.validator.CreditCardValidatorTest
 */
public class CheckCreditCardTest extends TestCase {

	private static final String VALID_VISA = "4417123456789113";
	private static final String VALID_SHORT_VISA = "4222222222222";
	private static final String VALID_AMEX = "378282246310005";
	private static final String VALID_MASTERCARD = "5105105105105100";
	private static final String VALID_DISCOVER = "6011000990139424";
	private static final String VALID_DINERS = "30569309025904";

	@Test
	public void testIsValid1a() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data());
		AbstractValidator<ITestData> validator = new Validator<ITestData>();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	@Test
	public void testIsValid1b() {
		List<ITestData> list = new ArrayList<ITestData>();
		for (int i = 0; i < 100; i++) {
			list.add(new DataVerifyWhenNull());
		}
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	@Test
	public void testIsValid2a() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data(""));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	@Test
	public void testIsValid2b() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new DataVerifyWhenNull(""));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	@Test
	public void testIsValid3() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("123456789012")); // too short
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	@Test
	public void testIsValid4() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("12345678901234567890")); // too long
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	@Test
	public void testIsValid5() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("4417123456789112"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	@Test
	public void testIsValid6() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("4417q23456w89113"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	@Test
	public void testIsValid7() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data(VALID_VISA));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	@Test
	public void testIsValid8() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data(VALID_SHORT_VISA));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	@Test
	public void testIsValid9() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data(VALID_AMEX));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	@Test
	public void testIsValid10() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data(VALID_MASTERCARD));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	@Test
	public void testIsValid11() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data(VALID_DISCOVER));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	@Test
	public void testIsValid12() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new DataAMEX("4417123456789113"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	@Test
	public void testAddAllowedCardType1() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new DataNONE(VALID_VISA));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	@Test
	public void testAddAllowedCardType2() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new DataNONE(VALID_AMEX));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	@Test
	public void testAddAllowedCardType3() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new DataNONE(VALID_MASTERCARD));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	@Test
	public void testAddAllowedCardType4() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new DataNONE(VALID_DISCOVER));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	@Test
	public void testAddAllowedCardType5() {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new DataNONE(VALID_DINERS));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	public class Data implements ITestData {
		private static final long serialVersionUID = 4559414533664290053L;
		@CheckCreditCard
		private String data;

		public Data() {
		}

		public Data(String data) {
			this.data = data;
		}

		public String getData() {
			return data;
		}

		public void setData(String data) {
			this.data = data;
		}
	}

	public class DataVerifyWhenNull implements ITestData {
		private static final long serialVersionUID = -5563397325639451664L;
		@CheckCreditCard(verifyWhenNull = true)
		private String data;

		public DataVerifyWhenNull() {
		}

		public DataVerifyWhenNull(String data) {
			this.data = data;
		}

		public String getData() {
			return data;
		}

		public void setData(String data) {
			this.data = data;
		}
	}

	public class DataAMEX implements ITestData {
		private static final long serialVersionUID = 2675508040515439365L;
		@CheckCreditCard(options = CreditCardValidator.AMEX)
		private String data;

		public DataAMEX() {
		}

		public DataAMEX(String data) {
			this.data = data;
		}

		public String getData() {
			return data;
		}

		public void setData(String data) {
			this.data = data;
		}
	}

	public class DataAMEXVerifyWhenNull implements ITestData {
		private static final long serialVersionUID = 3669783052293836663L;
		@CheckCreditCard(verifyWhenNull = true, options = CreditCardValidator.AMEX)
		private String data;

		public DataAMEXVerifyWhenNull() {
		}

		public DataAMEXVerifyWhenNull(String data) {
			this.data = data;
		}

		public String getData() {
			return data;
		}

		public void setData(String data) {
			this.data = data;
		}
	}

	public class DataNONE implements ITestData {
		private static final long serialVersionUID = 3628293252172988494L;
		@CheckCreditCard(options = CreditCardValidator.NONE)
		private String data;

		public DataNONE() {
		}

		public DataNONE(String data) {
			this.data = data;
		}

		public String getData() {
			return data;
		}

		public void setData(String data) {
			this.data = data;
		}
	}

	public class DataNONEVerifyWhenNull implements ITestData {
		private static final long serialVersionUID = -8643089108242414510L;
		@CheckCreditCard(verifyWhenNull = true, options = CreditCardValidator.NONE)
		private String data;

		public DataNONEVerifyWhenNull() {
		}

		public DataNONEVerifyWhenNull(String data) {
			this.data = data;
		}

		public String getData() {
			return data;
		}

		public void setData(String data) {
			this.data = data;
		}
	}
}
