package jp.sourceforge.masasa.architecture.framework.validation.resource;

import java.io.FileInputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;

import jp.sourceforge.masasa.architecture.framework.validation.util.StringUtil;

/**
 * メッセージリソースを管理するクラス.
 * 
 * @author masasa.
 */
public final class ResourceUtils {

    /** リソース. */
    private ResourceBundle bundle = null;
    /** リソース. */
    private Properties properties = null;
    /** 動的リソースマップ. */
    private final HashMap<String, String> dynamicResource = new HashMap<String, String>();

    /**
     * 指定されたリソースファイルを持つSigletonインスタンスを生成する.<br>
     * リソースファイルが存在しない場合、エラーにはしないが、メッセージ は取得できない.
     * 
     * @param file
     *            リソースファイル名.
     */
    public ResourceUtils(final String file) {
        this(file, null);
    }

    /**
     * 指定されたリソースファイルを持つインスタンスを生成する.<br>
     * リソースファイルが存在しない場合、エラーにはしないが、メッセージ は取得できない.
     * 
     * @param file
     *            リソースファイル名.
     * @param loader
     *            クラスローダ.
     */
    public ResourceUtils(final String file, final ClassLoader loader) {
        this.properties = null;
        try {
            if (loader != null) {
                this.bundle = ResourceBundle.getBundle(file, Locale.getDefault(), loader);
            } else {
                this.bundle = ResourceBundle.getBundle(file);
            }
        } catch (MissingResourceException e) {
            this.bundle = null;
            this.properties = new Properties();
            try {
                this.properties.load(new FileInputStream(file));
            } catch (IOException ioException) {
                // do nothing
            }
        }
    }

    /**
     * 指定されたキーワードのリソースを返却する.<br>
     * 以下の条件に合致する場合、キーワードをそのまま返却する.
     * <ol>
     * <li>動的リソースの中に値が存在しない</li>
     * <li>リソースファイルが存在しない</li>
     * <li>キーワードに該当するリソースが存在しない</li>
     * </ol>
     * 
     * @param key
     *            キーワード.
     * @return リソース.
     */
    public String getResourceString(final String key) {
        String ret = this.dynamicResource.get(key);
        if (StringUtil.isNotEmpty(ret)) return ret;
        if (this.bundle != null) {
            try {
                return this.bundle.getString(key);
            } catch (MissingResourceException e) {
                return null;
            }
        }
        if (this.properties != null) {
            ret = this.properties.getProperty(key);
            return StringUtil.isNotEmpty(ret) ? ret : key;
        }
        return null;
    }
}
