package jp.sourceforge.masasa.architecture.framework.validation;

import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.masasa.architecture.framework.validation.resource.PropertyManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * 入力チェック結果クラス({@link ValidateResult})を保持するクラス.
 *
 * @author masasa.
 */
public class ValidateResults {

    /** logger. */
    protected static final Log LOGGER = LogFactory.getLog(ValidateResults.class);
    /** 入力チェック結果クラス({@link ValidateResult})リスト. */
    private final List<ValidateResult> results = new ArrayList<ValidateResult>();

    /**
     * 最大エラーカウント. {@link ValidateProperties#TERMINATE_ERROR_COUNT}.
     * <ul>
     * <li>0 : 無制限にチェック実行.</li>
     * <li>1以上 : その値までチェック実行.</li>
     * </ul>
     */
    private int terminateCount = 0;

    /**
     * コンストラクタ.
     */
    public ValidateResults() {
        try {
            this.terminateCount = Integer
                    .parseInt(PropertyManager.getMessage(ValidateProperties.TERMINATE_ERROR_COUNT));
        } catch (NumberFormatException e) {
            // validation.propertiesが存在しなくてnull
            // またはvalidation.properties#terminate.error.countが存在しなくてnull
            // またはvalidation.properties#terminate.error.countが0以外
            // 何れの場合も正常運行させる
            this.terminateCount = 0;
            LOGGER.warn("\"terminate.error.count\" is not defined by \"validate.properties\"." + //$NON-NLS-1$
                    " AbstractValidator carry it out with terminateCount as 0."); //$NON-NLS-1$
        }
    }

    /**
     * 入力チェック結果クラス({@link ValidateResult})リストが存在するか否かを返却します.
     *
     * @return 存在可否.
     *         <ul>
     *         <li><code>true</code>:存在しない.</li>
     *         <li><code>false</code>:存在する.</li>
     *         </ul>
     */
    public boolean isEmpty() {
        return this.results.isEmpty();
    }

    /**
     * 入力チェック結果クラス({@link ValidateResult})を追加します.
     *
     * @param result
     *            入力チェック結果クラス({@link ValidateResult}).
     */
    public void add(ValidateResult result) {
        this.results.add(result);
    }

    /**
     * 入力チェック結果クラス({@link ValidateResult})リストを返却します.
     *
     * @return 入力チェック結果クラス({@link ValidateResult})リスト.
     */
    public List<ValidateResult> getResults() {
        return this.results;
    }

    /**
     * 最大エラー件数を超えたか検証します.
     *
     * @return 検証結果.
     *         <ul>
     *         <li><code>true</code>:最大エラー件数を超えた.</li>
     *         <li><code>false</code>:上記以外.</li>
     *         </ul>
     */
    public boolean isOverTerminate() {
        if (this.terminateCount == 0) return false;
        return this.results.size() >= this.terminateCount;
    }
}
