require 'ftools'
require 'extension/time'
require 'extension/file'
require 'extension/cgi'
require 'vikiwikiattach'
require 'vikiwikitheme'
require 'vikiwikirss'

module VikiWiki
	class Pages
		include Enumerable
		def initialize(sys)
			@sys = sys
			@regtextdir = /^#{Regexp::escape(@sys['TEXTDIR'])}\//
			@regtextfile = /^([^\.]+)\.([^\.]+)\.(?:([^\.]+)\.)?(#{Regexp::escape(@sys['FILEEXT'])})$/
			@entries = Hash::new
		end
		def each
			elements.each{|path, name, style, user, extension|
				yield self[name, style, path]
			}
		end
		def elements
			return @elements if @elements
			file = ['*', @sys['FILEEXT']].file
			subdir = @sys['PAGENODE'] ? '**/' : ''
			wc = "#{@sys['TEXTDIR']}/#{subdir}#{file}".untaint
			elm = Hash::new
			Dir::glob(wc) do |path|
				name, style, user, extension = split(path)
				next unless name
				next if /^#{@sys.site}\// === name
				elm[name] = [path.untaint, name, style, user, extension]
			end
			wc = "#{@sys['TEXTDIR']}/#{@sys.site}/#{subdir}#{file}".untaint
			Dir::glob(wc) do |path|
				name, style, user, extension = split(path)
				next unless name
				name = name.sub(/^#{@sys.site}\//, '')
				elm[name] = [path.untaint, name, style, user, extension]
			end
			@elements = Array::new
			@names = Array::new
			@paths = Array::new
			elm.keys.sort.each do |name|
				@elements << elm[name]
				@names << name
				@paths << elm[name][0]
			end
			return @elements
		end
		def names
			elements unless @names
			return @names
		end
		def paths
			elements unless @paths
			return @paths
		end
		def to_a
			unless @array then
				@array = Array::new
				each{|page|
					@array << page
				}
			end
			return @array
		end
		def split(path)
			return nil if path.include?('/.')
			return nil unless @regtextdir === path
			return nil unless @regtextfile === $'
			# page-name, style, user, extension
			return CGI::unescapePATH($1), $2, $3, $4
		end
		def convname(name)
			case name
			when nil, ''  then name = @sys['TOPPAGE']
			when SelfPage then name = @sys.page ? @sys.page.name : @sys['TOPPAGE']
			when TopPage  then name = @sys['TOPPAGE']
			when HelpPage then name = @sys.style + HelpPage
			end
			return name
		end
		def [](name, sty=nil, path=nil)
			name = convname(name)
			return @entries[name] if @entries.key?(name)
			page = Page::new(@sys, name, sty, path)
			@entries[name] = page
			@entries[page.name] = page if name != page.name
			return page
		end
		def findfile(name, pagenode=false)
			lang = @sys.lang
			lang = nil if @sys['LANG'] == lang
			lang = nil if /^([a-z]{2})\// === name && lang == $1
			lang = @sys['LANGSWITCH'] ? lang.capitalize : lang+'/' if lang
			subdir = pagenode ? '**/' : ''
			name = CGI::escapePATH(name)
			names = Array::new
			names << "#{@sys.site}/#{subdir}#{lang}#{name}" if lang
			names << "#{@sys.site}/#{subdir}#{name}"
			names << "#{subdir}#{lang}#{name}" if lang
			names << "#{subdir}#{name}"

			names.each do |schname|
				wc = "#{@sys['TEXTDIR']}/#{schname}.*.#{@sys['FILEEXT']}"
				file = Dir::glob(wc.untaint).first
				if file then
					sty = File::basename(file).split('.')[1]
					return file.untaint, sty
				end
			end
			return getpath(name, @sys.style), @sys.style
		end
		def getpath(name, sty)
			file = [name, sty, @sys.sec.user, @sys['FILEEXT']].file
			return [@sys['TEXTDIR'], file].path
		end
		def clear
			@elements = nil
			@names = nil
			@paths = nil
		end
		#===================================================
		# glob
		# Like a glob method at Dir class
		# patterns - Array class
		# - Regexp
		# - wildcard *,?,**
		# - string formatted Regexp "/.../"
		# - string
		#===================================================
		def glob(patterns)
			arr = Array::new
			del = Array::new
			patterns.each do |name|
				inv = false
				if String === name then
					name, inv = $', true if /^-/ === name
					name = name.to_any
				end
				if Regexp === name then
					cur = names.grep(name)
				elsif name.include?("*?") then
					esc = Regexp::escape(name)
					esc.gsub!("\\*\\*\\/", "(.+\\/)*")
					esc.gsub!("\\*", ".*")
					esc.gsub!("\\?", ".")
					cur = names.grep(Regexp::new(esc))
				elsif names.include?(name) then
					cur = [name]
				else
					next
				end
				cur -= del
				if inv then
					del += cur
				else
					arr += cur
				end
			end
			return arr.uniq
		end
	end
	class Page
		RE_BOM = /\A\xef\xbb\xbf/
		MAXPAGELEN = 512
		attr_reader :name, :static
		def initialize(sys, name, sty=nil, path=nil)
			@sys = sys
			Log.perform(__FILE__, __LINE__) if @sys['PERFORM']
			@name = name
			raise Message::new(:PAGE_INVALID_NAME, @name) unless @name
			raise Message::new(:PAGE_INVALID_NAME, @name) if @name.include?('.')
			raise Message::new(:PAGE_INVALID_NAME, @name) if /\s/ === @name
			raise Message::new(:PAGE_INVALID_NAME, @name) if not @sys['PAGENODE'] and @name.include?('/')
			raise Message::new(:PAGE_NAME_TOO_LONG) if CGI::escapePATH(@name).size > MAXPAGELEN
			@path = path if path
			@style = sty if sty
			@static = StaticHTML::new(@sys, @name) if @sys.staticable?
		end
		def alias
			return @alias if @alias
			return @alias = @sys.aliases.get(@name)
		end
		def path
			@path, @style = @sys.pages.findfile(@name) unless @path
			return @path
		end
		def style
			self.path unless @style
			return @style
		end
		def exist?
			File::exist?(self.path)
		end
		def uri(opt=nil, anc=nil, script=@sys.script_name)
			reqcode = opt['c'] if opt
			reqcode = @sys.code unless reqcode
			curcode = USEconv::kcode(@sys.encoding)
			name = @name
			name = $' if /^#{@sys.site}\// === name
			if reqcode != curcode then
				cnv = USEconv::cnvmethod(curcode, reqcode)
				name = cnv.call(name) if cnv
			end
			wopt = Hash::new
			wopt['p'] = name if name != @sys['TOPPAGE']
			wopt['c'] = reqcode if curcode != reqcode
			wopt['h'] = @sys.theme.name if @sys.theme.name != @sys['THEMECSS']
			wopt['l'] = @sys.lang if @sys.lang != @sys['LANG']
			wopt['m'] = @sys.cgi.params['m'][0] if @sys.cgi.params['m'][0] != LayoutPage
			wopt = wopt.update(opt) if opt
			wopt = wopt.to_a.map{|a,b|
				"#{a}=#{CGI::escapeURI(b,'')}" if b
			}.compact
			wuri = script
			wuri += '?'+wopt.join('&') unless wopt.empty?
			wuri += '#'+CGI::escapeURI(anc,'') if anc
			return wuri
		end
		def abs_name
			self.path.sub(/^#{Regexp::escape(@sys['TEXTDIR'])}\//, '').split('.').first
		end
		def load(version=nil)
			raise AccessDenied unless @sys.accessable?('page', @name, 'r')
			path = self.path
			return nil unless FileTest::exist?(path)
			version = @sys.version if version.nil? and @sys.page == self
			return @text if @text and version.nil?
			@time = File::mtime(path)
			text = @sys.cvs.readFile(path, version)
			text.sub!(RE_BOM, '') if text and USEconv::eql?(@sys.encoding, 'utf-8')
			@text = text unless version
			return text
		end
		def text; load; end
		def write(text, time=@time, staticflag=nil)
			raise AccessDenied unless @sys.accessable?('page', @name, 'w')
			text.replace_words!(@sys['REPLACE_WORDS']) if text and defined?(@sys['REPLACE_WORDS'])
			time = nil unless @sys['MODIFYCHECK']
			time = Time::parse(time) if String === time
			@sys.cvs.writeFile(self.path, text, time, @sys.user)
			@text = text || load
			if @static then
				if staticflag or @static.exist? then
					@static.write
				elsif staticflag == false then
					@static.delete
				end
			end
			Rss::new(@sys).update
			@sys.onwrite(self) if @sys.respond_to?(:onwrite)
		end
		def node
			return @node if @node
			@node = Parser::new(@sys).parse(self.load, self.style)
		end
		def html
			Parser::new(@sys).to_html(self.node, self.style)
		end
		def rename(newname)
			return if @name == newname
			newpage = @sys.pages[newname]
			raise Message::new(:PAGE_ALREADY_EXISTS, newname) if newpage.exist?
			raise AccessDenied unless @sys.accessable?('page', @name, 'w')
			raise AccessDenied unless @sys.accessable?('page', newname, 'w')
			@sys.cvs.renameFile(self.path, newpage.path)
			@static.rename(newname) if @static and @static.exist?
			resetup(@sys, newname, newpage.path)
			@static.write if @static and @static.exist?
			@sys.pages.clear
		end
		def altstyle(newstyle)
			raise AccessDenied unless @sys.accessable?('page', @name, 'w')
			return if self.style == newstyle
			time = File::mtime(self.path)
			text = Parser::trans_text(@sys, self.load, self.style, newstyle)
			newpath = @sys.pages.getpath(@name, newstyle)
			@sys.cvs.renameFile(self.path, newpath)
			resetup(@sys, @name, newstyle)
			write(text, time)
		end
		def resetup(sys, name, path)
			instance_variables.each{|var|
				remove_instance_variable(var)
			}
			initialize(sys, name, path)
		end
		def ==(other)
			return (self <=> other) == 0
		end
		def <=>(other)
			if other.respond_to?(:name) then
				name = other.name
			elsif String === other then
				name = other
			else
				name = other.to_s
			end
			return @name <=> name
		end
		def update_ref(oldname, newname)
			return unless self.exist?
			case @sys['ALTREFER']
			when 'none', false, nil then
				return
			when 'replace' then
				reg = /([A-Za-z0-9_\/\:]+)|(#{Regexp::escape(oldname)})/
				self.load.gsub(reg) do
					if ($1 and $1 == oldname) or $2 then
						newname
					else
						$&
					end
				end
			else
				self.links do |node|
					if node.nodeName == 'WikiName' then
						node.setAttribute('wri', newname)
					else
						node.nodeValue = newname
					end
				end
				text = Parser::new(@sys).to_text(self.node, self.style)
			end
			write(text, nil, true)
		end
		def diff(ver=nil)
			unless ver then
				vers = @sys.cvs.getVersions(self.path)
				return nil if vers.nil? or vers.empty?
				ver = @sys.version || vers.last
			end
			file = @sys.cvs.getDiffFile(self.path, ver)
			return nil unless file and File::exist?(file)
			return File::readlines(file).join
		end
		def links
			all_pages = @sys.pages.names
			arr = Array::new
			self.node.each_node(['WikiName','plugin']) do |node|
				if node.nodeName == 'WikiName' then
					wri = node.getAttribute('wri')
					yield node if iterator?
					next if wri.nil? or @name == wri
					arr << wri unless arr.include?(wri)
				else
					list = node.childNodes
					0.upto(list.length-1) do |i|
						item = list.item(i)
						wri = item.firstChild.nodeValue
						next if @name == wri
						next unless all_pages.include?(wri)
						yield item if iterator?
						arr << wri
					end
				end
			end
			return arr.uniq.sort
		end
	end
	class StaticHTMLs
		include Enumerable
		def initialize(sys)
			@sys = sys
			@dir = @sys['STATICDIR']
			File::mkpath(@dir) if @dir
		end
		def each(name=nil, subdir=true)
			each_path(name, subdir){|path|
				file = path[@dir.size+1, path.size]
				name, _ = file.split('.')
				name = CGI::unescapePATH(name)
				yield StaticHTML::new(@sys, name)
			}
		end
		def each_path(name=nil, subdir=true)
			name = CGI::escapePATH(name) if name
			file = [name, '*', 'html'].file
			subdir = subdir && @sys['PAGENODE'] ? '**' : nil
			wc = [@dir, subdir, file].path
			subpos = @dir ? @dir.split('/').size : 0
			Dir::glob(wc){|path|
				next if path.include?('/.')
				next if File::dirname(path) == @sys.basedir
				dsub = path.split('/')
				next if SYSDIRNAMES.include?(dsub)
				next if @sys['STATICEXCEPT'] and @sys['STATICEXCEPT'].include?(dsub)
				name = File::basename(path)
				next if name == 'index.html'
				yield path.untaint
			}
		end
		def index?
			File::exist?([@dir, 'index.html'].path)
		end
		def image?
			File::exist?([@dir, IMGNAM].path)
		end
		def theme?
			File::exist?([@dir, THEMENAM].path)
		end
		def delete_all
			each_path{|path| File::unlink(path)}
		end
		def generate_location_all
			@sys.pages.each{|page|
				uri = page.uri
				File::mkpath(File::dirname(page.static.path))
				File::open(page.static.path, "w"){|fout|
					fout << @sys.rhtml('location', binding)
				}
			}
		end
		def generate_index_html
			indexfile = [@sys['STATICDIR'], 'index.html'].path
			uri = [@sys['STATICURI'], [CGI::escapePATH(@sys['TOPPAGE']), 'html']].path
			File::open(indexfile, 'w'){|fout|
				fout.write(@sys.rhtml('location', binding))
			}
		end
		def copy_image
			return unless ['sync', 'copy'].include?(@sys['STATICIMG'])
			dir = [@sys['STATICDIR'], IMGNAM].path
			Dir::copy(@sys['IMGDIR'].first, dir)
		end
		def copy_theme
			return unless ['sync', 'copy'].include?(@sys['STATICTHEME'])
			dir = [@sys['STATICDIR'], THEMENAM].path
			Dir::copy(@sys['THEMEDIR'].first, dir)
		end
		def generate(page, flags, out_method=nil, finished=[])
			page = @sys.pages[page] if String === page
			name = page.name
			return if finished.include?(name)
			finished << name
			return unless page.exist?
			return unless page.static
			return if flags['exist'] and not File::exist?(page.static.path)
			return if flags['update'] and \
				File::exist?(page.static.path) and \
				File::mtime(page.path) <= File::mtime(page.static.path)
			out_method.call("Generate #{page.name}...") if out_method
			begin
				page.static.write
				out_method.call("Finished.\n") if out_method
			rescue StandardError, ScriptError
				Log.rescue(__FILE__, __LINE__, $!.to_s, $@.to_a.join("\n")) if @sys['DEBUG']
				$!.setmsg(@sys.msgs) if Message === $!
				out_method.call("Error #{$!}.\n") if out_method
			end
			if flags['link'] then
				if @sys.links[name] then
					@sys.links[name].each do |link_name|
						generate(link_name, flags, out_method, finished)
					end
				end
				page.links do |node|
					next unless node.nodeName == 'WikiName'
					wri = node.getAttribute('wri')
					next if wri.include?(':')
					generate(wri, flags, out_method, finished)
				end
			end
		end
	end
	class StaticHTML
		attr_reader :name
		def initialize(sys, name)
			@sys = sys
			name = $' if /^#{@sys.site}\// === name
			@name = name
			raise unless @name
			raise unless @sys.staticable?
		end
		def path
			name = CGI::escapePATH(@name)
			path = [
				@sys['STATICDIR'],
				[name, 'html'].file
			].path
			return path
		end
		def uri(anc=nil)
			name = CGI::escapePATH(@name)
			path = [
				@sys['STATICURI'],
				[name, 'html'].file
			].path
			path << '#'+CGI::escapeURI(anc,'') if anc
			return path
		end
		def write
			begin
				if @sys['STATICWIKI'] then
					page = @sys.pages[LayoutPage]
				else
					page = @sys.pages[StaticLayout]
				end
				raise Message::new(:PAGE_NOT_FOUND) unless page.exist?
				layout = page.load
				style = page.style
				File::mkpath(File::dirname(self.path))
				cgi = @sys.cgi.clone
				cgi.params.clear
				cgi.params['p'] = [@name]
				sys = Sys::new(@sys.local_file, @sys.conf, cgi)
				sys.setup
				sys.static_mode{
					sys.body = Parser::trans_html(sys, layout, style)
					html = sys.rhtml('body')
					File::fwrite(self.path, html)
				}
			rescue AuthorizationRequired
				raise Message::new(:PAGE_STATIC_DENIED, page.name) unless @sys['STATICWIKI']
			end
		end
		def delete
			File::unlink(self.path) if File::exist?(self.path)
		end
		def rename(newname)
			newpath = StaticHTML::new(@sys, newname).path
			raise Message::new(:FILE_AREADY_EXISTS, newpath) if FileTest::exist?(newpath)
			File::mkpath(File::dirname(newpath))
			File::rename(self.path, newpath)
		end
		def exist?
			File::exist?(self.path)
		end
	end
	class WRI
		attr_reader :name, :inter, :names, :anchor
		def WRI::wri(inter, names, anchor=nil)
			name = [inter, names].flatten.compact.join(':')
			name += "##{anchor}" if anchor
			return name
		end
		def initialize(sys, name)
			@sys = sys
			@name = name
			main, @anchor = @name.split('#')
			@names = main.split(':')
			if /\*/ === main then
				case @sys['SHORTWIKI']
				when 'wildcard' then
					@names.map!{|w| wildcard(w)}
				when 'abbrev' then
					@names.map!{|w| abbrev(w)}
				end
				@name = WRI::wri(nil, @names, @anchor)
			end
			@names.map!{|nam| @sys.pages.convname(nam) rescue nam}
			@inter = @names.shift if @names.size > 1
			@sibling = @sys.siblings[@inter] if @inter and names.size == 1
		end
		def fulluri(static_mode = @sys.static?)
			path = uri(static_mode)
			path = "http://#{@sys.cgi.host}#{path}" unless /^https?:/ === path
			return path
		end
		def uri(static_mode = @sys.static?)
			return @sibling.uri(names.first, @anchor) if @sibling
			unless @inter then
				page = @sys.pages[@names.first]
				if @sys.staticable? then
					static = StaticHTML::new(@sys, @names.first)
					return static.uri(@anchor) if static_mode and \
						(not @sys['STATICWIKI'] or \
						(page.exist? and static.exist?))
				end
				return page.uri(nil, @anchor)
			end
			wiki = @names.first
			@sys.interwikis.each_all{|name, value|
				return expand(value) if name == @inter
			}
			opt = @inter.include?('Layout') ? 'm' : 'r'
			return @sys.pages[wiki].uri({opt=>@inter}, anchor) if @sys.pages[@inter].exist?
			return nil
		end
		def expand(value)
			names = [File::basename(@sys.script_name)] + @names
			if value['code'] and value['code'] != @sys.encoding then
				from = USEconv::kcode(@sys.encoding)
				to = USEconv::kcode(value['code'])
				cnv = USEconv::cnvmethod(from, to)
				names.map!{|wiki|
					cnv.call(wiki) if wiki
				} if cnv
			end
			uri = (value['uri'] || value['url']).dup
			if uri == '$0' then
				opt = value['opt'].dup
				opt.keys.each{|k|
					opt[k] = names[$1.to_i] if /^\$(\d+)$/ === opt[k]
				}
				uri = @sys.pages[names[1]].uri(opt)
			elsif uri == '__FILE__' then
				uri = @sys.pages[names[1]].path
				from = @sys['EDITFILEFROM']
				to = @sys['EDITFILETO']
				uri = uri.sub(from, to) if from and to
			elsif /[\\\$]\d/ === uri then
				uri.gsub!(/([\\\$])(\d+)/){
					m, i = $1, $2.to_i
					next unless names[i]
					name = CGI::escapePATH(names[i])
					name = CGI::escapePATH(name) if m == "\\"
					name
				}
				if value['opt'] then
					opt = Hash::new
					value['opt'].each{|k,v|
						case v
						when nil then
							opt[k] = ''
						when /^\$(\d+)$/ then
							opt[k] = names[$1.to_i]
						else
							opt[k] = v
						end
					}
					opts = opt.to_a.map{|k,v|
						"#{k}=#{CGI::escapeURI(v,'')}"
					}
					uri += '?' + opts.join('&') unless opts.empty?
				end
			else
				uri += CGI::escapeURI(names[1])
			end
			return uri
		end
		def exist?
			return self.uri ? true : false if @inter
			return @sys.pages[@names.first].exist?
		end
		def alias
			if @sibling then
				@sibling['SITENAME'] + ':' +
				@sibling.aliases.get(@names.first)
			else
				@sys.pages[@name].alias
			end
		end
		def wildcard(name)
			return name unless /[\*\?]/ === name
			reg = Regexp::new('^'+name.gsub(/\*/, '.*').gsub(/\?/, '.')+'$')
			@sys.pages.names.each{|name|
				return name if reg === name
			}
			@sys.interwiki.each{|key, val|
				return key if reg === key
			}
			return name
		end
		def abbrev(name)
			return name unless /^\*/ === name
			name = $'
			if /^[A-Za-z0-9_]+$/ === name then
				reg = Regexp::new('^'+name.scan(/./).join('.*')+'.*$')
			else
				reg = Regexp::new('^.*'+name+'.*$')
			end
			@sys.pages.names.each{|name|
				return name if reg === name
			}
			@sys.interwiki.each{|key, val|
				return key if reg === key
			}
			return name
		end
	end
	class Siblings
		def initialize(sys)
			@sys = sys
			@siblings = Hash::new
		end
		def [](name)
			return nil unless @sys['SIBLINGWIKI']
			return nil unless @sys['SIBLINGWIKI'].include?(name)
			return @siblings[name] if @siblings[name]
			return @siblings[name] = Sibling::new(@sys, name)
		end
	end
	class Sibling
		def initialize(sys, name)
			@sys = sys
			@name = name
			@basedir = File::dirname(@sys.basedir) + "/#{@name}/"
			@baseuri = File::dirname(@sys.baseuri) + "/#{@name}/"
			@local_file = siblingpath(@sys.local_file).untaint
			confstr = File::readlines(@local_file).join[/^@conf = \{.+^\}\n/m]
			eval(confstr.untaint)
		end
		def siblingpath(path)
			path.sub(@sys.basedir, @basedir)
		end
		def siblinguri(path)
			path.sub(@sys.baseuri, @baseuri)
		end
		def aliases
			return @aliases if @aliases
			datadir = siblingpath(@sys['DATADIR'])
			file = datadir+'/AliasName.txt'
			@aliases = Aliases::new(file.untaint, @sys.lang)
			return @aliases
		end
		def uri(page, anc=nil)
			if @sys.static? then
				path = siblinguri(@sys['STATICURI'])
				path << "/#{CGI::escapePATH(page)}.html"
			else
				path = siblinguri(@sys.script_name)
				path << "?p=#{CGI::escapeURI(page,'')}"
			end
			path << '#'+CGI::escapeURI(anc,'') if anc
			return path
		end
		def [](key)
			@conf[key]
		end
	end
end
