=begin
	The class transforms any Wiki style text to the HTML text or any Wiki style text.

	VikiWiki::Parser::new(sys)

	parse(text, style)
		text - Wiki style text
		style - Style of Wiki style determinated by a style class.
		The method retrives DOM object generated by parsing.
		text - Wiki style text

	to_html(nodes=@nodes)
		The methid retrieves the string transformed to HTML.
		nodes - the parsed result with DOM object

	to_text(nodes=@nodes, style=@style)
		The methid retrieves the string transformed to Text with the Wiki style.
		nodes - the parsed result with DOM object

	to_xml(nodes=@nodes, style=@style)
		The methid retrieves the string transformed to XML.
		nodes - the parsed result with DOM object

	VikiWiki::Parser::trans_html(sys, text, pstyle='VikiWikiStyle')
		The methid retrieves the string transformed to HTML.
		pstyle - the style to parse

	VikiWiki::Parser::trans_text(sys, text, pstyle='VikiWikiStyle', tstyle='VikiWikiStyle')
		The methid retrieves the string transformed to Text with the Wiki style.
		pstyle - the style to parse
		tstyle - the style to transform to Text

	VikiWiki::Parser::trans_xml(sys, text, pstyle='VikiWikiStyle')
		The methid retrieves the string transformed to XML.
		pstyle - the style to parse

=end

require 'dom-mini' unless defined?(Document)
require 'vikiwikigenerator'

module VikiWiki
	class Parser
		attr_reader :text, :style, :nodes
		def Parser::trans_html(sys, text, pstyle='VikiWikiStyle')
			parser = Parser::new(sys)
			nodes = parser.parse(text, pstyle)
			html = parser.to_html(nodes, pstyle)
			return html
		end
		def Parser::trans_text(sys, text, pstyle='VikiWikiStyle', tstyle=pstyle)
			parser = Parser::new(sys)
			nodes = parser.parse(text, pstyle)
			text = parser.to_text(nodes, tstyle)
			return text
		end
		def Parser::trans_html_inline(sys, text, pstyle='VikiWikiStyle')
			parser = Parser::new(sys)
			nodes = parser.parse_inline(text, pstyle)
			html = parser.to_html_inline(nodes, pstyle)
			return html
		end
		def Parser::trans_text_inline(sys, text, pstyle='VikiWikiStyle', tstyle=pstyle)
			parser = Parser::new(sys)
			nodes = parser.parse_inline(text, pstyle)
			text = parser.to_text_inline(nodes, tstyle)
			return text
		end
		def Parser::trans_xml(sys, text, pstyle='VikiWikiStyle', root=nil)
			parser = Parser::new(sys)
			nodes = parser.parse(text, pstyle, root)
			xml = parser.to_xml(nodes)
			return xml
		end
		def initialize(sys=nil)
			@sys = sys
			@styles = @sys.styles if @sys and @sys.respond_to?(:styles)
			@styles = Styles::new(sys, 'VikiWiki') unless @styles
			@xml = @sys.xml if @sys and @sys.respond_to?(:xml)
			@xml = Document::new(DOMImplementation::new) unless @xml
		end
		def parse(lines, style=nil, root=nil)
			root = @xml.createDocumentFragment unless root
			@styles[style].parse(root, lines.to_a)
			return root
		end
		def parse_inline(line, style=nil, root=nil)
			root = @xml.createDocumentFragment unless root
			@styles[style].parse_inline(root, line)
			return root
		end
		def to_html(nodes, style=nil)
			generator = HtmlGenerator::new(@sys, @styles[style])
			return generator.generate(nodes) << "\n"
		end
		def to_html_inline(nodes, style=nil)
			generator = HtmlGenerator::new(@sys, @styles[style])
			return generator.generate(nodes) << "\n"
		end
		def to_xml(nodes)
			generator = XmlGenerator::new
			return generator.generate(nodes) << "\n"
		end
		def to_text(nodes, style=nil)
			return @styles[style].generate(nodes) << "\n"
		end
		def to_text_inline(nodes, style=nil)
			return @styles[style].generate(nodes)
		end
	end

	class Styles
		attr_reader :default
		def initialize(sys, default)
			@sys = sys
			@default = fullname(default)
			@entries = Hash::new
		end
		def [](name=nil)
			name = @default unless name
			return name unless String === name
			name = fullname(name)
			return @entries[name] if @entries.key?(name)
			require "wiki/#{name.downcase}".untaint
			cls = Wiki::const_get(name)
			raise "#{name} style not found." unless cls
			return @entries[name] = cls.new(@sys, @sys)
		end
private
		def fullname(name)
			name << 'Style' unless /Style$/ === name
			return name
		end
	end
end
