# Name of Wiki Engine
$WIKIENGINE     = 'VikiWiki'
# Version of Wiki Engine
$WIKIVERSION    = '1.9.7'

begin
	require 'builtin'
rescue LoadError
	require 'builtin-default'
end
require 'extension/time'
require 'extension/file'
require 'extension/cgi'
require 'useconv'
require 'dom-utility'
require 'erb-mini'
require 'log'
require 'cryptpassword'
require 'vikiwikierror'
require 'vikiwikiconst'
require 'vikiwikipage'
require 'vikiwikidata'
require 'vikiwikitheme'
require 'vikiwikiparser'
require 'vikiwikiplugin'

module VikiWiki
	class Sys
		include VikiWikiBuiltin

		attr_reader \
			:sec, :lang, :cvs, :cgi, :header, :script_name, \
			:acl, :interwikis, :links, :layout, \
			:aliases, :counter, :version, :pages, :statics, :siblings, \
			:indexes, :farms, :themes, :styles, :basedir, :baseuri, \
			:plugin, :conf, :local_file, :style, \
			:html_headers, :onload_scripts, :scripts, :admins, \
			:msgs, :encoding, :content_type, :xml, :site
		attr_accessor  :page, :error, :errinf, :code, :theme, :body

		def initialize(local_file, conf, cgi=nil)
			#---------------------------------------------------------------
			# Configuration
			#---------------------------------------------------------------
			conf = CONF.dup.update(conf)
			conf['DEBUG'] = true if $DEBUG
			#---------------------------------------------------------------
			# Get CGI parameter
			#---------------------------------------------------------------
			if cgi then
				@cgi = cgi
			else
				@cgi = CGI::new
				@cgi.easy!
			end
			kcode = USEconv::kcode(conf['ENCODING'])
			@code = @cgi.params['c'][0] || kcode
			@code = @cgi.params['c'][0] || kcode
			@conv_in = USEconv::cnvmethod(@code, kcode)
			@conv_out = USEconv::cnvmethod(kcode, @code)
			@cgi.convert!(@conv_in) if @conv_in
			#---------------------------------------------------------------
			# Prepare Instance Variable
			#---------------------------------------------------------------
			@out = $VIKIWIKIOUT || $>
			@header = {'cookie' => Array::new}
			@html_headers = Array::new
			@onload_scripts = Array::new
			@encoding = USEconv::encoding(@code)
			@lang = @cgi.params['l'][0]
			@lang = conf['LANG'] unless /^[a-z]{2}$/ === @lang
			@msgs = Message::load([$VIKIWIKIDIR, MSGDIR, "#{@lang}.rb"].path)
			#---------------------------------------------------------------
			# Based Path
			#---------------------------------------------------------------
			@local_file = local_file.gsub(/\\/,'/').untaint
			@basedir = conf['BASEDIR'] || File::dirname(@local_file)
			@basedir << '/' unless /\/$/ === @basedir
			@script_name = conf['SCRIPT_NAME'] || @cgi.script_name || ''
			@baseuri = conf['BASEURI'] || File::dirname(@script_name)
			@baseuri << '/' unless /\/$/ === @baseuri
			@site = File::basename(File::dirname(@script_name))
			#---------------------------------------------------------------
			# Dirctory and URI
			#---------------------------------------------------------------
			const = Hash::new
			const['ETCDIR']   = @basedir+ETCNAM
			const['TEXTDIR']  = @basedir+TEXTNAM
			const['DATADIR']  = @basedir+DATANAM
			const['FARMDIR']  = @basedir+(conf['WIKIFARMFLAT'] ? '' : FARMNAM)
			const['FARMURI']  = @baseuri+(conf['WIKIFARMFLAT'] ? '' : FARMNAM)
			const['IMGDIR']   = [@basedir+IMGNAM]
			const['IMGURI']   = [@baseuri+IMGNAM]
			const['THEMEDIR'] = [@basedir+THEMENAM]
			const['THEMEURI'] = [@baseuri+THEMENAM]
			#---------------------------------------------------------------
			# Dirctory and URI for ShareWiki
			#---------------------------------------------------------------
			if conf['SHAREWIKIDIR'] and conf['SHAREWIKIURI'] then
				shrdir = conf['SHAREWIKIDIR']
				shruri = conf['SHAREWIKIURI']
				shrdir << '/' unless /\/$/ === shrdir
				shruri << '/' unless /\/$/ === shruri
				const['ETCDIR']   = shrdir+ETCNAM
				const['TEXTDIR']  = shrdir+TEXTNAM
				const['DATADIR']  = shrdir+DATANAM
				const['IMGDIR']   = [shrdir+IMGNAM]
				const['IMGURI']   = [shruri+IMGNAM]
			end
			#---------------------------------------------------------------
			# Dirctory and URI for StaticHTML
			#---------------------------------------------------------------
			if conf['STATICDIR'] then
				stadir = [conf['STATICDIR'], IMGNAM].path
				stauri = [conf['STATICURI'], IMGNAM].path
				case conf['STATICIMG']
				when 'share' then
					const['IMGDIR'] << stadir
					const['IMGURI'] << stauri
				when 'static' then
					const['IMGDIR'] = [stadir]
					const['IMGURI'] = [stauri]
				end
				stadir = [conf['STATICDIR'], THEMENAM].path
				stauri = [conf['STATICURI'], THEMENAM].path
				case conf['STATICTHEME']
				when 'share' then
					const['THEMEDIR'] << stadir
					const['THEMEURI'] << stauri
				when 'static' then
					const['THEMEDIR'] = [stadir]
					const['THEMEURI'] = [stauri]
				end
			end
			#---------------------------------------------------------------
			# Set Configuration
			#---------------------------------------------------------------
			@conf = const.update(conf)
		end
		def setup
			#---------------------------------------------------------------
			# Security
			#---------------------------------------------------------------
			@sec = CryptPassword::new(@conf['ETCDIR'])
			begin
				login
			rescue JumpOperation
				@user = [$!, $@]
				Log.rescue(__FILE__, __LINE__, $!.to_s, $@.to_a.join("\n")) if @conf['DEBUG']
			end
			#---------------------------------------------------------------
			# Plugin
			#---------------------------------------------------------------
			@plugin = Plugin::new(self)
			#---------------------------------------------------------------
			# Pages
			#---------------------------------------------------------------
			@pages = Pages::new(self)
			#---------------------------------------------------------------
			# StaticHTMLs
			#---------------------------------------------------------------
			@statics = StaticHTMLs::new(self)
			#---------------------------------------------------------------
			# Styles
			#---------------------------------------------------------------
			@styles = Styles::new(self, @conf['WIKISTYLES'].first)
			#---------------------------------------------------------------
			# Themes
			#---------------------------------------------------------------
			@themes = Themes::new(self)
			#---------------------------------------------------------------
			# ACL
			#---------------------------------------------------------------
			acl = Array::new
			if @conf['ADMIN'] then
				acl << AccessControl::new(nil, 'plugin', @conf['ADMIN'], 'admin', true, true, true)
				PAGES.each{|name|
					acl << AccessControl::new(nil, 'page', nil, name, true, false, false)
				} unless @conf['EDITSYSTEM']
			end
			@acl = AccessControls::new(
				@conf['DATAFILE_ACL'] ||
				@conf['DATADIR']+'/AccessControlList.txt', acl)
			#---------------------------------------------------------------
			# Administrators
			#---------------------------------------------------------------
			@admins = Admins::new(
				@conf['DATAFILE_ADMIN'] ||
				@conf['DATADIR']+'/Admins.txt', @conf['ADMIN'])
			#---------------------------------------------------------------
			# WikiLink
			#---------------------------------------------------------------
			@links = Links::new(
				@conf['DATAFILE_LINK'] ||
				@conf['DATADIR']+'/LinksPage.txt')
			#---------------------------------------------------------------
			# AliasName
			#---------------------------------------------------------------
			@aliases = Aliases::new(
				@conf['DATAFILE_ALIAS'] ||
				@conf['DATADIR']+'/AliasName.txt', @lang)
			#---------------------------------------------------------------
			# AccessCounter
			#---------------------------------------------------------------
			@counter = Counter::new(
				@conf['DATAFILE_COUNT'] ||
				@conf['DATADIR']+'/CountPage.txt')
			#---------------------------------------------------------------
			# Index
			#---------------------------------------------------------------
			@indexes = Indexes::new(
				@conf['DATAFILE_INDEX'] ||
				@conf['DATADIR']+'/IndexPage.txt')
			#---------------------------------------------------------------
			# WikiFarm
			#---------------------------------------------------------------
			@farms = Farms::new(
				@conf['DATAFILE_FARM'] ||
				@conf['DATADIR']+'/WikiFarm.txt', @conf['FARMDIR'])
			#---------------------------------------------------------------
			# InterWikiName
			#---------------------------------------------------------------
			inter = {
				SjisPage => {'uri'=>'$0','opt'=>{'c'=>'s','p'=>'$1'},'code'=>'Shift_JIS'},
				EucPage => {'uri'=>'$0','opt'=>{'c'=>'e','p'=>'$1'},'code'=>'euc-jp'},
				UtfPage => {'uri'=>'$0','opt'=>{'c'=>'u','p'=>'$1'},'code'=>'utf-8'},
				LangPage => {'uri'=>'$0','opt'=>{'l'=>'$2','p'=>'$1'}, 'code'=>@conf['ENCODING']},
				EditPage => {'uri'=>'$0','opt'=>{'e'=>EditPage,'p'=>'$1'}, 'code'=>USEconv::encoding(@code)},
				WikiFarm => {'uri'=>"#{@conf['FARMURI']}/$1/$0?p=$2",'code'=>@conf['ENCODING']},
				StaticPage => {'uri'=>"#{@conf['STATICURI']}/\\1.html",'code'=>@conf['ENCODING']},
				UpdateStatic => {'uri'=>@script_name+'?o=on&p=','code'=>@conf['ENCODING']},
				CgiPage => {'uri'=>@script_name+'?p=','code'=>@conf['ENCODING']},
				CgiEdit => {'uri'=>@script_name+'?e='+EditPage+'&p=','code'=>@conf['ENCODING']},
				BaseUri => {'uri'=>@baseuri,'code'=>@conf['ENCODING']},
			}
			if @conf['EDITFILEFROM'] and @conf['EDITFILETO'] then
				inter[CgiEdit] = inter[EditPage]
				inter[EditPage] = {'uri'=>'__FILE__'}
			end
			@interwikis = InterWikis::new(
				@conf['DATAFILE_INTERWIKI'] ||
				@conf['DATADIR']+'/InterWikiName.txt', inter)
			#---------------------------------------------------------------
			# SiblingWiki
			#---------------------------------------------------------------
			@siblings = Siblings::new(self)
			#---------------------------------------------------------------
			# Get CGI parameters
			#---------------------------------------------------------------
			# Current Page
			begin
				@page = @pages[@cgi.params['p'][0]]
			rescue
				set_error
				@page = @pages[AnErrorOccurred]
			end
			# Style
			@style = @cgi.params['s'][0] || @styles.default
			@style = @page.style if @page.exist?
			# Version
			@version = @cgi.params['v'][0].to_i if @cgi.params['v'][0]
			# Theme
			@theme = @themes[@cgi.params['h'][0]]
			# Layout
			@layout = @page.name
			if @conf['SAFE'] < 9 then
				layout = @cgi.params['e'][0] || @cgi.params['m'][0] || LayoutPage
				layout = EditPage unless @page.exist?
				@layout = @pages[layout]
			end
			if @layout.name == XML then
				@content_type = XML
			else
				@content_type = HTML
			end
			#---------------------------------------------------------------
			# Call external setup method.
			#---------------------------------------------------------------
			onsetup
			#---------------------------------------------------------------
			# Finish Setup
			#---------------------------------------------------------------
			@sys = self
			#---------------------------------------------------------------
			# Hooking Processes
			#---------------------------------------------------------------
			static_update if @cgi.params['o'][0] # => Location
			#---------------------------------------------------------------
			# Return
			#---------------------------------------------------------------
			self
		end
		def setuped?; @sys == self; end
		def [](name); @conf[name]; end
		def getHTTPheader(type=@content_type)
			header = @header.clone
			case type
			when XML then
				return "Content-Type: application/xml; charset=#{@encoding}\n\n"
			else
				if @cgi.params['e'][0] == EditPage then
					header['Pragma'] = 'no-chache'
					header['Cache-control'] = 'no-cache'
				end
				header['charset'] = @encoding
			end
			return @cgi.header(header).gsub(/\r/, '')
		end
		def getHTTPbody(body=nil)
			if @sys.content_type == XML then
				wiki = @sys.xml.createElement('wiki')
				@body = Parser::trans_xml(@sys, @page.load, @page.style, wiki)
				return @sys.rhtml('xml-body')
			else
				@body = body
				return @sys.rhtml(@conf['LAYOUT'] || 'body')
			end
		end
		def import_layout
			return @body if @body
			begin
				if @layout.exist? then
					ltext = @layout.load
					lstyle = @layout.style
				else
					ltext = DEFAULT_LAYOUTPAGE
					lstyle = 'VikiWikiStyle'
				end
				body = Parser::trans_html(@sys, ltext, lstyle)
				if @layout.name == EditPage and \
					not @plugin.viewed?('edit') \
				then
					body << plugin_onview('edit')
				end
				return error_onview + body
			rescue StandardError, ScriptError, Message
				return error_onview($!, $@)
			end
		end
		def import(name=SelfPage)
			return name == SelfPage ? @body : '' if @body
			begin
				error = ''
				raise Message::new(:PAGE_INVALID_NAME) unless name
				if @sys.cgi.params['r'][0] and name == SelfPage then
					page = @sys.pages[@sys.cgi.params['r'][0]]
				elsif name == SelfPage then
					page = @sys.page
					error = error_onview
					return error + plugin_onview('edit') if not page.exist? or @layout.name == EditPage
				else
					page = @sys.pages[name]
				end
				raise Message::new(:PAGE_NOT_FOUND, page.name) unless page.exist?
				@import_pages = Array::new unless @import_pages
				raise Message::new(:FUNC_DUPLICATE_IMPORT) if @import_pages.include?(page.name)
				@import_pages.push(page.name)
				begin
					return error + page.html
				ensure
					@import_pages.pop
				end
			rescue StandardError, ScriptError, Message
				return error_onview($!, $@)
			end
		end
		def title(format=nil)
			rep = {'page'=>@page.name, 'alias'=>@page.alias}
			format = @static ? @conf['STATICTITLE'] : @conf['TITLE'] unless format
			format = @conf['TITLE'] unless format
			format.replace_words(rep)
		end
		def html_head
			rhtml('head')
		end
		def wikitext(text, style=nil)
			begin
				Parser::trans_html(self, text, style)
			rescue Exception
				error_onview($!, $@)
			end
		end
		def set_error(err=$!, inf=$@)
			@error = err
			@errinf = inf if @conf['DEBUG']
			return @error
		end
		def error_onview(err=@error, inf=@errinf, tag=:div)
			return '' unless err
			err.setmsg(@msgs) if Message === err
			eserr = "Error!! " + CGI::escapeHTML(err.to_s)
			esinf = CGI::escapeHTML(inf.to_a.join("\n"))
			if @conf['DEBUG'] then
				errbody = tag == :span ? <<SPAN : <<DIV
<span class="error"><em>#{eserr}</em></span>
<!-- #{esinf} -->
SPAN
<div class="error">
<p><em>#{eserr}</em></p>
<!-- #{esinf} -->
</div>
DIV
			else
				errbody = tag == :span ? <<SPAN : <<DIV
<span class="error"><em>#{eserr}</em></span>
SPAN
<div class="error">
<em>#{eserr}</em>
</div>
DIV
			end
			@error, @errinf = nil, nil if err == @error
			return errbody
		end
		def plugin_onview(name, id=nil, prms=[], plain='', inline=false)
			begin
				@plugin.onview(name, id, prms, plain, inline)
			rescue StandardError, ScriptError, Message
				error_onview($!, $@, inline ? :span : :div)
			end
		end
		def plugin_onpost
			return false unless @plugin
			name = @cgi.params['pi_name'][0]
			return false unless name
			return true if @cgi.params['f'][0] != PluginsPage
			id = @cgi.params['pi_id'][0]
			plain = @cgi.params['pi_plain'][0]
			inline = @cgi.params['pi_inline'][0] == 'true'
			prms = Array::new
			@cgi.params.each{|k,v|
				next unless /^pi_(\d+)$/ === k
				prms[$1.to_i] = v[0]
			}
			@plugin.onpost(name, id, prms, plain, inline)
			return true
		end
		def wri_onview(wri)
			node = @xml.createElementSimply('wikiname', {'wri'=>wri}, wri)
			Parser::new(@sys).to_html_inline(node)
		end
		def print(str)
			str = @conv_out.call(str) if @conv_out
			@out << str
		end
		def findfile(file)
			return file if File::exist?(file)
			$:.each do |dir|
				testpath = "#{dir.sub(/\/$/, '')}/#{file}"
				return testpath if File::exist?(testpath)
			end
			raise LoadError, "file not found #{file}."
		end
		def rhtml(name, bind=binding)
			if File::exist?(name) then
				file = name
			else
				name = "#{RHTMLNAM}/#{name}.rhtml"
				file = findfile(name)
			end
			esrc = File::readlines(file).join
			erb = ERB::new(esrc)
			erb.filename = name if erb.respond_to?(:filename=)
			return erb.result(bind)
		end
		def save_instance
			save = Hash::new
			instance_variables.map{|var|
				save[var] = eval("#{var}")
			}
			begin
				yield
			ensure
				instance_variables.each{|var|
					eval("#{var}=save[var]")
				}
			end
		end
		def static_mode
			begin
				@static = true
				yield
			ensure
				@static = false
			end
		end
		def static?; @static; end
		def staticable?; @conf['STATICDIR'] ? true : false; end
		BGN_UDEF = "# *** begin user's definitions ***\n"
		END_UDEF = "# *** end user's definitions ***\n"
		def config(conf)
			conf = @conf.dup.update(conf)
			flag = false
			udef = ''
			File::foreach(@local_file){|line|
				if line == BGN_UDEF then
					flag = true
				elsif line == END_UDEF then
					flag = false
				elsif flag and /^(\S+)\s*=>\s*([^,]+),?$/ === line then
					k, v = $1, $2
					k = k.inspect unless /^[\'\"]/ === k
					udef += "#{k}=>#{v},\n"
				end
			} if FileTest::exist?(@local_file)
			vikiwikiruby = conf['$VIKIWIKIRUBY'] || $VIKIWIKIRUBY
			vikiwikidir = conf['$VIKIWIKIDIR'] || $VIKIWIKIDIR
			vikiwikitmp = conf['$VIKIWIKITMP'] || $VIKIWIKITMP
			script = <<EOS
#!#{vikiwikiruby}
$KCODE='#{$KCODE[0].chr.downcase}'
# Environment Setting
$VIKIWIKIDIR=#{vikiwikidir.inspect}
$VIKIWIKITMP=#{vikiwikitmp.inspect}
$VIKIWIKIRUBY=#{vikiwikiruby.inspect}
$SAFE = 1
ENV['TMPDIR'] = $VIKIWIKITMP
ENV['TMP'] = $VIKIWIKITMP
unless $:.include?($VIKIWIKIDIR) then
$:.unshift($VIKIWIKIDIR)
$:.unshift($VIKIWIKIDIR+'/lib')
end
# System Configration
@conf = {
EOS
		CONF.keys.sort.each{|k|
			script << "#{k.inspect}=>#{conf[k].inspect},\n"
		}
		script << BGN_UDEF
		script << udef
		script << END_UDEF
		script << <<EOS
}
if $0 == __FILE__ then
require 'vikiwikiweb'
@sys = VikiWiki::Sys::new(__FILE__, @conf)
VikiWiki::Web::new(@sys)
end
EOS
			File::fwrite(@local_file, script)
			File::executable(@local_file)
		end
		def fulluri(path=nil)
			path = "#{@baseuri}#{path}" if path.nil? or path[0].chr != '/'
			path = "http://#{@cgi.host}#{path}" unless /^https?:/ === path
			return path
		end
		def location(uri, header=true)
			res = ''
			if @conf['LOCATION'] == 'direct' then
				@header['Location'] = uri
			end
			if header then
				res << @sys.getHTTPheader
			end
			res << rhtml('location', binding)
			return res
		end
		def user
			raise AuthorizationRequired if static?
			return @user if @user.nil? or String === @user
			raise @user[0], @user[0], @user[1] if Array === @user
			raise @user.inspect
		end
		def login
			login_user = nil
			case @conf['AUTHTYPE']
			when nil, 'none' then
				login_user=nil
			when 'www' then
				login_user=@cgi.remote_user.dup if @cgi.remote_user
			when 'vikiwiki' then
				auth = @sec.basic_auth
				return if auth.nil?
				raise AuthorizationRequired unless auth
				login_user = @sec.user
			when 'form' then
				cookie_tms = @cgi.cookies['tms'][0]
				cookie_id = @cgi.cookies['id'][0]
				login_user = @cgi.params['user'][0]
				login_pass = @cgi.params['password'][0]
				return if cookie_tms.nil? and cookie_id.nil? and login_user.nil? and login_pass.nil?
				cookie_tms, cookie_id, login_user = @sec.get_session(cookie_tms, cookie_id, login_user, login_pass, @conf['TIMEOUT'])
				raise AuthorizationRequired unless @sec.user
				login_user = @sec.user
				@header['cookie'] << c1 = CGI::Cookie::new('tms', cookie_tms)
				@header['cookie'] << c2 = CGI::Cookie::new('id', cookie_id)
				c1.expires = Time::parse(cookie_tms)
				c2.expires = Time::parse(cookie_tms)
				path = /^https?\:\/\/[^\/]+/ === @baseuri ? $' : @baseuri
				c1.path = path
				c2.path = path
			end
			@user = login_user.untaint
		end
		def authorization_required
			case @conf['AUTHTYPE']
			when 'form' then
				body = rhtml('authform')
			when 'vikiwiki' then
				realm = "VikiWiki"
				@header['nph']=true
				@header['status']='AUTH_REQUIRED'
				@header['WWW-Authenticate']=%Q!Basic realm="#{realm}"!
				body =rhtml('authbasic')
			else
				body = rhtml('deny')
			end
			return getHTTPheader(HTML) + getHTTPbody(body)
		end
		def access_denied
			return getHTTPheader(HTML) + getHTTPbody(rhtml('deny'))
		end
		def accessable?(type, target, mode)
			return true unless @acl
			return true if @conf['AUTHTYPE'] == 'none'
			res = true
			@acl.each_all{|ac|
				next unless ac.type === type
				next unless ac.target === target
				res = false
				case ac.users
				when Array then
					next unless ac.users.include?(self.user)
				when Regexp then
					next unless ac.users === self.user
				when String then
					next unless ac.users == self.user
				when nil then
				else
					next
				end
				res = true
				res &= ac.r if mode.include?('r')
				res &= ac.w if mode.include?('w')
				return true if res
			}
			if res == false then
				raise AuthorizationRequired unless self.user
				return true if @admins.include?(self.user)
			end
			return res
		end
		def index_uri
			@static ? @conf['STATICURI'] : @script_name
		end
		def add_onload_script(func)
			@onload_scripts << func unless @onload_scripts.include?(func)
		end
		def add_script(name, attr={})
			@scripts = Array::new unless @scripts
			return if @scripts.include?(name)
			@scripts << name
			attr['type'] = 'text/javascript'
			attr['src'] = @baseuri + "#{SCRIPTDIR}/#{name}.js"
			dfile = @basedir + "#{SCRIPTDIR}/#{name}.js"
			mfile = [MASTERDIR, SCRIPTDIR, "#{name}.js"].path
			if not File::exist?(dfile) or File::stat(dfile).mtime < File::stat(mfile).mtime then
				File::mkpath(File::dirname(dfile))
				File::copyp(mfile, dfile)
			end
			if @static then
				attr['src'] = "#{@conf['STATICURI']}/script/#{name}.js"
				sfile = "#{@conf['STATICDIR']}/script/#{name}.js"
				if not File::exist?(sfile) or File::stat(sfile).mtime < File::stat(dfile).mtime then
					File::mkpath(File::dirname(sfile))
					File::copyp(dfile, sfile)
				end
			end
			@html_headers << CGI::element('script', attr)
			@html_headers.uniq!
		end
		#
		# Hook processes
		#
		def static_update
			# StaticWiki
			return false unless @conf['STATICDIR']
			return false unless @page.exist?
			if @conf['EDITFILEFROM'] and @conf['EDITFILETO'] then
				@links.update(@page, @pages.names)
				#text = @page.load
				@page.write(nil, nil, true)
			elsif @page.static and @page.static.exist? then
				@page.static.write
			end
			@header['Pragma'] = 'no-chache'
			@header['Cache-control'] = 'no-cache'
			if @conf['STATICWIKI'] and @page.static.exist? then
				uri = @page.static.uri
			else
				uri = @page.uri
			end
			raise Location, uri
		end
	end
end
