require 'wiki/wikistyle'

module Wiki
	class VikiWikiStyle < WikiStyle
		#*******************************************************************
		# Format Rule
		#*******************************************************************
		#-------------------------------------------------------------------
		# Block
		#-------------------------------------------------------------------

		# <pre>
		FM_PRE_S         = '{{{'
		FM_PRE_E         = '}}}'
		RE_PRE_S         = /^\{{3}$/
		RE_PRE_E         = /^\}{3}$/

		# <h1> to <h6>

		# <hr />

		# <div class="">
		FM_DIV_S         = '<<<%s'
		FM_DIV_E         = '>>>%s'
		RE_DIV_S         = /^<<<([A-Za-z][A-Za-z\-_0-9]*)?$/
		RE_DIV_E         = /^>>>([A-Za-z][A-Za-z\-_0-9]*)?$/

		# <ul>
		FM_UL            = '*%s'
		RE_UL            = /^[\*-](.*)$/

		# <ol>
		FM_OL            = '+%s'
		RE_OL            = /^(?:\+|\d\.\s)(.*)$/

		# <dl>
		FM_DL            = ':%s:%s'
		RE_DL            = /^:([^:]+):(.*)$/
		FM_DL2           = '::%s::%s'
		RE_DL2           = /^::(.+)::(.*)$/

		# <blockquote>
		FM_BLOCKQUOTE    = '>%s'
		RE_BLOCKQUOTE    = /^>(.+)$/

		# <table border="1">
		TX_TABLE         = '||'
		RE_TABLE         = /^\|\|(.+)$/

		# <table>
		TX_TABLE_N       = '|'
		RE_TABLE_N       = /^\|([^\|].*)$/

		# <td>
		FM_TD_LEFT       = '%s '
		FM_TD_CENTER     = ' %s '
		FM_TD_RIGHT      = ' %s'
		RE_TD_LEFT       = /^(.+)\s$/
		RE_TD_CENTER     = /^\s(.+)\s$/
		RE_TD_RIGHT      = /^\s(.+)$/

		# <!-- -->
		FM_COMMENT       = '//%s'
		RE_COMMENT       = /^\/\/(.+)$/

		#-------------------------------------------------------------------
		# Inline
		#-------------------------------------------------------------------

		# WikiName

		# BracketName

		# AliasName

		# <em>

		# <strong>

		# <del>

		# <a>
		FM_A_IMG         = '[%s %s %s]'

		# URL

		# escape
		FM_ESCAPE        = '{{%s}}'
		RE_ESCAPE        = /^\{{2}(#{TX_NOPARE_M})\}{2}/

		# <img>
		FM_IMG           = '[%s %s]'
		RE_IMG           = /^\[(#{TX_NOBLANK}#{TX_IMGEXT})\s+(#{TX_NOPARE_L})\]/

		#-------------------------------------------------------------------
		# Plugin
		#-------------------------------------------------------------------

		# plugin

		# plugin (inline)

		# plugin parameters

		#*******************************************************************
		# Definitions
		#*******************************************************************
		BLOCK_PARSING_ORDER = [
			'plugin','div','pre','blockquote','comment',
			'h6','h5','h4','h3','h2',
			'hr','ul','ol','dl2','dl','table','table_n','p'
		]
		INLINE_PARSING_ORDER = [
			'wikiname','url','inplugin','em','strong','del',
			'escape','aliasname','bracketname','img','a'
		]
		ELEMENT_GENERATING_TAGS = [
			'wikiname', 'plugin', 'escape', 'del', 'strong',
			'em', 'img', 'a',
			'pre', 'h2', 'h3', 'h4', 'h5', 'h6', 'hr',
			'div', 'li', 'dl', 'blockquote',
			'table', 'tr', 'p'
		]
		INDENT_STYLE = true
		ALIAS_FORWARD = false

		#*******************************************************************
		# Block Parser
		#*******************************************************************
		def div_onparse(root, match, line, indent, lines)
			ndiv = appendChild_block(root, 'div', {'class'=>match[1]}, nil, indent)
			parse_block(ndiv, lines, @my::RE_DIV_E)
			return ndiv.parentNode
		end
		def pre_onparse(root, match, line, indent, lines)
			text = ''
			until lines.empty? do
				wline = lines.shift
				wsentence, windent = split_indent(wline)
				break if windent == indent and @my::RE_PRE_E === wsentence
				text << wline
			end
			npre = appendChild_block(root, 'pre', nil, nil, indent)
			npre.appendTextSimply(text)
			return npre.parentNode
		end
		def blockquote_onparse(root, match, line, indent, lines)
			text = [match[1]]
			while @my::RE_BLOCKQUOTE === lines.first do
				text << $1
				lines.shift
			end
			nbq = appendChild_block(root, 'blockquote', nil, nil, indent)
			appendChild_block(nbq, 'p', nil, text.join("\n"), indent, APPEND_CHILD)
			return nbq
		end
		def comment_onparse(root, match, line, indent, lines)
			root.appendCommentSimply(match[1])
			return root
		end
		def li_onparse(root, match, line, indent, lines, tag='ul')
			text = match[1]
			nlist = appendChild_block(root, tag, nil, nil, indent, APPEND_IF_NOT_EXIST)
			appendChild_block(nlist, 'li', nil, text, indent, APPEND_CHILD)
		end
		def dl_onparse(root, match, line, indent, lines)
			dttext, ddtext = match[1], match[2]
			root = appendChild_block(root, 'dl', nil, nil, indent, APPEND_IF_NOT_EXIST)
			appendChild_block(root, 'dt', nil, dttext, indent, APPEND_CHILD)
			appendChild_block(root, 'dd', nil, ddtext, indent, APPEND_CHILD)
		end
		alias dl2_onparse dl_onparse
		def table_n_onparse(root, match, line, indent, lines)
			table_onparse(root, match, line, indent, lines, false)
		end
		def table_onparse(root, match, line, indent, lines, border=true)
			nlast = root
			trtext = match[1]
			if border then
				attr = {'border'=>1}
				tdsplit = /#{Regexp::escape(@my::TX_TABLE)}/
				re_table = @my::RE_TABLE
			else
				attr = {'class'=>'noborder'}
				tdsplit = /#{Regexp::escape(@my::TX_TABLE_N)}/
				re_table = @my::RE_TABLE_N
			end
			ntable = appendChild_block(root, 'table', attr, nil, indent)
			while trtext do
				ntr = appendChild_block(ntable, 'tr', nil, nil, indent, APPEND_CHILD)
				tds = trtext.split(tdsplit, -1)
				tds.pop if tds.last.empty?
				tds.each do |tdtext|
					tag = 'td'
					if @my::RE_STRONG === tdtext.strip then
						tag, tdtext = 'th', $1
					end
					attr = nil
					case tdtext
					when @my::RE_TD_CENTER then
						tdtext, attr = $1, {'style' => 'text-align:center;'}
					when @my::RE_TD_LEFT then
						tdtext, attr = $1, {'style' => 'text-align:left;'}
					when @my::RE_TD_RIGHT then
						tdtext, attr = $1, {'style' => 'text-align:right;'}
					end
					nlast = appendChild_block(ntr, tag, attr, tdtext, indent, APPEND_CHILD)
				end
				break if lines.empty?
				wsentence, windent = split_indent(lines.first)
				break unless indent == windent and re_table === wsentence
				trtext = $1
				lines.shift
			end
			return nlast
		end

		#*******************************************************************
		# Inline Parser
		#*******************************************************************
		def escape_onparse(root, match)
			appendChild_inline(root, 'escape', nil, match[1])
		end

		#*******************************************************************
		# Wiki Text Block Generator
		#*******************************************************************
		def _comment_ontext(node, outobj='')
			node.nodeValue.each do |line|
				outobj << text_format(@my::FM_COMMENT, line)
			end
			outobj << "\n"
		end
		def pre_ontext(node, outobj='')
			outobj << indent_line(node)
			outobj << text_format(@my::FM_PRE_S)
			outobj << "\n"
			outobj << generateChild(node).chomp
			outobj << "\n"
			outobj << indent_line(node)
			outobj << text_format(@my::FM_PRE_E)
			outobj << "\n"
			return outobj
		end
		def div_ontext(node, outobj='')
			cls = node.getAttribute('class')
			outobj << indent_line(node)
			outobj << text_format(@my::FM_DIV_S, cls)
			outobj << "\n"
			generateBlock(node.childNodes, outobj)
			outobj << "\n"
			outobj << indent_line(node)
			outobj << text_format(@my::FM_DIV_E, cls)
			outobj << "\n"
			return outobj
		end
		def li_ontext(node, outobj='')
			parent = node.parentNode
			fmt = parent.nodeName == 'ol' ? @my::FM_OL : @my::FM_UL
			outobj << indent_line(node)
			text, ncur = generateInline(node)
			outobj << text_format(fmt, text)
			outobj << "\n"
			generateSurplus(ncur, outobj)
			return outobj
		end
		def dl_ontext(node, outobj='')
			list = node.childNodes
			dttext = ''
			0.upto(list.length-1) do |i|
				nelm = list.item(i)
				text, ncur = generateInline(nelm)
				if nelm.nodeName == 'dt' then
					dttext = text
				else
					outobj << indent_line(node)
					str = text_format(@my::FM_DL, dttext, text)
					if @my::RE_DL === str and $1 == dttext then
						outobj << str
					else
						outobj << text_format(@my::FM_DL2, dttext, text)
					end
					outobj << "\n"
					generateSurplus(ncur, outobj)
				end
			end
			return outobj
		end
		def blockquote_ontext(node, outobj='')
			split_if_same(node, outobj)
			indent = indent_line(node)
			text, ncur = generateFirst(node)
			text.each do |line|
				outobj << indent
				outobj << text_format(@my::FM_BLOCKQUOTE, line)
			end
			generateSurplus(ncur, outobj)
			return outobj
		end
		def table_ontext(node, outobj='')
			split_if_same(node, outobj)
			generateBlock(node.childNodes, outobj)
		end
		def tr_ontext(node, outobj='')
			ntbl = node
			ntbl = ntbl.parentNode while ntbl.parentNode and ntbl.nodeName != 'table'
			if ntbl.nodeName == 'table' and
				(ntbl.getAttribute('border').to_i > 0 or
				 ntbl.getAttribute('class') != 'noborder') then
				sep = @my::TX_TABLE
			else
				sep = @my::TX_TABLE_N
			end
			outobj << indent_line(node)
			list = node.childNodes
			0.upto(list.length-1) do |i|
				nelm = list.item(i)
				text = generateBlock(nelm.childNodes)
				align = get_align(nelm)
				case align
				when 'left' then
					text = text_format(@my::FM_TD_LEFT, text)
				when 'center' then
					text = text_format(@my::FM_TD_CENTER, text)
				when 'right' then
					text = text_format(@my::FM_TD_RIGHT, text)
				end
				outobj << sep
				if nelm.nodeName == 'th' then
					outobj << text_format(@my::FM_STRONG, text)
				else
					outobj << text
				end
			end
			outobj << "\n"
			return outobj
		end
		def p_ontext(node, outobj='')
			split_if_same(node, outobj)
			indent = indent_line(node)
			generateChild(node).each do |line|
				outobj << indent
				outobj << line
			end
			outobj << "\n"
			return outobj
		end

		#*******************************************************************
		# Wiki Text Inline Generator
		#*******************************************************************
		def escape_ontext(node, outobj='')
			outobj << text_format(@my::FM_ESCAPE, generateChild(node))
			return outobj
		end
		def a_ontext(node, outobj='')
			return outobj if wikiname_ontext(node, outobj)
			href = node.getAttribute('href')
			nchd = node.firstChild
			if nchd and nchd.nodeName == 'img' then
				src = nchd.getAttribute('src')
				alt = nchd.getAttribute('alt')
				outobj << text_format(@my::FM_A_IMG, href, src, alt)
			else
				text = generateChild(node)
				if href == text then
					outobj << text_format(@my::FM_URL, href)
				else
					outobj << text_format(@my::FM_A, href, text)
				end
			end
			return outobj
		end

		#*******************************************************************
		# Wiki Text Plugin Generator
		#*******************************************************************

	end
end
