/***************************************************************************
                          wvoxstorm.cpp  -  description
                             -------------------
    begin                : 2004ǯ  5  4  22:42:12 JST
    copyright            : (C) 2004 by Tomoaki Murakami
    email                : solarisphere@yahoo.co.jp
 ***************************************************************************/

/***************************************************************************
 *   Copyright (C) 2004 by Tomoaki Murakami                                *
 *   solarisphrere@yahoo.co.jp                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <unistd.h>
#include <sys/wait.h>
#include <signal.h>
#include <stdlib.h>
#include <qfileinfo.h>
#include <qstrlist.h>
#include <qstringlist.h>
#include <kaccel.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kprocess.h>
#include <qpopupmenu.h>
#include "wvoxstorm.h"
#include "wmixdowndialog.h"
#include "cvoxstormglobal.h"
#include "worderdialog.h"

/////////////////////////////////////////////////////////////////////////////

WVoxStorm::WVoxStorm(QWidget *WidgetParent, const char *cName)
  : KMainWindow(WidgetParent, cName), bPause(false)
{
  initialize();

  bDirty = false;
  slotNew();
}

/////////////////////////////////////////////////////////////////////////////

WVoxStorm::WVoxStorm(
  const QString& strFileName, 
  QWidget* WidgetParent, 
  const char* cName)
  : KMainWindow(WidgetParent, cName), bPause(false)
{
  initialize();

  loadFile(strFileName);
  bDirty = false;
}

/////////////////////////////////////////////////////////////////////////////

WVoxStorm::~WVoxStorm()
{
  writeConfigurationFile();

  if ( pDocVox != NULL ) {
    delete pDocVox;
    pDocVox = new QDomDocument;
  }

  delete ToggleActionRec;
  delete ToggleActionInteractive;
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::initialize()
{
  setupActions();
  setupStatusBar();

  readConfigurationFile();

  VoxBase = new WVoxBase(this, "VoxBase");
  setCentralWidget(VoxBase);

  connect(VoxBase, SIGNAL(playbackDeviceChanged(const QString&, const QString&)),
          this, SLOT(slotChangePlaybackDevice(const QString&, const QString&)));
  connect(VoxBase, SIGNAL(recordingDeviceChanged(const QString&, const QString&)),
          this, SLOT(slotChangeRecordingDevice(const QString&, const QString&)));
  connect(VoxBase, SIGNAL(trackRemoved(const QString&)),
          this, SLOT(slotRemoveTrack(const QString&)));
  connect(VoxBase, SIGNAL(fileNameChanged(const QString&, const QString&)),
          this, SLOT(slotChangeFileName(const QString&, const QString&)));
  connect(VoxBase, SIGNAL(levelChanged(const QString&, int)),
          this, SLOT(slotChangeLevel(const QString&, int)));
  connect(VoxBase, SIGNAL(panChanged(const QString&, int)),
          this, SLOT(slotChangePan(const QString&, int)));
  connect(VoxBase, SIGNAL(muteToggled(const QString&, bool)),
          this, SLOT(slotToggleMute(const QString&, bool)));
  connect(VoxBase, SIGNAL(recToggled(const QString&, bool)),
          this, SLOT(slotToggleRec(const QString&, bool)));
  connect(VoxBase, SIGNAL(startPositionChanged(const QString&, const QTime&)),
          this, SLOT(slotChangeStartPosition(const QString&, const QTime&)));
  connect(VoxBase, SIGNAL(offsetChanged(const QString&, const QTime&)),
          this, SLOT(slotChangeOffset(const QString&, const QTime&)));
  connect(VoxBase, SIGNAL(lengthChanged(const QString&, const QTime&)),
          this, SLOT(slotChangeLength(const QString&, const QTime&)));

  connect(VoxBase, SIGNAL(masterPositionChanged(const QTime&)),
          &ecaIAManager, SLOT(slotSetMasterPosition(const QTime&)));
  connect(VoxBase, SIGNAL(stateChanged()), 
          this, SLOT(slotChangedState()));
  connect(&ecaIAManager, SIGNAL(masterPositionChanged(const QTime&)),
          VoxBase, SLOT(slotChangePosition(const QTime&)));
  connect(&ecaIAManager, SIGNAL(finished()), this, SLOT(slotFinish()));
  connect(&ecaIAManager, SIGNAL(messageGenerated(const QString&)),
          VoxBase, SLOT(slotWriteConsoleMessage(const QString&)));
  connect(&ecaIAManager, SIGNAL(moved(const QTime&)), 
          VoxBase, SLOT(slotNotifyCurrentTime(const QTime&)));

  pDocVox = NULL;

  applyPreferences();
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::readConfigurationFile()
{
  KConfig  *Config = KGlobal::config();
  QString   strTmp;
  QStrList  strListTmp;

  // ---->>>> setting for preferences member
  Config->setGroup("Devices");
  preferences.strPlayDevice = Config->readEntry("PlaybackDevice", "/dev/dsp");
  preferences.strRecDevice  = Config->readEntry("RecDevice", "/dev/dsp");

  Config->setGroup("File");
  strTmp = Config->readEntry("FileType", "WAV");
  if ( strTmp == "WAV" ) {
    preferences.FileType = CVoxStormGlobal::ecaWAV;
  } else {
    preferences.FileType = CVoxStormGlobal::ecaMP3;
  }

  Config->setGroup("Project");
  preferences.strProjectPath = Config->readEntry("ProjectPath", "~/voxstorm");

  Config->setGroup("Sound Spec");
  preferences.format.setSampleFormat(
    CSourceFormat::sampleFormatIdentifier(
      Config->readEntry("SamplingFormat", "se16_le")));
  preferences.format.setChannelCount(Config->readNumEntry("Channels", 2));
  
  preferences.format.setSampleRate(Config->readNumEntry("SamplingRate", 44100));
  preferences.uiBuffer     = Config->readNumEntry("BufferSize", 1024);

  Config->setGroup("Track");
  Config->readListEntry("DefaultTracks", strListTmp, ',');
  preferences.strlDefaultTracks.clear();
  preferences.strlDefaultTracks = QStringList::fromStrList(strListTmp);

  Config->setGroup("Generic");
  preferences.iFrameSeconds = Config->readNumEntry("FrameSeconds", 5);
  QTime timeTmp;
  preferences.timeSessionMax = 
    timeTmp.addSecs(Config->readNumEntry("SessionMaxSecs", 1000));
  preferences.strMixerPath = Config->readEntry("MixerPath", "");
  // <<<<---- setting for preferences member

  // ---->>>> setting for Mainwindow
  Config->setGroup("Mainwindow");
  setGeometry(Config->readNumEntry("Left", 0),
              Config->readNumEntry("Top", 0),
              1, 1);  // Width and Height will be read after soon
  applyMainWindowSettings(Config);
  restoreWindowSize(Config);
  ToggleActionInteractive->setChecked(true);
  setWidgetsByIAState();
  // <<<<---- setting for Mainwindow
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::writeConfigurationFile()
{
  KConfig  *Config = KGlobal::config();

  // ---->>>> setting for Mainwindow
  Config->setGroup("Devices");
  Config->writeEntry("PlayDevice", preferences.strPlayDevice);
  Config->writeEntry("RecDevice", preferences.strRecDevice);

  Config->setGroup("File");
  if ( preferences.FileType == CVoxStormGlobal::ecaWAV ) {
    Config->writeEntry("FileType", "WAV");
  } else {
    Config->writeEntry("FileType", "MP3");
  }

  Config->setGroup("Project");
  Config->writeEntry("ProjectPath", preferences.strProjectPath);

  Config->setGroup("Sound Spec");
  Config->writeEntry(
    "SamplingFormat", 
    CSourceFormat::sampleFormatString(preferences.format.sampleFormat()));
  Config->writeEntry("Channels", preferences.format.channelCount());
  Config->writeEntry("SamplingRate", preferences.format.sampleRate());
  Config->writeEntry("BufferSize", preferences.uiBuffer);

  QStrList  strListTmp;
  int       i, nTrack;
  Config->setGroup("Track");
  nTrack = preferences.strlDefaultTracks.count();
  for ( i = 0; i < nTrack; i++ ) {
    strListTmp.append(preferences.strlDefaultTracks[i]);
  }
  Config->writeEntry("DefaultTracks", strListTmp, ',');

  Config->setGroup("Generic");
  Config->writeEntry("FrameSeconds", preferences.iFrameSeconds);
  QTime timeTmp;
  Config->writeEntry("SessionMaxSecs", 
                     timeTmp.secsTo(preferences.timeSessionMax));
  Config->writeEntry("MixerPath", preferences.strMixerPath);
  // <<<<---- setting for Mainwindow

  // ---->>>> setting for Mainwindow
  Config->setGroup("Mainwindow");
  Config->writeEntry("Left", geometry().x());
  Config->writeEntry("Top", geometry().y());
  saveMainWindowSettings(Config);
  saveWindowSize(Config);
  Config->writeEntry("Interactive", ToggleActionInteractive->isChecked());
  // <<<<---- setting for Mainwindow

  Config->sync();
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::applyPreferences()
{
  ecaIAManager.setBufferSize(preferences.uiBuffer);
  ecaIAManager.setSoundSourceFormat(preferences.format);
  VoxBase->setMaxSessionLength(preferences.timeSessionMax);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::setupActions()
{
  QPopupMenu* PopupMenu = new QPopupMenu;
  KAction* Action;

  Action = KStdAction::openNew(this, SLOT(slotNew()), actionCollection());
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  Action = KStdAction::open(this, SLOT(slotOpen()), actionCollection());
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  PopupMenu->insertSeparator();

  Action = KStdAction::save(this, SLOT(slotSave()), actionCollection());
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  Action = KStdAction::saveAs(this, SLOT(slotSaveAs()), actionCollection());
  Action->plug(PopupMenu);

  PopupMenu->insertSeparator();
      
  Action = KStdAction::quit(this, SLOT(close()), actionCollection());
  Action->plug(PopupMenu);

  menuBar()->insertItem(i18n("&File"), PopupMenu);
  toolBar()->insertLineSeparator();

  PopupMenu = new QPopupMenu;

  ActionPlay = new KAction(
                       i18n("&Play"),
                       "voxstorm_play",
                       KShortcut(CTRL+Key_P),
                       this,
                       SLOT(slotPlay()),
                       actionCollection(),
                       "Play");
  ActionPlay->plug(PopupMenu);
  ActionPlay->plug(toolBar());

  ToggleActionRec = new KToggleAction(
                       i18n("&Record"),
                       "voxstorm_record",
                       KShortcut(CTRL+Key_R),
                       this,
                       SLOT(slotRecord()),
                       this,
                       "Record");
  ToggleActionRec->plug(PopupMenu);
  ToggleActionRec->plug(toolBar());

  Action = new KAction(i18n("P&ause"),
                       "voxstorm_pause",
                       KShortcut(CTRL+Key_A),
                       this,
                       SLOT(slotPause()),
                       actionCollection(),
                       "Pause");
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  Action = new KAction(i18n("&Stop"),
                       "voxstorm_stop",
                       KShortcut(CTRL+Key_D),
                       this,
                       SLOT(slotStop()),
                       actionCollection(),
                       "Stop");
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  PopupMenu->insertSeparator();

  ActionJumpToStart = new KAction(i18n(
                       "&Jump to Start"),
                       "voxstorm_start",
                       0,
                       this,
                       SLOT(slotJumpToStart()),
                       actionCollection(),
                       "Jump To Start");
  ActionJumpToStart->plug(PopupMenu);
  ActionJumpToStart->plug(toolBar());

  ActionRewind = new KAction(
                       i18n("Re&wind"),
                       "voxstorm_rewind",
                       0,
                       this,
                       SLOT(slotRewind()),
                       actionCollection(),
                       "Rewind");
  ActionRewind->plug(PopupMenu);
  ActionRewind->plug(toolBar());

  ActionForward = new KAction(
                       i18n("&Forward"),
                       "voxstorm_forward",
                       0,
                       this,
                       SLOT(slotForward()),
                       actionCollection(),
                       "Forward");
  ActionForward->plug(PopupMenu);
  ActionForward->plug(toolBar());

  PopupMenu->insertSeparator();

  ActionMixdown = new KAction(
                       i18n("&Mixdown..."), 
                       "voxstorm_mixdown", 
                       0, 
                       this, 
                       SLOT(slotMixdown()), 
                       actionCollection(), 
                       "Mixdown");
  ActionMixdown->plug(PopupMenu);

  menuBar()->insertItem(i18n("&Operation"), PopupMenu);

  PopupMenu = new QPopupMenu;

  Action = new KAction(i18n("&Add Track..."),
                       NULL,
                       0,
                       this,
                       SLOT(slotAddTrack()),
                       actionCollection(),
                       "Add Track");
  Action->plug(PopupMenu);

  ActionTrackOrder = new KAction(i18n("&Track Order"),
                       NULL,
                       0,
                       this,
                       SLOT(slotTrackOrder()),
                       actionCollection(),
                       "Track Order");
  ActionTrackOrder->plug(PopupMenu);

  menuBar()->insertItem(i18n("&Track"), PopupMenu);

  PopupMenu = new QPopupMenu;
  
  toolBar()->insertLineSeparator();

  Action = new KAction(i18n("Run &Mixer Program"), 
                       "voxstorm_mixer", 
                       0, 
                       this, 
                       SLOT(slotRunMixer()), 
                       actionCollection(), 
                       "Run Mixer Program");
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  menuBar()->insertItem(i18n("&Tools"), PopupMenu);

  PopupMenu = new QPopupMenu;

  toolBar()->insertLineSeparator();

  ToggleActionInteractive = new KToggleAction(
                       i18n("&Interactive Mode"),
                       "voxstorm_interactive",
                       0,
                       this,
                       SLOT(slotInteractive()),
                       this,
                       "Interactive");
  ToggleActionInteractive->plug(PopupMenu);
  ToggleActionInteractive->plug(toolBar());
  ToggleActionInteractive->setEnabled(false);

  Action = KStdAction::preferences(this, SLOT(slotPreferences()), actionCollection());
  Action->plug(PopupMenu);

  menuBar()->insertItem(i18n("&Settings"), PopupMenu);

  QString strHelp =
            i18n("VoxStorm 0.3\n\n"
                 "(C) 2003-2004 Tomoaki Murakami\n"
                 "solarisphere@yahoo.co.jp\n"
                 "Simple MTR using Ecasound on KDE\n");

  PopupMenu = (QPopupMenu *)helpMenu(strHelp);

  menuBar()->insertItem(i18n("&Help"), PopupMenu);

  menuBar()->show();
  toolBar()->show();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::setupStatusBar()
{
  LabelStatusBar = new QLabel("xxxxxxxxxxxxxxxxxxxxx", this);
  LabelStatusBar->setMinimumSize(LabelStatusBar->sizeHint());
  LabelStatusBar->setText("");
  statusBar()->addWidget(LabelStatusBar);
  statusBar()->setSizeGripEnabled(true);
  statusBar()->show();
}

/////////////////////////////////////////////////////////////////////////////

bool
WVoxStorm::queryClose()
{
  if ( maybeSave() ) {
    QApplication::setOverrideCursor(waitCursor);
    slotStop();
    if ( ToggleActionInteractive->isChecked() ) {
      ecaIAManager.slotReleaseCurrentChainsetup();
    }
    QApplication::restoreOverrideCursor();

    return true;
  }

  return false;
}

/////////////////////////////////////////////////////////////////////////////

bool
WVoxStorm::maybeSave()
{
  if ( isChanged() ) {
    int iResult;

    iResult = KMessageBox::questionYesNoCancel(
                this,
                i18n("The session has been modified.\n"
                     "Do you want to save your change?"));

    switch ( iResult ) {
    case KMessageBox::Yes:
      return slotSave();
    case KMessageBox::No:
      break;
    case KMessageBox::Cancel:
      return false;
    }
  }

  return true;
}

/////////////////////////////////////////////////////////////////////////////

QString
WVoxStorm::sessionName() const
{
  if ( strSessionFile.isEmpty() ) {
    return kapp->caption();
  }

  QFileInfo FileInfo(strSessionFile);
  return FileInfo.fileName();
}

/////////////////////////////////////////////////////////////////////////////

bool 
WVoxStorm::isChanged() const
{
  return bDirty;
}

/////////////////////////////////////////////////////////////////////////////

QString
WVoxStorm::createNewUntitledName() const
{
  static unsigned int unID;

  unID++;
  return QString("%1%2").arg(i18n("Untitled")).arg(unID);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::loadFile(const QString& strFileName)
{
  QApplication::setOverrideCursor(waitCursor);
  LabelStatusBar->setText(i18n("Loading..."));

  QDomDocument doc;
  QFile File(strFileName);
  if ( !File.open(IO_ReadOnly) ) {
    LabelStatusBar->setText("");
    QApplication::restoreOverrideCursor();
    return;
  }
  if ( !doc.setContent(&File) ) {
    File.close();
    LabelStatusBar->setText("");
    QApplication::restoreOverrideCursor();
    return;
  }
  File.close();

  VoxBase->clearTracks();
  strSessionFile = strFileName;
  QFileInfo FileInfo(strFileName);
  setCaption(FileInfo.fileName());

  QDomElement eleRoot = doc.documentElement();
  VoxBase->addTracks(eleRoot.firstChild());

  enterIAMode(ToggleActionInteractive->isChecked());

  LabelStatusBar->setText("");
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::saveFile(const QString& strFileName)
{
  QApplication::setOverrideCursor(waitCursor);
  LabelStatusBar->setText("Saving...");

  if ( pDocVox != NULL ) {
    delete pDocVox;
  }
  pDocVox = new QDomDocument;

  // root node
  {
    QDomNode nodRoot(stateInDOM());
    pDocVox->appendChild(nodRoot);
  }

  // processing instruction
  {
    QDomNode nodPI = pDocVox->createProcessingInstruction(
                       "xml", 
                       "version=\"1.0\"");
    pDocVox->insertBefore(nodPI, pDocVox->firstChild());
  }

  QFile File(strFileName);
  if ( !File.open(IO_WriteOnly) ) {
    delete pDocVox;
    pDocVox = NULL;
    LabelStatusBar->setText("");
    QApplication::restoreOverrideCursor();
    return;
  }
  QTextStream out(&File);
  pDocVox->save(out, 4);
  delete pDocVox;
  pDocVox = NULL;

  bDirty = false;

  LabelStatusBar->setText("");
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::enterIAMode(bool bEnter)
{
  if ( bEnter ) {
    ecaIAManager.slotCreateNewChainsetup("VoxStorm");
  } else {
    // clear chainsetup
    ecaIAManager.slotReleaseCurrentChainsetup();
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::setWidgetsByIAState()
{
  if ( ToggleActionInteractive->isChecked() ) {
  } else {
  }
}

/////////////////////////////////////////////////////////////////////////////

QDomNode 
WVoxStorm::stateInDOM()
{
  QDomElement eleSession = pDocVox->createElement("Session");

  // child node
  {
    QDomNode nodChild = VoxBase->stateInDOM(pDocVox);
    eleSession.appendChild(nodChild);
  }

  return eleSession;
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::slotNew()
{
  if ( maybeSave() ) {
    setCaption(createNewUntitledName(), false);
    strSessionFile = "";

    VoxBase->clearTracks();

    int     i, nTrack;
    QString strNewTrack;
    nTrack = preferences.strlDefaultTracks.count();
    for ( i = 0; i < nTrack; i++ ) {
      strNewTrack = preferences.strlDefaultTracks[i];
      if ( !VoxBase->queryExistingTrackName(strNewTrack) ) {
        VoxBase->addTrack(preferences.strlDefaultTracks[i],
                          preferences.strPlayDevice,
                          preferences.strRecDevice);
      }
    }

    if ( ToggleActionInteractive->isChecked() ) {
      enterIAMode(true);
    }
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotOpen()
{
  if ( maybeSave() ) {
    QString strTmpPath = CVoxStormGlobal::absPath(preferences.strProjectPath);
    if ( !QFileInfo(strTmpPath).exists() ) {
      strTmpPath = "";
    }

    KURL url = KFileDialog::getOpenURL(
                 strTmpPath, 
                 i18n("*.voxs|VoxStorm session files (*.voxs)\n*|All files (*)"), 
                 NULL, 
                 i18n("Open VoxStorm session file"));
    if ( !url.isEmpty() ) {
      if ( url.isLocalFile() ) {
	loadFile(url.path());	
      } else {
	KMessageBox::sorry(this, i18n("VoxStorm cannot handle remote files yet."));
      }
    }
  }
}

/////////////////////////////////////////////////////////////////////////////

bool 
WVoxStorm::slotSave()
{
  if ( strSessionFile.isEmpty() ) {
    return slotSaveAs();
  } else {
    saveFile(strSessionFile);

    return true;
  }
}

/////////////////////////////////////////////////////////////////////////////

bool 
WVoxStorm::slotSaveAs()
{
  QString strTmpPath = preferences.strProjectPath;

  if ( !CVoxStormGlobal::queryAndCreateDir(
         this, strTmpPath) ) {
    strTmpPath = QString::null;
  } else {
    // to abs
    strTmpPath = CVoxStormGlobal::absPath(strTmpPath);
  }

  KFileDialog dialog(strTmpPath,
                     i18n("*.voxs|VoxStorm session files (*.voxs)"), 
                     NULL,
                     i18n("Save VoxStorm session file"), 
                     true);
  dialog.exec();
  QString strFileName = dialog.selectedFile();
  if ( strFileName.isEmpty() ) {
    return false;
  }

  QFileInfo FileInfo(strFileName);
  // modify extension if needed
  if ( FileInfo.extension(false) != "voxs" ) {
    FileInfo.setFile(strFileName + ".voxs");
  }

  if ( FileInfo.exists() ) {
    if ( KMessageBox::warningYesNo(
           this, 
           QString(i18n("A file named \"%1\" already exists. "
                "Are you sure you want to overwrite it?").arg(FileInfo.filePath())), 
           "Overwrite File?") != KMessageBox::Yes ) {
      return false;
    }
  }

  saveFile(FileInfo.filePath());
  strSessionFile = FileInfo.filePath();
  setCaption(FileInfo.fileName());

  return true;
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotPlay()
{
  QApplication::setOverrideCursor(waitCursor);
  if ( ToggleActionRec->isChecked() ) {
    LabelStatusBar->setText(i18n("Recording..."));
  } else {
    LabelStatusBar->setText(i18n("Playing..."));
  }
  if ( ToggleActionInteractive->isChecked() ) {
    if ( bPause ) {  // PAUSE -> PLAY
      ActionPlay->setEnabled(false);
      ToggleActionRec->setEnabled(false);
      ActionTrackOrder->setEnabled(false);
      ActionMixdown->setEnabled(false);
      VoxBase->slotChangeState(WVoxBase::Play);
      VoxBase->slotChangeMessageTab(WVoxBase::Console);
      ecaIAManager.slotResume();
      bPause = false;
    } else {         // STOP  -> PLAY
      ecaIAManager.slotRemoveAllChains();
      int nCount = VoxBase->getTrackCount();
      int i;
      for ( i = 0; i < nCount; i++ ) {
        ecaIAManager.slotAddChain(VoxBase->getTrackInf(i));
      }

      ActionPlay->setEnabled(false);
      ToggleActionRec->setEnabled(false);
      ActionTrackOrder->setEnabled(false);
      ActionMixdown->setEnabled(false);
      VoxBase->slotChangeState(WVoxBase::Play);
      VoxBase->slotChangeMessageTab(WVoxBase::Console);
      ecaIAManager.slotPlay(ToggleActionRec->isChecked());
    }
  } else {
  }
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotRecord()
{
  ;
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotPause()
{
  QApplication::setOverrideCursor(waitCursor);
  if ( ToggleActionInteractive->isChecked() ) {
    ecaIAManager.slotPause();
    ActionPlay->setEnabled(true);
    VoxBase->slotChangeState(WVoxBase::Pause);
    LabelStatusBar->setText(i18n("Pausing..."));
    bPause = true;
  }
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotStop()
{
  QApplication::setOverrideCursor(waitCursor);
  if ( ToggleActionInteractive->isChecked() ) {
    ecaIAManager.slotStop();
  }
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotJumpToStart()
{
  QApplication::setOverrideCursor(waitCursor);
  if ( ToggleActionInteractive->isChecked() ) {
    ecaIAManager.slotSetMasterPosition(0);
  }
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotRewind()
{
  QApplication::setOverrideCursor(waitCursor);
  if ( ToggleActionInteractive->isChecked() ) {    
    ecaIAManager.slotRewind(preferences.iFrameSeconds);
  }
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotForward()
{
  QApplication::setOverrideCursor(waitCursor);
  if ( ToggleActionInteractive->isChecked() ) {
    ecaIAManager.slotForward(preferences.iFrameSeconds);
  }
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::slotMixdown()
{
  if ( ToggleActionRec->isChecked() ) {
    KMessageBox::warningYesNo(
      this, 
      i18n("Recording action is enabled now.\n"
           "Please be disabled Recording action to mixdown."));
    return;
  }

  WMixdownDialog mixdownDialog(this);

  if ( mixdownDialog.exec() != QDialog::Accepted ) {
    return;
  }

  QApplication::setOverrideCursor(waitCursor);

  LabelStatusBar->setText(i18n("Mixing..."));

  ecaIAManager.slotRemoveAllChains();
  int nCount = VoxBase->getTrackCount();
  int i;
  for ( i = 0; i < nCount; i++ ) {
    ecaIAManager.slotAddChain(VoxBase->getTrackInf(i));
  }

  ActionPlay->setEnabled(false);
  ToggleActionRec->setEnabled(false);
  ActionJumpToStart->setEnabled(false);
  ActionRewind->setEnabled(false);
  ActionForward->setEnabled(false);
  ActionTrackOrder->setEnabled(false);
  ActionMixdown->setEnabled(false);
  VoxBase->slotChangeState(WVoxBase::Play);
  VoxBase->slotChangeMessageTab(WVoxBase::Console);
  ecaIAManager.slotMixdown(
        mixdownDialog.mixesWithPlaying(), 
        "Mixdown", 
        mixdownDialog.fileName());
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotRunMixer()
{
  KShellProcess process;

  process << preferences.strMixerPath;
  process.start(KProcess::DontCare);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotPreferences()
{
  WPrefDialog prefDialog(this);
  prefDialog.setFields(preferences);
  if ( prefDialog.exec() == QDialog::Accepted ) {
    preferences = prefDialog.fields();
    applyPreferences();
  }

  writeConfigurationFile();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotInteractive()
{
  if ( ToggleActionInteractive->isChecked() ) {
    setWidgetsByIAState();
    // enter intaractive mode
    enterIAMode(true);
  } else {
    setWidgetsByIAState();
    // exit interactive mode
    enterIAMode(false);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotAddTrack()
{
  VoxBase->addTrack(preferences.strPlayDevice, preferences.strRecDevice);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotTrackOrder()
{
  WOrderDialog dialog(this);

  // set track names
  {
    QStringList strl;
    int i;
    int nCount = VoxBase->getTrackCount();
    for ( i = 0; i < nCount; i++ ) {
      strl.push_back(VoxBase->trackInfByDisplayOrder(i).strTrackName);
    }
    dialog.slotSetList(strl);
  }

  if ( dialog.exec() == QDialog::Accepted ) {
    VoxBase->slotChangeTrackOrder(dialog.order());
    slotChangedState();
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::slotChangePlaybackDevice(const QString& strTrackName,
                                    const QString& strPlaybackDevice)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    ecaIAManager.slotChangePlaybackDevice(strTrackName, strPlaybackDevice);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotChangeRecordingDevice(const QString& strTrackName, 
                                     const QString& strRecordingDevice)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    ecaIAManager.slotChangeRecordingDevice(strTrackName, strRecordingDevice);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotRemoveTrack(const QString& strTrackName)
{
  if ( ToggleActionInteractive->isChecked() ) {
    ecaIAManager.slotRemoveChain(strTrackName);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotChangeFileName(const QString& strTrackName, const QString&)
{
  if ( ToggleActionInteractive->isChecked() ) {
    if ( !ecaIAManager.isExisting(strTrackName) ) {
      ;
    } else {
      ;
    }
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotChangeLevel(const QString& strTrackName, int nLevel)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    ecaIAManager.slotChangeLevel(strTrackName, nLevel);
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::slotChangePan(const QString& strTrackName, int nPan)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    ecaIAManager.slotChangePan(strTrackName, nPan);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotToggleMute(const QString& strTrackName, bool bMute)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    ecaIAManager.slotToggleMute(strTrackName, bMute);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotToggleRec(const QString& strTrackName, bool bRec)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    ecaIAManager.slotToggleRec(strTrackName, bRec);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotChangeStartPosition(
             const QString& strTrackName,
             const QTime& TimeStartPosition)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    ecaIAManager.slotChangeStartPosition(strTrackName, TimeStartPosition);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotChangeOffset(
             const QString& strTrackName,
             const QTime& TimeOffset)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    ecaIAManager.slotChangeOffset(strTrackName, TimeOffset);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotChangeLength(
             const QString& strTrackName,
             const QTime& TimeLength)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    ecaIAManager.slotChangeLength(strTrackName, TimeLength);
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::slotChangedState()
{
  this->bDirty = true;
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotFinish()
{
  if ( ToggleActionRec->isChecked() ) {
    VoxBase->slotNotifySucceedInRecording();
  }
  VoxBase->slotChangeState(WVoxBase::Stop);
  ActionMixdown->setEnabled(true);
  ActionTrackOrder->setEnabled(true);
  ActionForward->setEnabled(true);
  ActionRewind->setEnabled(true);
  ActionJumpToStart->setEnabled(true);
  ToggleActionRec->setEnabled(true);
  ActionPlay->setEnabled(true);
  LabelStatusBar->setText("");
  bPause = false;
}

/////////////////////////////////////////////////////////////////////////////

#include "wvoxstorm.moc"
