/*
  {}`̕`
  Satofumi KAMIMURA
  $Id$
*/

#include "basicDraw.h"
#include "sdlVideo.h"
#include "deleteObjects.h"
#include "viewComponent.h"


VXV::Grid BasicDrawInterface::view_pos;
VXV::Rect BasicDrawInterface::view_range;
double BasicDrawInterface::view_magnify;


void BasicDrawInterface::setDrawPosition(const VXV::Grid& pos,
					 const VXV::Rect& range,
					 double magnify) {
  view_pos = pos;
  view_range = range;
  view_magnify = magnify;
}


void BasicDrawInterface::draw(const VXV::Grid& pos, const VXV::Rect& range,
			      double magnify) {
  setDrawPosition(pos, range, magnify);
  draw();
}


VXV::Rect BasicDrawInterface::calcPixelRange(const VXV::Grid& pos,
					const VXV::Rect& range,
					double magnify) {
  VXV::Rect pixel_range;
  pixel_range.x = pos.x;
  pixel_range.y = pos.y;
  pixel_range.w = pixel_range.x + range.w;
  pixel_range.h = pixel_range.y + range.h;

  return pixel_range;
}


void BasicDrawInterface::drawRangedPixel(const VXV::Rect& pixel_range,
					 int x, int y) {
  if ((x > pixel_range.x) && (x < pixel_range.w) &&
      (y > pixel_range.y) && (y < pixel_range.h)) {
    PixelDraw::setPixel(x, y);
  }
}


unsigned long PixelDraw::static_draw_color = White;


PixelDraw::PixelDraw(int x, int y, unsigned long color)
  : draw_pos(VXV::Grid(x, y)), draw_color(color) {
}


void PixelDraw::draw(void) {
  VXV::Grid& pos = view_pos;
  VXV::Rect& range = view_range;
  double magnify = view_magnify;

  VXV::Rect pixel_range = calcPixelRange(pos, range, magnify);
  int x = static_cast<int>(rint((draw_pos.x - range.x) * magnify)) + pos.x;
  int y = static_cast<int>(rint((-draw_pos.y - range.y) * magnify)) + pos.y;
  setColor(draw_color);
  drawRangedPixel(pixel_range, x, y);
}


void PixelDraw::setColor(unsigned long color) {
  static_draw_color = color;
}


void PixelDraw::setPixel(int x, int y) {
  VXV::Rect rect;
  rect.x = x;
  rect.y = y;
  rect.w = 1;
  rect.h = 1;

  SDL_Video::fillRect(rect, static_draw_color);
}


PixelGroupDraw::PixelGroupDraw(std::vector<VXV::Grid>& points,
			       unsigned long color)
  : draw_points(points), draw_color(color) {
}


void PixelGroupDraw::draw(void) {
  VXV::Grid& pos = view_pos;
  VXV::Rect& range = view_range;
  double magnify = view_magnify;

  PixelDraw::setColor(draw_color);
  VXV::Rect pixel_range = calcPixelRange(pos, range, magnify);
  for (std::vector<VXV::Grid>::iterator it = draw_points.begin();
       it != draw_points.end(); ++it) {
    int x = static_cast<int>(rint((it->x - range.x) * magnify)) + pos.x;
    int y = static_cast<int>(rint((-it->y - range.y) * magnify)) + pos.y;
    drawRangedPixel(pixel_range, x, y);
  }
}


LineDraw::LineDraw(const VXV::Grid& p0, const VXV::Grid& p1,
		   unsigned long color)
  : draw_p0(p0), draw_p1(p1), draw_color(color) {
}


void LineDraw::draw(void) {
  VXV::Grid& pos = view_pos;
  VXV::Rect& range = view_range;
  double magnify = view_magnify;

  VXV::Grid p0, p1;
  p0.x = static_cast<int>(rint((draw_p0.x - range.x) * magnify)) + pos.x;
  p0.y = static_cast<int>(rint((-draw_p0.y - range.y) * magnify)) + pos.y;
  p1.x = static_cast<int>(rint((draw_p1.x - range.x) * magnify)) + pos.x;
  p1.y = static_cast<int>(rint((-draw_p1.y - range.y) * magnify)) + pos.y;

  VXV::Rect pixel_range = calcPixelRange(pos, range, magnify);

  int width = abs(p1.x - p0.x);
  int height = abs(p1.y - p0.y);
  int yy = (p0.y < p1.y) ? +1 : -1;
  int xx = (p0.x < p1.x) ? +1 : -1;

  PixelDraw::setColor(draw_color);

  if (width >= height) {
    int y = p0.y;
    int fraction = width;
    for (int x = p0.x; x != p1.x; x += xx) {
      drawRangedPixel(pixel_range, x, y);
      fraction += (height << 1);
      if (fraction >= (width << 1)) {
	fraction -= (width << 1);
	y += yy;
      }
    }

  } else {
    int x = p0.x;
    int fraction = height;
    for (int y = p0.y; y != p1.y; y += yy) {
      drawRangedPixel(pixel_range, x, y);
      fraction += (width << 1);
      if (fraction >= (height << 1)) {
	fraction -= (height << 1);
	x += xx;
      }
    }
  }
}


ContLineDraw::ContLineDraw(const std::deque<VXV::Grid>& points,
			   unsigned long color) {
  if (points.size() < 2) {
    return;
  }
  for (std::deque<VXV::Grid>::const_iterator it = points.begin();
       it != points.end() -1; ++it) {
    draw_lines.push_back(new LineDraw(*it, *(it+1), color));
  }
}


ContLineDraw::~ContLineDraw(void) {
  for_each(draw_lines.begin(), draw_lines.end(), DeleteObjects());
}


void ContLineDraw::draw(void) {
  for (std::deque<LineDraw*>::iterator it = draw_lines.begin();
       it != draw_lines.end(); ++it) {
    (*it)->draw();
  }
}


CircleDraw::CircleDraw(const VXV::Grid& p, int r, unsigned long color)
  : draw_p(p), draw_r(r), draw_color(color) {
}


void CircleDraw::draw(void) {
  VXV::Grid& pos = view_pos;
  VXV::Rect& range = view_range;
  double magnify = view_magnify;

  int r = static_cast<int>(draw_r * magnify);
  int d = 3 - (r << 1);
  int cy = r;

  // Jn_`
  PixelDraw::setColor(draw_color);
  VXV::Grid p;
  p.x = static_cast<int>(rint((draw_p.x - range.x) * magnify)) + pos.x;
  p.y = static_cast<int>(rint((-draw_p.y - range.y) * magnify)) + pos.y;

  VXV::Rect pixel_range = calcPixelRange(pos, range, magnify);
  drawRangedPixel(pixel_range, p.x, p.y + r);
  drawRangedPixel(pixel_range, p.x, p.y - r);
  drawRangedPixel(pixel_range, p.x + r, p.y);
  drawRangedPixel(pixel_range, p.x - r, p.y);

  // ~̕`
  for (int cx = 0; cx <= cy; ++cx) {
    if (d < 0) {
      d += 6 + (cx << 2);
    } else {
      d += 10 + (cx << 2) - (cy-- << 2);
    }

    drawRangedPixel(pixel_range, p.x + cx, p.y + cy);
    drawRangedPixel(pixel_range, p.x - cx, p.y + cy);
    drawRangedPixel(pixel_range, p.x + cy, p.y + cx);
    drawRangedPixel(pixel_range, p.x - cy, p.y + cx);
    drawRangedPixel(pixel_range, p.x + cy, p.y - cx);
    drawRangedPixel(pixel_range, p.x - cy, p.y - cx);
    drawRangedPixel(pixel_range, p.x - cx, p.y - cy);
    drawRangedPixel(pixel_range, p.x + cx, p.y - cy);
  }
}


TextDraw::TextDraw(TTF_Draw& ttf, const char* text, 
		   const VXV::Grid& text_pos, int pxSize, bool pin,
		   unsigned long color, unsigned long background,
		   bool transparent)
  : surface(new DrawSurface(ttf.createText(text, pxSize, color, background),
			    transparent)),
    label(new LabelComponent(surface)), draw_pos(text_pos), pined(pin) {
  label->setPosition(draw_pos);
}


TextDraw::~TextDraw(void) {
  delete surface;
  delete label;
}


void TextDraw::draw(void) {
  label->draw(view_pos, view_range, view_magnify, draw_pos, pined);
}
