/*************************************************************************
 *
 *  $RCSfile: BeanInfoAdapter.java,v $
 *
 *  $Revision: 1.3.20.1 $
 *
 *  last change: $Author: vg $ $Date: 2004/05/18 10:07:54 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  the BSD license.
 *  
 *  Copyright (c) 2003 by Sun Microsystems, Inc.
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of Sun Microsystems, Inc. nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 *  COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 *  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 *  OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 *  TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 *  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *     
 *************************************************************************/

import java.lang.reflect.Method;
import java.beans.*;
import java.util.Vector;
import java.util.Hashtable;

/** This is a helper class to make it easier to create BeanInfo classes. In the BeanInfo
  * classes derived of this class, only some methods have to be implemented. In the constructor,
  * the properties, methods and event sets have to be declared using the methods
  * add???PropertyDescriptor, addMethodDescriptor, add???EventSetDescriptor.
  */

public abstract class BeanInfoAdapter
	extends SimpleBeanInfo
{
    /** Member variable to hold the PropertyDescriptors.
      */
	private	Vector m_PropertyDescriptors = new Vector();

    /** Member variable to hold the EventSetDescriptors.
      */
	private Vector m_EventSetDescriptors = new Vector();

    /** Member variable to hold the MethodDescriptors. 
      */
	private Vector m_MethodDescriptors = new Vector();

    /** Member variable to hold the Methods of the Class. It is used for reasons of performance.
      */
   private Method m_Methods[] = null;

    /** Abstract method. Used to get the class of the bean for which the BeanInfo is created.
      */
   public abstract Class getBeanClass();

    /** Abstract method. Used to get the name which should be displayed for the bean.
      */
   public abstract String getDisplayName();

    /** Default constructor.
      */
	public BeanInfoAdapter()
	{
	}

	/** Method to get the BeanDescriptor.
	  * @return the BeanDescriptor of the bean
	  */
	public BeanDescriptor getBeanDescriptor()
	{
		BeanDescriptor bd =	new	BeanDescriptor(getBeanClass(), null);
		bd.setValue("CAN_ADD_CHILD", new Boolean(false));
		bd.setDisplayName(getDisplayName());
		return bd;
	}

	/** Method to add a PropertyDescriptor for a read/write property. The methods get<<PropertyName>>
	  * and set<<PropertyName>> must exist in order to access the property.
	  * @param propertyName the name of the property
	  * @param bound indicates if the property is a bound property
	  * @param constrained indicates if the property is a constrained property
	  */
	public void	addStandardRWPropertyDescriptor(String propertyName, 
		boolean bound,	boolean	constrained)
	{
      addRWPropertyDescriptor(propertyName, "get"+propertyName, 
	  	"set"+propertyName, bound, constrained);
	}

	public void	addStandardBooleanRWPropertyDescriptor(String propertyName, 
		boolean bound,	boolean	constrained)
	{
      addRWPropertyDescriptor(propertyName, "is"+propertyName, "set"+propertyName, 
	  	bound, constrained);
	}

	/** Method to add a PropertyDescriptor for a read property. The method get<<PropertyName>>
	  * must exist in order to access the property.
	  * @param propertyName the name of the property
	  * @param bound indicates if the property is a bound property
	  * @param constrained indicates if the property is a constrained property
	  */
	public void	addStandardRPropertyDescriptor(String propertyName,	boolean	bound, 
		boolean constrained)
	{
      addRWPropertyDescriptor(propertyName, "get"+propertyName, null, bound, constrained);
	}

	/** Method to add a PropertyDescriptor for a write property. The method set<<PropertyName>>
	  * must exist in order to access the property.
	  * @param propertyName the name of the property
	  * @param bound indicates if the property is a bound property
	  * @param constrained indicates if the property is a constrained property
	  */
	public void addStandardWPropertyDescriptor(String propertyName, boolean  bound, 
		boolean constrained)
	{
		addRWPropertyDescriptor(propertyName, null,	"set"+propertyName,	bound, 
			constrained);
	}

	/** Method to add a PropertyDescriptor for a read/write property. The methods must be given
	  * as parameters (getterName/setterName) in order to access the property.
	  * @param propertyName the name of the property
	  * @param getterName the name of the method to get the property
	  * @param setterName the name of the method to set the property
	  * @param bound indicates if the property is a bound property
	  * @param constrained indicates if the property is a constrained property
	  */
	public void	addRWPropertyDescriptor(String propertyName, String	getterName,	
		String setterName, boolean	bound, boolean constrained)
	{
		try {
			PropertyDescriptor pd =	new	PropertyDescriptor(
				propertyName, getBeanClass(), getterName, setterName);
			pd.setBound(bound);
			pd.setConstrained(constrained);
			m_PropertyDescriptors.addElement(pd);
		} catch (Exception e) {
			e.printStackTrace();
			System.out.println("Exception during adding	a PropertyDescriptor for "+
							   getBeanClass()+"."+propertyName+": "+e);
		}
	}

	/** Method to get the PropertyDescriptors.
	  * @return the PropertyDescriptors of the bean
	  */
	public PropertyDescriptor[]	getPropertyDescriptors() {
		PropertyDescriptor result[]	= new PropertyDescriptor[m_PropertyDescriptors.size()];
		for	(int i=0; i<m_PropertyDescriptors.size(); i++) {
			result[i] =	(PropertyDescriptor)m_PropertyDescriptors.elementAt(i);
		}
		return result;
	}

    /** Method to set a PropertyEditor class for a property. In IDEs the PropertyEditor
	 * is used to edit the property instead of the default PropertyEditor.
	 */
    public void setPropertyEditor(String propName, Class editClass) {
        java.util.Enumeration enumer = m_PropertyDescriptors.elements();
        while (enumer.hasMoreElements()) {
            PropertyDescriptor pd = (PropertyDescriptor)enumer.nextElement();
            if (pd.getName().equals(propName))
                pd.setPropertyEditorClass(editClass);
        }
    }

    /** Method to add MethodDescriptors for a given method name. All methods with this
	 * name (there may exist some methods with the same name and different parameters)
	 * are added to the MethodDescriptors.
	 * @param name the name of the method(s)
	 */
	public void	addMethodDescriptors(String name) {
	    try {
    		if (m_Methods == null)
    		    m_Methods = getBeanClass().getMethods();
    		boolean found = false;
    		for (int i=0; i < m_Methods.length; i++) {
    		    if (m_Methods[i].getName().equals(name)) {
    		        found = true;
    		        Class params[] = m_Methods[i].getParameterTypes();
    		        for (int j=0; j < params.length; j++) {
    		            m_MethodDescriptors.addElement(
    		                new MethodDescriptor(m_Methods[i], new ParameterDescriptor[0]));
    		        }
    		    }
    		}
			if (!found)
    		    System.out.println("The method " + name + " could not be found in class " + getBeanClass());
    	} catch (Exception e) {
			e.printStackTrace();
			System.out.println("Exception in 'addMethod()': "+e);
    	}
	}

	/** Method to get the MethodDescriptors.
	  * @return the MethodDescriptors of the bean
	  */
	public MethodDescriptor[] getMethodDescriptors() {
		MethodDescriptor result[]	= new MethodDescriptor[m_MethodDescriptors.size()];
		for	(int i=0; i< m_MethodDescriptors.size(); i++) {
			result[i] =	(MethodDescriptor) m_MethodDescriptors.elementAt(i);
		}
		return result;
    }

	/** Method to add a EventSetDescriptor for an EventSet with only one method.
	  * The listeners have to be added/removed with the methods add<<EventSetName>>Listener
	  * and remove<<EventSetName>>Listener.
	  * @param eventSetName the name of the EventSet
	  * @param listenerType the Class of the Event
	  * @param listenerMethodName the name of the method invoked when the event occurs
	  */
	public void addStandardEventSetDescriptor(String eventSetName, Class listenerType,
		String listenerMethodName)
	{
		try {
			m_EventSetDescriptors.addElement(new EventSetDescriptor(
				getBeanClass(), eventSetName, listenerType, listenerMethodName));
		} catch (Exception e) {
			e.printStackTrace();
			System.out.println("Exception during adding StandardEventSetDescriptor ("+
							   getBeanClass()+"."+eventSetName+") :"+e);
		}
	}

	/** Method to add a EventSetDescriptor for an EventSet.
	  * The listeners have to be added/removed with the methods add<<EventSetName>>Listener
	  * and remove<<EventSetName>>Listener.
	  * @param eventSetName the name of the EventSet
	  * @param listenerType the Class of the Event
	  * @param listenerMethodNames the names of the methods invoked when the event occurs
	  */
	public void addStandardEventSetDescriptor(String eventSetName, Class listenerType,
		String[] listenerMethodNames)
	{
		addEventSetDescriptor("", listenerType, listenerMethodNames,
			"add"+listenerType.getName(), "remove"+listenerType.getName());
	}

	/** Method to add a EventSetDescriptor for an EventSet.
	  * The listeners could be added/removed with methods with method names given as a parameter.
	  * @param eventSetName the name of the EventSet
	  * @param listenerType the Class of the Event
	  * @param listenerMethodNames the names of the methods invoked when the event occurs
	  * @param addListenerMethodName the name of the method to add a Listener
	  * @param removeListenerMethodName the name of the method to remove a Listener
	  */
	public void addEventSetDescriptor(String eventSetName, Class listenerType, 
		String[] listenerMethodNames, String addListenerMethodName, 
		String removeListenerMethodName)
	{
		try
		{
			m_EventSetDescriptors.addElement(new EventSetDescriptor(
				getBeanClass(), eventSetName, listenerType, listenerMethodNames,
				addListenerMethodName, removeListenerMethodName));
		} catch (Exception e) {
         e.printStackTrace();
			System.out.println("Exception during adding EventSetDescriptor ("+
								getBeanClass()+"."+eventSetName+") :"+e);
		}
	}

	/** Method to get the EventSetDescriptors.
	  * @return the EventSetDescriptors of the bean
	  */
	public EventSetDescriptor[]	getEventSetDescriptors()
	{
		EventSetDescriptor result[]	= new EventSetDescriptor[m_EventSetDescriptors.size()];
		for	(int i=0; i<m_EventSetDescriptors.size(); i++) {
			result[i] =	(EventSetDescriptor)m_EventSetDescriptors.elementAt(i);
		}
		return result;
	}

	/** Method to get the default event index. This method returns -1. If there is a
	  * default event index this method has to be implemented in the derived class.
	  */
	public int getDefaultEventIndex()
	{
	    return -1;
	}

	/** Method to get the default property index. This method returns -1. If there is a
	  * default property index this method has to be implemented in the derived class.
	  */
	public int getDefaultPropertyIndex()
	{
	    return -1;
	}

	/** Method to get additional bean info. This class returns null. If there is
	  * additional bean info this method has to be implemented in the derived class.
	  */
	public BeanInfo[] getAdditionalBeanInfo()
	{
        return null;
	}
}
