/*  NDKmol - Molecular Viewer on Android NDK

     (C) Copyright 2011, biochem_fan

     This file is part of NDKmol.

     NDKmol is free software: you can redistribute it and/or modify
     it under the terms of the GNU Lesser General Public License as published by
     the Free Software Foundation, either version 3 of the License, or
     (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU Lesser General Public License for more details.

     You should have received a copy of the GNU Lesser General Public License
     along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#include <Atom.h>
#include "NdkView.hpp"
#include <GLES/gl.h>
#include <android/log.h>
#include <cmath>
#include <vector>
#include <set>
#include <map>
#include "Color.hpp"
#include "PDBReader.hpp"
#include "SmoothCurve.hpp"
#include "RibbonStrip.hpp"
#include "Line.hpp"
#include "Renderable.hpp"
#include "MatRenderable.hpp"
#include "VBOSphere.hpp"
#include "VBOCylinder.hpp"
#include "ChemDatabase.hpp"
#include "Protein.hpp"
#include "View.hpp"

Atom *atoms = NULL;
Protein *protein = NULL;
Renderable *scene = NULL;

float sphereRadius = 1.5f;
float cylinderRadius = 0.2f;
float lineWidth = 0.5f;
float curveWidth = 2.0f;

std::vector<int> getAll();
std::vector<int> getHetatms(std::vector<int> &atomlist);
std::vector<int> removeSolvents(std::vector<int> &atomlist);
std::vector<int> getResiduesById(std::vector<int> &atomlist, std::set<int> &resi);
std::vector<int> getSideChain(std::vector<int> &atomlist);
std::vector<int> getChain(std::vector<int> &atomlist, std::set<std::string> &chain);
void colorByAtom(std::vector<int> &atomlist, std::map<std::string, unsigned int> &colors);
void colorByStructure(std::vector<int> &atomlist, Color helixColor, Color sheetColor);
void colorByChain(std::vector<int> &atomlist);
void colorChainbow(std::vector<int> &atomlist);
void drawMainchainCurve(Renderable &scene, std::vector<int> &atomlist, float curveWidth, std::string atomName);
void drawCartoon(Renderable &scene, std::vector<int> &atomlist);
bool isIdentity(Mat16 mat);
void drawBondsAsStick(Renderable &scene, std::vector<int> &atomlist, float bondR, float atomR);
void drawBondsAsLine(Renderable &scene, std::vector<int> &atomlist, float lineWidth);
void drawAtomsAsVdWSphere(Renderable &scene, std::vector<int> &atomlist);
void drawSymmetryMates(Renderable &scene, std::vector<int> &atomlist, std::map<int, Mat16> biomtMatrices);
void drawSymmetryMatesWithTranslation(Renderable &scene, std::vector<int> &atomlist, std::map<int, Mat16> matrices);
void drawUnitcell(Renderable &scene, float width);

// onSurfaceChanged
JNIEXPORT void JNICALL Java_jp_sfjp_webglmol_NDKmol_NdkView_nativeGLResize
(JNIEnv *env, jclass clasz, jint width, jint height) {
	__android_log_print(ANDROID_LOG_DEBUG, "NdkView", "resize");

	VBOSphere::prepareVBO();
	VBOCylinder::prepareVBO();
}

// onDrawFrame
JNIEXPORT void JNICALL Java_jp_sfjp_webglmol_NDKmol_NdkView_nativeGLRender
(JNIEnv *env, jclass clasz) {
	__android_log_print(ANDROID_LOG_DEBUG, "NdkView", "render");

	if (scene != NULL) scene->render();
}

// onSurfaceCreated
JNIEXPORT void JNICALL Java_jp_sfjp_webglmol_NDKmol_NdkView_nativeLoadProtein
(JNIEnv *env, jclass clasz, jstring path) {
	const char *filename = env->GetStringUTFChars(path, NULL);
	__android_log_print(ANDROID_LOG_DEBUG, "NdkView","opening %s", filename);
	PDBReader pdb;
	if (scene) {
		delete scene;
		scene = NULL;
	}
	if (protein) {
		delete protein;
		protein = NULL;
	}
	protein = pdb.parsePDB(filename);
	atoms = protein->atoms;
	__android_log_print(ANDROID_LOG_DEBUG,"NdkView","opened. protein at %p", protein);

	env->ReleaseStringUTFChars(path, filename);
}

// prepareScene
JNIEXPORT void JNICALL Java_jp_sfjp_webglmol_NDKmol_NdkView_buildScene
(JNIEnv *env, jclass clasz, jint proteinMode, jint hetatmMode, jint symmetryMode, jint colorMode, jboolean showSidechain, jboolean showUnitcell, jboolean resetView) {
	__android_log_print(ANDROID_LOG_DEBUG, "NdkView", "prepareScene");

	if (scene != NULL) {
		delete scene;
		scene = NULL;
	}
	if (protein == NULL) return;

	scene = new Renderable();
	if (true) { // FIXME: resetView is not necessary?
		scene->posx = -protein->centerx;
		scene->posy = -protein->centery;
		scene->posz = -protein->centerz;
	}

	std::vector<int> all = getAll();
	std::vector<int> tmp = getHetatms(all);
	std::vector<int> hetatm = removeSolvents(tmp);
	std::map<std::string, unsigned int> m;
	colorByAtom(all, m);

	switch (colorMode) {
	case 0:
		colorChainbow(all);
		break;
	case 1:
		colorByChain(all);
		break;
	case 2:
		colorByStructure(all, Color(0xCC00CC), Color(0x00CCCC));
	}

	switch (proteinMode) {
	case 0:
		drawCartoon(*scene, all);
		break;
	case 1:
		drawMainchainCurve(*scene, all, curveWidth, "CA");
	}
	drawMainchainCurve(*scene, all, curveWidth, "P");

	if (showSidechain) {
		std::vector<int> sidechains = getSideChain(all);
		drawBondsAsLine(*scene, sidechains, lineWidth);
	}

	switch (hetatmMode) {
	case 0:
		drawAtomsAsVdWSphere(*scene, hetatm);
		break;
	case 1:
		drawBondsAsStick(*scene, hetatm, cylinderRadius, cylinderRadius);
		break;
	case 2:
		drawBondsAsLine(*scene, hetatm, lineWidth * 8);
	}

	if (showUnitcell) {
		drawUnitcell(*scene, lineWidth);
	}

	switch (symmetryMode) {
	case 1:
		drawSymmetryMates(*scene, all, protein->biomtMatrices);
		break;
	case 2:
		drawSymmetryMatesWithTranslation(*scene, all, protein->symmetryMatrices);
	}
}

// onSurfaceCreated
JNIEXPORT void JNICALL Java_jp_sfjp_webglmol_NDKmol_NdkView_nativeGLInit
(JNIEnv *env, jclass clasz) {
	__android_log_print(ANDROID_LOG_DEBUG,"NdkView","init");

	VBOSphere::prepareVBO(); // FIXME: Why is it necessary here when resize is also called?
	VBOCylinder::prepareVBO();

	glClearColor(0, 0, 0, 1);
	glEnable(GL_DEPTH_TEST);
	glShadeModel(GL_SMOOTH);
//	glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);
	glEnable(GL_POINT_SMOOTH);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	//		glEnable(GL_LINE_SMOOTH); // FIXME: Check if this is working.
	glLightModelx(GL_LIGHT_MODEL_TWO_SIDE, 1); // double sided
	glEnable(GL_COLOR_MATERIAL); // glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE) ?
	glDepthFunc(GL_LEQUAL);
	glDisable(GL_DITHER);
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	float f1[] = {0.4f, 0.4f, 0.4f, 1};
	glLightfv(GL_LIGHT0, GL_AMBIENT, f1);
	float f2[] = {0, 0, 1, 0};
	glLightfv(GL_LIGHT0, GL_POSITION, f2);
	float f3[] = {0.8f, 0.8f, 0.8f, 1};
	glLightfv(GL_LIGHT0, GL_DIFFUSE, f3);
	glEnable(GL_LIGHT1);
	//		glLightfv(GL_LIGHT0, GL_AMBIENT, Geometry.getFloatBuffer(new float[] {0.4f, 0.4f, 0.4f, 1}));
	float f4[] = {0, 0, -1, 0};
	glLightfv(GL_LIGHT1, GL_POSITION, f4);
	float f5[] = {0.1f, 0.1f, 0.1f, 1};
	glLightfv(GL_LIGHT1, GL_DIFFUSE, f3);
	glLightfv(GL_LIGHT0, GL_SPECULAR, f5);
	glLightfv(GL_LIGHT1, GL_SPECULAR, f5);
	//	glPointParameterfv(GL11.GL_POINT_DISTANCE_ATTENUATION, {0, 0, 1}));
}

std::vector<int> getAll() { // TODO: should return reference?
	std::vector<int> ret;
	for (int i = 0; i < 100001; i++) {
		if (atoms[i].valid) ret.push_back(i);
	}
	return ret;
}

std::vector<int> getHetatms(std::vector<int> &atomlist) {
	std::vector<int> ret;
	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom = atoms + atomlist[i];
		if (!atom->valid) continue;

		if (atom->hetflag)	ret.push_back(atom->serial);
	}
	return ret;
}

std::vector<int> removeSolvents(std::vector<int> &atomlist) {
	std::vector<int> ret;
	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom = atoms + atomlist[i];
		if (!atom->valid) continue;

		if (atom->resn !="HOH") ret.push_back(atom->serial);
	}
	return ret;
}

std::vector<int> getResiduesById(std::vector<int> &atomlist, std::set<int> &resi) {
	std::vector<int> ret;
	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom = atoms + atomlist[i];
		if (!atom->valid) continue;

		if (resi.count(atom->resi) > 0) ret.push_back(atom->serial);
	}
	return ret;
}

std::vector<int> getSideChain(std::vector<int> &atomlist) {
	std::vector<int> ret;
	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom = atoms + atomlist[i];
		if (!atom->valid) continue;


		if (atom->hetflag) continue;
		if (atom->atom == "O" || atom->atom == "N" || atom->atom == "C") continue;
		ret.push_back(atom->serial);
	}

	return ret;
}


std::vector<int> getChain(std::vector<int> &atomlist, std::set<std::string> &chain) {
	std::vector<int> ret;
	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom = atoms + atomlist[i];
		if (!atom->valid) continue;

		if (chain.count(atom->chain) > 0) ret.push_back(atom->serial);
	}
	return ret;
}

void colorByAtom(std::vector<int> &atomlist, std::map<std::string, unsigned int> &colors) {
	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom = atoms + atomlist[i];
		if (!atom->valid) continue;

		std::map<std::string, unsigned int>::iterator it = colors.find(atom->elem);
		if (it != colors.end()) {
			atom->color = it->second;
		} else {
			atom->color = ChemDatabase::getColor(atom->elem);
		}
		//		__android_log_print(ANDROID_LOG_DEBUG,"CbyA","%d %s colored %f %f %f", atom->serial, atom->elem.c_str(), atom->color.r, atom->color.g, atom->color.b);
	}
}

void colorByStructure(std::vector<int> &atomlist, Color helixColor, Color sheetColor) {
	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom = atoms + atomlist[i];
		if (!atom->valid) continue;

		if (atom->atom != "CA" || atom->hetflag) continue;
		if (atom->ss[0] == 's') atom->color = sheetColor;
		else if (atom->ss[0] == 'h') atom->color = helixColor;
	}
}

void colorByChain(std::vector<int> &atomlist) {
	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom = atoms + atomlist[i];
		if (!atom->valid) continue;

		if (atom->atom != "CA" || atom->hetflag) continue;
		atom->color.setHSV((float)(atom->chain[0] % 15) / 15, 1, 0.9f);
	}
}

void colorChainbow(std::vector<int> &atomlist) {
	int cnt = 0;

	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom = atoms + atomlist[i];
		if (!atom->valid) continue;

		if (atom->atom != "CA" || atom->hetflag) continue;
		cnt++;
	}

	int total = cnt;
	cnt = 0;

	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom = atoms + atomlist[i];
		if (!atom->valid) continue;

		if (atom->atom != "CA" || atom->hetflag) continue;

		atom->color.setHSV((float)240 / 360 * cnt / total, 1, 0.9f);
		cnt++;
	}
}

void drawMainchainCurve(Renderable &scene, std::vector<int> &atomlist, float curveWidth, std::string atomName) {
	std::vector<Vector3> points;
	std::vector<Color> colors;

	std::string currentChain = "A";
	int currentResi = -1;

	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom = atoms + atomlist[i];
		if (!atom->valid) continue;

		if ((atom->atom == atomName) && !atom->hetflag) {
			if (currentChain != atom->chain || currentResi + 1 != atom->resi) {
				scene.children.push_back(new SmoothCurve(points, colors, curveWidth));
				points.clear();
				colors.clear();
			}
			points.push_back(Vector3(atom->x, atom->y, atom->z));
			colors.push_back(atom->color);
			currentChain = atom->chain;
			currentResi = atom->resi;
		}
	}
	scene.children.push_back(new SmoothCurve(points, colors, curveWidth));

	// TODO: Arrow heads
}


void drawCartoon(Renderable &scene, std::vector<int> &atomlist) {
	std::vector<Vector3> points1;
	std::vector<Vector3> points2;
	std::vector<Color> colors;

	std::string currentChain = "A";
	int currentResi = -1;

	Vector3 prevCO, currentCA;

	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom = atoms + atomlist[i];
		if (!atom->valid) continue;

		if ((atom->atom =="O" || atom->atom == "CA") && !atom->hetflag) {
			if (atom->atom == "CA") {
				if (currentChain != atom->chain || currentResi + 1 != atom->resi) {
					scene.children.push_back(new RibbonStrip(points1, points2, colors));
					scene.children.push_back(new SmoothCurve(points1, colors, curveWidth));
					scene.children.push_back(new SmoothCurve(points2, colors, curveWidth));
					points1.clear(); points2.clear();
					colors.clear();
					prevCO.set(0, 0, 0); // FIXME: is this OK?
				}
				currentCA.set(atom->x, atom->y, atom->z);
				currentChain = atom->chain;
				currentResi = atom->resi;
				colors.push_back(atom->color);
			} else { // O
				float ox = atom->x - currentCA.x, oy = atom->y - currentCA.y, oz = atom->z - currentCA.z;
				float width = (atom->ss[0] == 'c') ? 0.1f : 0.5f;
				ox *= width; oy *= width; oz *= width;
				if (ox * prevCO.x + oy * prevCO.y + oz * prevCO.z < 0) {
					ox *= -1; oy *= -1; oz *= -1;
				}
				prevCO.set(ox, oy, oz);
				points1.push_back(Vector3(currentCA.x + prevCO.x, currentCA.y + prevCO.y, currentCA.z + prevCO.z));
				points2.push_back(Vector3(currentCA.x - prevCO.x, currentCA.y - prevCO.y, currentCA.z - prevCO.z));
			}
		}
	}
	scene.children.push_back(new RibbonStrip(points1, points2, colors));
	scene.children.push_back(new SmoothCurve(points1, colors, curveWidth));
	scene.children.push_back(new SmoothCurve(points2, colors, curveWidth));
}

bool isIdentity(Mat16 mat) {
	for (int i = 0; i < 4; i++) {
		for (int j = 0; j < 4; j++) {
			if (i == j && std::abs(mat.m[i * 4 + j] - 1) > 0.001) return false;
			if (i != j && std::abs(mat.m[i * 4 + j]) > 0.001) return false;
		}
	}
	return true;
}


void drawBondsAsStick(Renderable &scene, std::vector<int> &atomlist, float bondR, float atomR) {
	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom1 = atoms + atomlist[i];
		if (!atom1->valid) continue;

		for (int j = i + 1; j < i + 40 && j < lim; j++) { // TODO: Check if 40 is sufficient for DNA and HETATMs., port to GLmol
			Atom *atom2 = atoms + atomlist[j];
			if (!atom2->valid) continue;
			if (!atom1->isConnected(*atom2)) continue;

			Renderable *cylinder = new VBOCylinder(atom1->x, atom1->y, atom1->z,
					(atom1->x + atom2->x) / 2, (atom1->y + atom2->y) / 2, (atom1->z + atom2->z) / 2, bondR, atom1->color);
			scene.children.push_back(cylinder);

			cylinder = new VBOCylinder((atom1->x + atom2->x) / 2, (atom1->y + atom2->y) / 2, (atom1->z + atom2->z) / 2,
					atom2->x, atom2->y, atom2->z, bondR, atom2->color);
			scene.children.push_back(cylinder);
		}
		Renderable *sphere = new VBOSphere(atom1->x, atom1->y, atom1->z, atomR, atom1->color);
		scene.children.push_back(sphere);
	}
}

// ȤƤ٤Τϡstd::vector<int> μФǡ֥Ȥ뤫?
void drawBondsAsLine(Renderable &scene, std::vector<int> &atomlist, float lineWidth) {
	std::vector<Vector3> points;
	std::vector<Color> colors;

	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		for (int j = i + 1; j < i + 40 && j < lim; j++) {
			Atom *atom1 = atoms + atomlist[i];
			Atom *atom2 = atoms + atomlist[j];
			if (!atom1->valid|| !atom2->valid) continue;
			if (!atom1->isConnected(*atom2)) continue;

			Vector3 mid((atom1->x + atom2->x) / 2, (atom1->y + atom2->y) / 2, (atom1->z + atom2->z) / 2);
			points.push_back(mid);
			points.push_back(Vector3(atom1->x, atom1->y, atom1->z));
			colors.push_back(atom1->color); colors.push_back(atom1->color);

			points.push_back(mid);
			points.push_back(Vector3(atom2->x, atom2->y, atom2->z));
			colors.push_back(atom2->color); colors.push_back(atom2->color);
		}
	}

	Line *line = new Line(points, colors);
	line->width = lineWidth;
	line->discrete = true;
	scene.children.push_back(line);
}

void drawAtomsAsVdWSphere(Renderable &scene, std::vector<int> &atomlist) {
	std::vector<Vector3> points;
	std::vector<Color> colors;
	std::vector<float> radii;

	for (int i = 0, lim = atomlist.size(); i < lim; i++) {
		Atom *atom = atoms + atomlist[i];
		if (!atom->valid) continue;

		Renderable *sphere = new VBOSphere(atom->x, atom->y, atom->z, ChemDatabase::getVdwRadius(atom->elem), atom->color);
		scene.children.push_back(sphere);
		//		points.push_back(new Vector3(atom->x, atom->y, atom->z));
		//		colors.push_back(atom->color);
		//		radii.push_back(ChemDatabase.getVdwRadius(atom->elem));
	}
	//	Renderable sphere = new VBOSpheres(points, colors, radii);
	//	scene.children.push_back(sphere);
}

void drawSymmetryMates(Renderable &scene, std::vector<int> &atomlist, std::map<int, Mat16> biomtMatrices) {
//	__android_log_print(ANDROID_LOG_DEBUG,"NdkView", "drawSymmetryMates N = %d", biomtMatrices.size());
	for (std::map<int, Mat16>::iterator it = biomtMatrices.begin(); it != biomtMatrices.end(); ++it) {
		Mat16 mat = it->second;
		if (isIdentity(mat)) continue;

//		__android_log_print(ANDROID_LOG_DEBUG,"NdkView", "{(%f, %f, %f, %f),\n (%f, %f, %f, %f),\n (%f, %f, %f, %f),\n (%f, %f, %f, %f))}",
//				mat.m[0], mat.m[1], mat.m[2], mat.m[3], mat.m[4], mat.m[5], mat.m[6], mat.m[7], mat.m[8], mat.m[9],
//				mat.m[10], mat.m[11], mat.m[12], mat.m[13], mat.m[14], mat.m[15], mat.m[16]);
		MatRenderable *symmetryMate = new MatRenderable();
		drawMainchainCurve(*symmetryMate, atomlist, curveWidth, "CA");
		drawMainchainCurve(*symmetryMate, atomlist, curveWidth, "P");
		symmetryMate->setupMatrix(mat);
		scene.children.push_back(symmetryMate);
	}
}

void drawSymmetryMatesWithTranslation(Renderable &scene, std::vector<int> &atomlist, std::map<int, Mat16> matrices) {
	for (std::map<int, Mat16>::iterator it = matrices.begin(); it != matrices.end(); ++it) {
		Mat16 mat = it->second;

		for (int a = -1; a <= 0; a++) {
			for (int b = -1; b <= 0; b++) {
				for (int c = -1; c <= 0; c++) {
					Mat16 translated = mat;
					translated.m[3] += protein->ax * a + protein->bx * b + protein->cx * c;
					translated.m[7] += protein->ay * a + protein->by * b + protein->cy * c;
					translated.m[11] += protein->az * a + protein->bz * b + protein->cz * c;
					if (isIdentity(translated)) continue;

					MatRenderable *symmetryMate = new MatRenderable();
					symmetryMate->setupMatrix(translated);

					drawMainchainCurve(*symmetryMate, atomlist, curveWidth, "CA");
					drawMainchainCurve(*symmetryMate, atomlist, curveWidth, "P");

					scene.children.push_back(symmetryMate);
				}
			}
		}
	}
}


void drawUnitcell(Renderable &scene, float width) {
	if (protein->a == 0) return;

	float vertices[][3] = {{0, 0, 0},
			{protein->ax, protein->ay, protein->az},
			{protein->bx, protein->by, protein->bz},
			{protein->ax + protein->bx, protein->ay + protein->by, protein->az + protein->bz},
			{protein->cx, protein->cy, protein->cz},
			{protein->cx + protein->ax, protein->cy + protein->ay,  protein->cz + protein->az},
			{protein->cx + protein->bx, protein->cy + protein->by, protein->cz + protein->bz},
			{protein->cx + protein->ax + protein->bx, protein->cy + protein->ay + protein->by, protein->cz + protein->az + protein->bz}};
	int edges[] = {0, 1, 0, 2, 1, 3, 2, 3, 4, 5, 4, 6, 5, 7, 6, 7, 0, 4, 1, 5, 2, 6, 3, 7};

	float *points = new float[24 * 3];
	for (int i = 0; i < 24; i++) {
		points[i * 3] = vertices[edges[i]][0];
		points[i * 3 + 1] = vertices[edges[i]][1];
		points[i * 3 + 2] = vertices[edges[i]][2];
	}
	Line *line = new Line(points, 24);
	line->objectColor = Color(0.8f, 0.8f, 0.8f, 1);
	line->discrete = true;
	line->width = width;
	scene.children.push_back(line);
}

