package jp.ac.dendai.cdl.mori.wikie.util;

import info.bliki.wiki.filter.*;
import info.bliki.wiki.model.*;

import java.util.*;
import java.util.regex.*;

import jp.ac.dendai.cdl.mori.wikie.main.*;

import org.apache.commons.lang.*;

/**
 * リンク等に関するメソッドをまとめたクラス
 * @author Mori
 *
 */
public class WLinkUtils {
    /**
     * 使用するWNormalizer
     */
    private WNormalizer normalizer;
    /**
     * 使用するWikiModel
     */
    private WikiModel wmodel;

    public WLinkUtils(WNormalizer normalizer) {
        this.normalizer = normalizer;
        wmodel = new WikiModel(normalizer.getBaseURL() + "${image}", normalizer.getBaseURL() + "${title}");
    }

    /**
     * 二重括弧[[]]の中身を取得する
     * @param text text要素
     * @return 二重括弧[[]]の中身をStringで格納したList
     */
    public List<String> getDoubleBracket(String text) {
        List<String> result = new ArrayList<String>();
        Pattern pattern = Pattern.compile("\\[\\[(.+?)\\]\\]");
        Matcher matcher = pattern.matcher(text);
        while (matcher.find()) {
            String entry = matcher.group(1).trim();
            if (StringUtils.isNotBlank(entry)) {
                result.add(entry);
            }
        }
        return result;
    }

    /**
     * Category付けのリンクを取得する。
     * @param text text要素
     * @return Category付けされているCategoryページのWEntry型オブジェクト
     */
    public List<WEntry> getCategory(String text) {
        List<WEntry> result = new ArrayList<WEntry>();
        Iterator<String> linkItr = getDoubleBracket(text).iterator();
        while (linkItr.hasNext()) {
            String link = linkItr.next();
            try {
                if (link.charAt(0) != ':') {
                    WEntry linkEntry = normalizer.normalize(StringUtils.splitPreserveAllTokens(link, "\\|")[0]);
                    if (linkEntry.getNsNumber() == WikIE.CATEGORY_NS_NUM && StringUtils.isNotBlank(linkEntry.toString())) {
                        result.add(linkEntry);
                    }
                }
            }
            catch (ArrayIndexOutOfBoundsException e) {
            }
        }
        return result;
    }

    /**
     * Wiki内リンクを取得する。
     * @param text text要素
     * @return　WLink型オブジェクトのList
     */
    public List<WLink> getLink(String text) {
        List<WLink> result = new ArrayList<WLink>();
        Iterator<String> linkItr = getDoubleBracket(text).iterator();
        while (linkItr.hasNext()) {
            WLink wLink = new WLink();
            String anchor[] = linkItr.next().split("\\|");
            if (anchor.length == 2) {
                wLink.setAnchorText(anchor[1].trim());
            }
            String section[] = anchor[0].split("#");
            if (section.length == 2) {
                wLink.setSection(WNormalizer.decodeSectionLink(section[1].trim()));
            }
            WEntry entry = normalizer.normalize(section[0].trim());
            if (entry.getNsNumber() == WikIE.ARTICLE_NS_NUM ||
                entry.getNsNumber() == WikIE.CATEGORY_NS_NUM && section[0].charAt(0) == ':') {
                if (StringUtils.isBlank(entry.getLang()) && StringUtils.isBlank(entry.getProject())) {
                    wLink.setEntry(entry);
                    result.add(wLink);
                }
            }
        }
        return result;
    }

    /**
     * 言語間リンクを取得する。
     * @param text text要素
     * @return　WLink型オブジェクトのList
     */
    public List<WLink> getInterLangLink(String text) {
        List<WLink> result = new ArrayList<WLink>();
        Iterator<String> linkItr = getDoubleBracket(text).iterator();
        while (linkItr.hasNext()) {
            String link = linkItr.next();
            if (link.indexOf(":") > 0) {
                WEntry entry = normalizer.normalize(link);
                if (StringUtils.isNotBlank(entry.getLang())) {
                    result.add(new WLink(entry, "", ""));
                }
            }
        }
        return result;
    }

    /**
     * エントリの種別を判定する。
     * @param entry 調べたいエントリのWEntry
     * @param text そのエントリのtext要素
     * @return 普通の記事ならWikIE.LEAF<br>
     *         リダイレクトならWikIE.REDIRECT<br>
     *         カテゴリならWikIE.NODE<br>
     *         それ以外ならWikIE.OTHER
     */
    public String getKind(WEntry entry, String text) {
        int ns = entry.getNsNumber();
        String kind = WikIE.OTHER;
        if (ns == WikIE.ARTICLE_NS_NUM) {
            WEntry redirect = isRedirect(text);
            if (redirect != null) {
                kind = WikIE.REDIRECT;
            }
            else {
                kind = WikIE.LEAF;
            }
        }
        else if (ns == WikIE.CATEGORY_NS_NUM){
            kind = WikIE.NODE;
        }
        return kind;
    }

    /**
     * Blinkライブラリを使ってエントリがリダイレクトかどうか判定する。
     * @param text text要素
     * @return リダイレクトならリダイレクト先エントリのWEntry<br>
     *         リダイレクトでないならnull
     */
    public WEntry isRedirect(String text) {
        String target = WikipediaParser.parseRedirect(text, wmodel);
        if (StringUtils.isNotBlank(target)) {
            return normalizer.normalize(target);
        }
        return null;
    }
}
