package hiro.yoshioka.sql;

import hiro.yoshioka.ast.sql.oracle.WolfSQLParserConstants;
import hiro.yoshioka.sdh.ResultSetDataHolder;
import hiro.yoshioka.sdh2.ReflectionPreparedStatement;
import hiro.yoshioka.sdh2.ResultSetDataHolder2;
import hiro.yoshioka.sql.engine.Request;
import hiro.yoshioka.sql.engine.ResourceCaptionRequest;
import hiro.yoshioka.sql.engine.SQLOperationType;
import hiro.yoshioka.sql.engine.TransactionRequest;
import hiro.yoshioka.sql.resource.DBCrossRefference;
import hiro.yoshioka.sql.resource.DBSequence;
import hiro.yoshioka.sql.resource.DBTable;
import hiro.yoshioka.sql.resource.IDBColumn;
import hiro.yoshioka.sql.resource.IDBResource;
import hiro.yoshioka.sql.resource.IDBSchema;
import hiro.yoshioka.sql.resource.IDBSequence;
import hiro.yoshioka.sql.resource.IDBTable;
import hiro.yoshioka.util.StringUtil;

import java.io.File;
import java.sql.Driver;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Savepoint;
import java.sql.Statement;
import java.sql.Types;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class PostgresSQL extends AbsTransactionSQL implements IPostgresDBConst,
		IDoExplainPlan {

	private HashMap _tablesComments;
	Savepoint fSavePoint;

	public static String getSuggestURL() {
		return JDBC_URL_EXAMPLE;
	}

	protected PostgresSQL(Driver ds) {
		super(ds);
	}

	@Override
	public ResultSetDataHolder2 explainPlan(String sql_statement)
			throws SQLException {
		Statement statement = null;
		ResultSetDataHolder2 sdh = null;
		try {
			statement = _extra_con.createStatement();
			sdh = executePrepareQuery(_extra_con, "EXPLAIN ANALYZE "
					+ sql_statement, StringUtil.EMPTY_STRING_ARRAY);

			return sdh;
		} finally {
			_extra_con.commit();
			if (statement != null) {
				statement.close();
			}
		}
	}

	protected String getSupportToken() {
		StringBuffer buf = new StringBuffer();
		Pattern p = Pattern.compile("\"(\\w+)\"");
		String[] str = WolfSQLParserConstants.tokenImage;
		for (int i = 0; i < str.length; i++) {
			Matcher m = p.matcher(str[i]);
			if (m.matches()) {
				buf.append(m.group(1)).append(",");
			}
		}
		if (buf.length() > 0) {
			buf.setLength(buf.length() - 1);
		}
		return buf.toString();
	}

	public boolean load(File f) {
		boolean ret = super.load(f);
		_root.setDefaultSchema((IDBSchema) _root.getResource("public"));
		return ret;
	}

	private void setSavePoint() {
		if (_con != null) {
			try {
				fSavePoint = _con.setSavepoint(savepointName);
			} catch (Exception e) {
				fLogger.info(
						"you must execute rollback ,after every sqlstatement executed.",
						e);
			}
		}
	}

	private void releaseSavePoint() throws SQLException {
		if (fSavePoint != null) {
			_con.rollback(fSavePoint);
		}
	}

	public boolean doOperation(SQLOperationType operation, Request request)
			throws SQLException {
		boolean retCode = true;
		if (operation != SQLOperationType.CLOSE
				&& operation != SQLOperationType.ROLLBACK
				&& operation != SQLOperationType.CANSEL) {
			setSavePoint();
		}

		try {
			switch (operation) {
			case SELECT_SESSION:
				((TransactionRequest) request).setRDH(getSessionInfo());
				break;
			default:
				retCode = super.doOperation(operation, request);
				break;
			}
		} catch (SQLException e) {
			fLogger.warn("MES:" + e.getMessage());
			fLogger.warn("MES:" + e.getLocalizedMessage());
			fLogger.warn("MES:" + e.getSQLState());
			releaseSavePoint();
			throw e;
		}
		return retCode;
	}

	/**
	 * SAVEPOINT
	 */
	public boolean execute(ReflectionPreparedStatement ref) throws SQLException {
		setSavePoint();
		try {
			return super.execute(ref);
		} catch (SQLException e) {
			fLogger.warn("MES:" + e.getMessage());
			fLogger.warn("MES:" + e.getLocalizedMessage());
			fLogger.warn("MES:" + e.getSQLState());
			releaseSavePoint();
			throw e;
		}
	}

	/**
	 * @param operationCode
	 * @return
	 */
	public boolean canDoOperation(SQLOperationType operation) {
		switch (operation) {
		case SELECT_SESSION:
			return connected();
		case SELECT_LOCK:
			return false;
		default:
			return super.canDoOperation(operation);
		}
	}

	protected void setComments(ResourceCaptionRequest request)
			throws SQLException {

		request.begtinTask("Grab comment（POSTGRE　NATIVE）",
				3 * _root.getSchemas().length);
		for (IDBSchema mschema : _root.getSchemas()) {
			setTableComments(mschema);

			request.worked(1);

			String[] params = new String[] { mschema.getName() };
			ResultSetDataHolder rdh = executePrepareQuery(
					_SELECT_TAB_COL_COMMENTS, params);

			request.worked(1);
			IDBColumn icolumn = null;
			for (int i = 0; i < rdh.getRowCount(); i++) {
				String str_table = rdh.getStringData(i, "TABLE_NAME");
				String str_column = rdh.getStringData(i, "COLUMN_NAME");
				String str_column_com = rdh.getStringData(i, "COMMENTS");

				// TODO: add grab condition
				IDBTable t = mschema.getTable(str_table);
				if (t != null) {
					icolumn = (IDBColumn) t.getResource(str_column
							.toUpperCase());
					if (icolumn != null) {
						icolumn.setComment(str_column_com);
						request.subTask(str_table + "." + str_column + "["
								+ str_column_com + "]");
					}
				}
			}
			request.worked(1);
		}

	}

	private void setTableComments(IDBSchema schema) throws SQLException {
		ResultSet rsC = null;

		try {
			_tablesComments = new HashMap();
			PreparedStatement pre = _con
					.prepareStatement(_SELECT_TAB_COMMENTS,
							ResultSet.TYPE_SCROLL_SENSITIVE,
							ResultSet.CONCUR_READ_ONLY);
			pre.setString(1, schema.getName());
			rsC = pre.executeQuery();

			for (int i = 0; rsC.next(); i++) {
				String name = rsC.getString("TABLE_NAME");

				IDBResource res = schema.getTable(name);
				if (res != null) {
					res.setComment(rsC.getString("COMMENTS"));
				}
			}
		} finally {
			if (rsC != null) {
				rsC.close();
			}
		}
	}

	public ResultSetDataHolder2 getSessionInfo() throws SQLException {
		return executePrepareQuery(_extra_con, _SELECT_SESSION, new String[] {});
	}

	public ResultSetDataHolder2 getLockInfo() throws SQLException {
		return null;
	}

	protected ResultSetDataHolder createDBTableDef(
			ResourceCaptionRequest request) throws SQLException {
		try {
			fLogger.debug("start");
			ResultSetDataHolder rdh = null;

			request.begtinTask("Grab Table Definitions ]",
					_root.getSchemas().length);
			for (IDBSchema mschema : _root.getSchemas()) {

				request.worked(1);

				rdh = RS2RDH(
						_meta.getTables(null, mschema.getName(), "%", null),
						true, null, null);
				for (int i = 0; i < rdh.getRowCount(); i++) {
					String tableName = rdh.getStringData(i, "TABLE_NAME");
					String type = rdh.getStringData(i, "TABLE_TYPE")
							.toUpperCase();
					if (type == null || type.endsWith("INDEX")
							|| type.endsWith("SEQUENCE")) {
						continue;
					}
					// TODO: add grab condition
					DBTable dbTable = new DBTable(mschema);
					dbTable.setName(tableName);
					System.out.println(tableName + "/" + dbTable.getName());
					dbTable.setTableType(type);
					if (_info.isCaptureWithColumnInfo()) {
						setTableColumns(mschema.getName(), dbTable);
					}
					mschema.putTable(dbTable);

					request.subTask(dbTable.toString());

					setResourceProperties(dbTable, i, rdh);
				}
				if (request.canceld()) {
					return null;
				}
			}

			fLogger.trace("end rdh is null ?" + (rdh == null));
			return rdh;
		} catch (RuntimeException e) {
			fLogger.trace(e);
			return null;
		}
	}

	@Override
	protected void setTableText(ResourceCaptionRequest request)
			throws SQLException {
		IDBSchema[] schemas = _root.getSchemas();

		for (int i = 0; i < schemas.length; i++) {
			IDBTable[] prs = schemas[i].getProcedures();
			// for (int j = 0; j < prs.length; j++) {
			// setUSText(prs[j]);
			// }
			// prs = schemas[i].getFunctions();
			// for (int j = 0; j < prs.length; j++) {
			// setUSText(prs[j]);
			// }
			IDBTable[] tbl = schemas[i].getTables();
			for (int j = 0; j < tbl.length; j++) {
				if (tbl[j].isView()) {
					setViewText(tbl[j]);
				}
			}
			if (i % 3 == 0 && request.canceld()) {
				return;
			}
		}
	}

	protected void setViewText(IDBTable table) throws SQLException {
		ResultSetDataHolder rdh = null;

		rdh = executePrepareQuery(_SELECT_VIEW_TEXT, new String[] {
				table.getParent().getUName(), table.getUName() });
		for (int i = 0; i < rdh.getRowCount(); i++) {
			String tn = rdh.getStringData(i, "TEXT");
			table.setText(tn);
		}
	}

	@Override
	protected void getTrigger() throws SQLException {
	}

	public static String getLimitString() {
		return "LIMIT 0, 30";
	}

	@Override
	protected void getSequence() throws SQLException {
		ResultSetDataHolder rdh = null;
		ResultSetDataHolder rdh2 = null;
		IDBSchema[] schemas = _root.getSchemas();
		for (int i = 0; i < schemas.length; i++) {
			if (schemas[i].getName().length() == 0) {
				continue;
			}
			rdh = executePrepareQuery(_SELECT_ALL_SEQUENCE,
					new String[] { schemas[i].getName() });
			DBSequence sequence = null;
			for (int j = 0; j < rdh.getRowCount(); j++) {
				rdh2 = null;
				sequence = new DBSequence(schemas[i]);
				String name = rdh.getStringData(j, "NAME");
				sequence.setName(name);
				String text = _SELECT_SEQUENCE_BODY + schemas[i].getName()
						+ "." + name;
				rdh2 = executePrepareQuery(text, EMPTY);
				if (rdh2.getRowCount() > 0) {
					setResourceProperties(sequence, 0, rdh2);
				}
				schemas[i].putSequence(sequence);
				// setViewText(trigger);
			}

		}
	}

	@Override
	public boolean migration(ITransactionSQL osql, DBCrossRefference cross,
			boolean drop, boolean cascade) throws SQLException {
		fLogger.info("migration DBCrossRefference " + cross);

		IDBColumn[] columns = cross.pkList.toArray(new IDBColumn[0]);
		String tn = columns[0].getParent().getName();
		String sn = columns[0].getParent().getParent().getName();
		ResultSetDataHolder rdh = executePrepareQuery(_SELECT_INDEX_CNT,
				new String[] { sn, cross.getName() });
		if (rdh.getIntData(0, "CNT") == 0) {
			StringBuilder buf = new StringBuilder();
			buf.append("ALTER TABLE ");
			buf.append(sn).append(".").append(tn)
					.append(StringUtil.LINE_SEPARATOR);
			buf.append("  add constraint ").append(cross.getName())
					.append(StringUtil.LINE_SEPARATOR);
			buf.append("  foreign key(");

			executePrepare(buf.toString(), EMPTY);
		}

		commit();

		return true;
	}

	@Override
	public boolean migration(ITransactionSQL osql, IDBSchema schema,
			boolean drop, boolean cascade) throws SQLException {
		fLogger.info("migration DBSchema " + schema);
		String sn = schema.getName();
		ResultSetDataHolder rdh = executePrepareQuery(_con, _SELECT_SCHEMA_CNT,
				new String[] { sn });
		boolean existsSchema = rdh.getIntData(0, "CNT") > 0;
		if (existsSchema && drop) {
			String st = String.format("DROP SCHEMA %s", sn);
			if (cascade) {
				st += " CASCADE";
			}
			executePrepare(st, EMPTY);
		}
		if (!existsSchema || drop) {
			String st = String.format("CREATE SCHEMA %s", sn);
			executePrepare(st, EMPTY);
		}
		commit();

		return true;
	}

	@Override
	public boolean migration(ITransactionSQL osql, IDBSequence sequence,
			boolean drop, boolean cascade, boolean noSchema)
			throws SQLException {
		fLogger.info("migration DBSequence " + sequence);
		ResultSetDataHolder rdh = null;
		String sn = sequence.getParent().getName();
		if (noSchema) {
			sn = _root.getDefaultSchema().getUName();
		}
		rdh = executePrepareQuery(_con, _SELECT_SEQUENCE_CNT, new String[] {
				sn, sequence.getName() });
		boolean existsSequence = rdh.getIntData(0, "CNT") > 0;
		if (existsSequence && drop) {
			String st = String.format("DROP SEQUENCE %s.%s", sn,
					sequence.getName());
			executePrepare(st, EMPTY);
		}
		if (!existsSequence || drop) {
			if (!createSequenceBody(sn, sequence)) {
				return false;
			}
		}
		commit();

		return true;
	}

	private boolean createSequenceBody(String schemaName, IDBSequence sequence)
			throws SQLException {
		StringBuilder buf = new StringBuilder();
		buf.append("CREATE SEQUENCE ");
		buf.append(schemaName).append(".");
		buf.append(sequence.getName()).append(" ")
				.append(StringUtil.LINE_SEPARATOR);
		buf.append("  INCREMENT ").append("1")
				.append(StringUtil.LINE_SEPARATOR);
		buf.append("  MINVALUE ").append("1").append(StringUtil.LINE_SEPARATOR);
		buf.append("  MAXVALUE ").append("10000")
				.append(StringUtil.LINE_SEPARATOR);
		buf.append("  START ").append("200").append(StringUtil.LINE_SEPARATOR);
		buf.append("  CACHE ").append("1").append(StringUtil.LINE_SEPARATOR);
		// CYCLE;
		executePrepare(buf.toString(), EMPTY);

		return true;
	}

	@Override
	public boolean migration(ITransactionSQL osql, IDBTable table,
			boolean drop, boolean cascade, boolean noSchema)
			throws SQLException {
		fLogger.info("migration DBTable " + table);
		ResultSetDataHolder rdh = null;
		String sn = table.getParent().getName();
		if (noSchema) {
			sn = _root.getDefaultSchema().getUName();
		}
		if (table.isTable()) {
			rdh = executePrepareQuery(_SELECT_TABLE_CNT, new String[] { sn,
					table.getName() });
		} else {
			rdh = executePrepareQuery(_SELECT_VIEW_CNT, new String[] { sn,
					table.getName() });
		}
		boolean existsTable = rdh.getIntData(0, "CNT") > 0;
		if (existsTable && drop) {
			String st = null;
			if (table.isTable()) {
				st = String.format("DROP TABLE %s.%s", sn, table.getName());
				if (cascade) {
					st += " CASCADE";
				}
			} else {
				st = String.format("DROP VIEW %s.%s", sn, table.getName());
			}
			executePrepare(st, EMPTY);
		}
		if (!existsTable || drop) {
			if (createTableBody(sn, table)) {
				createTableComment(sn, table);
			} else {
				return false;
			}
		}
		if (!migrateTableValue(osql, table, noSchema)) {
			return false;
		}
		commit();

		return true;
	}

	private boolean migrateTableValue(ITransactionSQL osql, IDBTable table,
			boolean noSchema) {
		StringBuilder buf = new StringBuilder();
		PreparedStatement st = null;
		ResultSet rs = null;
		fLogger.info("." + table);
		try {
			buf.append("SELECT * FROM ");
			buf.append(table.getName());
			AbsTransactionSQL asql = (AbsTransactionSQL) osql;
			st = asql.getPrepareStatement(asql._con, buf.toString());
			rs = st.executeQuery();

			doInsertFromRs(rs, table.getColumns(), noSchema);
		} catch (SQLException e) {
			fLogger.warn(e);
			return false;
		} finally {
			try {
				if (rs != null) {
					rs.close();
				}
				if (st != null) {
					st.close();
				}
			} catch (SQLException e) {
				fLogger.warn(e);
				return false;
			}
		}
		return true;

	}

	private boolean createTableBody(String schemaName, IDBTable table)
			throws SQLException {
		StringBuilder buf = new StringBuilder();
		if (table.isTable()) {
			buf.append("CREATE TABLE ");
			buf.append(schemaName).append(".").append(table.getName());
			buf.append(" ( ").append(StringUtil.LINE_SEPARATOR);
			IDBColumn[] columns = table.getColumns();
			for (int j = 0; j < columns.length; j++) {
				StringBuilder buf2 = new StringBuilder();
				buf2.append("  ").append(getTypeString(columns[j]));
				buf2.append(",").append(StringUtil.LINE_SEPARATOR);
				buf.append(buf2);
			}
			if (columns.length > 0) {
				buf.setLength(buf.length() - 1
						- StringUtil.LINE_SEPARATOR.length());
			}
			buf.append(StringUtil.LINE_SEPARATOR);
			if (table.hasPk()) {
				buf.append("  ,PRIMARY KEY (");
				int[] pk = table.getPkPositions();
				for (int j = 0; j < pk.length; j++) {
					if (j > 0) {
						buf.append(",");
					}
					buf.append(table.getColumns()[pk[j]].getName());
				}
				buf.append(")").append(StringUtil.LINE_SEPARATOR);
			}
			buf.append(")");
		} else {
			if (StringUtil.nvl(table.getText()).length() == 0) {
				return true;
			}
			buf.append("CREATE VIEW ");
			buf.append(schemaName).append(".").append(table.getName());
			buf.append(" AS ").append(StringUtil.LINE_SEPARATOR);
			buf.append(table.getText());
			buf.append(StringUtil.LINE_SEPARATOR);
		}

		executePrepare(buf.toString(), EMPTY);
		return true;
	}

	private boolean createTableComment(String schemaName, IDBTable table)
			throws SQLException {
		StringBuilder buf = new StringBuilder();
		if (table.getComment().length() > 0) {
			buf = new StringBuilder();
			buf.append("COMMENT ON ");
			if (table.isTable()) {
				buf.append("TABLE ");
			} else {
				buf.append("VIEW ");
			}
			buf.append(schemaName).append(".").append(table.getName());
			buf.append(" IS '").append(table.getComment()).append("'");
			executePrepare(buf.toString(), EMPTY);
		}
		IDBColumn[] columns = table.getColumns();
		for (int j = 0; j < columns.length; j++) {
			if (columns[j].getComment().length() > 0) {
				buf = new StringBuilder();
				buf.append("COMMENT ON COLUMN ");
				buf.append(schemaName).append(".");
				buf.append(table.getName()).append(".");
				buf.append(columns[j].getName()).append(" IS '");
				buf.append(columns[j].getComment()).append("'");
				executePrepare(buf.toString(), EMPTY);
			}
		}
		return true;
	}

	private String getTypeString(IDBColumn column) {
		StringBuilder buf = new StringBuilder();
		buf.append(column.getName()).append(" ");
		switch (column.getDataType()) {
		case Types.CHAR:
		case Types.NCHAR:
			buf.append(String.format("CHAR(%d)", column.getSize()));
			break;
		case Types.VARCHAR:
		case Types.NVARCHAR:
			buf.append(String.format("VARCHAR(%d)", column.getSize()));
			break;
		case Types.NUMERIC:
			buf.append(String.format("NUMERIC(%d)", column.getSize()));
			break;
		case Types.INTEGER:
			buf.append(String.format("INTEGER(%d)", column.getSize()));
			break;
		case Types.FLOAT:
			buf.append(String.format("FLOAT(%d)", column.getSize()));
			break;
		case Types.DATE:
			buf.append("DATE");
			break;
		case Types.TIMESTAMP:
			buf.append("TIMESTAMP");
			break;
		case Types.BLOB:
			buf.append("BYTEA");
			break;
		case Types.BINARY:
			buf.append("BYTEA");
			break;
		default:
		}
		if (column.isNotNull() && !column.isPkey()) {
			buf.append(" NOT NULL");
		}
		return buf.toString();
	}
}