/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.config;

import java.util.LinkedList;

/**
 * {@link #interpret(ConfigContext)}が呼び出された回数を数えるテスト用{@link ConfigExpression}。
 * このクラスはスレッドセーフではない。
 * @author nakamura
 *
 */
public class ConfigCount implements ConfigExpression {
	private final int max;
	private final ConfigExpression expression;
	private int count = 0;
	private final LinkedList<ConfigContext> contextList;
	
	/**
	 * コンストラクタ。
	 * @param max 一覧の最大件数。
	 * @param expression 委譲先。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public ConfigCount(final int max, final ConfigExpression expression){
		expression.getClass();
		this.max = max;
		this.expression = expression;
		contextList = new LinkedList<ConfigContext>();
	}
	
	/**
	 * コンストラクタ。
	 * 一覧の最大件数を4とする。
	 * @param expression 委譲先。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public ConfigCount(final ConfigExpression expression){
		this(4, expression);
	}

	public void interpret(final ConfigContext context) {
		count++;
		if(max > 0){
			if(count > max){
				contextList.removeFirst();
			}
			contextList.add(context);
		}
		expression.interpret(context);
	}

	/**
	 * {@link #interpret(ConfigContext)}が呼び出された回数を返す。
	 * @return {@link #interpret(ConfigContext)}が呼び出された回数。
	 */
	public int getCount() {
		return count;
	}

	/**
	 * 呼び出された{@link #interpret(ConfigContext)}の引数の一覧を返す。
	 * @return 呼び出された{@link #interpret(ConfigContext)}の引数の一覧。
	 */
	public LinkedList<ConfigContext> getContextList() {
		return contextList;
	}
}
