/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * リフレクションを使用した、プロパティにアクセスする{@link GettingEL}と{@link EL}のファクトリ。
 * getter/setterのオーバライドには対応していない。
 * @author nakamura
 *
 */
public class ReflectionPropertyELFactory implements PropertyELFactory {
	private final TypeConverter converter;
	
	/**
	 * コンストラクタ。
	 * @param converter 型変換器。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public ReflectionPropertyELFactory(final TypeConverter converter){
		converter.getClass();
		this.converter = converter;
	}
	
	private static boolean equalsStoic(final String s0, final String s1){
		return (s0 == s1) || s0.equals(s1);
	}
	
	private static boolean equals(final String s0, final String s1){
		return equalsStoic(s0, s1) || (
				(Character.toUpperCase(s0.charAt(0)) == Character.toUpperCase(s1.charAt(0))) && 
				equalsStoic(s0.substring(1), s1.substring(1))
		);
	}

	public GettingEL newGetter(final Class clazz, final String propertyName) {
		final Method[] methList = clazz.getMethods();
		for (int i=0; i<methList.length; i++) {
			final Method method = methList[i];
			{
				final Class[] paramTypes = method.getParameterTypes();
				if(paramTypes != null && paramTypes.length > 0){
					continue;
				}
			}
			if(Void.TYPE.equals(method.getReturnType())){
				continue;
			}
			final String key = method.getName();
			if(
					(key.startsWith("get") && equals(key.substring(3), propertyName)) || 
					(key.startsWith("is") && equals(key.substring(2), propertyName))
					){
				return new GettingELAdapter(){
					@Override public Object getValue(final Object root, final Class clazz)
							throws ELTargetRuntimeException {
						try {
							final Object result = method.invoke(root, new Object[0]);
							return converter.convert(propertyName, result, clazz);
						} catch (final IllegalArgumentException e) {
							throw new RuntimeException(e);
						} catch (final IllegalAccessException e) {
							throw new RuntimeException(e);
						} catch (final InvocationTargetException e) {
							throw new ELTargetRuntimeException(e);
						}
					}
				};
			}
		}
		throw new RuntimeException("property not found: " + clazz + ": " + propertyName);
	}

	public EL newSetter(final Class clazz, final String propertyName) {
		final Method[] methList = clazz.getMethods();
		for (int i=0; i<methList.length; i++) {
			final Method method = methList[i];
			{
				final Class[] paramTypes = method.getParameterTypes();
				if(paramTypes != null && paramTypes.length != 1){
					continue;
				}
			}
			final String key = method.getName();
			if(key.startsWith("set") && equals(key.substring(3), propertyName)){
				return new ELAdapter(){	
					@Override public void setValue(final Object root, final Object value) throws ELTargetRuntimeException {
						try {
							method.invoke(root, new Object[]{converter.convert(propertyName, value, method.getParameterTypes()[0])});
						} catch (final IllegalArgumentException e) {
							throw new RuntimeException(e);
						} catch (final IllegalAccessException e) {
							throw new RuntimeException(e);
						} catch (final InvocationTargetException e) {
							throw new ELTargetRuntimeException(e);
						}
					}
					@Override
					public Object getValue(Object root, Class clazz) throws ELTargetRuntimeException {
						// カバレージがここを通過してはいけない
						return null;
					}
				};
			}
		}
		throw new RuntimeException("property not found: " + clazz + ": " + propertyName);
	}
	
	@Override public int hashCode(){
		return converter.hashCode();
	}
	
	@Override public boolean equals(final Object other){
		if(other instanceof ReflectionPropertyELFactory){
			final ReflectionPropertyELFactory o = (ReflectionPropertyELFactory)other;
			return converter.equals(o.converter);
		}
		return false;
	}
}
