/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.io.IOException;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;


/**
 * {@link ValidatorContext#getInputMap()}から{@link ValidatorContext#getTmpKey()}を
 * キーとして取得した値一覧に対し順次委譲先を実行する{@link ValidatorExpression}。
 * @author nakamura
 *
 */
public class ValueLoopValidator extends DelegationValidator {
	private final ValidatorIterable validatorIterable;
	
	/**
	 * コンストラクタ。
	 * @param validatorIterable 値検証一覧の呼び出し方。
	 * @param expression 委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public ValueLoopValidator(final ValidatorIterable validatorIterable, final ValidatorExpression expression){
		super(expression);
		validatorIterable.getClass();
		this.validatorIterable = validatorIterable;
	}
	
	/**
	 * コンストラクタ。
	 * 値検証一覧の呼び出し方に{@link ValidatorConstants#AND}を使用する。
	 * @param expression 委譲先。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public ValueLoopValidator(final ValidatorExpression expression){
		this(ValidatorConstants.AND, expression);
	}
	
	/**
	 * @throws NullPointerException 引数または{@link ValidatorContext#getInputMap()}が null の場合。
	 */
	@Override public boolean interpret(final ValidatorContext context) {
		final int length;
		{
			final List<Object> list = context.getInputMap().get(context.getTmpKey());
			if(list == null){
				length = 1;
			}else{
				length = list.size();
			}
		}

		return validatorIterable.interpret(context, new Iterable<ValidatorExpression>(){
			public Iterator<ValidatorExpression> iterator() {
				return new Iterator<ValidatorExpression>(){
					int i=0;
					public boolean hasNext() {
						return (i < length);
					}
					public ValidatorExpression next() {
						return new ValidatorExpression(){
							public boolean interpret(final ValidatorContext context) {
								final ValidatorContext newContext = context.copy();
								newContext.setTmpIndex(i++);
								return ValueLoopValidator.super.interpret(newContext);
							}
							public void appendTo(final Appendable sb) {
								throw new UnsupportedOperationException();
							}
							public void addMessageTo(final Collection<String> messageCollection) {
								throw new UnsupportedOperationException();
							}
						};
					}
					public void remove() {
						throw new UnsupportedOperationException();
					}
				};
			}
		});
	}

	@Override public void appendArgumentTo(final Appendable sb) throws IOException {
		validatorIterable.appendTo(sb);
		sb.append(",");
		super.appendArgumentTo(sb);
	}
}
