/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.text;

import java.text.Format;
import java.text.ParseException;
import java.util.HashMap;
import java.util.Map;

import junit.framework.TestCase;

public class LimitedValueFormatTest extends TestCase {

	public void testConstructor(){
		try{
			new LimitedValueFormat(null, "defaultParsedValue", null, "defaultFormattedValue");
			fail();
		}catch(final NullPointerException expected){
		}
		new LimitedValueFormat(null, "defaultFormattedValue", new HashMap<String,Object>(), "defaultParsedValue");
		new LimitedValueFormat(new HashMap<Object,String>(), null, new HashMap<String,Object>(), "defaultParsedValue");
		new LimitedValueFormat(new HashMap<Object,String>(), "defaultFormattedValue", null, "defaultParsedValue");
		new LimitedValueFormat(new HashMap<Object,String>(), "defaultFormattedValue", new HashMap<String,Object>(), null);
	}
	
	public void testDuplicate0(){
		final Map<String,String> m = new HashMap<String,String>();
		m.put("key0", "value0");
		m.put("key1", "value1");
		m.put("key2", "value1");
		try{
			new LimitedValueFormat(null, "defaultFormattedValue", m, "defaultParsedValue");
			fail();
		}catch(final IllegalStateException expected){
			assertEquals("already registered: value1", expected.getMessage());
		}
		try{
			new LimitedValueFormat(m, "defaultFormattedValue", null, "defaultParsedValue");
			fail();
		}catch(final IllegalStateException expected){
			assertEquals("already registered: value1", expected.getMessage());
		}
	}
	
	private void scenarioParse(final Format format) throws ParseException{
		assertEquals("key0", format.parseObject("value0"));
		assertEquals("key1", format.parseObject("value1"));
		assertEquals("key2", format.parseObject("value2"));
	}

	private void scenarioParseException(final Format format) throws ParseException{
		try{
			assertEquals("defaultParsedValue", format.parseObject("value3"));
		}catch(final ParseException expected){
		}
	}

	private void scenarioParseDefault(final Format format) throws ParseException{
		assertEquals("defaultParsedValue", format.parseObject("value3"));
	}

	private void scenarioFormat(final Format format) throws ParseException{
		assertEquals("value0", format.format("key0"));
		assertEquals("value1", format.format("key1"));
		assertEquals("value2", format.format("key2"));
	}

	private void scenarioFormatNothing(final Format format) throws ParseException{
		assertEquals("key3", format.format("key3"));
	}

	private void scenarioFormatDefault(final Format format) throws ParseException{
		assertEquals("defaultFormattedValue", format.format("key3"));
	}

	public void testNormal() throws ParseException{
		final Map<String,String> m0 = new HashMap<String,String>();
		m0.put("key0", "value0");
		m0.put("key1", "value1");
		m0.put("key2", "value2");
		
		final Map<String,String> m1 = new HashMap<String,String>();
		m1.put("value0", "key0");
		m1.put("value1", "key1");
		m1.put("value2", "key2");
		{
			final Format format0 = new LimitedValueFormat(m0);
			scenarioParse(format0);
			scenarioParseException(format0);
			scenarioFormat(format0);
			scenarioFormatNothing(format0);
			
			final Format format1 = (Format)format0.clone();
			assertTrue(format0 != format1);
			scenarioParse(format0);
			scenarioParseException(format0);
			scenarioFormat(format0);
			scenarioFormatNothing(format0);
		}
		{
			final Format format0 = new LimitedValueFormat(m0, "defaultFormattedValue");
			scenarioParse(format0);
			scenarioParseException(format0);
			scenarioFormat(format0);
			scenarioFormatDefault(format0);
			final Format format1 = (Format)format0.clone();
			assertTrue(format0 != format1);
			scenarioParse(format0);
			scenarioParseException(format0);
			scenarioFormat(format0);
			scenarioFormatDefault(format0);
		}
		{
			final Format format0 = new LimitedValueFormat(m0, m1);
			scenarioParse(format0);
			scenarioParseException(format0);
			scenarioFormat(format0);
			scenarioFormatNothing(format0);
			final Format format1 = (Format)format0.clone();
			assertTrue(format0 != format1);
			scenarioParse(format0);
			scenarioParseException(format0);
			scenarioFormat(format0);
			scenarioFormatNothing(format0);
		}
		{
			final Format format0 = new LimitedValueFormat(m0, "defaultFormattedValue", m1, "defaultParsedValue");
			scenarioParse(format0);
			scenarioParseDefault(format0);
			scenarioFormat(format0);
			scenarioFormatDefault(format0);
			final Format format1 = (Format)format0.clone();
			assertTrue(format0 != format1);
			scenarioParse(format0);
			scenarioParseDefault(format0);
			scenarioFormat(format0);
			scenarioFormatDefault(format0);
		}
	}
	
	public void testToString(){
		final Map<String,String> m0 = new HashMap<String,String>();
		m0.put("key0", "value0");
		m0.put("key1", "value1");
		m0.put("key2", "value2");
		
		final Map<String,String> m1 = new HashMap<String,String>();
		m1.put("value0", "key0");
		m1.put("value1", "key1");
		m1.put("value2", "key2");
		
		final Format format0 = new LimitedValueFormat(m0, "defaultFormattedValue", m1, "defaultParsedValue");
		assertEquals("new LimitedValueFormat(" +
				"{\"key1\":\"value1\",\"key0\":\"value0\",\"key2\":\"value2\"}," +
				"\"defaultFormattedValue\"," +
				"{\"value1\":\"key1\",\"value2\":\"key2\",\"value0\":\"key0\"}," +
				"\"defaultParsedValue\")", format0.toString());
	}
}
