/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.samples.crud;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import junit.framework.TestCase;
import woolpack.adapter.JXPUtils;
import woolpack.samples.SamplesAllTests;
import woolpack.test.TestDomContextBuilder;
import woolpack.utils.Utils;
import woolpack.validator.SimpleMessageCollector;
import woolpack.validator.ValidatorContext;

public class SampleServletTest extends TestCase {
	
	public void testNormal() throws Exception {
		String userId = null;
		final SampleServlet servlet = new SampleServlet();
		servlet.toNode.setFn(SamplesAllTests.getFileToNode("^(.*)$", "html/sample/crud/$1.html"));
		final TestDomContextBuilder builder = new TestDomContextBuilder(JXPUtils.ONE);

		// 登録値検証ブロック
		builder.init();
		builder.get().setId("common_top");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'top\']"));

		builder.initXPath("//A[@href=\'user_registerInput\']");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_registerInput\']"));
		
		builder.initXPath("//FORM");
		builder.getInput().get("name").set(0, "");
		builder.getInput().put("sex", Arrays.<Object>asList("1"));
		builder.getInput().put("address", Arrays.<Object>asList("1"));
		builder.getInput().get("age").set(0, "17");
		builder.getInput().put("hobby", Arrays.<Object>asList("1"));
		builder.getInput().get("comment").set(0, "comment0");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'error\']"));
		assertTrue(builder.exists("//P[@id=\'errorValidate\' and text() = \'name required.\']"));

		// 登録正常系
		builder.init();
		builder.get().setId("common_top");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'top\']"));
		
		builder.initXPath("//A[@href=\'user_registerInput\']");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_registerInput\']"));
		
		builder.initXPath("//FORM");
		builder.getInput().get("name").set(0, "name0");
		builder.getInput().put("sex", Arrays.<Object>asList("1"));
		builder.getInput().put("address", Arrays.<Object>asList("1"));
		builder.getInput().get("age").set(0, "17");
		builder.getInput().put("hobby", Arrays.<Object>asList("1"));
		builder.getInput().get("comment").set(0, "comment0");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_registerConfirm\']"));
		assertEquals(Utils
				.map("name", Arrays.asList("name0"))
				.map("sex", Arrays.asList("fimale"))
				.map("address", Arrays.asList("Naichi"))
				.map("age", Arrays.asList("17"))
				.map("hobby", Arrays.asList("car"))
				.map("comment", Arrays.asList("comment0")), builder.selectEmbedded());

		builder.initXPath("//FORM");
		assertEquals(Utils
				.map("name", Arrays.asList("name0"))
				.map("sex", Arrays.asList("1"))
				.map("address", Arrays.asList("1"))
				.map("age", Arrays.asList("17"))
				.map("hobby", Arrays.asList("1"))
				.map("comment", Arrays.asList("comment0")), builder.getInput());
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_registerResult\']"));

		// 検索値検証ブロック
		builder.initXPath("//A[@href=\'common_top\']");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'top\']"));
		
		builder.initXPath("//A[@href=\'user_searchInput\']");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_searchInput\']"));

		builder.initXPath("//FORM");
		builder.getInput().get("name").set(0, "name0");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_searchResult\']"));
		
		userId = builder.selectNode("//INPUT[@name=\'userId\' and ../..//text() = \'name0\']/@value").getNodeValue();
		builder.initXPath("//FORM");
		builder.get().setId("user_updateInput");
		builder.getInput().remove("userId");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'error\']"));
		assertTrue(builder.exists("//P[@id=\'errorValidate\' and text() = \'user id required.\']"));
		
		// 訂正値検証ブロック
		builder.init();
		builder.get().setId("common_top");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'top\']"));
		
		builder.initXPath("//A[@href=\'user_searchInput\']");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_searchInput\']"));

		builder.initXPath("//FORM");
		builder.getInput().get("name").set(0, "name0");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_searchResult\']"));
		
		userId = builder.selectNode("//INPUT[@name=\'userId\' and ../..//text() = \'name0\']/@value").getNodeValue();
		builder.initXPath("//FORM");
		builder.get().setId("user_updateInput");
		builder.getInput().put("userId", Arrays.<Object>asList(userId));
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_updateInput\']"));

		builder.initXPath("//FORM");
		builder.getInput().get("name").set(0, "");
		builder.getInput().get("sex").set(0, "0");
		builder.getInput().get("address").set(0, "0");
		builder.getInput().get("age").set(0, "18");
		builder.getInput().get("hobby").set(0, "0");
		builder.getInput().get("hobby").add("2");
		builder.getInput().get("comment").set(0, "comment0u");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'error\']"));
		assertTrue(builder.exists("//P[@id=\'errorValidate\' and text() = \'name required.\']"));

		// 訂正正常系
		builder.init();
		builder.get().setId("common_top");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'top\']"));
		
		builder.initXPath("//A[@href=\'user_searchInput\']");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_searchInput\']"));

		builder.initXPath("//FORM");
		builder.getInput().get("name").set(0, "name0");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_searchResult\']"));
		
		userId = builder.selectNode("//INPUT[@name=\'userId\' and ../..//text() = \'name0\']/@value").getNodeValue();
		builder.initXPath("//FORM");
		builder.get().setId("user_updateInput");
		builder.getInput().put("userId", Arrays.<Object>asList(userId));
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_updateInput\']"));

		builder.initXPath("//FORM");
		assertEquals(Utils
				.map("userId", Arrays.asList("0"))
				.map("name", Arrays.asList("name0"))
				.map("sex", Arrays.asList("1"))
				.map("address", Arrays.asList("1"))
				.map("age", Arrays.asList("17"))
				.map("hobby", Arrays.asList("1"))
				.map("comment", Arrays.asList("comment0")), builder.getInput());
		builder.getInput().get("name").set(0, "name0u");
		builder.getInput().get("sex").set(0, "0");
		builder.getInput().get("address").set(0, "0");
		builder.getInput().get("age").set(0, "18");
		builder.getInput().get("hobby").set(0, "0");
		builder.getInput().get("hobby").add("2");
		builder.getInput().get("comment").set(0, "comment0u");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_updateConfirm\']"));
		assertEquals(Utils
				.map("name", Arrays.asList("name0u"))
				.map("sex", Arrays.asList("male"))
				.map("address", Arrays.asList("Okinawa"))
				.map("age", Arrays.asList("18"))
				.map("hobby", Arrays.asList("bycycle", "bike"))
				.map("comment", Arrays.asList("comment0u")), builder.selectEmbedded());

		builder.initXPath("//FORM");
		assertEquals(Utils
				.map("userId", Arrays.asList("0"))
				.map("name", Arrays.asList("name0u"))
				.map("sex", Arrays.asList("0"))
				.map("address", Arrays.asList("0"))
				.map("age", Arrays.asList("18"))
				.map("hobby", Arrays.asList("0", "2"))
				.map("comment", Arrays.asList("comment0u")), builder.getInput());
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_updateResult\']"));

		// 削除正常系
		builder.initXPath("//A[@href=\'common_top\']");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'top\']"));
		
		builder.initXPath("//A[@href=\'user_searchInput\']");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_searchInput\']"));

		builder.initXPath("//FORM");
		builder.getInput().get("name").set(0, "name0");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_searchResult\']"));

		userId = builder.selectNode("//INPUT[@name=\'userId\' and ../..//text() = \'name0u\']/@value").getNodeValue();
		builder.initXPath("//FORM");
		builder.get().setId("user_deleteConfirm");
		builder.getInput().put("userId", Arrays.<Object>asList(userId));
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_deleteConfirm\']"));
		assertEquals(Utils
				.map("name", Arrays.asList("name0u"))
				.map("sex", Arrays.asList("male"))
				.map("address", Arrays.asList("Okinawa"))
				.map("age", Arrays.asList("18"))
				.map("hobby", Arrays.asList("bycycle", "bike"))
				.map("comment", Arrays.asList("comment0u")), builder.selectEmbedded());

		builder.initXPath("//FORM");
		assertEquals(Utils
				.map("userId", Arrays.asList("0")), builder.getInput());
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_deleteResult\']"));

		builder.initXPath("//A[@href=\'common_top\']");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'top\']"));

		builder.initXPath("//A[@href=\'user_searchInput\']");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_searchInput\']"));

		builder.initXPath("//FORM");
		builder.getInput().get("name").set(0, "name0");
		servlet.fn.exec(builder.get());
		assertTrue(builder.exists("//H4[text() = \'user_searchResult\']"));
		assertTrue(builder.exists("//FORM/TABLE[normalize-space(text()) = \'\']"));
	}
	
	public void testValidator() {
		final SampleServlet servlet = new SampleServlet();
		final ValidatorContext context = new ValidatorContext();
		final SimpleMessageCollector collector = new SimpleMessageCollector();
		context.setCollectable(collector);
		context.setId("");
		{
			collector.getList().clear();
			context.setInputMap(Collections.singletonMap("userId2", Arrays.asList((Object) "a")));
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.acceptable"), collector.getList());
		}
		{
			collector.getList().clear();
			context.setInputMap(Collections.singletonMap("userId", Arrays.asList((Object) "a")));
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.userId.parse"), collector.getList());
		}
		{
			collector.getList().clear();
			final StringBuilder sb = new StringBuilder();
			for (int i = 0; i < 40; i++) {
				sb.append('a');
			}
			context.setInputMap(Collections.singletonMap("name", Arrays.asList((Object) sb.toString())));
			assertTrue(servlet.validatorFn.exec(context));
		}
		{
			collector.getList().clear();
			final StringBuilder sb = new StringBuilder();
			for (int i = 0; i < 41; i++) {
				sb.append('a');
			}
			context.setInputMap(Collections.singletonMap("name", Arrays.asList((Object) sb.toString())));
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.name.maxLength"), collector.getList());
		}
		{
			collector.getList().clear();
			context.setInputMap(Collections.singletonMap("sex", Arrays.asList((Object) '2')));
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.sex.acceptable"), collector.getList());
		}
		{
			collector.getList().clear();
			final StringBuilder sb = new StringBuilder();
			for (int i = 0; i < 80; i++) {
				sb.append('a');
			}
			context.setInputMap(Collections.singletonMap("address", Arrays.asList((Object) sb.toString())));
			assertTrue(servlet.validatorFn.exec(context));
		}
		{
			collector.getList().clear();
			final StringBuilder sb = new StringBuilder();
			for (int i = 0; i < 81; i++) {
				sb.append('a');
			}
			context.setInputMap(Collections.singletonMap("address", Arrays.asList((Object) sb.toString())));
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.address.maxLength"), collector.getList());
		}
		{
			collector.getList().clear();
			context.setInputMap(Collections.singletonMap("age", Arrays.asList((Object) "a")));
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.age.parse"), collector.getList());
		}
		{
			collector.getList().clear();
			context.setInputMap(Collections.singletonMap("age", Arrays.asList((Object) "-1")));
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.age.min"), collector.getList());
		}
		{
			collector.getList().clear();
			context.setInputMap(Collections.singletonMap("age", Arrays.asList((Object) "0")));
			assertTrue(servlet.validatorFn.exec(context));
		}
		{
			collector.getList().clear();
			context.setInputMap(Collections.singletonMap("age", Arrays.asList((Object) "200")));
			assertTrue(servlet.validatorFn.exec(context));
		}
		{
			collector.getList().clear();
			context.setInputMap(Collections.singletonMap("age", Arrays.asList((Object) "201")));
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.age.max"), collector.getList());
		}
		{
			collector.getList().clear();
			context.setInputMap(Collections.singletonMap("hobby", Arrays.asList((Object) "a")));
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.hobby.acceptable"), collector.getList());
		}
		{
			collector.getList().clear();
			final StringBuilder sb = new StringBuilder();
			for (int i = 0; i < 200; i++) {
				sb.append('a');
			}
			context.setInputMap(Collections.singletonMap("comment", Arrays.asList((Object) sb.toString())));
			assertTrue(servlet.validatorFn.exec(context));
		}
		{
			collector.getList().clear();
			final StringBuilder sb = new StringBuilder();
			for (int i = 0; i < 201; i++) {
				sb.append('a');
			}
			context.setInputMap(Collections.singletonMap("comment", Arrays.asList((Object) sb.toString())));
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.comment.maxLength"), collector.getList());
		}
		{
			context.setId("user_updateInput");
			collector.getList().clear();
			context.setInputMap(new HashMap<String, List<Object>>());
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.userId.required"), collector.getList());
		}
		{
			context.setId("user_deleteConfirm");
			collector.getList().clear();
			context.setInputMap(new HashMap<String, List<Object>>());
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.userId.required"), collector.getList());
		}
		{
			context.setId("user_registerConfirm");
			collector.getList().clear();
			context.setInputMap(new HashMap<String, List<Object>>());
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.name.required", "validator.sex.required", "validator.address.required", "validator.age.required"), collector.getList());
		}
		{
			context.setId("user_registerResult");
			collector.getList().clear();
			context.setInputMap(new HashMap<String, List<Object>>());
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.name.required", "validator.sex.required", "validator.address.required", "validator.age.required"), collector.getList());
		}
		{
			context.setId("user_updateConfirm");
			collector.getList().clear();
			context.setInputMap(new HashMap<String, List<Object>>());
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.name.required", "validator.sex.required", "validator.address.required", "validator.age.required"), collector.getList());
		}
		{
			context.setId("user_updateResult");
			collector.getList().clear();
			context.setInputMap(new HashMap<String, List<Object>>());
			assertFalse(servlet.validatorFn.exec(context));
			assertEquals(Arrays.asList("validator.name.required", "validator.sex.required", "validator.address.required", "validator.age.required"), collector.getList());
		}
	}
}
